/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imarkerviewobject.h"


#include "icaption.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatareader.h"
#include "ierror.h"
#include "ipointglyph.h"
#include "ireplicatedactor.h"
#include "ivectorfieldviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"
#include "iviewobjectobserver.h"

#include <vtkProperty.h>
#include <vtkProperty2D.h>
#include <vtkTextProperty.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iMarkerViewObject,Marker,m);

IOBJECT_DEFINE_KEY(iMarkerViewObject,ColorAutomatic,ca,Bool,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,CaptionText,ct,String,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,CaptionPosition,cx,Float,2);
IOBJECT_DEFINE_KEY(iMarkerViewObject,InteractiveMove,im,Bool,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,Scaled,sc,Bool,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,Type,t,Int,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,TextColor,tc,Color,1);
IOBJECT_DEFINE_KEY(iMarkerViewObject,Transform,tf,Float,6);

IOBJECT_DEFINE_POSITION_KEY(iMarkerViewObject,Position,x);
IOBJECT_DEFINE_DISTANCE_KEY(iMarkerViewObject,Size,s);

IOBJECT_DEFINE_KEY(iMarkerViewObject,TypeAsString,-tas,String,1);

//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iMarkerViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(iMarkerViewObject,1);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(iMarkerViewObject,1);


iMarkerViewObject* iMarkerViewObject::New(iViewModule *vm)
{
	return new iMarkerViewObject(vm); // non-inheritable, so no need to use Object Factory
}


iMarkerViewObject::iMarkerViewObject(iViewModule *vm) : iSolidViewObject(vm,iDataType::Null(),1,_ViewObjectAlwaysShaded|_ViewObjectNoReplicating|_ViewObjectNoPalette), mSize(vm,true), mAttachedObjects(0)
{
	mName = "iMarkerViewObject";
	mObjectType = _ObjectTypeMarker;

	mSize = 0.05;
	mColorAuto = true;

	mTextColor = iColor(0,0,0);
	mCaptionText = "";

	mInteractiveMove = false;

	mTransform[0] = mTransform[1] = mTransform[2] = 0.0;
	mTransform[3] = mTransform[4] = mTransform[5] = 1.0;
	//
	//  Do VTK stuff
	//	
	mActors[0]->SetScaled(true);
	mObject = iPointGlyph::New(); IERROR_ASSERT_NULL_POINTER(mObject);
	mCaption = iCaption::New(vm->GetRenderTool()); IERROR_ASSERT_NULL_POINTER(mCaption);

	mObject->SetType(_PointTypeSphere);

	mActors[0]->SetInput(mObject->GetOutput());
	
	//
	//  Add observer to keep information about this object
	//
	const iPosition x(this->GetViewModule());
	float xc[2] = { 0.0, 0.0 };

	this->SetColor(0,iColor(255,0,0));
	this->SetPosition(x);
	this->SetCaptionPosition(xc);
	this->SetTextColor(mTextColor);
	this->SetSize(mSize);
	this->SetFontSize(this->GetViewModule()->GetFontSize());

	this->GetViewModule()->AddObject(mCaption);
	mCaption->VisibilityOff();
}


iMarkerViewObject::~iMarkerViewObject()
{
	if(!mInit) 
	{
		this->GetViewModule()->RemoveObject(mCaption);
		//
		//  Update attached objects
		//
		for(int j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerDelete(this);
	}
	mObject->Delete();
	mCaption->Delete();
}


void iMarkerViewObject::FinishInitialization()
{
	if(mColorAuto) this->SetTextColor(this->GetViewModule()->GetBackgroundColor());
}


void iMarkerViewObject::SetType(int m)
{
	mObject->SetType(m);
	this->ClearCache();
}


int iMarkerViewObject::GetType() const
{
	return mObject->GetType();
}


const iString iMarkerViewObject::GetTypeAsString() const
{
	return mObject->GetName();
}


void iMarkerViewObject::SetScaled(bool s)
{
	mActors[0]->SetScaled(s);
	this->ClearCache();
}


bool iMarkerViewObject::GetScaled() const
{
	return mActors[0]->GetScaled();
}


void iMarkerViewObject::SetPosition(const iPosition &p)
{
	this->Move(p);
	if(!mInteractiveMove) this->FinishMoving(); // always finish
	this->ClearCache();
}


void iMarkerViewObject::Move(const iPosition& p)
{
	mPosition = p;
	mActors[0]->SetPosition(mPosition);
	mCaption->SetAttachmentPoint(mPosition);
	if(mInteractiveMove) this->FinishMoving();
	this->ClearCache();
}


void iMarkerViewObject::FinishMoving()
{
	//
	//  Update attached objects
	//
	for(int j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerMove();
}


void iMarkerViewObject::SetCaptionPosition(const float *x)
{
	int j;
	for(j=0; j<2; j++) mCapPos[j] = x[j];
	mCaption->SetPosition(x[0],x[1]);
	this->ClearCache();
}


void iMarkerViewObject::SetSize(double s)
{
	if(s>0.0 && s<1.01)
	{
		iDistance tmp(this->GetViewModule(),true);
		tmp = s;
		this->SetSize(tmp);
	}
}


void iMarkerViewObject::SetSize(const iDistance &s)
{
	if(s < 1.01)
	{
		mSize = s;
		mActors[0]->SetBasicScale(mSize);
		float ps = 10.0*pow(s,0.3);
		if(ps < 1.0) ps = 1.0;
		if(ps > 10.0) ps = 10.0;
		mActors[0]->GetProperty()->SetPointSize(ps);
		this->ClearCache();
	}
}


void iMarkerViewObject::SetFontSize(int s)
{
	mCaption->SetFontSize(s);
	this->ClearCache();
}


void iMarkerViewObject::SetTextColor(iColor c, bool withAuto)
{
	if(withAuto && c==this->GetViewModule()->GetBackgroundColor()) mColorAuto = true; else mColorAuto = false;
	if(mColorAuto) 
	{
		c = this->GetViewModule()->GetBackgroundColor().Reverse();
	}
	mTextColor = c;
	mCaption->GetProperty()->SetColor(c.ToVTK());
	mCaption->GetCaptionTextProperty()->SetColor(c.ToVTK());
	this->ClearCache();
}


void iMarkerViewObject::SetCaptionText(const iString &s)
{
	mCaptionText = s;
	mCaption->SetCaption(s.ToCharPointer());
	if(s.Length() > 0) mCaption->VisibilityOn(); else mCaption->VisibilityOff();
	this->ClearCache();
}


void iMarkerViewObject::SetInteractiveMove(bool s)
{
	mInteractiveMove = s;
	this->ClearCache();
}


void iMarkerViewObject::SetTransform(const float *t)
{
	int i;
	for(i=0; i<6; i++) mTransform[i] = t[i];
	mActors[0]->SetAxisScale(t[3],t[4],t[5]);
	mActors[0]->SetOrientation(t[0],t[1],t[2]);
	this->ClearCache();
}


void iMarkerViewObject::ShowBody(bool show)
{
	if(show)
	{
		mActors[0]->VisibilityOn();
		if(mCaptionText.Length() > 0) mCaption->VisibilityOn(); else mCaption->VisibilityOff(); 
	} 
	else 
	{
		mActors[0]->VisibilityOff();
		mCaption->VisibilityOff();
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iMarkerViewObject::SolidViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyColorAutomatic(),mColorAuto);
	this->PackValue(s,KeyInteractiveMove(),mInteractiveMove);
	this->PackValue(s,KeyScaled(),this->GetScaled());

	this->PackValue(s,KeyType(),this->GetType());
	this->PackValue(s,KeyTextColor(),mTextColor);

	this->PackValue(s,KeyCaptionText(),mCaptionText);

	this->PackValueDistance(s,KeySize(),mSize);
	this->PackValuePosition(s,KeyPosition(),mPosition);
	this->PackValue(s,KeyCaptionPosition(),mCapPos,2);

	this->PackValue(s,KeyTransform(),mTransform,6);

	this->PackValue(s,KeyTypeAsString(),this->GetTypeAsString());
}


void iMarkerViewObject::SolidViewObjectUnPackStateBody(const iString &s)
{
	int i; bool b; float fv[6]; iColor c; iString z;

	if(this->UnPackValue(s,KeyColorAutomatic(),b))
	{
		mColorAuto = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyInteractiveMove(),b)) this->SetInteractiveMove(b);
	if(this->UnPackValue(s,KeyScaled(),b)) this->SetScaled(b);

	if(this->UnPackValue(s,KeyType(),i)) this->SetType(i);

	if(this->UnPackValue(s,KeyTextColor(),c)) this->SetTextColor(c,false);

	if(this->UnPackValue(s,KeyCaptionText(),z)) this->SetCaptionText(z);

	if(this->UnPackValueDistance(s,KeySize(),mSize)) this->SetSize(mSize);
	if(this->UnPackValuePosition(s,KeyPosition(),mPosition)) this->SetPosition(mPosition);

	fv[0] = mCapPos[0]; fv[1] = mCapPos[1];
	if(this->UnPackValue(s,KeyCaptionPosition(),fv,2)) this->SetCaptionPosition(fv);

	for(i=0; i<6; i++) fv[i] = mTransform[i];
	if(this->UnPackValue(s,KeyTransform(),fv,6)) this->SetTransform(fv);
}


void iMarkerViewObject::ShowColorBarsBody(bool)
{
}


void iMarkerViewObject::ResetInput()
{
}


void iMarkerViewObject::ResetPipeline()
{
}


float iMarkerViewObject::GetMemorySize() const
{ 
	return 0.0; 
}


void iMarkerViewObject::AttachViewObject(iViewObject *o)
{
	mAttachedObjects.AddUnique(o);
}


void iMarkerViewObject::DetachViewObject(iViewObject *o)
{
	mAttachedObjects.Remove(o);
}


bool iMarkerViewObject::CanBeShown() const
{
	return true;
}


void iMarkerViewObject::ViewObjectSyncWithData(int, void *)
{
	//
	//  Attach all vectors
	//
	int j;
	for(j=0; j<=this->GetViewModule()->GetVectorFieldFamily()->GetMaxMemberIndex(); j++)
	{
		this->AttachViewObject(this->GetViewModule()->GetVectorFieldFamily()->GetMember(j));
	}
	//
	//  Update attached objects
	//
	for(j=0; j<mAttachedObjects.Size(); j++) mAttachedObjects[j]->UpdateAfterMarkerAdd(this);
}

