/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iorthoslicer.h"


#include "idata.h"
#include "ierror.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


iOrthoSlicer* iOrthoSlicer::New(iViewObject *vo)
{ 
	return new iOrthoSlicer(vo);
}


iOrthoSlicer::iOrthoSlicer(iViewObject *vo) : iViewObjectComponent(vo)
{
	mPos[0] = mPos[1] = mPos[2] = 0;
	mDir = 2; 
	mResample = 1;
	mInterpolate = true;
}


iOrthoSlicer::~iOrthoSlicer() 
{
}


void iOrthoSlicer::SetInterpolation(bool s)
{
	mInterpolate = s;
	this->Modified();
}


void iOrthoSlicer::SetPos(double p)
{
	mPos[mDir] = p;
	this->Modified();
}


void iOrthoSlicer::SetDir(int d)
{
	if(d>=0 && d<3)
	{
		mDir = d;
		this->Modified();
	}
}


void iOrthoSlicer::ProduceOutput(vtkImageData *input, vtkStructuredPoints *output)
{
    int j, dim[3], dim0[3], numPts;
    double org[3], spa[3];
    double orgOut[3], spaOut[3];
    int Uidx=(mDir+1)%3,  Vidx=(mDir+2)%3;
    
	output->Initialize();

    input->GetOrigin(org);
    input->GetSpacing(spa);
    input->GetDimensions(dim0);

	int ijk[3], ijk1[3], ijk2[3];
    float ht = (mPos[mDir]-org[mDir])/spa[mDir];	//Normalized height
    ijk[mDir] = (int)floor(ht);						//Grid Point below
    float ss1, ss2 = (ht-ijk[mDir]);				//interpolation factor

	//
	//  Add a little bit of a cushion
	//
	if(ijk[mDir]==-1 && ss2>0.99)
	{
		ijk[mDir] = 0;
		ss2 = 0.0;
	}

	if(ijk[mDir]==dim0[mDir]-1 && ss2<0.01)
	{
		ijk[mDir] = dim0[mDir] - 2;
		ss2 = 1.0;
	}
	ss1 = 1.0 - ss2;

	//
	//  Must do that for parallel work
	//
	if(ijk[mDir]<0 || ijk[mDir]>=dim0[mDir]-1) return;
	
	int step = mResample;
	step = ( dim0[Uidx]/8 > step ) ? step : dim0[Uidx]/8;
	step = ( dim0[Vidx]/8 > step ) ? step : dim0[Uidx]/8;
	if(step < 1) step = 1;

	dim[Uidx] = dim0[Uidx]/step;
	dim[Vidx] = dim0[Vidx]/step;
	dim[mDir] = 1;

	float Ufact = (dim0[Uidx]-1.0)/(dim[Uidx]-1.0);
	float Vfact = (dim0[Vidx]-1.0)/(dim[Vidx]-1.0);

	spaOut[Uidx] = spa[Uidx]*Ufact;
	spaOut[Vidx] = spa[Vidx]*Vfact;
	spaOut[mDir] = spa[mDir];

	orgOut[Uidx] = org[Uidx];
	orgOut[Vidx] = org[Vidx];
	orgOut[mDir] = mPos[mDir];

	output->SetDimensions(dim);
	output->SetOrigin(orgOut);
	output->SetSpacing(spaOut);

	//
	//  Prepare output scalars
	//
    numPts = dim[Uidx]*dim[Vidx];
	if(!this->ScalarsInit(input->GetPointData(),numPts)) return;

	vtkIdType loffOut, loffIn1, loffIn2;

	ijk1[mDir] = ijk[mDir];
	ijk2[mDir] = ijk[mDir] + 1;
	for(ijk[Vidx]=0; ijk[Vidx]<dim[Vidx]; ijk[Vidx]++)
	{
		this->UpdateProgress((float)ijk[Vidx]/dim[Vidx]);
		if(this->GetAbortExecute()) break;

		for(ijk[Uidx]=0; ijk[Uidx]<dim[Uidx]; ijk[Uidx]++)
		{
			loffOut = wScalarDimOut*(ijk[Uidx]+ijk[Vidx]*(vtkIdType)dim[Uidx]);

			if(mInterpolate)
			{
				ijk1[Uidx] = ijk[Uidx]*step + step/2;
				ijk1[Vidx] = ijk[Vidx]*step + step/2;
				loffIn1 = wScalarDimIn*(ijk1[0]+(vtkIdType)dim0[0]*(ijk1[1]+(vtkIdType)dim0[1]*ijk1[2]));
				ijk2[Uidx] = ijk1[Uidx];
				ijk2[Vidx] = ijk1[Vidx];
				loffIn2 = wScalarDimIn*(ijk2[0]+(vtkIdType)dim0[0]*(ijk2[1]+(vtkIdType)dim0[1]*ijk2[2]));
				for(j=0; j<wScalarDimOut; j++)
				{
					wScalarPtrOut[loffOut+j] = ss1*wScalarPtrIn[loffIn1+j] + ss2*wScalarPtrIn[loffIn2+j];
				}
			}
			else
			{
				if(ss2 < 0.5f)
				{
					ijk1[Uidx] = ijk[Uidx]*step + step/2;
					ijk1[Vidx] = ijk[Vidx]*step + step/2;
					loffIn1 = wScalarDimOut*(ijk1[0]+(vtkIdType)dim0[0]*(ijk1[1]+(vtkIdType)dim0[1]*ijk1[2]));
				}
				else
				{
					ijk2[Uidx] = ijk[Uidx]*step + step/2;
					ijk2[Vidx] = ijk[Vidx]*step + step/2;
					loffIn1 = wScalarDimOut*(ijk2[0]+(vtkIdType)dim0[0]*(ijk2[1]+(vtkIdType)dim0[1]*ijk2[2]));
				}
				for(j=0; j<wScalarDimOut; j++)
				{
					wScalarPtrOut[loffOut+j] = wScalarPtrIn[loffIn1+j];
				}
			}
		}
    }
    
    this->ScalarsDone(output->GetPointData());
}

