/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparallelpipeline.h"


#include "iappendpolydatafilter.h"
#include "idatadistributor.h"
#include "ierror.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iserialpipeline.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "ivtk.h"

#include <vtkImageAppend.h>
#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  main class
//
template<class Filter, class InputType, class OutputType>
iParallelPipeline<Filter,InputType,OutputType>::iParallelPipeline(int numInputs, iViewObject *vo) : Filter(numInputs), iViewObjectComponent(vo), iParallelWorker(vo->GetViewModule()->GetParallelManager()), mWorkInputs(0), mWorkPipelines(0)
{
	this->mConfigureTime = 0UL;
	this->mNumProcs = 0;  // zero procs means we are in a non-set state
	this->mPolish = true;
}


template<class Filter, class InputType, class OutputType>
iParallelPipeline<Filter,InputType,OutputType>::~iParallelPipeline()
{
	this->FinishInitialization(); // in case it hasn't been called

	this->ResetPipelines();
	//
	//  Remove master thread
	//
	this->mWorkPipelines.Remove(0)->Delete();

	mDistributor->Delete();
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::FinishInitialization()
{
	int j;

	if(this->mWorkPipelines.Size() > 0) return;
	//
	//  Create the master thread which is always present
	//
	InputType *d = this->CreateSerialInput(); IERROR_ASSERT_NULL_POINTER(d);
	this->mWorkInputs.Add(d);
	iSerialPipeline<Filter,InputType> *f = this->CreateSerialPipeline(); IERROR_ASSERT_NULL_POINTER(f);
	this->mWorkPipelines.Add(f);
	f->SetInput(d);
	d->Delete();
	f->AttachObservers(true); 

	OutputType* tmp;
	for(j=1; j<f->GetNumberOfOutputs(); j++)
	{
		tmp = OutputType::New(); IERROR_ASSERT_NULL_POINTER(tmp);
		this->SetNthOutput(j,tmp);
		tmp->Delete();
	}
	//
	//  Create data distributor
	//
	this->mDistributor = this->CreateDataDistributor(); IERROR_ASSERT_NULL_POINTER(this->mDistributor);
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::Modified()
{
	int i;
	for(i=0; i<this->mWorkPipelines.Size(); i++) 
	{
		this->mWorkPipelines[i]->Modified();
	}
	this->Filter::Modified();
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::Reconfigure()
{
	//
	//  Actual reconfiguring is done at run-time
	//
	this->mConfigureTime = 0UL;
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::ResetPipelines()
{
	while(this->mWorkPipelines.Size() > 1) this->mWorkPipelines.RemoveLast()->Delete();
	while(this->mWorkInputs.Size() > 1) this->mWorkInputs.RemoveLast(); // don't need to delete inputs, filters own them.
	this->mNumProcs = 0;
	this->Modified();
}


template<class Filter, class InputType, class OutputType>
InputType* iParallelPipeline<Filter,InputType,OutputType>::CreateSerialInput()
{
	return InputType::New();
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::ConfigurePipelines()
{
	int i;
	this->ResetPipelines(); // Must reset all pipelines - if the old ones are preserved, then they may not be exact copies of the Master pipeline.
							// Alternative, we can add a new function to bring them in sync with the Master pipeline

	if(this->mWorkPipelines.Size() != 1)
	{
		vtkErrorMacro("Incorrect number of pipelines");
		return;
	}

	//
	//  Save our number of processor in case it changes during our execution
	//
	this->mNumProcs = this->GetManager()->GetNumberOfProcessors();

 	for(i=1; i<this->mNumProcs; i++)
	{
		InputType *d = this->CreateSerialInput(); IERROR_ASSERT_NULL_POINTER(d);
		this->mWorkInputs.Add(d);
		iSerialPipeline<Filter,InputType> *f = this->mWorkPipelines[0]->Copy(); IERROR_ASSERT_NULL_POINTER(f);
		this->mWorkPipelines.Add(f);
		f->SetInput(d);
		d->Delete();
		f->AttachObservers(false); // i==0: master thread, i>0: slave thread
	}

	//
	//  Assign global bounds and inputs to parallel filters
	//
 	for(i=0; i<this->mNumProcs; i++)
	{
		this->mWorkPipelines[i]->SetGlobalInput(this->GetInput());
	}

	//
	//  Ask distributor to distribute the data 
	//
	this->mDistributor->DistributeData(this->GetInput(),this->mWorkInputs);

	//
	//  If the filters accept more than one input, just pass other inputs
	//
	if(this->GetNumberOfInputs() > 1)
	{
		int j;
		for(i=0; i<this->mNumProcs; i++)
		{
			for(j=1; j<this->GetNumberOfInputs(); j++) this->mWorkPipelines[i]->SetNthInput(j,this->GetInputs()[j]);
		}
	}

	this->Filter::Modified();
	this->mConfigureTime = this->GetMTime();
}


template<class Filter, class InputType, class OutputType>
int iParallelPipeline<Filter,InputType,OutputType>::ExecuteStep(int, iParallel::ProcessorInfo &p)
{
	if(p.NumProcs != this->mNumProcs) return 1;

	this->mWorkPipelines[p.ThisProc]->Update();
	return 0;
}


template<class Filter, class InputType, class OutputType>
void iParallelPipeline<Filter,InputType,OutputType>::ProduceOutput(InputType *, OutputType *)
{
	int j;

	for(j=0; j<this->GetNumberOfOutputs(); j++)
	{
		this->GetOutput(j)->Initialize();
	}

	//
	//  Do we need to re-distribute the data?
	//
	if(this->mNumProcs==0 || this->mNumProcs!=this->GetManager()->GetNumberOfProcessors() || this->GetInput()->GetMTime()>this->mConfigureTime)// || this->GetMTime()>this->mConfigureTime) // the latter is needed when DataTypes are changed
	{
		this->ConfigurePipelines();
	}

	//
	//  Now run the pipelines
	//
	if(this->mNumProcs == 1)
	{
		//
		//  Serial execution
		//
		this->mWorkPipelines[0]->Update();
	}
	else
	{
		//
		//  Parallel execution
		//
		this->ParallelExecute(0);
//		mWorkPipelines[0]->Update();
	}

	if(this->GetAbortExecute()) return;
	this->Append();

	if(this->IsOptimizedForMemory()) this->ResetPipelines();

	if(!this->mPolish || this->GetAbortExecute()) return;

	for(j=0; j<this->GetNumberOfOutputs(); j++)
	{
		this->mDistributor->PolishOutput(this->mWorkPipelines[0]->GetViewObject(),this->GetOutput(j));
	}
}


template<class Filter, class InputType, class OutputType>
float iParallelPipeline<Filter,InputType,OutputType>::GetMemorySize() const
{
	int i, j;
	float s = 0.0;

	for(j=0; j<this->NumberOfOutputs; j++)
	{
		if(this->IsOptimizedForMemory())
		{
			s += this->Outputs[j]->GetActualMemorySize();
		}
		else
		{
			for(i=0; i<this->mWorkPipelines.Size(); i++) 
			{
				s += this->mWorkPipelines[i]->GetMemorySize();
			}
		}
	}
	return s;
}


//
//  iParallelDataToPolyDataPipeline template
//
template<class Filter, class InputType>
iParallelDataToPolyDataPipeline<Filter,InputType>::iParallelDataToPolyDataPipeline(int numInputs, iViewObject *vo) : iParallelPipeline<Filter,InputType,vtkPolyData>(numInputs,vo)
{
	this->mPatchStitches = true;

	this->mAppend = iAppendPolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(this->mAppend);
	this->mAppend->UserManagedInputsOff();
}


template<class Filter, class InputType>
iParallelDataToPolyDataPipeline<Filter,InputType>::~iParallelDataToPolyDataPipeline()
{
	this->mAppend->Delete();
}


template<class Filter, class InputType>
void iParallelDataToPolyDataPipeline<Filter,InputType>::Append()
{
	int i, j;
	vtkPolyData *tmpOutput;

	for(j=0; j<this->GetNumberOfOutputs(); j++)
	{
		if(iParallelPipeline<Filter,InputType,vtkPolyData>::mNumProcs == 1)
		{
			this->GetOutput(j)->ShallowCopy(iParallelPipeline<Filter,InputType,vtkPolyData>::mWorkPipelines[0]->GetOutput(j));
		}
		else
		{
			this->mAppend->RemoveAllInputs();
			for(i=0; i<iParallelPipeline<Filter,InputType,vtkPolyData>::mWorkPipelines.Size(); i++) 
			{
				tmpOutput = vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(tmpOutput);
				tmpOutput->ShallowCopy(iParallelPipeline<Filter,InputType,vtkPolyData>::mWorkPipelines[i]->GetOutput(j));
				this->mAppend->AddInput(tmpOutput);
				tmpOutput->Delete();
			}
			this->mAppend->Update();
			this->mAppend->RemoveAllInputs();
			this->GetOutput(j)->ShallowCopy(this->mAppend->GetOutput());
		}	
	}
}


template<class Filter, class InputType>
void iParallelDataToPolyDataPipeline<Filter,InputType>::SetPatchStitches(bool s)
{
	this->mPatchStitches = s;
	this->Modified();
}


//
//  iParallelDataToGridDataPipeline template
//
template<class Filter, class InputType>
iParallelDataToGridDataPipeline<Filter,InputType>::iParallelDataToGridDataPipeline(int numInputs, iViewObject *vo) : iParallelPipeline<Filter,InputType,vtkStructuredPoints>(numInputs,vo)
{
	this->mAppend = vtkImageAppend::New(); IERROR_ASSERT_NULL_POINTER(this->mAppend);
}


template<class Filter, class InputType>
iParallelDataToGridDataPipeline<Filter,InputType>::~iParallelDataToGridDataPipeline()
{
	this->mAppend->Delete();
}


template<class Filter, class InputType>
void iParallelDataToGridDataPipeline<Filter,InputType>::Append()
{
	int i, j;
	vtkImageData *tmpOutput;

	for(j=0; j<this->GetNumberOfOutputs(); j++)
	{
		if(iParallelPipeline<Filter,InputType,vtkStructuredPoints>::mNumProcs == 1)
		{
			this->GetOutput(j)->ShallowCopy(iParallelPipeline<Filter,InputType,vtkStructuredPoints>::mWorkPipelines[0]->GetOutput(j));
		}
		else
		{
			this->mAppend->RemoveAllInputs();
			this->mAppend->SetNumberOfThreads(iParallelPipeline<Filter,InputType,vtkStructuredPoints>::mNumProcs);
			for(i=0; i<this->mWorkPipelines.Size(); i++) 
			{
				tmpOutput = vtkImageData::New(); IERROR_ASSERT_NULL_POINTER(tmpOutput);
				tmpOutput->ShallowCopy(iParallelPipeline<Filter,InputType,vtkStructuredPoints>::mWorkPipelines[i]->GetOutput(j));
				this->mAppend->AddInput(tmpOutput);
				tmpOutput->Delete();
			}
			this->mAppend->Update();
			this->mAppend->RemoveAllInputs();
			this->GetOutput(j)->ShallowCopy(this->mAppend->GetOutput());
		}	
	}
}

