/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipicker.h"


#include "iactor.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "istreamlinefilter.h"
#include "iuniformgriddatasubject.h"
#include "iviewmodule.h"
#include "iviewobjectobserver.h"
#include "ivtk.h"

#include <vtkCellPicker.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPointLocator.h>
#include <vtkPointPicker.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProp.h>
#include <vtkProperty.h>
#include <vtkPropPicker.h>
#include <vtkSphereSource.h>
#include <vtkStructuredPoints.h>


//
//  Picker must be an object so that it can be created by the ObjectFactory
//
IOBJECT_DEFINE_TYPE(iPicker,Picker,pi,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iPicker,Accuracy,a,Float,1);
IOBJECT_DEFINE_KEY(iPicker,PickMethod,m,Int,1);
IOBJECT_DEFINE_KEY(iPicker,PointSize,ps,Float,1);


namespace iPicker_Private
{
	class CellPicker : public vtkCellPicker
	{

	public:

		static CellPicker* New()
		{ 
			return new CellPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class PointPicker : public vtkPointPicker
	{

	public:

		static PointPicker* New()
		{ 
			return new PointPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class ObjectPicker : public vtkPropPicker
	{

	public:

		static ObjectPicker* New()
		{ 
			return new ObjectPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};
};


using namespace iParameter;
using namespace iPicker_Private;


iPicker* iPicker::New(iViewModule *vm)
{
	return iPointerCast<iPicker,iObject>(iExtensionFactory::CreateObject(Type(),vm));
}


iPicker::iPicker(iViewModule *vm) : iDataFormatter(vm), mPos(vm), mNumPickMethods(3)
{
	int i;

	mObjName = "";
	mObjType = _ObjectTypeUndefined;
	mDataTypePointer = 0;
	mPickMethod = _PickMethodObject;

	//
	//  Pickers
	//
	mDevices[0] = CellPicker::New(); IERROR_ASSERT_NULL_POINTER(mDevices[0]);
	mDevices[1] = PointPicker::New(); IERROR_ASSERT_NULL_POINTER(mDevices[1]);
	mDevices[2] = ObjectPicker::New(); IERROR_ASSERT_NULL_POINTER(mDevices[2]);

	this->SetAccuracy(0.02f);

	mObserver = iPickEventObserver::New(this->GetViewModule()); IERROR_ASSERT_NULL_POINTER(mObserver);
	this->AddObserver(vtkCommand::StartPickEvent,mObserver);
	this->AddObserver(vtkCommand::EndPickEvent,mObserver);
	for(i=0; i<mNumPickMethods; i++)
	{
		mDevices[i]->AddObserver(vtkCommand::AbortCheckEvent,this->GetViewModule()->GetAbortRenderEventObserver());
	}

	//
	//  Graphical representation
	//
	mPointActor = iActor::New(); IERROR_ASSERT_NULL_POINTER(mPointActor);
	mPointSource = vtkSphereSource::New(); IERROR_ASSERT_NULL_POINTER(mPointSource);

	mPointSize = 0.1;
	this->SetPointSize(mPointSize);
	mPointSource->SetRadius(0.5);
	mPointActor->SetInput(mPointSource->GetOutput());
	mPointActor->GetProperty()->SetColor(0.9,0.9,0.9);
	mPointActor->VisibilityOff();
	mPointActor->PickableOff();

	this->GetViewModule()->AddObject(mPointActor);
}


iPicker::~iPicker()
{
	int i;
	mObserver->Delete();
	for(i=0; i<mNumPickMethods; i++) mDevices[i]->Delete();
	this->GetViewModule()->RemoveObject(mPointActor);
	mPointActor->Delete();
	mPointSource->Delete();
}


int iPicker::IsA(const char *type)
{
	if(strcmp(type,"iPicker") == 0) return 1; else return vtkAbstractPropPicker::IsA(type);
}


int iPicker::Pick(double x, double y, double z, vtkRenderer *ren)
{
	this->InvokeEvent(vtkCommand::StartPickEvent,0);
	int ret = mDevices[mPickMethod]->Pick(x,y,z,ren);
	this->InvokeEvent(vtkCommand::EndPickEvent,0);
	return ret;
}


void iPicker::Modified()
{
	int i;
	for(i=0; i<mNumPickMethods; i++) mDevices[i]->Modified();
	vtkAbstractPropPicker::Modified();
	this->ClearCache();
}


void iPicker::UpdateReport()
{
	mObjName = "Unknown";
	mPos = mDevices[mPickMethod]->GetPickPosition();

	this->SetPointSize(mPointSize);
	mPointActor->SetInput(mPointSource->GetOutput());
	mPointActor->SetPosition(mPos);

	this->ClearReport();
	this->UpdateReportBody();

	this->ShowPickedPoint(true);
}


void iPicker::UpdateReportBody()
{
	int j;
	iString s1, s2;

	//
	//  Get the object info
	//
#ifdef IVTK_4
	vtkProp *obj = mDevices[mPickMethod]->GetProp();
#else
	vtkProp *obj = mDevices[mPickMethod]->GetViewProp();
#endif
	if(obj == 0) 
	{
		mObjName = "";
		mPointActor->VisibilityOff(); 
		return;
	}
	iViewObjectObserver *obs = iPointerCast<iViewObjectObserver,vtkCommand>(obj->GetCommand(1));

	if(obs != 0)
	{
		mObjType = obs->GetObjectType();
		mDataTypePointer = &obs->GetDataType();
	}
	else
	{
		mObjType = _ObjectTypeUndefined;
		mDataTypePointer = 0;
	}
	//
	//  Display the object info
	//
	switch (mObjType)
	{
	case _ObjectTypeSurface: { mObjName = "Probe surface"; break; }
	case _ObjectTypeCrossSection: { mObjName = "Cross section"; break; }
	case _ObjectTypeVolume: { mObjName = "Volume rendering"; break; }
	case _ObjectTypeParticles: { mObjName = "Particles"; break; }
	case _ObjectTypeVectorField: { mObjName = "Vector field"; break; }
	case _ObjectTypeTensorField: { mObjName = "Tensor field"; break; }
	case _ObjectTypeMarker: { mObjName = "Marker symbol"; break; }
	}
	//
	//  Display the data info
	//
	vtkPointLocator *locator;
	locator = vtkPointLocator::New(); IERROR_ASSERT_NULL_POINTER(locator);
	
	if(mDataTypePointer != 0)
	{
		iDataLimits *lim = this->GetViewModule()->GetReader()->GetLimits(*mDataTypePointer);
		if(*mDataTypePointer == iDataType::UniformScalars())
		{
			int nvar = lim->GetNumVars();
			//
			//  Do not need any locator - extract mesh data from the point location directly - and interpolate too.
			//  First find 8 neighbors
			//
			int dims[3];
			double org[3], spa[3];
			double d1[3], d2[3];

			vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
			data->GetDimensions(dims);
			data->GetOrigin(org);
			data->GetSpacing(spa);

			vtkIdType npoi = (vtkIdType)dims[0]*dims[1]*dims[2];
			if(nvar>0 && npoi>0)
			{
				int ijk1[3], ijk2[3];
				bool per[3];
				lim->GetPeriodicities(per);
				iGridDataSubject::GetCICInterpolation(per,dims,org,spa,mPos,ijk1,ijk2,d1,d2);
				//
				//  Interpolate mesh data keeping in mind that the data array is horizontal, not vertical
				//
				float *v = new float[nvar]; IERROR_ASSERT_NULL_POINTER(v);
				float *ptr;
				vtkFloatArray *array;
				for(j=0; j<nvar; j++)
				{
					array = dynamic_cast<vtkFloatArray*>(data->GetPointData()->GetArray(j));
					if(array == 0)
					{
#ifdef I_CHECK1
						IERROR_REPORT_BUG;
#endif
					}
					else
					{
						ptr = array->GetPointer(0);
						v[j] = 
							ptr[ijk1[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk1[2])]*d1[0]*d1[1]*d1[2] +
							ptr[ijk1[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk2[2])]*d1[0]*d1[1]*d2[2] +
							ptr[ijk1[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk1[2])]*d1[0]*d2[1]*d1[2] +
							ptr[ijk1[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk2[2])]*d1[0]*d2[1]*d2[2] +
							ptr[ijk2[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk1[2])]*d2[0]*d1[1]*d1[2] +
							ptr[ijk2[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk2[2])]*d2[0]*d1[1]*d2[2] +
							ptr[ijk2[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk1[2])]*d2[0]*d2[1]*d1[2] +
							ptr[ijk2[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk2[2])]*d2[0]*d2[1]*d2[2];
					}
				}
				//
				//  Make the correction for the surface
				//
				if(mObjType == _ObjectTypeSurface)
				{
					j = obs->GetIsoSurfaceData().Var;
					if(j>=0 && j<nvar) v[j] = obs->GetIsoSurfaceData().Level;
				}

				this->FormatScalarData(lim,nvar,v);

				delete [] v;
			}
		}

		if(*mDataTypePointer == iDataType::UniformVectors())
		{ 
			//
			//  Do not need any locator - extract vector data from the point location directly - and interpolate too.
			//  First find 8 neighbors
			//
			int dims[3];
			double org[3], spa[3];
			float v[3], vort[3], div[1];

			vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformVectors()));
			data->GetDimensions(dims);
			data->GetOrigin(org);
			data->GetSpacing(spa);
			float *ptr = (float *)data->GetPointData()->GetVectors()->GetVoidPointer(0);

			if(ptr != 0)
			{
				iStreamLineFilter::GetVector(mPos,v,ptr,dims,org,spa,lim,vort,div);

				this->FormatVectorData(lim,-1,v,div,vort);
			}
		}

		if(*mDataTypePointer == iDataType::UniformTensors())
		{ 
			//
			//  Do not need any locator - extract vector data from the point location directly - and interpolate too.
			//  First find 8 neighbors
			//
			int dims[3];
			double org[3], spa[3];
			double d1[3], d2[3];

			vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformTensors()));
			data->GetDimensions(dims);
			data->GetOrigin(org);
			data->GetSpacing(spa);

			int ijk1[3], ijk2[3];
			bool per[3];
			lim->GetPeriodicities(per);
			iGridDataSubject::GetCICInterpolation(per,dims,org,spa,mPos,ijk1,ijk2,d1,d2);
			//
			//  Interpolate vector data keeping in mind that the data array is vertical, not horizontal
			//
			float v[9];
			float *ptr = (float *)data->GetPointData()->GetTensors()->GetVoidPointer(0);

			if(ptr != 0)
			{
				for(j=0; j<9; j++)
				{
					v[j] = 
						ptr[j+9*(ijk1[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk1[2]))]*d1[0]*d1[1]*d1[2] +
						ptr[j+9*(ijk1[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk2[2]))]*d1[0]*d1[1]*d2[2] +
						ptr[j+9*(ijk1[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk1[2]))]*d1[0]*d2[1]*d1[2] +
						ptr[j+9*(ijk1[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk2[2]))]*d1[0]*d2[1]*d2[2] +
						ptr[j+9*(ijk2[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk1[2]))]*d2[0]*d1[1]*d1[2] +
						ptr[j+9*(ijk2[0]+dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk2[2]))]*d2[0]*d1[1]*d2[2] +
						ptr[j+9*(ijk2[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk1[2]))]*d2[0]*d2[1]*d1[2] +
						ptr[j+9*(ijk2[0]+dims[0]*(ijk2[1]+(vtkIdType)dims[1]*ijk2[2]))]*d2[0]*d2[1]*d2[2];
				}

				this->FormatTensorData(lim,-1,v);

			} 
		}

		if(*mDataTypePointer == iDataType::BasicParticles())
		{
			int natt = lim->GetNumVars();
			vtkPolyData *data = iPointerCast<vtkPolyData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::BasicParticles()));

			locator->SetDataSet(data);
			vtkIdType pid = locator->FindClosestPoint(mDevices[mPickMethod]->GetPickPosition()); 
			float *p = iPointerCast<vtkFloatArray,vtkDataArray>(data->GetPointData()->GetScalars())->GetPointer(0);

			if(p!=0 && natt>0 && pid>=0 && pid<data->GetNumberOfPoints())
			{
				this->FormatScalarData(lim,natt,p+pid*natt);
			}
		}

	}

	if(locator != 0) locator->Delete();
}


void iPicker::SetPointSize(float s)
{
	if(s > 0.0)
	{
		mPointSize = s;
		mPointActor->SetScaled(true);
		mPointActor->SetBasicScale(mPointSize);
		this->Modified();
	}
}


void iPicker::ShowPickedPoint(bool s)
{
	if(!mObjName.IsEmpty() && s)
	{
		mPointActor->VisibilityOn();
	}
	if(!s)
	{
		mPointActor->VisibilityOff();
	}
	this->GetViewModule()->Render();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iPicker::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyPickMethod(),mPickMethod);
	this->PackValue(s,KeyPointSize(),mPointSize);
	this->PackValue(s,KeyAccuracy(),mAccuracy);
}


void iPicker::UnPackStateBody(const iString &s)
{
	int i; float f;

	if(this->UnPackValue(s,KeyPickMethod(),i)) this->SetPickMethod(i);
	if(this->UnPackValue(s,KeyPointSize(),f)) this->SetPointSize(f);
	if(this->UnPackValue(s,KeyAccuracy(),f)) this->SetAccuracy(f);
}


void iPicker::SetPickMethod(int s)
{
	if(s>=0 && s<mNumPickMethods)
	{
		mPickMethod = s;
		this->ClearCache();
	}
}


void iPicker::SetAccuracy(float s)
{
	if(s > 0.0f)
	{
		mAccuracy = s;
		
		int j;
		vtkPicker *p;
		for(j=0; j<mNumPickMethods; j++)
		{
			p = dynamic_cast<vtkPicker*>(mDevices[j]);
			if(p != 0) p->SetTolerance(s);
		}
		this->ClearCache();
	}
}

