/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iruler.h"


#include "ierror.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkTextMapper.h>
#include <vtkTextProperty.h>


iRuler* iRuler::New(iViewModule *vm)
{
	return new iRuler(vm);
}


iRuler::iRuler(iViewModule *vm) : iOverlayBase(vm->GetRenderTool()), iViewModuleComponent(vm)
{
	mFontSize = 12;
	mStarted = false;
	mScale = mBaseScale = 1.0;
	mLineWidth = this->AxisActor->GetProperty()->GetLineWidth();
	mTickLength = TickLength;

	this->GetLabelTextProperty()->BoldOn();
	this->GetLabelTextProperty()->ItalicOff();
	this->GetLabelTextProperty()->ShadowOn();
	this->GetTitleTextProperty()->BoldOn();
	this->GetTitleTextProperty()->ItalicOff();
	this->GetTitleTextProperty()->ShadowOn();
	this->LabelMappers[0]->SetTextProperty(this->LabelTextProperty);
	this->LabelMappers[1]->SetTextProperty(this->LabelTextProperty);
	this->LabelTextProperty->SetVerticalJustificationToTop();

	this->SetPoint1(0.25,0.9);
	this->SetPoint2(0.75,0.9);
	this->SetRange(0.0,1.0);
	this->AdjustLabelsOff();
	this->SetNumberOfLabels(2);

	this->PickableOff();

	iColor black;
	this->SetColor(black);
}


iRuler::~iRuler()
{
}


void iRuler::ReleaseGraphicsResources(vtkWindow *win)
{
	this->vtkAxisActor2D::ReleaseGraphicsResources(win);
}


int iRuler::RenderOverlay(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;

	int mag = this->GetRenderingMagnification();

	if(mag == 1)
	{
		this->LabelTextProperty->SetFontSize(mFontSize);
	}
	else
	{
		this->LabelTextProperty->SetFontSize(mag*mFontSize);
	}
	
	return this->vtkAxisActor2D::RenderOverlay(viewport);
}


int iRuler::RenderOpaqueGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;
	
	int mag = this->GetRenderingMagnification();

	double x0[3], x1[3];
	double dummy = 0.0;
	
	int k;
	double vx[3], vy[3], vz[3];
	
	if(mag == 1)
	{
		cam->GetViewUp(vy);
		cam->GetDirectionOfProjection(vz);
		for(k=0; k<3; k++) vz[k] = -vz[k];
		
		vtkMath::Cross(vy,vz,vx);
		
		viewport->SetWorldPoint(vx[0],vx[1],vx[2],1.0);
		viewport->WorldToDisplay();
		viewport->GetDisplayPoint(x0);
		viewport->SetWorldPoint(-vx[0],-vx[1],-vx[2],1.0);
		viewport->WorldToDisplay();
		viewport->GetDisplayPoint(x1);
		
		double d = sqrt(vtkMath::Distance2BetweenPoints(x1,x0));
		viewport->DisplayToNormalizedDisplay(d,dummy);
		
		if(!mStarted)
		{
			mStarted = true;
			mBaseScale = cam->GetParallelScale();
		}
		
		float s = cam->GetParallelScale()/mBaseScale;
		mDx = 0.5*d*s;
		mDx = (mDx > 0.3) ? 0.3 : mDx;

		if(!this->GetViewModule()->GetOpenGLCoordinates()) s *= this->GetViewModule()->GetBoxSize();
		this->SetRange(0.0,s);
		mScale = s;
		
		this->SetPoint1(0.5-mDx,0.9);
		this->SetPoint2(0.5+mDx,0.9);

		this->AxisActor->GetProperty()->SetLineWidth(mLineWidth);
		this->SetTickLength(mTickLength);
	}
	else
	{
		int winij[2];
		this->ComputePositionShiftsUnderMagnification(cam,winij,viewport);
		
		this->SetPoint1(mag*(0.5-mDx)-winij[0],mag*0.9-winij[1]);
		this->SetPoint2(mag*(0.5+mDx)-winij[0],mag*0.9-winij[1]);
		
		this->AxisActor->GetProperty()->SetLineWidth(mLineWidth*mag);
		this->SetTickLength(mag*mTickLength);
	}
	
	this->vtkAxisActor2D::RenderOpaqueGeometry(viewport);
	
	return 1;
}


void iRuler::SetFontSize(int s)
{ 
	if(s>0 && s<99) mFontSize = s; 
}


void iRuler::SetColor(iColor &c)
{ 
	this->GetProperty()->SetColor(c.ToVTK());
	this->GetLabelTextProperty()->SetColor(c.ToVTK());
	this->GetTitleTextProperty()->SetColor(c.ToVTK());
}


void iRuler::SetBaseScale(float s)
{
	mStarted = true;
	mBaseScale = s;
}


void iRuler::SetScale(float s)
{
	if(!mStarted)
	{
		mStarted = true;
		mBaseScale = this->GetViewModule()->GetRenderer()->GetActiveCamera()->GetParallelScale();
	}

	if(!this->GetViewModule()->GetOpenGLCoordinates()) s /= this->GetViewModule()->GetBoxSize();
	this->GetViewModule()->GetRenderer()->GetActiveCamera()->SetParallelScale(s*mBaseScale);
}
