/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "isurfaceserialpipeline.h"


#include "iboundedplanesource.h"
#include "iboundedspheresource.h"
#include "icontourfilter.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iflipnormalsfilter.h"
#include "igriddataconverter.h"
#include "ioptimizepolydatafilter.h"
#include "imath.h"
#include "iprobefilter.h"
#include "ireducepolydatafilter.h"
#include "ireducepolydatafilter2.h"
#include "ireplicatedpolydata.h"
#include "ishiftpolydatafilter.h"
#include "ismoothpolydatafilter.h"
#include "isurfaceviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectreplicator.h"

#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>


//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"


using namespace iParameter;


//
// iSurfaceSerialPipeline class
//
iSurfaceSerialPipeline::iSurfaceSerialPipeline(iSurfaceViewObject *s) : iGridDataToPolyDataPipeline(1,s)
{
	int i;

	mSpecialVar = 0;
	mParent = s;

	for(i=0; i<2; i++)
	{
		mOutput[i] = vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(mOutput[i]);
	}
//
//  Do VTK stuff
//	
	mDataConverter = iGridDataConverter::New(mParent); IERROR_ASSERT_NULL_POINTER(mDataConverter);
	mIsoSurfaceSource = iContourFilter::New(); IERROR_ASSERT_NULL_POINTER(mIsoSurfaceSource);
	mShiftFilter = iShiftPolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(mShiftFilter);
	mGateway = iProbeFilter::New(mParent); IERROR_ASSERT_NULL_POINTER(mGateway);
	mSmoothFilter = iSmoothPolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(mSmoothFilter);
	mReduceFilter = iReducePolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(mReduceFilter);
	mReduce2Filter = iReducePolyDataFilter2::New(); IERROR_ASSERT_NULL_POINTER(mReduce2Filter);
	mOptimizeFilter = iOptimizePolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(mOptimizeFilter);

	mSphereSource = iBoundedSphereSource::New(mParent); IERROR_ASSERT_NULL_POINTER(mSphereSource);
	mPlaneSource = iBoundedPlaneSource::New(mParent); IERROR_ASSERT_NULL_POINTER(mPlaneSource);

	for(i=0; i<2; i++)
	{
		mProbe[i] = iProbeFilter::New(mParent); IERROR_ASSERT_NULL_POINTER(mProbe[i]);
		mDataReplicated[i] = iReplicatedPolyData::New(mParent->GetReplicator()); IERROR_ASSERT_NULL_POINTER(mDataReplicated[i]);
		mDataConverterPaint[i] = iGridDataConverter::New(mParent); IERROR_ASSERT_NULL_POINTER(mDataConverterPaint[i]);
		mFlipNormalsFilter[i] = iFlipNormalsFilter::New(); IERROR_ASSERT_NULL_POINTER(mFlipNormalsFilter[i]);
	}

	mIsoSurfaceSource->SetInput(mDataConverter->GetOutput());
	mIsoSurfaceSource->UseScalarTreeOn();
	mIsoSurfaceSource->ComputeNormalsOn();
	mIsoSurfaceSource->ComputeGradientsOff();

	mGateway->SetInput(mIsoSurfaceSource->GetOutput());
	mGateway->SetSource(mDataConverter->GetOutput());
	mGateway->SetActive(false);  //  a dummy filter

	for(i=0; i<2; i++)
	{
		mDataConverterPaint[i]->SetCurrentVar(mParent->GetPaintVar(i));

		mProbe[i]->SetSource(mDataConverterPaint[i]->GetOutput());

		mFlipNormalsFilter[i]->ReverseCellsOff();
		mFlipNormalsFilter[i]->SetInput(mProbe[i]->GetOutput());
	}

	mSmoothFilter->BoundarySmoothingOff();
	mSmoothFilter->NonManifoldSmoothingOn();
	mSmoothFilter->FeatureEdgeSmoothingOn();

	mReduceFilter->PreserveTopologyOn();
	mReduceFilter->BoundaryVertexDeletionOff(); 
	
	mReduceFilter->SetInput(mGateway->GetOutput());
	mReduce2Filter->SetInput(mGateway->GetOutput());
	mShiftFilter->SetInput(mFlipNormalsFilter[1]->GetOutput());

	mDataReplicated[0]->SetInput(mFlipNormalsFilter[0]->GetOutput());
	mDataReplicated[1]->SetInput(mShiftFilter->GetOutput());

	this->SetNthOutput(0,mOutput[0]);
	this->SetNthOutput(1,mOutput[1]);
	mOutput[0]->Delete();
	mOutput[1]->Delete();

	this->UpdateAll();
	mDataConverter->SetScalarInput(0);
	for(i=0; i<2; i++) mDataConverterPaint[i]->SetScalarInput(0);
}


iSurfaceSerialPipeline::~iSurfaceSerialPipeline()
{
	int i;
	for(i=0; i<2; i++)
	{
		mDataReplicated[i]->Delete();
		mProbe[i]->Delete();
		mDataConverterPaint[i]->Delete();
		mFlipNormalsFilter[i]->Delete();
	}
	mSmoothFilter->Delete();
	mReduceFilter->Delete();
	mReduce2Filter->Delete();
	mOptimizeFilter->Delete();
	mSphereSource->Delete();
	mPlaneSource->Delete();
	mDataConverter->Delete();
	mGateway->Delete();
	mIsoSurfaceSource->Delete();
	mShiftFilter->Delete();
}


void iSurfaceSerialPipeline::AttachObservers(bool master)
{
	int i;

	iEventObserver *obsAbortRender;
	
	if(master)
	{
		obsAbortRender = mParent->GetViewModule()->GetAbortRenderEventObserver();
	}
	else
	{
		obsAbortRender = mParent->GetViewModule()->GetSlaveAbortRenderEventObserver();
	}

	for(i=0; i<2; i++)
	{
		mProbe[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		mDataReplicated[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		mFlipNormalsFilter[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		mDataConverterPaint[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	}
	mSmoothFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mReduceFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mReduce2Filter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mOptimizeFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mSphereSource->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mPlaneSource->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mIsoSurfaceSource->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mShiftFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
}


void iSurfaceSerialPipeline::ProduceOutput(vtkImageData *input, vtkPolyData *)
{
	int dims[3];
	double spa[3], org[3];

	mOutput[0]->Initialize();
	mOutput[1]->Initialize();

	if(input == 0) return;

	if(mDataConverter->GetInput() != input)
	{
		mDataConverter->SetScalarInput(input);
		mDataConverterPaint[0]->SetScalarInput(input);
		mDataConverterPaint[1]->SetScalarInput(input);
	}
	
	//
	//  Check that fixed probe surface need to be adjusted to correspond to the input resolution
	//
	bool redo = false;
	if(mSphereSource->GetMTime() < input->GetMTime())
	{
		this->UpdateSphereResolution();
		redo = true;
	}
	if(mPlaneSource->GetMTime() < input->GetMTime())
	{
		this->UpdatePlaneResolution();
		redo = true;
	}

	if(redo)
	{
		input->GetDimensions(dims);
		input->GetOrigin(org);
		input->GetSpacing(spa);
		mPlaneSource->SetBounds(org[0],org[0]+spa[0]*(dims[0]-1),org[1],org[1]+spa[1]*(dims[1]-1),org[2],org[2]+spa[2]*(dims[2]-1));
		mSphereSource->SetBounds(org[0],org[0]+spa[0]*(dims[0]-1),org[1],org[1]+spa[1]*(dims[1]-1),org[2],org[2]+spa[2]*(dims[2]-1));
	}

	//
	//  Update the pipeline
	//
	mDataReplicated[0]->Update();
	mOutput[0]->ShallowCopy(mDataReplicated[0]->GetOutput());

	if(mParent->GetPaintVar(1) > 0)
	{
		mDataReplicated[1]->Update();
		mOutput[1]->ShallowCopy(mDataReplicated[1]->GetOutput());
	}
	else mOutput[1]->Initialize();
}


float iSurfaceSerialPipeline::GetMemorySize() const
{
	int i;
	float s, s1;

	s = mIsoSurfaceSource->GetOutput()->GetActualMemorySize();

	for(i=0; i<2; i++)
	{
		s1 = (float)mProbe[i]->GetOutput()->GetActualMemorySize();
		if(s1 < 0.0) s1 = 0.0;
		s += s1;
		s += mDataReplicated[i]->GetMemorySize();
	}

	for(i=0; i<2; i++)
	{
		s1 = (float)mFlipNormalsFilter[i]->GetOutput()->GetActualMemorySize();
		if(s1 < 0.0) s1 = 0.0;
		s += s1;
	}

	s += mSmoothFilter->GetOutput()->GetActualMemorySize();
	s += mReduceFilter->GetOutput()->GetActualMemorySize();
	s += mReduce2Filter->GetOutput()->GetActualMemorySize();
	s += mOptimizeFilter->GetOutput()->GetActualMemorySize();

	s += mSphereSource->GetMemorySize();
	s += mPlaneSource->GetMemorySize();

	return s;
}


iGridDataToPolyDataPipeline* iSurfaceSerialPipeline::Copy() const
{
	iSurfaceSerialPipeline *tmp = mParent->CreateSerialPipeline();	
	if(tmp == 0) return 0;

	//
	//  Update the pipeline
	//
	tmp->UpdateAll();

	return tmp;
}


void iSurfaceSerialPipeline::UpdateAll()
{
	//
	//  Update the pipeline
	//
	this->UpdateMethod();
	this->UpdateIsoSurfacePipeline();
	this->UpdateIsoSurfaceVar();
	this->UpdateIsoSurfaceLevel();
	this->UpdateVarPaint(0);
	this->UpdateVarPaint(1);
	this->UpdateNormalsFlipped();
	this->UpdateProbeSize();
	this->UpdateProbePosition();
	this->UpdatePlaneDirection();
	this->UpdateReplicas();

	this->UpdateSphereResolution();
	this->UpdatePlaneResolution();
}


void iSurfaceSerialPipeline::UpdateIsoSurfacePipeline(bool force)
{ 
	int i;

	if(!force && mParent->GetMethod()!=_SurfaceMethodIsoSurface) return;

	switch(mParent->GetIsoSurfaceReduction())
	{
	case 1:
		{
			mReduceFilter->SetTargetReduction(0.75);
			mReduce2Filter->SetTargetReduction(0.75);
			break;
		}
	case 2:
		{
			mReduceFilter->SetTargetReduction(0.90);
			mReduce2Filter->SetTargetReduction(0.90);
			break;
		}
	case 3:
		{
			mReduceFilter->SetTargetReduction(0.99);
			mReduce2Filter->SetTargetReduction(0.99);
			break;
		}
	}

	mSmoothFilter->SetNumberOfIterations(2<<(2*mParent->GetIsoSurfaceSmoothing()));

	//
	//  Create the pipeline
	//
	vtkPolyData *step1, *step2, *step3;

	if(mParent->GetIsoSurfaceReduction() > 0) 
	{
		step1 = mParent->GetAlternativeIsoSurfaceReductionMethod() ? mReduce2Filter->GetOutput() : mReduceFilter->GetOutput();
	}
	else
	{
		step1 = mGateway->GetOutput();
	}

	mSmoothFilter->SetInput(step1);

	if(mParent->GetIsoSurfaceSmoothing() > 0)
	{
		step2 = mSmoothFilter->GetOutput();
	}
	else
	{
		step2 = step1;
	}

	mOptimizeFilter->SetInput(step2);

	if(mParent->GetIsoSurfaceOptimization())
	{
		step3 = mOptimizeFilter->GetOutput();
	}
	else
	{
		step3 = step2;
	}

	for(i=0; i<2; i++) mProbe[i]->SetInput(step3);

	this->Modified();
}


void iSurfaceSerialPipeline::UpdateIsoSurfaceVar()
{
	mDataConverter->SetCurrentVar(mParent->GetIsoSurfaceVar());
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateIsoSurfaceLevel()
{ 
	mIsoSurfaceSource->SetValue(0,mParent->GetDataIsoSurfaceLevel());
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateVarPaint(int s)
{
	if(s<0 || s>1) return;
	mDataConverterPaint[s]->SetCurrentVar(mParent->GetPaintVar(s));
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateNormalsFlipped()
{ 
	int var = mParent->GetIsoSurfaceVar();
	bool s = mParent->GetNormalsFlipped();

	if(mParent->GetMethod() != _SurfaceMethodIsoSurface) var = -1;

	if((s && var!=mSpecialVar) || (!s && var==mSpecialVar)) 
	{
		mFlipNormalsFilter[0]->ReverseNormalsOn(); 
		mFlipNormalsFilter[1]->ReverseNormalsOff();
		mShiftFilter->SetShift(fabs(mShiftFilter->GetShift()));
	} 
	else 
	{
		mFlipNormalsFilter[0]->ReverseNormalsOff();
		mFlipNormalsFilter[1]->ReverseNormalsOn();
		mShiftFilter->SetShift(-fabs(mShiftFilter->GetShift()));
	}
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateReplicas()
{
	mDataReplicated[0]->Modified();
	mDataReplicated[1]->Modified();
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateMethod()
{
	switch(mParent->GetMethod())
	{
	case _SurfaceMethodIsoSurface:
		{
			this->UpdateIsoSurfacePipeline();
			this->Modified();
			break;
		}
	case _SurfaceMethodSphere:
		{
			mProbe[0]->SetInput(mSphereSource->GetOutput());
			mProbe[1]->SetInput(mSphereSource->GetOutput());
			this->Modified();
			break;
		}
	case _SurfaceMethodPlane:
		{
			mProbe[0]->SetInput(mPlaneSource->GetOutput());
			mProbe[1]->SetInput(mPlaneSource->GetOutput());
			this->Modified();
			break;
		}
	}
	this->UpdateNormalsFlipped();
}


void iSurfaceSerialPipeline::UpdateProbeSize()
{
	mSphereSource->SetRadius(mParent->GetSize());
	this->UpdateSphereResolution();
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateProbePosition()
{
	mSphereSource->SetCenter(mParent->GetPosition());
	mPlaneSource->SetCenter(mParent->GetPosition());
	this->Modified();
}


void iSurfaceSerialPipeline::UpdatePlaneDirection()
{
	const float *n = mParent->GetPlaneDirection();
	mPlaneSource->SetNormal(n);
	this->Modified();
}


void iSurfaceSerialPipeline::UpdateSphereResolution()
{
	int dims[3];
	double spa[3], org[3], dx;

	if(this->GetInput() == 0) return;

	this->GetInput()->GetDimensions(dims);
	this->GetInput()->GetOrigin(org);
	this->GetInput()->GetSpacing(spa);

	dx = spa[0];
	if(dx < spa[1]) dx = spa[1];
	if(dx < spa[2]) dx = spa[2];
	if(dx < 1.0e-3) dx = 1.0e-3;

	float s = mSphereSource->GetRadius();
	int n = round(3.14*s/dx);
	if(n < 6) n = 6;
	mSphereSource->SetResolution(n);
	this->Modified();
}


void iSurfaceSerialPipeline::UpdatePlaneResolution()
{
	int dims[3];
	double spa[3], org[3], dx;

	if(this->GetInput() == 0) return;

	this->GetInput()->GetDimensions(dims);
	this->GetInput()->GetOrigin(org);
	this->GetInput()->GetSpacing(spa);

	dx = spa[0];
	if(dx < spa[1]) dx = spa[1];
	if(dx < spa[2]) dx = spa[2];
	if(dx < 1.0e-3) dx = 1.0e-3;

	float s = 7.0;
	int n = round(s/dx);
	mPlaneSource->SetResolution(n);
	this->Modified();
}
