/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "itextvector.h"


#include "ierror.h"
#include "imath.h"
#include "itransform.h"
#include "iviewmodule.h"

#include <vtkActor2D.h>
#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkTransformPolyDataFilter.h>
#include <vtkViewport.h>
#include <vtkVectorText.h>


iTextVector::iTextVector(const iRenderTool *rv) : iText(rv)
{
	mSource = vtkVectorText::New(); IERROR_ASSERT_NULL_POINTER(mSource);
	mTransform = iTransform::New(); IERROR_ASSERT_NULL_POINTER(mTransform);
	mFilter = vtkTransformPolyDataFilter::New(); IERROR_ASSERT_NULL_POINTER(mFilter);
	mMapper = vtkPolyDataMapper2D::New(); IERROR_ASSERT_NULL_POINTER(mMapper);
	mActor = vtkActor2D::New(); IERROR_ASSERT_NULL_POINTER(mActor);
	mActorShadow = vtkActor2D::New(); IERROR_ASSERT_NULL_POINTER(mActorShadow);
	
	mFilter->SetInput(mSource->GetOutput());
	mFilter->SetTransform(mTransform);
	mMapper->SetInput(mFilter->GetOutput());
	mActor->SetMapper(mMapper);
	mActorShadow->SetMapper(mMapper);

	mActor->GetProperty()->SetPointSize(4.0);
	mActorShadow->GetProperty()->SetPointSize(4.0);

	mActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	mActorShadow->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
}


iTextVector::~iTextVector()
{
	mSource->Delete();
	mTransform->Delete();
	mFilter->Delete();
	mMapper->Delete();
	mActor->Delete();
	mActorShadow->Delete();
}


void iTextVector::GetSize(vtkViewport *viewport, float s[2])
{
	int mag = this->GetRenderingMagnification();

	float cosa = cos(mAngle*0.017453292);
	float sina = sin(mAngle*0.017453292);

	mSource->SetText(mText.ToCharPointer());
	mSource->Update();
	double *b = mSource->GetOutput()->GetBounds();
	
	float w = 0.618*(b[1]-b[0])*mFontSize;
	float h = 0.953*mFontSize;
	float wt = w*fabs(cosa) + h*fabs(sina);
	float ht = h*fabs(cosa) + w*fabs(sina);

	float c[4][2], x[2];

	c[0][0] = c[3][0] = 0.0;
	c[1][0] = c[2][0] = w;
	c[0][1] = c[1][1] = 0.0;
	c[2][1] = c[3][1] = h;

	float dx = 0.0, dy = 0.0;
	int j;
	for(j=0; j<4; j++)
	{
		x[0] = c[j][0]*cosa - c[j][1]*sina;
		x[1] = c[j][0]*sina + c[j][1]*cosa;
		if(x[0] < dx) dx = x[0];
		if(x[1] < dy) dy = x[1];
	}

	mTransform->Identity();
	mTransform->Translate(-dx*mag,-dy*mag,0.0);
	mTransform->RotateZ(mAngle);
	mTransform->Scale(0.6*mag*mFontSize,0.65*mag*mFontSize,1.0);
	mTransform->Translate(-0.17,0.09,0.0);
	mMapper->GetInput()->Update();

	int *ws = viewport->GetSize();
	mSize[0] = s[0] = wt/ws[0]; 
	mSize[1] = s[1] = ht/ws[1];

}


void iTextVector::ReleaseGraphicsResources(vtkWindow *win)
{
	mActor->ReleaseGraphicsResources(win);
	mActorShadow->ReleaseGraphicsResources(win);
	this->iText::ReleaseGraphicsResources(win);
}


int iTextVector::RenderOverlay(vtkViewport* viewport)
{
	mActorShadow->RenderOverlay(viewport);
	mActor->RenderOverlay(viewport);
	return 1;
}


int iTextVector::RenderOpaqueGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam == 0) return 0;

	int *ws = viewport->GetSize();
	float cosa = cos(mAngle*0.017453292);
	float sina = sin(mAngle*0.017453292);

	float pshadow[2];
	pshadow[0] = 1.0/ws[0]*(cosa+sina);
	pshadow[1] = -1.0/ws[1]*(cosa-sina);

	this->GetSize(viewport,mSize);

	int mag = this->GetRenderingMagnification();

	if(mag == 1)
	{
		this->SetActorPosition(viewport);

		mActor->GetProperty()->SetColor(mColor.ToVTK());
		mActorShadow->GetProperty()->SetColor(mColor.Shadow().ToVTK());

		mActor->SetPosition(mActorPos[0],mActorPos[1]);
		mActor->SetPosition2(mSize[0],mSize[1]);
		mActorShadow->SetPosition(mActorPos[0]+pshadow[0],mActorPos[1]+pshadow[1]);
		mActorShadow->SetPosition2(mSize[0],mSize[1]);

	}
	else
	{
		int winij[2];
		this->ComputePositionShiftsUnderMagnification(cam,winij,viewport);
		
		mActor->SetPosition(mag*mActorPos[0]-winij[0],mag*mActorPos[1]-winij[1]);
		mActor->SetPosition2(mag*mSize[0],mag*mSize[1]);
		mActorShadow->SetPosition(mag*(mActorPos[0]+pshadow[0])-winij[0],mag*(mActorPos[1]+pshadow[1])-winij[1]);
		mActorShadow->SetPosition2(mag*mSize[0],mag*mSize[1]);

	}
	//
	//  shadow color
	//
	mActorShadow->RenderOpaqueGeometry(viewport);
	mActor->RenderOpaqueGeometry(viewport);

	return 1;
}
