/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  DataSubject for reading uniform mesh data
//
#ifndef IUNIFORMGRIDDATASUBJECT_H
#define IUNIFORMGRIDDATASUBJECT_H


#include "idatasubject.h"


#include "iarray.h"
#include "ibuffer.h"

#include <vtkObject.h>

class iExpressionParser;
class iUniformScalarsDataHelper;
class iGridDataHelper;

class vtkStructuredPoints;


class iGridDataSubject : public iDataSubject
{

public:

	void SetCellToPointMode(int m);
	inline int GetCellToPointMode() const { return mCellToPointMode; }

	void SetScaledDimension(int v);
	inline int GetScaledDimension() const { return mScaledDim; }

	void SetVoxelLocation(int s);
	inline int GetVoxelLocation() const { return mVoxelLocation; }

	inline int GetDim(int d) const { if (d>=0 && d<3) return mDims[d]; else return 0; }
	virtual void ComputeSpacing();

	static void GetCICInterpolation(bool periodic[3], int dims[3], double org[3], double spa[3],  float pos[3], int ijk1[3], int ijk2[3], double d1[3], double d2[3]);
	static void GetCICInterpolation(bool periodic[3], int dims[3], double org[3], double spa[3], double pos[3], int ijk1[3], int ijk2[3], double d1[3], double d2[3]);

protected:

	iGridDataSubject(iDataReader *r, const iDataType &type);
	virtual ~iGridDataSubject();

	inline float* GetDataPointer() const { return mBuffer; }

	virtual void DataSubjectPackStateBody(iString &s) const;
	virtual void DataSubjectUnPackStateBody(const iString &s);

	virtual void ReadFileBody(const iString &suffix, const iString &fname);
	virtual void FinalizeBody();
	virtual void ShiftDataBody(int d, double dx);
	virtual void Polish();

	virtual void ReadBinFile(const iString &fname);
    virtual void ReadTxtFile(const iString &fname);

	virtual void GetCellDims(int dimCell[3]) const;

	virtual void SetIndicies(int &nvar, int &ncomp) = 0;
	virtual void AssignBinData(int n, vtkIdType ntot, float *d) = 0;
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f) = 0;
	virtual void AttachBuffer() = 0;
	virtual void SetArraysAndComponents() = 0;

	void EraseBuffer();

	int mCellToPointMode, mVoxelLocation, mScaledDim;
	int mNumVars, mDims[3], mNumArrays, mNumComponents;
	const int mMaxDimension;
	iBuffer<float> mVarMin;
	iBuffer<float> mVarMax;

	long mBufferSize;
	float *mBuffer;
	vtkStructuredPoints *mData;
	iGridDataHelper *mHelper;
};


//
//  Concrete classes
//
class iUniformVectorsDataSubject : public iGridDataSubject
{

	friend class iDataReader;
	friend class iUniformScalarsDataHelper;

public:

	static const iObjectType& Type();

	//
	//  Inherited members
	//
	IDATASUBJECT_DECLARE_INHERITED_KEYS;

	virtual const iObjectType& RealType();

protected:

	iUniformVectorsDataSubject(iDataReader *r);

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int n, vtkIdType ntot, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer();
	virtual void SetArraysAndComponents();

	virtual iDataLimits* CreateLimits() const;
};


class iUniformTensorsDataSubject : public iGridDataSubject
{

	friend class iDataReader;

public:

	static const iObjectType& Type();

	//
	//  Inherited members
	//
	IDATASUBJECT_DECLARE_INHERITED_KEYS;

	virtual const iObjectType& RealType();

protected:

	iUniformTensorsDataSubject(iDataReader *r);

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int n, vtkIdType ntot, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer();
	virtual void SetArraysAndComponents();

	virtual iDataLimits* CreateLimits() const;
};


class iUniformScalarsDataSubject : public iGridDataSubject
{

	friend class iDataReader;
	friend class iUniformScalarsDataHelper;

public:

	static const iObjectType& Type();

	//
	//  Inherited members
	//
	IDATASUBJECT_DECLARE_INHERITED_KEYS;

	virtual const iObjectType& RealType();

	void SetCalculatorFunction(const iString &s);
	inline const iString& GetCalculatorFunction() const { return mCalculatorFunction; }

	void SetCalculatorOutputVariable(int n);
	inline int GetCalculatorOutputVariable() const { return mCalculatorOutputVariable; }

protected:

	iUniformScalarsDataSubject(iDataReader *r, iUniformVectorsDataSubject *vs);
	virtual ~iUniformScalarsDataSubject();

    virtual void OperateOnData();

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int n, vtkIdType ntot, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer();
	virtual void SetArraysAndComponents();

	virtual iDataLimits* CreateLimits() const;

	iString mCalculatorFunction;
	int mCalculatorOutputVariable;
	iExpressionParser *mCalculator;

	iUniformVectorsDataSubject *mVectorFieldSubject;
	iUniformScalarsDataHelper *mHelper2;
};

#endif

