/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivectorfieldglyphserialpipeline.h"


#include "idatalimits.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ireplicatedpolydata.h"
#include "ivectorglyphfilter.h"
#include "ivectorfieldviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectreplicator.h"

#include <vtkExtractVOI.h>
#include <vtkImageData.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"


//
// iVectorFieldGlyphSerialPipeline1 class
//
iVectorFieldGlyphSerialPipeline::iVectorFieldGlyphSerialPipeline(iVectorFieldViewObject *s) : iGridDataToPolyDataPipeline(1,s)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mDataReplicated = iReplicatedPolyData::New(mParent->GetReplicator()); IERROR_ASSERT_NULL_POINTER(mDataReplicated);
	mGlyph = iVectorGlyphFilter::New(mParent); IERROR_ASSERT_NULL_POINTER(mGlyph);
	mFilter = vtkExtractVOI::New(); IERROR_ASSERT_NULL_POINTER(mFilter);
	
	mGlyph->SetInput(mFilter->GetOutput());
	mDataReplicated->SetInput(mGlyph->GetOutput());

	this->UpdateAll();
}


iVectorFieldGlyphSerialPipeline::~iVectorFieldGlyphSerialPipeline()
{
	mDataReplicated->Delete();
	mGlyph->Delete();
	mFilter->Delete();
}


void iVectorFieldGlyphSerialPipeline::AttachObservers(bool master)
{
	iEventObserver *obsAbortRender;
	
	if(master)
	{
		obsAbortRender = mParent->GetViewModule()->GetAbortRenderEventObserver();
	}
	else
	{
		obsAbortRender = mParent->GetViewModule()->GetSlaveAbortRenderEventObserver();
	}

	mDataReplicated->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mGlyph->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
}


void iVectorFieldGlyphSerialPipeline::ProduceOutput(vtkImageData *input, vtkPolyData *output)
{
	if(mFilter->GetInput() != input)
	{
		mFilter->SetInput(input);
	}
	//
	//  Needed so that the glyph size is correct
	//
	this->UpdateSize();
	//
	//  Update the pipeline
	//
	mDataReplicated->Update();
	output->ShallowCopy(mDataReplicated->GetOutput());
}


float iVectorFieldGlyphSerialPipeline::GetMemorySize() const
{
	float s = 0.0;
	s += mDataReplicated->GetOutput()->GetActualMemorySize();
	s += mGlyph->GetOutput()->GetActualMemorySize();
	s += mFilter->GetOutput()->GetActualMemorySize();
	return s;
}


iGridDataToPolyDataPipeline* iVectorFieldGlyphSerialPipeline::Copy() const
{
	iVectorFieldGlyphSerialPipeline *tmp = mParent->CreateGlyphSerialPipeline();	
	if(tmp == 0) return 0;

	//
	//  Update the pipeline
	//
	tmp->UpdateAll();

	return tmp;
}


void iVectorFieldGlyphSerialPipeline::ComputeSpacing()
{
	if(this->GetInput() != 0)
	{
		mCurSpacing = this->GetInput()->GetSpacing()[0];
	}
	else
	{
		mCurSpacing = 1.0;
	}
}


void iVectorFieldGlyphSerialPipeline::UpdateAll()
{
	//
	//  Update the pipeline
	//
	this->UpdateSize();
	this->UpdateSampleRate();
	this->UpdateReplicas();
}


void iVectorFieldGlyphSerialPipeline::UpdateSize()
{
	this->ComputeSpacing();
	//
	//  Do not update unless necessary
	//
	if(mParent->CanBeShown())
	{
		double ns = 10.0*mParent->GetGlyphSize()*mParent->GetGlyphSampleRate()*mCurSpacing/this->GetLimits()->GetUpperLimit(mParent->GetDataOffset());
		if(fabs(ns/mGlyph->GetScaleFactor()-1.0) > 1.0e-10)
		{
			mGlyph->SetScaleFactor(ns);
			this->Modified();
		}
	}
}


void iVectorFieldGlyphSerialPipeline::UpdateSampleRate()
{ 
	mFilter->SetSampleRate(mParent->GetGlyphSampleRate(),mParent->GetGlyphSampleRate(),mParent->GetGlyphSampleRate());
	this->Modified();
}


void iVectorFieldGlyphSerialPipeline::UpdateReplicas()
{
	mDataReplicated->Modified();
	this->Modified();
}


void iVectorFieldGlyphSerialPipeline::SyncWithData(int, void *)
{
	this->UpdateSize();
}

