/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivectorfieldstreamlineserialpipeline.h"


#include "idatalimits.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ilinetobandfilter.h"
#include "ilinetotubefilter.h"
#include "ireplicatedpolydata.h"
#include "istreamlinefilter.h"
#include "ivectorfieldviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectreplicator.h"

#include <vtkImageData.h>
#include <vtkPolyData.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"


using namespace iParameter;


//
// iVectorFieldStreamLineSerialPipeline class
//
iVectorFieldStreamLineSerialPipeline::iVectorFieldStreamLineSerialPipeline(iVectorFieldViewObject *s) : iPolyDataToPolyDataPipeline(2,s)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mLine = iStreamLineFilter::New(mParent); IERROR_ASSERT_NULL_POINTER(mLine);
	mTubeFilter = iLineToTubeFilter::New(); IERROR_ASSERT_NULL_POINTER(mTubeFilter);
	mBandFilter = iLineToBandFilter::New(); IERROR_ASSERT_NULL_POINTER(mBandFilter);
	mDataReplicated = iReplicatedPolyData::New(mParent->GetReplicator()); IERROR_ASSERT_NULL_POINTER(mDataReplicated);
	
	mTubeFilter->SetNumberOfSides(6);
	mTubeFilter->SetVaryRadiusToVaryRadiusByScalar();

	mBandFilter->SetDistanceFactor(15.0);
	mBandFilter->SetRuledModeToPointWalk();
	mBandFilter->SetOffset(0);
	mBandFilter->SetOnRatio(2);
	mBandFilter->PassLinesOn();

	this->UpdateAll();
}


iVectorFieldStreamLineSerialPipeline::~iVectorFieldStreamLineSerialPipeline()
{
	mLine->Delete();
	mTubeFilter->Delete();
	mDataReplicated->Delete();
	mBandFilter->Delete();
}


void iVectorFieldStreamLineSerialPipeline::AttachObservers(bool master)
{
	iEventObserver *obsAbortRender;
	
	if(master)
	{
		obsAbortRender = mParent->GetViewModule()->GetAbortRenderEventObserver();
	}
	else
	{
		obsAbortRender = mParent->GetViewModule()->GetSlaveAbortRenderEventObserver();
	}

	mLine->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mTubeFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mDataReplicated->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mBandFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
}


void iVectorFieldStreamLineSerialPipeline::ProduceOutput(vtkPolyData *input, vtkPolyData *output)
{
	if(mLine->GetInput() != input)
	{
		mLine->SetInput(input);
	}
	vtkImageData *gridInput = dynamic_cast<vtkImageData*>(this->GetInputs()[1]);
	if(mLine->GetGridData() != gridInput)
	{
		mLine->SetGridData(gridInput);
	}
	//
	//  Update the pipeline
	//
	mDataReplicated->Update();
	output->ShallowCopy(mDataReplicated->GetOutput());
}


float iVectorFieldStreamLineSerialPipeline::GetMemorySize() const
{
	float s = 0.0;
	s += mLine->GetOutput()->GetActualMemorySize();
	s += mTubeFilter->GetOutput()->GetActualMemorySize();
	s += mDataReplicated->GetOutput()->GetActualMemorySize();
	s += mBandFilter->GetOutput()->GetActualMemorySize();
	return s;
}


iPolyDataToPolyDataPipeline* iVectorFieldStreamLineSerialPipeline::Copy() const
{
	iVectorFieldStreamLineSerialPipeline *tmp = mParent->CreateStreamLineSerialPipeline();	
	if(tmp == 0) return 0;

	//
	//  Update the pipeline
	//
	tmp->UpdateAll();

	return tmp;
}


void iVectorFieldStreamLineSerialPipeline::UpdateAll()
{
	//
	//  Update the pipeline
	//
	this->UpdatePipeline();
	this->UpdateDirection();
	this->UpdateLength();
	this->UpdateQuality();
	this->UpdateTubeSize();
	this->UpdateTubeRangeFactor();
	this->UpdateTubeVariationFactor();
	this->UpdateReplicas();
}


void iVectorFieldStreamLineSerialPipeline::UpdatePipeline()
{
	switch(mParent->GetMethod())
	{
	case _VectorMethodGlyph:
		{
			//
			//  mode=0 is handled by iVectorFieldSerialPipeline1, ignore
			//
			break;
		}
	case _VectorMethodStreamLine:
		{
			mLine->SetSplitLines(false);
			mDataReplicated->SetInput(mLine->GetOutput()); 
			break;
		}
	case _VectorMethodStreamTube:
		{
			mLine->SetSplitLines(false);
			mTubeFilter->SetInput(mLine->GetOutput());
			mDataReplicated->SetInput(mTubeFilter->GetOutput());
			break;
		}
	case _VectorMethodStreamBand:
		{
			mLine->SetSplitLines(true);
			mBandFilter->SetInput(mLine->GetOutput());
			mDataReplicated->SetInput(mBandFilter->GetOutput());
			break;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
		}
	}
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateDirection()
{ 
	mLine->SetDirection(mParent->GetLineDir());
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateLength()
{ 
	mLine->SetLength(2.0*mParent->GetLineLength());
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateQuality()
{ 
	mLine->SetQuality(mParent->GetLineQuality());
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateTubeSize()
{ 
	mTubeFilter->SetRadius(0.005*mParent->GetTubeSize());
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateTubeRangeFactor()
{ 
	mTubeFilter->SetRadiusFactor(mParent->GetTubeRangeFactor());
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::UpdateTubeVariationFactor()
{
	//
	//  Do not update unless necessary
	//
	if(mParent->CanBeShown())
	{
		mLine->SetMinimumSpeed(mParent->GetTubeVariationFactor()*this->GetLimits()->GetUpperLimit(mParent->GetDataOffset()));
		this->Modified();
	}
}


void iVectorFieldStreamLineSerialPipeline::UpdateReplicas()
{
	mDataReplicated->Modified();
	this->Modified();
}


void iVectorFieldStreamLineSerialPipeline::SyncWithData(int, void *)
{
	this->UpdateTubeVariationFactor();
}

