/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivolumeviewobject.h"


#include "iclipplane.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ifunctionmapping.h"
#include "igriddataconverter.h"
#include "ihistogrammaker.h"
#include "ipalette.h"
#include "ipiecewisefunction.h"
#include "ireplicatedgriddata.h"
#include "ireplicatedvolume.h"
#include "iviewmodule.h"
#include "iviewobjectobserver.h"
#include "iviewobjectreplicator.h"
#include "ivolumedataconverter.h"
#include "ivtk.h"

#include <vtkStructuredPoints.h>
#include <vtkVolumeProperty.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkVolumeRayCastMIPFunction.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkVolumeTextureMapper2D.h>
#ifndef IVTK_4
#include <vtkVolumeTextureMapper3D.h>
#endif

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iVolumeViewObject,Volume,w);

IOBJECT_DEFINE_KEY(iVolumeViewObject,BlendMode,bm,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,CompositeMethod,cm,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,InterpolationType,it,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,OpacityFunction,of,Any,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,Palette,p,OffsetInt,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,Var,v,OffsetInt,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,XYResample,rxy,Float,1);
IOBJECT_DEFINE_KEY(iVolumeViewObject,ZResample,rz,Float,1);

//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iVolumeViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_SHADING(iVolumeViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iVolumeViewObject);


//
// iVolumeViewObject class
// 
iVolumeViewObject* iVolumeViewObject::New(iViewModule *vm)
{
	iVolumeViewObject *tmp = iPointerCast<iVolumeViewObject,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	tmp->Reset();
	return tmp;
}


iVolumeViewObject::iVolumeViewObject(iViewModule *vm, const iDataInfo &info) : iViewObject(vm,info,0U), mVar(0), mFunctionMappings(0), mHistogramMakers(0)
{
	int i;

	mName = "iVolumeViewObject";
	mObjectType = _ObjectTypeVolume;

	if(mPrimaryDataInfo->Index(iDataType::UniformScalars()) == -1)
	{
		IERROR_REPORT_FATAL_ERROR("iVolumeViewObject is configured incorrectly.");
	}
	for(i=0; i<mPrimaryDataInfo->Count(); i++) mVar.Add(0);

	mPalette = 0;
	
#ifdef VTK_USE_VOLUMEPRO
	mMethod = _VolumeMethodVolumePro;
#else
	mMethod = _VolumeMethod2DTexture;
#endif
	
	mOF = iPiecewiseFunction::New(0.0,1.0); IERROR_ASSERT_NULL_POINTER(mOF);
	
	mBlendMode = 0;
	mCompositeMethod = 0;
	mInterpolationType = 1; 

	mResXY = mResZ = 1.0;
	mBlendMode = 0;
	
	mVolume = iReplicatedVolume::New(this->GetReplicator()); IERROR_ASSERT_NULL_POINTER(mVolume);
	mVolume->SetPosition(0.0,0.0,0.0);
	
	mVolume->GetProperty()->SetShade(mShading?1:0);
	mVolume->GetProperty()->SetScalarOpacity(mOF->GetVTKFunction());
	mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(mPalette)->GetColorTransferFunction());
	mVolume->GetProperty()->SetInterpolationTypeToLinear();
	this->UpdateMaterialProperties();
	
	mDataConverter = iVolumeDataConverter::New(this); IERROR_ASSERT_NULL_POINTER(mDataConverter);
	mDataReplicated = iReplicatedGridData::New(this); IERROR_ASSERT_NULL_POINTER(mDataReplicated);
	mHistogramMakers.Add(iHistogramMaker::New(this->GetViewModule())); IERROR_ASSERT_NULL_POINTER(mHistogramMakers[0]);
	mFunctionMappings.Add(new iFunctionMapping(mOF,mHistogramMakers[0],0)); IERROR_ASSERT_NULL_POINTER(mFunctionMappings[0]);
	mHistogramConverter = iGridDataConverter::New(this); IERROR_ASSERT_NULL_POINTER(mHistogramConverter);
	
	mDataReplicated->SetInput(mDataConverter->GetOutput());

	mHistogramConverter->SetCurrentVar(this->GetVar());
	mHistogramMakers[0]->SetInput(mHistogramConverter->GetOutput(),0);

	iAbortRenderEventObserver *obsAbortRender = this->GetViewModule()->GetAbortRenderEventObserver();

	mRaycastMapper = vtkVolumeRayCastMapper::New(); IERROR_ASSERT_NULL_POINTER(mRaycastMapper);
	mRaycastMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
//	mRaycastMapper->AutoAdjustSampleDistancesOff();
	mCF = vtkVolumeRayCastCompositeFunction::New(); IERROR_ASSERT_NULL_POINTER(mCF);
	mCF->SetCompositeMethodToInterpolateFirst();
	mMF = vtkVolumeRayCastMIPFunction::New(); IERROR_ASSERT_NULL_POINTER(mMF);

	switch(mBlendMode)
	{
	case 0: { mRaycastMapper->SetVolumeRayCastFunction(mCF); break; }
	case 1: { mRaycastMapper->SetVolumeRayCastFunction(mMF); break; }
	}
	//  mRaycastMapper->SetMinimumImageSampleDistance(0.125);
	//  mRaycastMapper->SetImageSampleDistance(1.0);
	
	m2DTextureMapper = vtkVolumeTextureMapper2D::New(); IERROR_ASSERT_NULL_POINTER(m2DTextureMapper);
	m2DTextureMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

#ifdef IVTK_4
	m3DTextureMapper = 0;
#else
	m3DTextureMapper = vtkVolumeTextureMapper3D::New(); IERROR_ASSERT_NULL_POINTER(m3DTextureMapper);
	m3DTextureMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
#endif

	mDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mDataReplicated->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper = iVolumeProMapper::New(); IERROR_ASSERT_NULL_POINTER(mVolumeProMapper);

	mVolumeProMapper->IntermixIntersectingGeometryOn();
	mVolumeProMapper->UseImageClipperOff();
	mVolumeProMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
#else
	mVolumeProMapper = 0;
#endif

#ifdef IVTK_4
	mRaycastMapper->UseImageClipperOff();
	m2DTextureMapper->UseImageClipperOff();
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->UseImageClipperOff();
#endif
#endif

	m2DTextureMapper->SetInput(mDataReplicated->GetOutput());
#ifndef IVTK_4
	m3DTextureMapper->SetInput(mDataReplicated->GetOutput());
#endif
	mRaycastMapper->SetInput(mDataReplicated->GetOutput());
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetInput(mDataReplicated->GetOutput());
#endif

	//
	//  Add observer to keep information about this object
	//
	mVolume->AddObserver(vtkCommand::UserEvent,mObjectObserver);

	mVolume->SetVisibility(false);
	this->GetViewModule()->AddObject(mVolume);
}


iVolumeViewObject::~iVolumeViewObject()
{
	int i;

	this->GetViewModule()->RemoveObject(mVolume);
	mOF->Delete();
	mVolume->Delete();
	mDataConverter->Delete();
	mDataReplicated->Delete();

	for(i=0; i<mHistogramMakers.Size(); i++) mHistogramMakers[i]->Delete();
	for(i=0; i<mFunctionMappings.Size(); i++) delete mFunctionMappings[i];

	mHistogramConverter->Delete();
	mRaycastMapper->Delete();
	m2DTextureMapper->Delete();
#ifndef IVTK_4
	m3DTextureMapper->Delete();
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->Delete();
#endif
	mCF->Delete();
	mMF->Delete();
}


void iVolumeViewObject::FinishInitialization()
{
	this->SetMethod(mMethod);
}


void iVolumeViewObject::Reset()
{
	vtkImageData *input = iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
	mHistogramConverter->SetScalarInput(input);
	mDataConverter->SetInput(input);
	this->SyncWithData(0,0);

	float r = mResZ;
	mResZ = -1.0;
	this->SetZResample(r);

	//
	//  Un-initialized, if needed
	//
	if(!mInit)
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(false);
		this->GetViewModule()->RemoveObject(mVolume);
		mInit = true;
	}
}


void iVolumeViewObject::ShowClipPlane(bool s)
{
	if(s && !mClipPlaneOn)
	{
		mRaycastMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
		m2DTextureMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#ifndef IVTK_4
		m3DTextureMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
#ifdef VTK_USE_VOLUMEPRO
		mVolumeProMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
		mClipPlaneOn = true;
	} 
	if(!s && mClipPlaneOn)
	{
		mRaycastMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
		m2DTextureMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#ifndef IVTK_4
		m3DTextureMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
#ifdef VTK_USE_VOLUMEPRO
		mVolumeProMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
		mClipPlaneOn = false;
	}
	this->ClearCache();
}


void iVolumeViewObject::ShowColorBars(bool show)
{
	if(!this->IsVisible()) return;
	this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityVolume,this->GetVar(),this->GetActiveDataType(),mPalette,show);
	this->ClearCache();
}


void iVolumeViewObject::SetVar(int v)
{
	if(v>=0 && v<this->GetLimits()->GetNumVars() && v!=this->GetVar())
	{
		this->ShowColorBars(false);
		mVar[mActiveDataTypeIndex] = v;
		this->ShowColorBars(true);
		this->UpdateVar();
		this->SyncWithData(v+1,0);
		this->ClearCache();
	}
}


void iVolumeViewObject::UpdateVar()
{
    mFunctionMappings[mActiveDataTypeIndex]->AttachToLimits(this->GetLimits(),this->GetVar());
	mDataConverter->SetCurrentVar(this->GetVar());
	mHistogramConverter->SetCurrentVar(this->GetVar());
	mHistogramConverter->Update();
}


void iVolumeViewObject::SetPalette(int p)
{ 
	if(p>=0 && p<this->GetViewModule()->GetControlModule()->GetNumberOfPalettes())
	{
		this->ShowColorBars(false);
		mPalette = p;
		this->ShowColorBars(true);
		mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(mPalette)->GetColorTransferFunction());
		this->ClearCache();
	}
}


void iVolumeViewObject::UpdateMaterialProperties()
{
	if(mShading) mVolume->GetProperty()->SetShade(1); else mVolume->GetProperty()->SetShade(0);

	mVolume->GetProperty()->SetAmbient(mAmbient);
	mVolume->GetProperty()->SetDiffuse(mDiffuse);
	mVolume->GetProperty()->SetSpecular(mSpecular);
	mVolume->GetProperty()->SetSpecularPower(mSpecularPower);
	this->ClearCache();
}


void iVolumeViewObject::ShowBody(bool show)
{
	if(show)
	{
		mVolume->SetVisibility(true);
		this->ShowColorBars(true);
		mHistogramConverter->Update();
	} 
	else 
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(false);
	}
}


int iVolumeViewObject::SetMethod(int m)
{
	if(m>=0 && m<__NumVolumeMethods)
	{
		int mo = mMethod;
		mMethod = m;
		switch(m) 
		{
		case _VolumeMethodRayCast: 
			{
				if(!mInit) 
				{
					vtkImageData *input = iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
					double *s = input->GetSpacing();
					mRaycastMapper->SetSampleDistance(s[2]*mResZ);
				}
				mVolume->SetMapper(mRaycastMapper);
				break; 
			}
		case _VolumeMethod2DTexture:
			{
				mVolume->SetMapper(m2DTextureMapper);
				break; 
			}
		case _VolumeMethod3DTexture:
			{
#ifndef IVTK_4
				if(!mInit) 
				{
					vtkImageData *input = iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
					double *s = input->GetSpacing();
					mRaycastMapper->SetSampleDistance(s[2]*mResZ);
				}
				mVolume->SetMapper(m3DTextureMapper);
#else
				mMethod = mo;
#endif
				break; 
			}
		case _VolumeMethodVolumePro: 
			{
#ifdef VTK_USE_VOLUMEPRO
				mVolume->SetMapper(mVolumeProMapper);
#else
				mMethod = mo;
#endif
				break; 
			}
		default: 
			{
				mMethod = mo; 
			}
		}      			
		this->ClearCache();
	}
	return mMethod;
}


bool iVolumeViewObject::HasVolumePro() const
{
#ifdef VTK_USE_VOLUMEPRO
	return true;
#else
	return false;
#endif
}


bool iVolumeViewObject::Has3DTexture() const
{
#ifdef IVTK_4
	return false;
#else
	return m3DTextureMapper->IsRenderSupported(mVolume->GetProperty()) != 0;
#endif
}


void iVolumeViewObject::SetCompositeMethod(int m) 
{
	if(m>=0 && m<2 && m!=mCompositeMethod) 
	{
		mCompositeMethod = m;
		switch(m)
		{
		case 0:
			{ 
				mCF->SetCompositeMethodToInterpolateFirst();
				break;
			}
		case 1:
			{
				mCF->SetCompositeMethodToClassifyFirst();
				break;
			}
		}
		this->ClearCache();
	}
}


void iVolumeViewObject::SetInterpolationType(int m) 
{
	if(m>=0 && m<2 && m!=mInterpolationType) 
	{
		mInterpolationType = m;
		switch(m)
		{
		case 0:
			{
				mVolume->GetProperty()->SetInterpolationTypeToNearest();
				break;
			}
		case 1:
			{
				mVolume->GetProperty()->SetInterpolationTypeToLinear();
				break;
			}
		}
		this->ClearCache();
	}
}


void iVolumeViewObject::SetZResample(float m) 
{
	if(m == mResZ) return;
	
	mResZ = m;

	double s = 0.01;
	if(!mInit)
	{
		vtkImageData *input = iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
		double s = input->GetSpacing()[2];
	}

	mRaycastMapper->SetSampleDistance(s*mResZ);
#ifndef IVTK_4
	m3DTextureMapper->SetSampleDistance(s*mResZ);
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetSuperSamplingFactor(mResXY,mResXY,mResZ);
#endif

	this->ClearCache();
}


void iVolumeViewObject::SetXYResample(float m) 
{
	if(m == mResXY) return;
	
	mResXY = m;
	
	mRaycastMapper->SetImageSampleDistance(mResXY);
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetSuperSamplingFactor(mResXY,mResXY,mResZ);
#endif

	this->ClearCache();
}


void iVolumeViewObject::SetBlendMode(int m) 
{
	if(m==mBlendMode || m<0 || m>1) return;
	
	mBlendMode = m;

	switch(mBlendMode)
	{
	case 0:
		{
			mRaycastMapper->SetVolumeRayCastFunction(mCF);
			break;
		}
	case 1:
		{
			mRaycastMapper->SetVolumeRayCastFunction(mMF);
			break;
		}
	}

#ifndef IVTK_4
	m3DTextureMapper->SetBlendMode(mBlendMode);
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetBlendMode(mBlendMode);
#endif

	this->ClearCache();
}


bool iVolumeViewObject::IsVisible() const
{
	return (mVolume->GetVisibility() == 1);
}


float iVolumeViewObject::GetMemorySize() const
{
	float s = 0.0;
	s += mDataConverter->GetMemorySize();
	s += mDataReplicated->GetMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVolumeViewObject::ViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyVar(),this->GetVar());
	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyPalette(),mPalette);
	this->PackValue(s,KeyBlendMode(),mBlendMode);
	this->PackValue(s,KeyCompositeMethod(),mCompositeMethod);
	this->PackValue(s,KeyInterpolationType(),mInterpolationType);

	this->PackValue(s,KeyXYResample(),mResXY);
	this->PackValue(s,KeyZResample(),mResZ);

	this->PackValuePiecewiseFunction(s,KeyOpacityFunction(),mOF);
}


void iVolumeViewObject::ViewObjectUnPackStateBody(const iString &s)
{
	int i; float f;
		
	if(this->UnPackValue(s,KeyVar(),i)) this->SetVar(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	if(this->UnPackValue(s,KeyPalette(),i)) this->SetPalette(i);
	if(this->UnPackValue(s,KeyBlendMode(),i)) this->SetBlendMode(i);
	if(this->UnPackValue(s,KeyCompositeMethod(),i)) this->SetCompositeMethod(i);
	if(this->UnPackValue(s,KeyInterpolationType(),i)) this->SetInterpolationType(i);
	
	if(this->UnPackValue(s,KeyXYResample(),f)) this->SetXYResample(f);
	if(this->UnPackValue(s,KeyZResample(),f)) this->SetZResample(f);

	this->UnPackValuePiecewiseFunction(s,KeyOpacityFunction(),mOF); // it resets the function only if successfull
}


vtkImageData* iVolumeViewObject::GetMapperInput()
{
	switch(mMethod) 
	{
	case _VolumeMethodRayCast:
		{
			return mRaycastMapper->GetInput();
		}
	case _VolumeMethod2DTexture:
		{
			return m2DTextureMapper->GetInput();
		}
	case _VolumeMethod3DTexture:
		{
#ifndef IVTK_4
			return m2DTextureMapper->GetInput();
#endif
		}
	case _VolumeMethodVolumePro: 
		{
#ifdef VTK_USE_VOLUMEPRO
			return mVolumeProMapper->GetInput();
#endif
		}
	}      			
	
	return 0;
}


iFunctionMapping* iVolumeViewObject::GetOpacityFunction() const
{
	return mFunctionMappings[mActiveDataTypeIndex];
}


bool iVolumeViewObject::CanBeShown() const
{
	return (this->GetVar() < this->GetLimits()->GetNumVars());
}


void iVolumeViewObject::ViewObjectSyncWithData(int, void *)
{
	vtkImageData *input = iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars()));
	double *s = input->GetSpacing();
	mRaycastMapper->SetSampleDistance(s[2]*mResZ);

    mFunctionMappings[mActiveDataTypeIndex]->AttachToLimits(this->GetLimits(),this->GetVar());
}

