/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggmainwindow.h"


#include "ianimator.h"
#include "icontrolmodule.h"
#include "icrosssectionviewobject.h"
#include "idata.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ihelpfactory.h"
#include "iimagefactory.h"
#include "iparallelmanager.h"
#include "iparticlesviewobject.h"
#include "ishell.h"
#include "isurfaceviewobject.h"
#include "isystem.h"
#include "itensorfieldviewobject.h"
#include "itextactor.h"
#include "ivectorfieldviewobject.h"
#include "iversion.h"
#include "iviewmodule.h"
#include "ivolumeviewobject.h"

#include "iggdialoganimatingprogress.h"
#include "iggdialogcommandline.h"
#include "iggdialogdataexplorer.h"
#include "iggdialogeventrecorder.h"
#include "iggdialogfilesetexplorer.h"
#include "iggdialoghelp.h"
#include "iggdialogimagecomposer.h"
#include "iggdialogloadfile.h"
#include "iggdialogpaletteeditor.h"
#include "iggdialogparallelcontroller.h"
#include "iggdialogperformancemeter.h"
#include "iggdialogpickerwindow.h"
#include "iggdialogrenderingprogress.h"
#include "iggdialogscriptdebugger.h"
#include "iggextensionwindow.h"
#include "iggframetopparent.h"
#include "iggrenderwindow.h"
#include "iggpagecrosssection.h"
#include "iggpagedata.h"
#include "iggpageparticles.h"
#include "iggpagesurface.h"
#include "iggpagetensorfield.h"
#include "iggpagevectorfield.h"
#include "iggpageview.h"
#include "iggpagevolume.h"
#include "iggshell.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeyhandler.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetprogressbar.h"
#include "iggwidgetrendermodebutton.h"
#include "iggwidgettext.h"

#include "ibgextensionwindowsubject.h"
#include "ibgframesubject.h"
#include "ibgmainwindowsubject.h"
#include "ibgmenuwindowsubject.h"
#include "ibgrenderwindowsubject.h"
#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwindowhelper.h"

#include <vtkPolyData.h>
#include <vtkRenderWindowCollection.h>
#include <vtkStructuredPoints.h>
#include <vtkVersion.h>

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  templates
//
#include "iarraytemplate.h"


IOBJECT_DEFINE_TYPE(iggMainWindow,MainWindow,-mw,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iggMainWindow,Geometry,g,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,ExtensionGeometry,eg,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,Docked,d,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,InteractorHelp,ih,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,ToolTips,tt,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,OptionsAreGlobal,og,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,WindowUnderFocusCurrent,fc,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,IsIdiosyncratic,ii,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,AttachWindows,aw,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,TabMode,tm,Int,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Theme,t,String,1);


namespace iggMainWindow_Private
{
	//
	//  Menu ids
	//
	const int _MenuFileOpenUniformScalars = 1;
	const int _MenuFileOpenBasicParticles = 2;
	const int _MenuFileOpenUniformVectors = 3;
	const int _MenuFileOpenUniformTensors = 4;
	const int _MenuFileExit = 6;
	const int _MenuFileMax = 100;

	const int _MenuDialogScriptDebugger = 101;
	const int _MenuDialogPaletteEditor = 102;
	const int _MenuDialogPickerWindow = 103;
	const int _MenuDialogEventRecorder = 104;
	const int _MenuDialogParallelController = 105;
	const int _MenuDialogFileSetExplorer = 106;
	const int _MenuDialogDataExplorer = 107;
	const int _MenuDialogImageComposer = 108;
	const int _MenuDialogCommandLine = 109;
	const int _MenuDialogPerformanceMeter = 110;
	const int _MenuDialogMax = 200;

	const int _MenuStyleDockWindow = 201;
	const int _MenuStyleInteractorHelp = 202;
	const int _MenuStyleToolTips = 203;
	const int _MenuStyleTabText = 211;
	const int _MenuStyleTabIcon = 212;
	const int _MenuStyleTabBoth = 213;
	const int _MenuStyleRenderImmediate = 221;
	const int _MenuStyleRenderDelayed = 222;
	const int _MenuStyleRenderResetAll = 223;
	const int _MenuStyleSlidersEditable = 224;
	const int _MenuStyleIsIdiosyncratic = 225;
	const int _MenuStyleAttachWindows = 226;
	const int _MenuStyleDetailLoadedDataInfo = 227;
	const int _MenuStyleTheme = 230;
	const int _MenuStyleMax = 300;

	const int _MenuOptionAntialiasing = 301;
	const int _MenuOptionIfritBox = 311;
	const int _MenuOptionClassicBox = 312;
	const int _MenuOptionHairBox = 313;
	const int _MenuOptionAxesBox = 314;
	const int _MenuOptionFontTypeBitmap = 322;
	const int _MenuOptionFontTypeVector = 323;
	const int _MenuOptionFontSize10 = 331;
	const int _MenuOptionFontSize12 = 332;
	const int _MenuOptionFontSize14 = 333;
	const int _MenuOptionFontSize16 = 334;
	const int _MenuOptionFontSize20 = 335;
	const int _MenuOptionFontSize24 = 336;
	const int _MenuOptionFontSize32 = 337;
	const int _MenuOptionFontSize40 = 338;
	const int _MenuOptionFontSize48 = 339;
	const int _MenuOptionFontSize64 = 340;
	const int _MenuOptionFontSize80 = 341;
	const int _MenuOptionFontSize96 = 342;
	const int _MenuOptionImageFormatPNG = 351;
	const int _MenuOptionImageFormatJPG = 352;
	const int _MenuOptionImageFormatPPM = 353;
	const int _MenuOptionImageFormatBMP = 354;
	const int _MenuOptionImageFormatTIF = 355;
	const int _MenuOptionImageFormatEPS = 356;
	const int _MenuOptionImageZoom001 = 361;
	const int _MenuOptionImageZoom002 = 362;
	const int _MenuOptionImageZoom003 = 363;
	const int _MenuOptionImageZoom004 = 364;
	const int _MenuOptionImageZoom005 = 365;
	const int _MenuOptionImageZoom006 = 366;
	const int _MenuOptionImageZoom008 = 367;
	const int _MenuOptionImageZoom010 = 368;
	const int _MenuOptionImageZoom015 = 369;
	const int _MenuOptionImageZoom020 = 370;
	const int _MenuOptionImageZoom030 = 371;
	const int _MenuOptionImageZoom040 = 372;
	const int _MenuOptionImageZoom050 = 373;
	const int _MenuOptionImageZoom060 = 374;
	const int _MenuOptionImageZoom080 = 375;
	const int _MenuOptionImageZoom100 = 376;
	const int _MenuOptionPostScriptPaperFormatA0 = 401;
	const int _MenuOptionPostScriptPaperFormatA1 = 402;
	const int _MenuOptionPostScriptPaperFormatA2 = 403;
	const int _MenuOptionPostScriptPaperFormatA3 = 404;
	const int _MenuOptionPostScriptPaperFormatA4 = 405;
	const int _MenuOptionPostScriptPaperFormatA5 = 406;
	const int _MenuOptionPostScriptPaperFormatA6 = 407;
	const int _MenuOptionPostScriptPaperFormatA7 = 408;
	const int _MenuOptionPostScriptPaperFormatA8 = 409;
	const int _MenuOptionPostScriptPaperFormatL1 = 410;
	const int _MenuOptionPostScriptPaperFormatL2 = 411;
	const int _MenuOptionPostScriptOrientationPortrait = 421;
	const int _MenuOptionPostScriptOrientationLandscape = 422;
	const int _MenuOptionSettingsGlobal = 498;
	const int _MenuOptionSaveState = 499;
	const int _MenuOptionMax = 500;

	const int _MenuHelpContents = 501;
	const int _MenuHelpAbout = 502;
	const int _MenuHelpMax = 599;

	const int _ToolBarOpenWindowsPage = 801;
	const int _ToolBarMinimizeWindowsPage = 802;
	const int _ToolBarMaximizeWindowsPage = 803;

	const int _ToolBarShowSurface = 901;
	const int _ToolBarShowCrossSection = 902;
	const int _ToolBarShowVolume = 903;
	const int _ToolBarShowParticleSet = 904;
	const int _ToolBarShowVectorField = 905;
	const int _ToolBarShowTensorField = 906;
	const int _ToolBarMax = 999;

	class InteractorHelpTextView : public iggWidgetTextBrowser
	{

	private:

		struct Entry
		{
			const char* Name;
			const char* Body;
			Entry(const char *n, const char *b) : Name(n), Body(b) {}
		};

	public:

		InteractorHelpTextView(iggFrame *parent) : iggWidgetTextBrowser(true,false,parent)
		{
			mShownData = 0;
		}

		virtual void Show(bool s)
		{
			this->UpdateWidget();
			iggWidgetTextBrowser::Show(s);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			static Entry data0[] = 
			{ 
				Entry("U key","dump the current view into an image."),
					Entry("F key","toggle the Full Screen mode on and off."),
					Entry("","(use Style->Show interactor help menu to turn this help off)")
			};

			static Entry dataD[] = 
			{ 
				Entry(   "Trackball/Joystick mode",""),
					Entry("","-------------------------"),
					Entry("Left button","rotate the camera around its focal point."),
					Entry("[Ctrl]+Left button","spin the camera around the Z-axis (axis perpendicular to the screen)."),
					Entry("Middle button","pan (move sideways) the camera."),
					Entry("Right button","zoom in/out."),
					Entry("3 key","toggle into and out of stereo mode."),
					Entry("P key","perform a pick operation."),
					Entry("R key","reset the camera view along the current view direction."),
					Entry("S key","show all solid objects as surfaces."),
					Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataF[] = 
			{ 
				Entry(   "Fly-by mode",""),
					Entry("","-------------"),
					Entry("Left button","forward motion."),
					Entry("Right button","reverse motion."),
					Entry("[Shift] key","accelerator in mouse and key modes."),
					Entry("[Ctrl]+[Shift] keys","causes sidestep instead of steering."),
					Entry("+/- keys","increase/deacrease normal speed.")
			};

			static Entry dataK[] = 
			{ 
				Entry(   "Keyboard mode",""),
					Entry("","---------------"),
					Entry("Arrow keys (or HJKL keys)","rotate the camera around its focal point."),
					Entry("[Ctrl]+Arrow keys (or [Ctrl]+HJKL keys)","spin the camera around the Z-axis (axis perpendicular to the screen)."),
					Entry("[Shift]+Arrow keys (or [Shift]+HJKL keys)","pan (move sideways) the camera."),
					Entry("+/- keys","zoom in/out."),
					Entry("A/Z keys","speed up/slow down the interaction."),
					Entry("3 key","toggle into and out of stereo mode."),
					Entry("P key","perform a pick operation."),
					Entry("R key","reset the camera view along the current view direction."),
					Entry("S key","show all solid objects as surfaces."),
					Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataM[] = 
			{
				Entry(   "Measuring box mode",""),
					Entry("","--------------------"),
					Entry("Left button","rotate the measuring box around its origin."),
					Entry("Middle button","translate the measuring box in/out of screen."),
					Entry("Right button","translate the measuring box along scene axes."),
					Entry("A/Z keys","scale the measuring box down/up."),
					Entry("S/X keys","adjust box opacity."),
					Entry("C key","shift the camera focal point to the box center.")
			};

			static Entry dataC[] = 
			{ 
				Entry(   "Camera path mode",""),
					Entry("","------------------"),
					Entry("Left button","move a handle or the whole path."),
					Entry("Middle button","move the path (if you seem unable to grab the line, change the number of steps slightly)."),
					Entry("[Ctrl]+Middle button","spin the path."),
					Entry("Right button","scale the path.")
			};

			int i, n = 0;
			Entry *data = 0;
			bool b;
			if(this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyInteractorStyle(),i))
			{
				switch(i)
				{
				case _InteractorStyleTrackball:
				case _InteractorStyleJoystick:
					{
						data = dataD;
						n = sizeof(dataD)/sizeof(Entry);
						break;
					}
				case _InteractorStyleFlight:
					{
						data = dataF;
						n = sizeof(dataF)/sizeof(Entry);
						break;
					}
				case _InteractorStyleKeyboard:
					{
						data = dataK;
						n = sizeof(dataK)/sizeof(Entry);
						break;
					}
				}
			}
			if(this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyMeasuringBox(),b) && b)
			{
				data = dataM;
				n = sizeof(dataM)/sizeof(Entry);
			}
			if(this->GetShell()->GetControlModule()->QueryValue(iAnimator::KeyMode(),i) && i==4)
			{
				data = dataC;
				n = sizeof(dataC)/sizeof(Entry);
			}
			if(n == 0)
			{
#ifdef I_CHECK1
				IERROR_REPORT_ERROR("Should not be here");
#endif
				return false;
			}

			if(data != mShownData)
			{
				this->Clear();
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data[i].Name,data[i].Body,iColor(0,0,255));
				}
				n = sizeof(data0)/sizeof(Entry);
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data0[i].Name,data0[i].Body,iColor(0,0,255));
				}
				mShownData = data;
			}
			return iggWidgetTextBrowser::UpdateWidgetBody();
		}

		Entry* mShownData;
	};


	class BusyIndicator : public iggWidgetImageFlipper
	{

	public:

		BusyIndicator(iggFrame *parent) : iggWidgetImageFlipper(parent)
		{
			mSubject->AddImage(*iImageFactory::FindIcon("light_green.png"),true);
			mSubject->AddImage(*iImageFactory::FindIcon("light_red.png"),true);

			this->SetBaloonHelp("Ready/busy indicator","A green light indicates that IFrIT is ready for user input. The red light comes on during the time IFrIT is busy doing some work. User input is blocked while the red light is on.");
		}

		void SetBusy(bool s)
		{
			if(s) mSubject->ShowImage(1); else mSubject->ShowImage(0);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}
	};


	class VisitedFileList : public iggWidget
	{

	public:

		VisitedFileList(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"File:");

			this->SetBaloonHelp("List of visited files","This drop-down box shows the list of recently visited files. A file from the list can be reloaded by simply selecting it.");
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			int i, k, n = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetNumberOfVisitedFiles();
			const iDataReader::VisitedFile *list = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList();

			k = 0;
			do
			{
				mSubject->Clear();
				for(i=n-1; i>=0; i--)
				{
					mSubject->InsertItem(iString((k>0)?"...":"")+list[i].Name.Part(k));
				}
				if(k == 0) k += 4; else k++;
			}
			while(!mSubject->DoesContentFit());

			return true;
		}

		virtual void OnInt1Body(int v)
		{
			int n = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetNumberOfVisitedFiles();
			const iDataReader::VisitedFile *list = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList();

			if(v>=0 && v<n)
			{
				this->GetMainWindow()->LoadData(list[n-1-v].Subject->GetDataType(),list[n-1-v].Name);
			}
		}

		ibgWidgetComboBoxSubject *mSubject;
	};


	class ImageButton : public iggWidgetSimpleButton
	{

	public:

		ImageButton(iggFrame *parent) : iggWidgetSimpleButton("Image",parent)
		{
			this->SetBaloonHelp("Dump an image","Dump the scene in the current visualization window as an image. Image format and zoom factor are set in the Options menu.");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetShell()->GetControlModule()->GetViewModule()->DumpImages();
			this->GetMainWindow()->Block(false);
		}
	};


	class Book : public iggFrameBook
	{
		
	public:
		
		Book(iggFrame *parent) : iggFrameBook(parent,true)
		{
		}

	protected:

		virtual void OnInt1Body(int i)
		{
			if(this->GetMainWindow()->GetExtensionWindow() != 0) this->GetMainWindow()->GetExtensionWindow()->OpenBookPageByIndex(i);
		}
	};


	class DialogAbout : public iggDialog
	{

	public:

		DialogAbout(iggMainWindow *parent) : iggDialog(parent,_DialogModal,0,"About",0,3,true,"Ok")
		{
			int i, n;
			iString s;

			mFrame->AddLine(new iggWidgetTextArea("%b%+IFrIT - Ionization FRont Interactive Tool",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("Version "+iVersion::GetVersion(),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea(iEdition::GetEditionName(),mFrame),3);

			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);

			mFrame->AddLine(new iggWidgetTextArea(iString("VTK version ")+vtkVersion::GetVTKVersion(),mFrame),3);

			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following extensions:",mFrame),3);
			s = iVersion::GetIncludedExtensions();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following shells:",mFrame),3);
			s = iVersion::GetIncludedShells();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("IFrIT is created and maintained by Nick Gnedin",mFrame),3);
		}

		virtual void Show(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::Show(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	class DialogDocking : public iggDialog
	{

	public:

		DialogDocking(iggMainWindow *parent) : iggDialog(parent,_DialogBlocking|_DialogNoTitleBar,0,"Docking...",0,3,false)
		{
			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);
			mFrame->AddLine(new iggWidgetTextArea("%b%+Rearranging windows...",mFrame),3);
		}

		virtual void Show(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::Show(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	//
	//  A small dialog with three line edit to set the axes labels
	//
	class AxesLabelsDialog : public iggDialog
	{

	public:

		AxesLabelsDialog(iggMainWindow *parent) : iggDialog(parent,0U,0,"Axes Labels",0,3,false)
		{
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"X: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,1));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Y: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,1),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,3));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Z: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,4),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,5));
		}
	};

#ifdef I_DEBUG
	class DebugHelperEmphasizeAllWidgetsCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeAllWidgetsCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Show all layouts",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}

		void OnVoid1Body()
		{
			iggWidget::EmphasizeLayouts(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperEmphasizeUnderCursorCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeUnderCursorCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Show under cursor",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}

		void OnVoid1Body()
		{
			ibgMainWindowSubject::SetEmphasizeUnderCursor(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperFlipAllPagesButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperFlipAllPagesButton(iggFrame *sf, iggFrame *parent) : iggWidgetSimpleButton("Flip all pages",parent)
		{
			mStartFrame = sf;
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			if(mStartFrame != 0) mStartFrame->FlipThroughAllChildren();
		}

		iggFrame *mStartFrame;
	};

	class DebugHelperCreateUGButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperCreateUGButton(iggFrame *parent) : iggWidgetSimpleButton("Create User Guide(s)",parent)
		{
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			iHelpFactory::CreateUserGuide(false);
			iHelpFactory::CreateUserGuide(true);
#if defined(I_DEVEL) && defined(_WIN32)
			system("Work\\scripts\\createug.csh");
#endif
		}
	};

	class DebugHelperDialog : public iggDialog
	{

	public:

		DebugHelperDialog(iggFrame *sf, iggMainWindow *parent) : iggDialog(parent,0U,0,"Debug Helper",0,1,false)
		{
			mFrame->AddLine(new DebugHelperEmphasizeAllWidgetsCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperEmphasizeUnderCursorCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperFlipAllPagesButton(sf,mFrame));
			mFrame->AddLine(new DebugHelperCreateUGButton(mFrame));
		}
	};
#endif
};


using namespace iggMainWindow_Private;


//
//  **************************************************************************************
//
//      INITIALIZATION
//
//  **************************************************************************************
//
iggMainWindow::iggMainWindow(iShell *shell) : iggMenuWindow(shell), mThemeList(iString()), mReloadDataInfo(iDataInfo::Null()), mEraseDataInfo(iDataInfo::Null())
{
	mInitialized = mDocked = mDoNotOfferToReload = mDetailLoadedDataInfo = false;
	mInteractorHelp = mToolTips = mOptionsAreGlobal = mWindowUnderFocusCurrent = true;
	mIdiosyncratic = mAttachWindows = true;

	mProgressBarMode = 0;
	mCurrentTheme = -1;

	mStateFileName = "ifrit.ini";

	mInitialGeometry[0] = mInitialGeometry[1] = mInitialGeometry[2] = mInitialGeometry[3] = 0;

	mInMove = false;
	mPrevPos[0] = mPrevPos[1] = -1;

	mMainWindow = this;
}


void iggMainWindow::StartInitialization()
{
	//
	//  MainSubject must be created first
	//
	mMainSubject = iggSubjectFactory::CreateMainWindowSubject(this,3);

	//
	//  Have we created MenuSubject as well?
	//
	if(this->GetSubject() == 0)
	{
		mTwoSubjectsAreTheSame = false;
		this->AttachSubject(iggSubjectFactory::CreateMenuWindowSubject(this,iImageFactory::FindIcon("genie1gui.png"),"Ionization Front Interactive Tool"),3);
	}
	else
	{
		mTwoSubjectsAreTheSame = true;
	}

	//
	//  Top frames
	//
	mBusyIndicatorFrame = new iggFrameTopParent(this->GetShell());
	mVisitedFileListFrame = new iggFrameTopParent(this->GetShell());

	mMainSubject->SetTopParentSubjects(mBusyIndicatorFrame,mVisitedFileListFrame);

	IERROR_ASSERT_NULL_POINTER(mBusyIndicatorFrame->GetSubject());
	IERROR_ASSERT_NULL_POINTER(mVisitedFileListFrame->GetSubject());

	//
	//  Other components
	//
	mProgressBar = new iggWidgetProgressBar(mGlobalFrame,true);

	//
	//  Status bar
	//
	mBusyIndicator = new BusyIndicator(mBusyIndicatorFrame);
	mBusyIndicatorFrame->AddLine(mBusyIndicator);
	mVisitedFileList = new VisitedFileList(mVisitedFileListFrame);
	mVisitedFileListFrame->AddLine(mVisitedFileList);

	mMainSubject->PopulateStatusBar(mBusyIndicatorFrame,mProgressBar->GetSubject(),mVisitedFileListFrame);

	mGlobalFrame->SetPadding(true);

	//
	//  Extension
	//
	mExtensionWindow = new iggExtensionWindow(this);

	//
	//  Base frame
	//
	mBaseFrame = new iggFrameFlip(mGlobalFrame);

	//
	//  Create the book frame but not layout it
	//
	iggFrame *tmpf = new iggFrame(mBaseFrame);
	mLog = new iggWidgetTextBrowser(false,false,mGlobalFrame);
	mLog->SetBaloonHelp("Log window","In this window IFrIT logs information about the data files, non-critical execution error, performance information, etc.");
	
	//
	//  Create dialogs (needed for book pages)
	//
	mDialogAbout = new DialogAbout(this); IERROR_ASSERT_NULL_POINTER(mDialogAbout);
	mDialogAnimatingProgress = new iggDialogAnimatingProgress(this); IERROR_ASSERT_NULL_POINTER(mDialogAnimatingProgress);
	mDialogAxesLabels = new AxesLabelsDialog(this); IERROR_ASSERT_NULL_POINTER(mDialogAxesLabels);
	mDialogCommandLine = new iggDialogCommandLine(this); IERROR_ASSERT_NULL_POINTER(mDialogCommandLine);
	mDialogDataExplorer = new iggDialogDataExplorer(this); IERROR_ASSERT_NULL_POINTER(mDialogDataExplorer);
	mDialogDocking = new DialogDocking(this); IERROR_ASSERT_NULL_POINTER(mDialogDocking);
	mDialogEventRecorder = new iggDialogEventRecorder(this); IERROR_ASSERT_NULL_POINTER(mDialogEventRecorder);
	mDialogFileSetExplorer = new iggDialogFileSetExplorer(this); IERROR_ASSERT_NULL_POINTER(mDialogFileSetExplorer);
	mDialogHelp = new iggDialogHelp(this); IERROR_ASSERT_NULL_POINTER(mDialogHelp);
	mDialogImageComposer = new iggDialogImageComposer(this); IERROR_ASSERT_NULL_POINTER(mDialogImageComposer);
	mDialogLoadFile = new iggDialogLoadFile(this); IERROR_ASSERT_NULL_POINTER(mDialogLoadFile);
	mDialogPaletteEditor = new iggDialogPaletteEditor(this); IERROR_ASSERT_NULL_POINTER(mDialogPaletteEditor);
	mDialogParallelController = new iggDialogParallelController(this); IERROR_ASSERT_NULL_POINTER(mDialogParallelController);
	mDialogPerformanceMeter = new iggDialogPerformanceMeter(this); IERROR_ASSERT_NULL_POINTER(mDialogPerformanceMeter);
	mDialogPickerWindow = new iggDialogPickerWindow(this); IERROR_ASSERT_NULL_POINTER(mDialogPickerWindow);
	mDialogRenderingProgress = 0; // we create this one later, so that it does not pop up during the initialization process
	mDialogScriptDebugger = new iggDialogScriptDebugger(this); IERROR_ASSERT_NULL_POINTER(mDialogScriptDebugger);

	//
	//  Create the book
	//
	mBook = new Book(tmpf);

//	iggPage::SetDelayedConstruction(false);
	mViewPage = new iggPageView(mBook);
	mParticlesPage = new iggPageParticles(mBook);
	mDataPage = new iggPageData(mBook);
//	iggPage::SetDelayedConstruction(true);

	iParticlesViewObject::UseFullState(false); // optimization, since we use iParticleGroup keys

	mBook->AddPage("View",iImageFactory::FindIcon("view.png"),mViewPage);
	mBook->AddPage("Surface",iImageFactory::FindIcon("surf.png"),new iggPageSurface(mBook));
	mBook->AddPage("Cross section",iImageFactory::FindIcon("xsec.png"),new iggPageCrossSection(mBook));
	mBook->AddPage("Volume",iImageFactory::FindIcon("volv.png"),new iggPageVolume(mBook));
	mBook->AddPage("Particles",iImageFactory::FindIcon("part.png"),mParticlesPage);
	mBook->AddPage("Vector field",iImageFactory::FindIcon("vect.png"),new iggPageVectorField(mBook));
	mBook->AddPage("Tensor field",iImageFactory::FindIcon("tens.png"),new iggPageTensorField(mBook));
	mBook->AddPage("Data",iImageFactory::FindIcon("data.png"),mDataPage);

	//
	//  Add additional particle icons
	//
	mExtensionWindow->AddParticleSetIcons();

	//
	//  Layout the book frame
	//
	tmpf->AddLine(mBook);
	mBaseFrame->AddLayer(tmpf);

	//
	//  Interactor help frame
	//
	tmpf = new iggFrame("",mBaseFrame);
	tmpf->AddLine(new iggWidgetTextArea("%b%+Interactor Help",tmpf));
	tmpf->AddLine(new InteractorHelpTextView(tmpf));
	tmpf->SetRowStretch(0,1);
	tmpf->SetRowStretch(1,10);
	mBaseFrame->AddLayer(tmpf);
	mBaseFrame->ShowLayer(0);

	mGlobalFrame->AddLine(mBaseFrame,3);
	mGlobalFrame->AddLine(mLog,3);
	iggFrame *tmp1 = new iggFrame(mGlobalFrame);
	iggFrame *tmp2 = new iggFrame(mGlobalFrame);
	tmp1->AddSpace(1);
	tmp1->AddLine(new ImageButton(tmp1));
	mDataTypeFrame = new iggFrameFlip(tmp2,false);
	tmp2->AddSpace(1);
	tmp2->AddLine(mDataTypeFrame);
	mGlobalFrame->AddLine(tmp1,new iggWidgetTextArea("",mGlobalFrame),tmp2);
	mGlobalFrame->SetRowStretch(0,0);
	mGlobalFrame->SetRowStretch(1,10);
	mGlobalFrame->SetRowStretch(2,0);

	iEdition::ApplySettings(this,this->Type());

	this->ShowToolTips(mToolTips);

#if defined(I_DEBUG) && defined(I_EDITION)
	mDialogDebugHelper = new DebugHelperDialog(mGlobalFrame,this);
#endif

	//
	//  Connect pre-existing RenderWindows
	//
	int i;
	for(i=0; i<this->GetShell()->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		iPointerCast<iggRenderWindow,vtkRenderWindow>(this->GetShell()->GetControlModule()->GetViewModule(i)->GetRenderWindow())->AttachToMainWindow(this);
	}
}


void iggMainWindow::PreShowInitialization()
{
	this->BuildMenus();
	if(mInitialGeometry[2]>0 && mInitialGeometry[3]>0) this->GetSubject()->SetWindowGeometry(mInitialGeometry);
}


void iggMainWindow::PostShowInitialization()
{
	mDialogRenderingProgress = new iggDialogRenderingProgress(this); IERROR_ASSERT_NULL_POINTER(mDialogRenderingProgress);

	int wg[4];
	this->GetSubject()->GetWindowGeometry(wg);
	mPrevPos[0] = wg[0];
	mPrevPos[1] = wg[1];

	mInitialized = true;
}


iggMainWindow::~iggMainWindow()
{
	if(mDocked)
	{
		mMainSubject->RestoreWindowsFromDockedPositions();
	}

	delete mExtensionWindow;
	mExtensionWindow = 0; // it does not exist any more

	if(mDialogAbout != 0) delete mDialogAbout;
	if(mDialogAnimatingProgress != 0) delete mDialogAnimatingProgress;
	if(mDialogAxesLabels != 0) delete mDialogAxesLabels;
	if(mDialogCommandLine != 0) delete mDialogCommandLine;
	if(mDialogDataExplorer != 0) delete mDialogDataExplorer;
	if(mDialogDocking != 0) delete mDialogDocking;
	if(mDialogEventRecorder != 0) delete mDialogEventRecorder;
	if(mDialogFileSetExplorer != 0) delete mDialogFileSetExplorer;
	if(mDialogHelp != 0) delete mDialogHelp;
	if(mDialogImageComposer != 0) delete mDialogImageComposer;
	if(mDialogLoadFile != 0) delete mDialogLoadFile;
	if(mDialogPaletteEditor != 0) delete mDialogPaletteEditor;
	if(mDialogParallelController != 0) delete mDialogParallelController;
	if(mDialogPerformanceMeter != 0) delete mDialogPerformanceMeter;
	if(mDialogPickerWindow != 0) delete mDialogPickerWindow;
	if(mDialogRenderingProgress != 0) delete mDialogRenderingProgress;
	if(mDialogScriptDebugger != 0) delete mDialogScriptDebugger;
#if defined(I_DEBUG) && defined(I_EDITION)
	if(mDialogDebugHelper != 0) delete mDialogDebugHelper;
#endif

	delete mProgressBar;

	delete mGlobalFrame;
	delete mBusyIndicatorFrame;
	delete mVisitedFileListFrame;

	if(!mTwoSubjectsAreTheSame)
	{
		delete mMainSubject;
	}
}


void iggMainWindow::UpdateContents()
{
	mGlobalFrame->UpdateWidget();

	//
	//  Also update widgets that are not children of mGlobalFrame
	//
	mBusyIndicator->UpdateWidget();
	mVisitedFileList->UpdateWidget();

	//
	//  Update some of the dialogs
	//
	if(mDialogDataExplorer != 0) mDialogDataExplorer->UpdateDialog();
	if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->UpdateDialog();
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateDialog();
	if(mDialogPickerWindow != 0) mDialogPickerWindow->UpdateDialog();

	//
	//  Update script debugger
	//
	if(mDialogScriptDebugger != 0) mDialogScriptDebugger->UpdateAll();

	//
	//  Update extension too
	//
	mExtensionWindow->UpdateAll();
}


void iggMainWindow::Register(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.AddUnique(window);
}


void iggMainWindow::UnRegister(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.Remove(window);
}


//
//  **************************************************************************************
//
//      DATA MANIPULATION
//
//  **************************************************************************************
//
void iggMainWindow::LoadData(const iDataType &type, const iString &filename, int mod)
{
	iDataInfo info(type);
	
	mProgressBarMode = 1;
	mDialogLoadFile->LoadData(info,filename,mod);
	mProgressBarMode = 0;
}


void iggMainWindow::AddReloadDataType(const iDataType &type)
{
	mReloadDataInfo += type;
}


void iggMainWindow::ReloadData()
{
	mProgressBarMode = 1;
	mDialogLoadFile->ReloadData(mReloadDataInfo);
	mProgressBarMode = 0;
	mReloadDataInfo.Erase();
}


void iggMainWindow::RequestReloadData(const iDataInfo &info)
{
	int i;

	for(i=0; i<info.Count(); i++) this->AddReloadDataType(info.Type(i));

	if(!mDoNotOfferToReload && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(info))
	{
		int ret = this->PopupWindow("This control only takes effect when the data set is reloaded.",_PopupWindowMessage,"Reload now","Later","Do not ask again");
		if(ret == 0) this->ReloadData();
		if(ret == 2) mDoNotOfferToReload = true;
	}
}


void iggMainWindow::AddEraseDataType(const iDataType &type)
{
	mEraseDataInfo += type;
}


void iggMainWindow::RemoveEraseDataType(const iDataType &type)
{
	mEraseDataInfo -= type;
}


void iggMainWindow::EraseData()
{
	int i;

	this->Block(true);
	
	for(i=0; i<mEraseDataInfo.Count(); i++)
	{
		this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->EraseData(mEraseDataInfo.Type(i));
		this->AfterEraseData(mEraseDataInfo.Type(i));
	}
	
	this->Block(false);
	mEraseDataInfo.Erase();
}


void iggMainWindow::AfterLoadData(const iDataType &type, const iString &)
{
	iDataReader *r = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();

	if(r->GetErrorStatus()->NoError() || r->GetErrorStatus()->GetLevel()<0)  // warning only
	{
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(_RenderModeClones);
		this->LogDataInfo(type,mDetailLoadedDataInfo);

		//
		//  Provide the FileSet info
		//
		if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsSet())
		{
			int m = 0;
			while(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m) != iDataType::Null())
			{
				this->WriteToLog((m==0)?"File Set":"        ",this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m).GetTextName());
				m++;
			}
		}
		mDataPage->UpdateOnDataLoad();

		if(r->GetErrorStatus()->IsError()) this->PopupWindow(r->GetErrorMessage());
	}
	else
	{
		if(!r->GetErrorStatus()->IsAborted()) this->PopupWindow("Error in loading data:\n"+r->GetErrorMessage(),_PopupWindowError);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterReloadData(const iDataType &)
{
	if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorMessage().IsEmpty())
	{
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(_RenderModeClones);
	}
	else
	{
		this->PopupWindow("Reloading of the current file set failed:\n"+this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorMessage(),_PopupWindowWarning);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterEraseData(const iDataType &)
{
	if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
	this->GetShell()->GetControlModule()->Render(_RenderModeClones);

	this->UpdateAll();
}


void iggMainWindow::LogDataInfo(const iDataType &type, bool details)
{
	iDataReader *dr = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();
	iDataLimits *lim = 0;
	iDataSubject *sub = 0;

	iDataExplorer::Info di;
	iDataExplorer *de = 0;
	if(details) de = iDataExplorer::New(this->GetShell()->GetControlModule()->GetViewModule());

	this->WriteToLog("","");

	if(type==iDataType::UniformScalars() && dr->IsThereData(iDataType::UniformScalars()))
	{
		int i, dims[3];

		sub = dr->GetSubject(iDataType::UniformScalars());
		lim = dr->GetLimits(iDataType::UniformScalars());
		vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(dr->GetOutput(iDataType::UniformScalars()));
		data->GetDimensions(dims);

		this->WriteToLog("Uniform Scalars file","");
		this->WriteToLog("Number of variables in the file",iString::FromNumber(lim->GetNumVars()));
		this->WriteToLog("Dimensions",iString::FromNumber(dims[0])+" x "+iString::FromNumber(dims[1])+" x "+iString::FromNumber(dims[2]));

		if(de != 0)
		{
			de->SetActiveDataType(iDataType::UniformScalars());
            for(i=0; i<de->GetNumberOfInputComponents(); i++)
			{
				de->SetInputComponent(i);
				di = de->GetInfo(true);
				this->WriteToLog("Variable "+iString::FromNumber(i+1),"from "+iString::FromNumber(di.Minimum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Minimum)))+" dex) to "+iString::FromNumber(di.Maximum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Maximum)))+" dex)");
			}
		}
	}

	if(type==iDataType::UniformVectors() && dr->IsThereData(iDataType::UniformVectors()))
	{
		int dims[3];

		sub = dr->GetSubject(iDataType::UniformVectors());
		lim = dr->GetLimits(iDataType::UniformVectors());
		vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(dr->GetOutput(iDataType::UniformVectors()));
		data->GetDimensions(dims);

		this->WriteToLog("Uniform Vectors file","");
		this->WriteToLog("Dimensions",iString::FromNumber(dims[0])+" x "+iString::FromNumber(dims[1])+" x "+iString::FromNumber(dims[2]));
		if(de != 0)
		{
			de->SetActiveDataType(iDataType::UniformVectors());
			de->SetInputComponent(0);
			di = de->GetInfo(true);
			this->WriteToLog("Vector amplitude",iString::FromNumber(di.Maximum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Maximum)))+" dex)");
		}
	}

	if(type==iDataType::UniformTensors() && dr->IsThereData(iDataType::UniformTensors()))
	{
		int i, dims[3];

		sub = dr->GetSubject(iDataType::UniformTensors());
		lim = dr->GetLimits(iDataType::UniformTensors());
		vtkStructuredPoints *data = iPointerCast<vtkStructuredPoints,vtkDataSet>(dr->GetOutput(iDataType::UniformTensors()));
		data->GetDimensions(dims);

		this->WriteToLog("Uniform Tensors file","");
		this->WriteToLog("Dimensions",iString::FromNumber(dims[0])+" x "+iString::FromNumber(dims[1])+" x "+iString::FromNumber(dims[2]));
		if(de != 0)
		{
			de->SetActiveDataType(iDataType::UniformTensors());
            for(i=0; i<3; i++)
			{
				de->SetInputComponent(i);
				di = de->GetInfo(true);
				this->WriteToLog("Eigenvalue "+iString::FromNumber(i+1),"from "+iString::FromNumber(di.Minimum)+" to "+iString::FromNumber(di.Maximum));
			}
		}
	}

	if(type==iDataType::BasicParticles() && dr->IsThereData(iDataType::BasicParticles()))
	{
		int i;

		sub = dr->GetSubject(iDataType::BasicParticles());
		lim = dr->GetLimits(iDataType::BasicParticles());
		vtkPolyData *data = iPointerCast<vtkPolyData,vtkDataSet>(dr->GetOutput(iDataType::BasicParticles()));

		this->WriteToLog("Basic Particles file","");
		this->WriteToLog("Number of particles",iString::FromNumber(long(data->GetNumberOfPoints())));
		this->WriteToLog("Number of attributes",iString::FromNumber(lim->GetNumVars()));
		if(de != 0)
		{
			de->SetActiveDataType(iDataType::BasicParticles());
            for(i=0; i<de->GetNumberOfInputComponents(); i++)
			{
				de->SetInputComponent(i);
				di = de->GetInfo(true);
				this->WriteToLog("Attribute "+iString::FromNumber(i+1),"from "+iString::FromNumber(di.Minimum)+" to "+iString::FromNumber(di.Maximum));
			}
		}
	}

	mExtensionWindow->LogDataInfo(type,details);

	if(sub!=0 && sub->IsBoxPeriodic())
	{
		iString ws;
		if(sub->IsDirectionPeriodic(0)) ws += "X, ";
		if(sub->IsDirectionPeriodic(1)) ws += "Y, ";
		if(sub->IsDirectionPeriodic(2)) ws += "Z";
		this->WriteToLog("Data periodic in",ws);
	}

	if(de != 0) de->Delete();
}


//
//  **************************************************************************************
//
//      MISC FUNCTIONS
//
//  **************************************************************************************
//
iggWidgetProgressBar* iggMainWindow::GetProgressBar() const
{
	switch(mProgressBarMode)
	{
	case 1:
		{
			return mDialogLoadFile->GetProgressBar();
		}
	case 0:
	default:
		{
			return mProgressBar;
		}
	}
}


int iggMainWindow::PopupWindow(const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(mGlobalFrame,text,type,b0,b1,b2);
}


int iggMainWindow::PopupWindow(const iggFrame *parent, const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(parent,text,type,b0,b1,b2);
}


void iggMainWindow::ShowToolTips(bool s)
{
	mToolTips = s;
	mMainSubject->ShowToolTips(s);
	this->ClearCache();
}


void iggMainWindow::ShowInteractorHelp(bool s, iViewModule* /*vm*/)
{
	mBaseFrame->ShowLayer(s?1:0);
}


void iggMainWindow::Block(bool s)
{
	iPointerCast<BusyIndicator,iggWidget>(mBusyIndicator)->SetBusy(s);
	ibgWindowSubject::Block(s);
}


bool iggMainWindow::IsExitAllowed()
{
	return this->AskForConfirmation("Are you sure you want to exit IFrIT?","Exit");
}


void iggMainWindow::WriteToLog(const iString &prefix, const iString &message, const iColor &color)
{
	mLog->AppendTextLine(prefix,message,color);

	if(mLog->GetNumberOfLines() > 1000)
	{
		while(mLog->GetNumberOfLines() > 700) mLog->RemoveLine(0);
	}
}


void iggMainWindow::ClearLog()
{
	mLog->Clear();
}


void iggMainWindow::OpenBookPage(int n)
{
	mBook->OpenPage(n);
}


void iggMainWindow::SetTabMode(int m)
{
	switch(m)
	{
	case 0:
		{
			mBook->SetTabMode(_BookTitleOnly);
			break;
		}
	case 1:
		{
			mBook->SetTabMode(_BookImageOnly);
			break;
		}
	case 2:
		{
			mBook->SetTabMode(_BookTitleAndImage);
			break;
		}
	}
	if(mExtensionWindow != 0) mExtensionWindow->SetTabMode(m);
	this->ClearCache();
}


void iggMainWindow::ShowAll(bool s)
{
	this->GetSubject()->Show(s);
	if(!mDocked)
	{
		mExtensionWindow->GetSubject()->Show(s);
		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->Show(s);
		}
	}
}


void iggMainWindow::DockWindows(bool s, bool show)
{
	if(s == mDocked) return;
	
	this->ClearCache();

	if(show) mDialogDocking->Show(true);

	if(s)
	{
		mDocked = true;

		//
		//  Save all geometries
		//
		this->GetSubject()->GetHelper()->SaveWindowGeometry();
		mExtensionWindow->GetSubject()->GetHelper()->SaveWindowGeometry();

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->SaveWindowGeometry();
		}

		mGlobalFrame->SetPadding(false);
		mExtensionWindow->GetSubject()->SetPadding(false);

		mMainSubject->PlaceWindowsInDockedPositions();
	} 
	else 
	{
		mMainSubject->RestoreWindowsFromDockedPositions();

		mGlobalFrame->SetPadding(true);
		mExtensionWindow->GetSubject()->SetPadding(true);

		this->GetSubject()->GetHelper()->RestoreWindowGeometry();
		this->GetSubject()->Show(true);
		mExtensionWindow->GetSubject()->GetHelper()->RestoreWindowGeometry();
		mExtensionWindow->GetSubject()->Show(true);

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->RestoreWindowGeometry();
			w->GetSubject()->Show(true);
		}

		mDocked = false;
	}

	FOR_EVERY_RENDER_WINDOW(w)
	{
		w->Render();
	}

	if(show) mDialogDocking->Show(false);
}


void iggMainWindow::PackStateBody(iString &s) const
{
	int wg[4];

	this->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyGeometry(),wg,4);

	mExtensionWindow->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyExtensionGeometry(),wg,4);

	this->PackValue(s,KeyDocked(),mDocked);
	this->PackValue(s,KeyInteractorHelp(),mInteractorHelp);
	this->PackValue(s,KeyToolTips(),mToolTips);
	this->PackValue(s,KeyOptionsAreGlobal(),mOptionsAreGlobal);
	this->PackValue(s,KeyWindowUnderFocusCurrent(),mWindowUnderFocusCurrent);
	this->PackValue(s,KeyIsIdiosyncratic(),mIdiosyncratic);
	this->PackValue(s,KeyAttachWindows(),mAttachWindows);
	this->PackValue(s,KeyTabMode(),mBook->GetTabMode());
	this->PackValue(s,KeyTheme(),(mCurrentTheme>=0 && mCurrentTheme<mThemeList.Size())?mThemeList[mCurrentTheme]:"");
}


void iggMainWindow::UnPackStateBody(const iString &s)
{
	int i, wg[4]; bool b; iString t;

	if(this->UnPackValue(s,KeyGeometry(),wg,4))
	{
		if(mInitialized)
		{
			this->GetSubject()->SetWindowGeometry(wg);
			this->ClearCache();
		}
		else
		{
			for(i=0; i<4; i++) mInitialGeometry[i] = wg[i];
		}
	}

	if(this->UnPackValue(s,KeyExtensionGeometry(),wg,4))
	{
		mExtensionWindow->GetSubject()->SetWindowGeometry(wg);
		this->ClearCache();
	}

	if(this->UnPackValue(s,KeyDocked(),b)) this->DockWindows(b,false);
	if(this->UnPackValue(s,KeyInteractorHelp(),b))
	{
		mInteractorHelp = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyToolTips(),b)) this->ShowToolTips(b);
	if(this->UnPackValue(s,KeyOptionsAreGlobal(),b))
	{
		mOptionsAreGlobal = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyWindowUnderFocusCurrent(),b))
	{
		mWindowUnderFocusCurrent = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyIsIdiosyncratic(),b))
	{
		mIdiosyncratic = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyAttachWindows(),b))
	{
		mAttachWindows = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyTabMode(),i))	this->SetTabMode(i);
	if(this->UnPackValue(s,KeyTheme(),t))
	{
		for(i=0; i<mThemeList.Size(); i++)
		{
			if(t == mThemeList[i])
			{
				this->SetTheme(i);
				break;
			}
		}
	}
}


bool iggMainWindow::AskForConfirmation(const iString &message, const char *action)
{
	return !mIdiosyncratic || this->PopupWindow(message,_PopupWindowMessage,action,"Cancel")==0;
}


void iggMainWindow::UpdateOnPick()
{
	mExtensionWindow->UpdateOnPick();
}


void iggMainWindow::UpdateMarkerWidgets()
{
	if(mViewPage != 0) mViewPage->UpdateMarkerWidgets();
}


void iggMainWindow::UpdateParticleSetWidgets(const iImage *icon)
{
	if(icon != 0)
	{
		mBook->ChangeIcon(4,icon);
	}
	if(mParticlesPage != 0)
	{
		if(icon != 0)
		{
			mParticlesPage->GetBook()->ChangeIcon(-1,icon);
		}
		mParticlesPage->UpdateWidget();
	}
	if(icon != 0)
	{
		this->GetSubject()->SetToolBarIcon(_ToolBarShowParticleSet,*icon);
	}
	this->GetSubject()->UpdateMenus();

	mExtensionWindow->UpdateParticleSetWidgets(icon);
}


void iggMainWindow::AddTheme(const iString &name)
{
	mThemeList.Add(name);
}


void iggMainWindow::SetTheme(int n)
{
	if(mCurrentTheme!=n && n>=0 && n<mThemeList.Size())
	{
		mCurrentTheme = n;
		mMainSubject->SetTheme(mThemeList[n]);
		this->ClearCache();
	}
}


void iggMainWindow::ProcessEvents(bool sync) const
{
	iPointerCast<iggShell,iShell>(this->GetShell())->ProcessEvents(sync);
}


void iggMainWindow::OnRenderWindowMove(int /*wn*/)
{
}


void iggMainWindow::OnRenderWindowResize(int /*wn*/)
{
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateView();
}


void iggMainWindow::OnRenderWindowFocusIn(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mWindowUnderFocusCurrent && wn>=0)
	{
		if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(wn)) this->UpdateAll();
	}

//	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetView(wn));
}


void iggMainWindow::OnRenderWindowFocusOut(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::OnRenderWindowEnter(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetViewModule(wn));
}


void iggMainWindow::OnRenderWindowLeave(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::DisplayWindowsAsIcons()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++)
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsIcon();
		}
	}
}


void iggMainWindow::DisplayWindowsAsWindows()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++) 
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsWindow();
		}
	}
}


void iggMainWindow::MoveWindows(bool all)
{
	if(mInMove) return; // ignore rapidly arriving events
	mInMove = true;

	int pos[2], wg[4];

	this->GetSubject()->GetWindowGeometry(wg);
	pos[0] = wg[0];
	pos[1] = wg[1];

	if(all && mInitialized && !mDocked && mAttachWindows) // do not move the first render window
	{
		int i, dp[2];
		dp[0] = pos[0] - mPrevPos[0];
		dp[1] = pos[1] - mPrevPos[1];
		if(dp[0]!=0 || dp[1]!=0)
		{
			ibgWindowSubject::Block(true); // needed to disable rendering while moving
			for(i=0; i<mWindowList.Size(); i++) if(mWindowList[i]->IsVisible())
			{
				mWindowList[i]->GetWindowGeometry(wg);
				wg[0] += dp[0];
				wg[1] += dp[1];
				mWindowList[i]->SetWindowGeometry(wg);
			}
			ibgWindowSubject::Block(false);
		}
	}

	mPrevPos[0] = pos[0];
	mPrevPos[1] = pos[1];

	mInMove = false;
}


void iggMainWindow::Exit()
{
	if(this->IsExitAllowed())
	{
		if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(false);
		this->GetShell()->Exit();
	}
}



//
//  **************************************************************************************
//
//      MENU INTERACTION
//
//  **************************************************************************************
//
void iggMainWindow::BuildMenus()
{
	//
	//  1. File menu
	//
	this->GetSubject()->BeginMenu("&File",false);
	{
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformScalars,"Open Uniform&Scalars data file",iImageFactory::FindIcon("fileopenmesh.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenBasicParticles,"Open Basic&Particles data file",iImageFactory::FindIcon("fileopenpart.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformVectors,"Open Uniform&Vectors data file",iImageFactory::FindIcon("fileopenvect.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformTensors,"Open Uniform&Tensors data file",iImageFactory::FindIcon("fileopentens.png"),"",false,false,&iViewModule::KeyNoClone());

		mExtensionWindow->PopulateFileMenu();

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(_MenuFileExit,"&Exit",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  2. Dialog menu
	//
	this->GetSubject()->BeginMenu("&Tools",false);
	{
		if(mDialogScriptDebugger != 0) this->GetSubject()->AddMenuItem(_MenuDialogScriptDebugger,"&Animation Script Debugger",iImageFactory::FindIcon("debug.png"),"Ctrl+A",false,false);
		if(mDialogPaletteEditor != 0) this->GetSubject()->AddMenuItem(_MenuDialogPaletteEditor,"Palette &Editor",iImageFactory::FindIcon("paled.png"),"Ctrl+E",false,false);
		if(mDialogPickerWindow != 0) this->GetSubject()->AddMenuItem(_MenuDialogPickerWindow,"&Picker Window",iImageFactory::FindIcon("picks.png"),"Ctrl+P",false,false);
		if(mDialogEventRecorder != 0) this->GetSubject()->AddMenuItem(_MenuDialogEventRecorder,"Event &Recorder",iImageFactory::FindIcon("recorder.png"),"Ctrl+R",false,false);
		if(mDialogFileSetExplorer != 0) this->GetSubject()->AddMenuItem(_MenuDialogFileSetExplorer,"&File Set Explorer",iImageFactory::FindIcon("setexp.png"),"Ctrl+F",false,false,&iDataReader::KeyIsSet());
		mExtensionWindow->PopulateLocalDialogMenu();
		this->GetSubject()->AddMenuSeparator();
		if(mDialogPerformanceMeter != 0) this->GetSubject()->AddMenuItem(_MenuDialogPerformanceMeter,"Performance &Meter",iImageFactory::FindIcon("perf.png"),"Ctrl+M",false,false);
		if(mDialogParallelController!=0 && this->GetSubject()->GetShell()->GetControlModule()->GetParallelManager()->GetMaxNumberOfProcessors()>1) this->GetSubject()->AddMenuItem(_MenuDialogParallelController,"Parallel &Controller",iImageFactory::FindIcon("parallel.png"),"Ctrl+C",false,false);
		if(mDialogDataExplorer != 0) this->GetSubject()->AddMenuItem(_MenuDialogDataExplorer,"&Data Explorer",iImageFactory::FindIcon("dataexp.png"),"Ctrl+D",false,false);
		if(mDialogImageComposer != 0) this->GetSubject()->AddMenuItem(_MenuDialogImageComposer,"&Image Composer",iImageFactory::FindIcon("imcomp.png"),"Ctrl+I",false,false);
		if(mDialogCommandLine != 0) this->GetSubject()->AddMenuItem(_MenuDialogCommandLine,"Command I&nterpreter",iImageFactory::FindIcon("comline.png"),"Ctrl+N",false,false);
		mExtensionWindow->PopulateGlobalDialogMenu();
	}
	this->GetSubject()->EndMenu();
	//
	//  3. Style menu
	//
	this->GetSubject()->BeginMenu("&Style",false);
	{
		this->GetSubject()->AddMenuItem(_MenuStyleInteractorHelp,"Show &interactor help",0,"",true,mInteractorHelp);
		this->GetSubject()->AddMenuItem(_MenuStyleToolTips,"Show t&ooltips",0,"",true,mToolTips);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->BeginMenu("&Tab style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleTabText,"&Text only",0,"",true,mBook->GetTabMode()==_BookTitleOnly);
			this->GetSubject()->AddMenuItem(_MenuStyleTabIcon,"&Icon only",0,"",true,mBook->GetTabMode()==_BookImageOnly);
			this->GetSubject()->AddMenuItem(_MenuStyleTabBoth,"Icon &and text",0,"",true,mBook->GetTabMode()==_BookTitleAndImage);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Slider render style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleRenderImmediate,"&Immediate render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==_RenderModeImmediate);
			this->GetSubject()->AddMenuItem(_MenuStyleRenderDelayed,"&Delayed render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==_RenderModeDelayed);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(_MenuStyleRenderResetAll,"&Reset all sliders to global mode",0,"",false,false);

		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->BeginMenu("&Advanced",false);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleDockWindow,"&Dock windows",0,"",true,mDocked);
			this->GetSubject()->AddMenuItem(_MenuStyleSlidersEditable,"&Sliders are editable",0,"",true,false);
			this->GetSubject()->AddMenuItem(_MenuStyleIsIdiosyncratic,"&Ask for confirmations",0,"",true,mIdiosyncratic);
			this->GetSubject()->AddMenuItem(_MenuStyleAttachWindows,"&Move windows together",0,"",true,mAttachWindows);
			this->GetSubject()->AddMenuItem(_MenuStyleDetailLoadedDataInfo,"&Print detail info for loaded data",0,"",true,mDetailLoadedDataInfo);
		}
		this->GetSubject()->EndMenu();
		this->GetSubject()->BeginMenu("The&me",true);
		{
			int i;
			for(i=0; i<mThemeList.Size(); i++) this->GetSubject()->AddMenuItem(_MenuStyleTheme+i,mThemeList[i],0,"",true,false);
		}
		this->GetSubject()->EndMenu();
	}
	this->GetSubject()->EndMenu();
	//
	//  4. Options menu
	//
	this->GetSubject()->BeginMenu("&Options",false);
	{
		this->GetSubject()->AddMenuItem(_MenuOptionAntialiasing,"Use antialiasing",0,"Shift+Ctrl+Z",true,false,0,false,&iViewModule::KeyAntialiasing());

		this->GetSubject()->BeginMenu("&Bounding box style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuOptionIfritBox,"&Ifrit style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeDefault);
			this->GetSubject()->AddMenuItem(_MenuOptionClassicBox,"&Classic style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeClassic);
			this->GetSubject()->AddMenuItem(_MenuOptionHairBox,"&Hair-thin style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeHairThin);
			this->GetSubject()->AddMenuItem(_MenuOptionAxesBox,"&Axes style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeAxes);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Fonts",false);
		{
			this->GetSubject()->BeginMenu("Font &type",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeBitmap,"&Bitmap",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeBitmap);
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeVector,"&Vector",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeVector);
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Font &size",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize10,"10",0,"",true,false,0,false,&iViewModule::KeyFontSize(),10);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize12,"12",0,"",true,false,0,false,&iViewModule::KeyFontSize(),12);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize14,"14",0,"",true,false,0,false,&iViewModule::KeyFontSize(),14);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize16,"16",0,"",true,false,0,false,&iViewModule::KeyFontSize(),16);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize20,"20",0,"",true,false,0,false,&iViewModule::KeyFontSize(),20);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize24,"24",0,"",true,false,0,false,&iViewModule::KeyFontSize(),24);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize32,"32",0,"",true,false,0,false,&iViewModule::KeyFontSize(),32);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize40,"40",0,"",true,false,0,false,&iViewModule::KeyFontSize(),40);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize48,"48",0,"",true,false,0,false,&iViewModule::KeyFontSize(),48);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize64,"64",0,"",true,false,0,false,&iViewModule::KeyFontSize(),64);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize80,"80",0,"",true,false,0,false,&iViewModule::KeyFontSize(),80);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSize96,"96",0,"",true,false,0,false,&iViewModule::KeyFontSize(),96);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Images",false);
		{
			this->GetSubject()->BeginMenu("Image &format",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatPNG,"PNG (&Portable Network Graphics)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatPNG);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatJPG,"JPG (&Joint Photographic Experts Group)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatJPG);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatPPM,"PPM (Portable Pi&xmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatPNM);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatBMP,"BMP (Window &Bitmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatBMP);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatTIF,"TIF (&Tag Image File Format)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatTIF);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatEPS,"EPS (&Encapsulated PostScript)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatEPS);

				this->GetSubject()->BeginMenu("  Postscript paper format",true);
				{
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA0,"A0",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),0);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA1,"A1",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),1);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA2,"A2",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),2);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA3,"A3",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),3);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA4,"A4",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),4);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA5,"A5",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),5);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA6,"A6",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),6);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA7,"A7",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),7);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA8,"A8",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),8);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatL1,"Letter",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),9);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatL2,"10x17",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),10);
				}
				this->GetSubject()->EndMenu();

				this->GetSubject()->BeginMenu("  Postscript orientation",true);
				{
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptOrientationPortrait,"Portrait",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),0);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptOrientationLandscape,"Landscape",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),1);
				}
				this->GetSubject()->EndMenu();
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Image &zoom",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom001,"x 1",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),1);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom002,"x 2",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),2);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom003,"x 3",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),3);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom004,"x 4",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),4);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom005,"x 5",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),5);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom006,"x 6",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),6);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom008,"x 8",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),8);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom010,"x 10",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),10);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom015,"x 15",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),15);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom020,"x 20",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),20);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom030,"x 30",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),30);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom040,"x 40",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),40);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom050,"x 50",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),50);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom060,"x 60",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),60);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom080,"x 80",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),80);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom100,"x 100",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),100);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(_MenuOptionSettingsGlobal,"Options apply to all windows",0,"",true,mOptionsAreGlobal);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(_MenuOptionSaveState,"Save state",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  5. Help menu
	//
	this->GetSubject()->BeginMenu("&Help",false);
	{
		this->GetSubject()->AddMenuItem(_MenuHelpContents,"Contents",0,"F1",false,false);
#if defined(I_DEBUG) && defined(I_EDITION)
		this->GetSubject()->AddMenuItem(_MenuHelpMax,"Debug Helper",0,"",false,false);
#endif
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuHelpAbout,"About IFrIT",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	this->GetSubject()->CompleteMenu();

	//
	//  Build a tool bar
	//
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformScalars,"Open UniformScalars data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenBasicParticles,"Open BasicParticles data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformVectors,"Open UniformVectors data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformTensors,"Open UniformTensors data file");
	mExtensionWindow->PopulateFileToolBar();
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_ToolBarShowSurface,"Show surface",iImageFactory::FindIcon("surf.png"),true,&iSurfaceViewObject::KeyReady(),false,&iSurfaceViewObject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowCrossSection,"Show cross section",iImageFactory::FindIcon("xsec.png"),true,&iCrossSectionViewObject::KeyReady(),false,&iCrossSectionViewObject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowVolume,"Show volume",iImageFactory::FindIcon("volv.png"),true,&iVolumeViewObject::KeyReady(),false,&iVolumeViewObject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowParticleSet,"Show particles",iImageFactory::FindIcon("part.png"),true,&iParticlesViewObject::KeyReady(),false,&iParticlesViewObject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowVectorField,"Show vector field",iImageFactory::FindIcon("vect.png"),true,&iVectorFieldViewObject::KeyReady(),false,&iVectorFieldViewObject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowTensorField,"Show tensor field",iImageFactory::FindIcon("tens.png"),true,&iTensorFieldViewObject::KeyReady(),false,&iTensorFieldViewObject::KeyVisible());
	mExtensionWindow->PopulateShowToolBar();
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_ToolBarOpenWindowsPage,"Switch to page with multiple windows controls",iImageFactory::FindIcon("wins.png"),false);
	this->GetSubject()->AddToolBarButton(_ToolBarMinimizeWindowsPage,"Minimize all windows",iImageFactory::FindIcon("minimize.png"),false);
	this->GetSubject()->AddToolBarButton(_ToolBarMaximizeWindowsPage,"Restore all windows to normal size",iImageFactory::FindIcon("maximize.png"),false);

	this->GetSubject()->AddToolBarSeparator();
	if(mDialogPickerWindow != 0) this->GetSubject()->AddToolBarButton(_MenuDialogPickerWindow,"Open picker window");
	if(mDialogFileSetExplorer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogFileSetExplorer,"Open file set explorer window");
	if(mDialogDataExplorer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogDataExplorer,"Open data explorer window");
	if(mDialogImageComposer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogImageComposer,"Open image composer window");
}


void iggMainWindow::OnMenuBody(int id, bool on)
{
	iString fname;

	if(id < 0)
	{
		IERROR_REPORT_ERROR("Invalid menu item id.");
	}

	if(id <= _MenuFileMax)
	{
		switch(id)
		{
		case _MenuFileOpenUniformScalars:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iDataType::UniformScalars());
				if(ws.IsEmpty()) ws = this->GetShell()->GetControlModule()->GetEnvironment(_EnvironmentScalarFieldData);
				fname = this->GetFileName("Choose a file",ws,"Scalar field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iDataType::UniformScalars(),fname);
				break;
			}
		case _MenuFileOpenUniformVectors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iDataType::UniformVectors());
				if(ws.IsEmpty()) ws = this->GetShell()->GetControlModule()->GetEnvironment(_EnvironmentVectorFieldData);
				fname = this->GetFileName("Choose a file",ws,"Vector field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iDataType::UniformVectors(),fname);
				break;
			}
		case _MenuFileOpenUniformTensors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iDataType::UniformTensors());
				if(ws.IsEmpty()) ws = this->GetShell()->GetControlModule()->GetEnvironment(_EnvironmentTensorFieldData);
				fname = this->GetFileName("Choose a file",ws,"Tensor field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iDataType::UniformTensors(),fname);
				break;
			}
		case _MenuFileOpenBasicParticles:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iDataType::BasicParticles());
				if(ws.IsEmpty()) ws = this->GetShell()->GetControlModule()->GetEnvironment(_EnvironmentParticleSetData);
				fname = this->GetFileName("Choose a file",ws,"Particle set files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iDataType::BasicParticles(),fname);
				break;
			}
		case _MenuFileExit:
			{
				if(this->IsExitAllowed()) this->GetShell()->Exit();
				break;
			}
		default:
			{
				IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuDialogMax)
	{
		switch(id)
		{
		case _MenuDialogScriptDebugger:
			{
				if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(true);
				break;
			}
		case _MenuDialogPaletteEditor:
			{
				if(mDialogPaletteEditor != 0) mDialogPaletteEditor->Show(true);
				break;
			}
		case _MenuDialogPickerWindow:
			{
				if(mDialogPickerWindow != 0) mDialogPickerWindow->Show(true);
				break;
			}
		case _MenuDialogEventRecorder:
			{
				if(mDialogEventRecorder != 0) mDialogEventRecorder->Show(true);
				break;
			}
		case _MenuDialogParallelController:
			{
				if(mDialogParallelController != 0) mDialogParallelController->Show(true);
				break;
			}
		case _MenuDialogPerformanceMeter:
			{
				if(mDialogPerformanceMeter != 0) mDialogPerformanceMeter->Show(true);
				break;
			}
		case _MenuDialogFileSetExplorer:
			{
				if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->Show(true);
				break;
			}
		case _MenuDialogDataExplorer:
			{
				if(mDialogDataExplorer != 0) mDialogDataExplorer->Show(true);
				break;
			}
		case _MenuDialogImageComposer:
			{
				if(mDialogImageComposer != 0) mDialogImageComposer->Show(true);
				break;
			}
		case _MenuDialogCommandLine:
			{
				if(mDialogCommandLine != 0) mDialogCommandLine->Show(true);
				break;
			}
		default:
			{
				IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuStyleMax)
	{
		switch(id)
		{
		case _MenuStyleDockWindow:
			{
				this->DockWindows(on,true);
				break;
			}
		case _MenuStyleInteractorHelp:
			{
				mInteractorHelp = on;
				break;
			}
		case _MenuStyleToolTips:
			{
				this->ShowToolTips(on);
				break;
			}
		case _MenuStyleTabText:
			{
				if(on) mBook->SetTabMode(_BookTitleOnly);
				break;
			}
		case _MenuStyleTabIcon:
			{
				if(on) mBook->SetTabMode(_BookImageOnly);
				break;
			}
		case _MenuStyleTabBoth:
			{
				if(on) mBook->SetTabMode(_BookTitleAndImage);
				break;
			}
		case _MenuStyleRenderImmediate:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(_RenderModeImmediate);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case _MenuStyleRenderDelayed:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(_RenderModeDelayed);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case _MenuStyleRenderResetAll:
			{
				iggWidgetRenderModeButton::ResetAllToGlobal();
				break;
			}
		case _MenuStyleSlidersEditable:
			{
				iggWidgetSlider::SetAllEditable(on);
				break;
			}
		case _MenuStyleIsIdiosyncratic:
			{
				mIdiosyncratic = on;
				break;
			}
		case _MenuStyleAttachWindows:
			{
				mAttachWindows = on;
				break;
			}
		case _MenuStyleDetailLoadedDataInfo:
			{
				mDetailLoadedDataInfo = on;
				break;
			}
		default:
			{
				if(id>=_MenuStyleTheme && id<_MenuStyleTheme+mThemeList.Size())
				{
					if(on) this->SetTheme(id-_MenuStyleTheme);
				}
				else IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuOptionMax)
	{
		iString ws;

		int option = _ObjectModeCurrent|_RenderModeAuto;
		if(mOptionsAreGlobal)
		{
			option |= _ModuleModeAll;
		}
		else
		{
			option |= _ModuleModeCurrent;
		}

		switch(id)
		{
		case _MenuOptionAntialiasing:
			{
				this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyAntialiasing(),on);
				break;
			}
		case _MenuOptionIfritBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeDefault);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionClassicBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeClassic);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionHairBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeHairThin);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionAxesBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeAxes);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(true);
				}
				break;
			}
		case _MenuOptionFontTypeBitmap:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeBitmap);
				break;
			}
		case _MenuOptionFontTypeVector:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeVector);
				break;
			}
		case _MenuOptionFontSize10:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),10);
				break;
			}
		case _MenuOptionFontSize12:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),12);
				break;
			}
		case _MenuOptionFontSize14:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),14);
				break;
			}
		case _MenuOptionFontSize16:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),16);
				break;
			}
		case _MenuOptionFontSize20:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),20);
				break;
			}
		case _MenuOptionFontSize24:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),24);
				break;
			}
		case _MenuOptionFontSize32:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),32);
				break;
			}
		case _MenuOptionFontSize40:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),40);
				break;
			}
		case _MenuOptionFontSize48:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),48);
				break;
			}
		case _MenuOptionFontSize64:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),64);
				break;
			}
		case _MenuOptionFontSize80:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),80);
				break;
			}
		case _MenuOptionFontSize96:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontSize(),96);
				break;
			}
		case _MenuOptionImageFormatPNG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatPNG);
				break;
			}
		case _MenuOptionImageFormatJPG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatJPG);
				break;
			}
		case _MenuOptionImageFormatPPM:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatPNM);
				break;
			}
		case _MenuOptionImageFormatBMP:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatBMP);
				break;
			}
		case _MenuOptionImageFormatTIF:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatTIF);
				break;
			}
		case _MenuOptionImageFormatEPS:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatEPS);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA0:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),0);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),1);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),2);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),3);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),4);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),5);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA6:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),6);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA7:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),7);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA8:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),8);
				break;
			}
		case _MenuOptionPostScriptPaperFormatL1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),9);
				break;
			}
		case _MenuOptionPostScriptPaperFormatL2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),10);
				break;
			}
		case _MenuOptionPostScriptOrientationPortrait:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),0);
				break;
			}
		case _MenuOptionPostScriptOrientationLandscape:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),1);
				break;
			}
		case _MenuOptionImageZoom001:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),1);
				break;
			}
		case _MenuOptionImageZoom002:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),2);
				break;
			}
		case _MenuOptionImageZoom003:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),3);
				break;
			}
		case _MenuOptionImageZoom004:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),4);
				break;
			}
		case _MenuOptionImageZoom005:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),5);
				break;
			}
		case _MenuOptionImageZoom006:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),6);
				break;
			}
		case _MenuOptionImageZoom008:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),8);
				break;
			}
		case _MenuOptionImageZoom010:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),10);
				break;
			}
		case _MenuOptionImageZoom015:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),15);
				break;
			}
		case _MenuOptionImageZoom020:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),20);
				break;
			}
		case _MenuOptionImageZoom030:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),30);
				break;
			}
		case _MenuOptionImageZoom040:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),40);
				break;
			}
		case _MenuOptionImageZoom050:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),50);
				break;
			}
		case _MenuOptionImageZoom060:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),60);
				break;
			}
		case _MenuOptionImageZoom080:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),80);
				break;
			}
		case _MenuOptionImageZoom100:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),100);
				break;
			}
		case _MenuOptionSettingsGlobal:
			{
				mOptionsAreGlobal = on;
				return;
			}
		case _MenuOptionSaveState:
			{
				if(!this->GetShell()->GetControlModule()->SaveStateToFile(this->GetShell()->GetControlModule()->GetFileName(_EnvironmentBase,mStateFileName))) this->PopupWindow("Saving options failed for unknown reason.\n The current state will not be saved",_PopupWindowError);
				return;
			}
		default:
			{
				IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		if(!ws.IsEmpty()) this->GetShell()->GetControlModule()->Execute(ws,true,option);
		return;
	}

	if(id <= _MenuHelpMax)
	{
		switch(id)
		{
		case _MenuHelpContents:
			{
				if(mDialogHelp != 0) mDialogHelp->Show(true);
				break;
			}
		case _MenuHelpAbout:
			{
				if(mDialogAbout != 0) mDialogAbout->Show(true);
				break;
			}
#if defined(I_DEBUG) && defined(I_EDITION)
		case _MenuHelpMax:
			{
				if(mDialogDebugHelper != 0) mDialogDebugHelper->Show(true);
				break;
			}
#endif
		default:
			{
				IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _ToolBarMax)
	{
		switch(id)
		{
		case _ToolBarOpenWindowsPage:
			{
				mBook->OpenPage(0);
				mViewPage->ShowPage(4);
				break;
			}
		case _ToolBarMinimizeWindowsPage:
			{
				this->DisplayWindowsAsIcons();
				this->GetSubject()->GetHelper()->ShowAsIcon();
				break;
			}
		case _ToolBarMaximizeWindowsPage:
			{
				this->DisplayWindowsAsWindows();
				break;
			}
		case _ToolBarShowSurface:
			{
				mBook->OpenPage(1);
				mExtensionWindow->OpenBookPageByIndex(1);
				this->GetShell()->GetControlModule()->Show(iSurfaceViewObject::Type(),on);
				mBook->GetPage(1)->UpdateWidget();
				break;
			}
		case _ToolBarShowCrossSection:
			{
				mBook->OpenPage(2);
				mExtensionWindow->OpenBookPageByIndex(2);
				this->GetShell()->GetControlModule()->Show(iCrossSectionViewObject::Type(),on);
				mBook->GetPage(2)->UpdateWidget();
				break;
			}
		case _ToolBarShowVolume:
			{
				mBook->OpenPage(3);
				mExtensionWindow->OpenBookPageByIndex(3);
				this->GetShell()->GetControlModule()->Show(iVolumeViewObject::Type(),on);
				mBook->GetPage(3)->UpdateWidget();
				break;
			}
		case _ToolBarShowParticleSet:
			{
				mBook->OpenPage(4);
				mExtensionWindow->OpenBookPageByIndex(4);
				this->GetShell()->GetControlModule()->Show(iParticlesViewObject::Type(),on);
				mBook->GetPage(4)->UpdateWidget();
				break;
			}
		case _ToolBarShowVectorField:
			{
				mBook->OpenPage(5);
				mExtensionWindow->OpenBookPageByIndex(5);
				this->GetShell()->GetControlModule()->Show(iVectorFieldViewObject::Type(),on);
				mBook->GetPage(5)->UpdateWidget();
				break;
			}
		case _ToolBarShowTensorField:
			{
				mBook->OpenPage(6);
				mExtensionWindow->OpenBookPageByIndex(6);
				this->GetShell()->GetControlModule()->Show(iTensorFieldViewObject::Type(),on);
				mBook->GetPage(6)->UpdateWidget();
				break;
			}
		default:
			{
				IERROR_REPORT_ERROR("Invalid menu item id.");
			}
		}
		return;
	}

	if(!mExtensionWindow->OnMenuBody(id,on))
	{
		IERROR_REPORT_ERROR("Invalid menu item id.");
	}
}
