/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggpageview.h"


#include "ianimator.h"
#include "icamera.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iimagefactory.h"
#include "imarkerviewobject.h"
#include "ipalette.h"
#include "ipointglyph.h"
#include "irendertool.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"
#include "ishell.h"
#include "ivtk.h"
#include "iwriter.h"

#include "iggdatatypeprovider.h"
#include "iggdialoganimatingprogress.h"
#include "iggdialogimagecomposer.h"
#include "iggdialogscriptdebugger.h"
#include "iggeventobserver.h"
#include "iggextensionfactory.h"
#include "iggframedatatypeselector.h"
#include "iggframedatavariablelist.h"
#include "iggframedoublebutton.h"
#include "iggframematerialproperties.h"
#include "iggframeobjectcontrols.h"
#include "iggframepaletteselection.h"
#include "iggframeposition.h"
#include "iggmainwindow.h"
#include "iggrenderwindowobserver.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeycolorselection.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetkeytrackball.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetmisc.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwidgethelper.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "iggwidgetkeyhandlertemplate.h"
#include "iggwidgetkeyslidertemplate.h"



namespace iggPageView_Private
{
	//
	//  Show color bar check box
	//
	class ShowColorBarCheckBox : public iggWidgetKeyHandler<int>
	{
		
	public:
		
		ShowColorBarCheckBox(const iObjectKey &key, iggFrame *parent) : iggWidgetKeyHandler<int>(_WidgetTypeOther,key,_RenderModeImmediate,parent,3,0)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Show",1);
		}
		
	protected:

		virtual void OnVoid1Body()
		{
			this->ExecuteControl(true);
		}

		virtual void QueryValue(int &v) const
		{
			if(mSubject->IsDown()) v = 1; else v = 0;
		}
		
		virtual void UpdateValue(int v)
		{
			mSubject->SetDown(v != 0);
		}
		
		ibgWidgetButtonSubject *mSubject;
	};


	//
	//  Color bar frame
	//
	class ColorBarFrame : public iggFrame
	{
		
	public:
		
		ColorBarFrame(iggWidgetKeyCheckBox *autobox, const iString &title, const iObjectKey &key, iggFrame *parent) : iggFrame(title,parent,1), mKey(key)
		{
			mProvider = new iggKeywordDataTypeProvider("ScalarField",this); 

			mAutomaticBox = autobox;
			autobox->AddDependent(this);

			this->AddLine(new ShowColorBarCheckBox(key,this));

			iggFrameDataVariableList *vl = new iggFrameDataVariableList(mProvider,"Variable",key,0,this,0);
			vl->Complete(-1);
			this->AddLine(vl);
			this->AddSpace(2);

			this->AddLine(new iggWidgetTextArea("Palette",this));
			mPaletteList = new iggWidgetKeyComboBox("",-1,key,this,2);
			this->AddLine(mPaletteList);

			this->AddLine(new iggWidgetTextArea("Data type",this));
			this->AddLine(new iggFrameDataTypeSelector(mProvider,false,"",this));
		}

		virtual ~ColorBarFrame()
		{
			delete mProvider;
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			//
			//  Are we enabled?
			//
			bool b;
			mAutomaticBox->QueryValue(b);
			this->Enable(!b);

			if(!b)
			{
				//
				//  Update palettes
				//
				int i, np = this->GetShell()->GetControlModule()->GetNumberOfPalettes();
				mPaletteList->Clear();
				for(i=0; i<np; i++)
				{
					mPaletteList->InsertItem(this->GetShell()->GetControlModule()->GetPalette(i)->GetName().ToCharPointer());
				}
				//
				//  Update data type
				//
				iString ws;
				this->GetShell()->GetControlModule()->PackCommand(ws,mKey,1,mProvider->GetActiveDataType().GetId());
				if(!this->GetShell()->GetControlModule()->Execute(ws,false)) return this->UpdateFailed();
			}
			//
			//  Update the rest
			//
			return iggFrame::UpdateWidgetBody();
		}

		const iObjectKey &mKey;
		iggWidgetKeyComboBox *mPaletteList;
		iggWidgetKeyCheckBox *mAutomaticBox;
		iggKeywordDataTypeProvider *mProvider;
	};


	//
	//  Animation controls
	//
	class AnimateButton : public iggWidgetSimpleButton
	{

	public:

		AnimateButton(iggFrame *parent) : iggWidgetSimpleButton("Animate",parent)
		{
			this->SetBaloonHelp("Start animation");
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsFileAnimatable());
			return true;
		}

		virtual void Execute()
		{
			this->GetMainWindow()->GetDialogAnimatingProgress()->Animate();
			this->GetShell()->GetControlModule()->Render();

			if(this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetErrorStatus()->IsError())
			{
				this->GetMainWindow()->PopupWindow("Animation completed with the following error message:\n"+this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetErrorStatus()->GetMessage());
			}
		}
	};


	//
	//  Helper classes that create a dialog with a single image if Area is clicked
	//
	class ViewImageDialog : public iggDialog
	{

	public:

		ViewImageDialog(iggMainWindow *parent, const iImage &image) : iggDialog(parent,_DialogModal,0,"View Image",0,1)
		{
			mArea = new iggWidgetImageArea(image,false,mFrame);
			mFrame->AddLine(mArea);
		}

		void SetImage(const iImage& image)
		{
			mArea->SetImage(image,false);
			mArea->SetFixedSize(image.Width(),image.Height());
		}

		virtual const iString& GetToolTip() const 
		{
			static const iString tmp = "Shows a full image";
			return tmp; 
		}

	protected:

		iggWidgetImageArea *mArea;
	};


	//
	//  View image area
	//
	class ViewImageArea : public iggWidgetImageArea
	{

	public:

		ViewImageArea(iggFrame *parent) : iggWidgetImageArea(iImage(),false,parent)
		{
			mSubject->SetBackgroundColor(iColor(100,100,100));
		}
	};


	//
	//  Load title/logo button
	//
	class LoadButton : public iggWidgetKeyFileNameButton
	{

	public:

		LoadButton(iggWidgetImageArea *view, const iString &header, const iObjectKey &filekey, const iObjectKey &imagekey, iggFrame *parent) : iggWidgetKeyFileNameButton("Load",header,"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",filekey,parent), mImageKey(imagekey), mView(view)
		{
		}

		virtual void QueryValue(iString& val) const
		{
			static const iImage nullImage = iImage();

			iggWidgetKeyFileNameButton::QueryValue(val);
			if(val.IsEmpty()) return;

			iImage im;
			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,this->GetKey(),val);
			this->GetShell()->GetControlModule()->Execute(ws,false);
			if(this->GetShell()->GetControlModule()->QueryValue(mImageKey,im) && !im.IsEmpty())
			{
				mView->SetImage(im,false);
			}
			else
			{
				mView->SetImage(nullImage,false);
				this->GetMainWindow()->PopupWindow("Unable to load image from the file.\n The file may be corrupted.");
			}
			mView->UpdateWidget();
		}

	protected:

		const iObjectKey &mImageKey;
		iggWidgetImageArea *mView;
	};


	//
	//  Remove title/logo button
	//
	class RemoveButton : public iggWidgetKeyFileNameButton
	{

	public:

		RemoveButton(iggWidgetImageArea *view, const iObjectKey &key, iggFrame *parent) : iggWidgetKeyFileNameButton("Remove","","",key,parent), mView(view)
		{
		}

		virtual void QueryValue(iString&) const
		{
			static const iImage nullImage = iImage();

			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,this->GetKey(),iString());
			this->GetShell()->GetControlModule()->Execute(ws,false);
			mView->SetImage(nullImage,false);
			mView->UpdateWidget();
			mWidgetHelper->Enable(!mView->GetImage().IsEmpty());
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			mWidgetHelper->Enable(!mView->GetImage().IsEmpty());
			return iggWidgetKeyFileNameButton::UpdateWidgetBody();
		}

		iggWidgetImageArea *mView;
	};


	//
	//  View title/logo button
	//
	class ViewButton : public iggWidgetLaunchButton
	{

	public:

		ViewButton(iggWidgetImageArea *view, iggFrame *parent) : iggWidgetLaunchButton((iggDialog *)0,"View",parent,true), mView(view)
		{
			mDialog = mWindow = new ViewImageDialog(this->GetMainWindow(),mView->GetImage());
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			mWidgetHelper->Enable(!mView->GetImage().IsEmpty());
			return iggWidgetLaunchButton::UpdateWidgetBody();
		}

		virtual void OnVoid1Body()
		{
			mWindow->SetImage(mView->GetImage());
			iggWidgetLaunchButton::OnVoid1Body();
		}

		iggWidgetImageArea *mView;
		ViewImageDialog *mWindow;
	};


	//
	//  Window manipulation widgets
	//
	class WindowNumberComboBox : public iggWidget
	{

	public:

		WindowNumberComboBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"Set current window");
			this->SetBaloonHelp("Select the current window","This control selects a visualization window to be the \"current\" one. The current window is labeled by a marker placed in the upper left corner of the window, and is the window controled by the widgets.");
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			mSubject->Clear();

			int i, n = this->GetShell()->GetControlModule()->GetNumberOfViewModules();
			for(i=0; i<n; i++) mSubject->InsertItem("Window "+iString::FromNumber(i+1));
			mSubject->SetValue(this->GetShell()->GetControlModule()->GetCurrentViewModuleIndex());

			return true;
		}

		void OnInt1Body(int v)
		{
			if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(v)) this->GetMainWindow()->UpdateAll();
		}

		ibgWidgetComboBoxSubject *mSubject;
	};


	class DialogWait : public iggDialog
	{

	public:

		DialogWait(bool create, iggMainWindow *parent) : iggDialog(parent,_DialogBlocking|_DialogNoTitleBar,0,"Working...",0,3,false)
		{
			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);
			mFrame->AddLine(new iggWidgetTextArea(iString("%b%+")+(create?"Creat":"Delet")+"ing window...",mFrame),3);
		}

		virtual void Show(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::Show(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	class CreateWindowButton : public iggWidgetSimpleButton
	{

	public:

		CreateWindowButton(int type, iggFrame *parent) : iggWidgetSimpleButton("",parent)
		{
			mType = type;
			mDialog = new DialogWait(true,this->GetMainWindow());

			switch(mType)
			{
			case _ModuleTypeNew:
				{
					mSubject->SetText("Create new window");
					this->SetBaloonHelp("Create a new independent window","This button creates a new visualization window. The new window is independent of other windows and has default values of all parameters.");
					break;
				}
			case _ModuleTypeCopy:
				{
					mSubject->SetText("Copy current window");
					this->SetBaloonHelp("Copy the current window","This button creates a new visualization window which is a copy of the current visualization window, i.e. it has the same values of all parameters for all objects. The new window is independent of other windows, and has no data associated with it.");
					break;
				}
			case _ModuleTypeClone:
				{
					mSubject->SetText("Clone current window");
					this->SetBaloonHelp("Clone the current window","This button creates a clone of the current visualization window. The clone shares the data with the parent window, and at birth has the same values of all parameters for all visualization objects. It does have a separate set of visualization objects however, so the parameters of the parent and the clone window can be changed independently.");
					break;
				}
			default: IERROR_REPORT_FATAL_ERROR("Incorrect button type");
			}
        }

		virtual ~CreateWindowButton()
		{
			delete mDialog;
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			mDialog->Show(true);
			if(!this->GetShell()->GetControlModule()->CreateObject(iViewModule::Type(),true,-1,mType))
			{
				this->GetMainWindow()->PopupWindow("Unable to create new window.");
			}
			else
			{
				if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(this->GetShell()->GetControlModule()->GetNumberOfViewModules()-1)) this->GetMainWindow()->UpdateAll();
			}
			mDialog->Show(false);
			this->GetMainWindow()->Block(false);
		}

		int mType;
		DialogWait *mDialog;
	};


	class DeleteWindowButton : public iggWidgetSimpleButton
	{

	public:

		DeleteWindowButton(iggFrame *parent) : iggWidgetSimpleButton("Delete current window",parent)
		{
			mDialog = new DialogWait(false,this->GetMainWindow());

			this->SetBaloonHelp("Delete the current window","This button deletes the current visualization window and all the objects and the data associated with it.");
        }

		virtual ~DeleteWindowButton()
		{
			delete mDialog;
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetNumberOfViewModules() > 1);
			return true;
		}

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			mDialog->Show(true);
			if(!this->GetShell()->GetControlModule()->DeleteObject(iViewModule::Type(),true))
			{
				this->GetMainWindow()->PopupWindow("Unable to delete current window.");
			}
			else
			{
				if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(this->GetShell()->GetControlModule()->GetNumberOfViewModules()-1)) this->GetMainWindow()->UpdateAll();
			}
			mDialog->Show(false);
			this->GetMainWindow()->Block(false);
		}

		DialogWait *mDialog;
	};


	class WindowUnderFocusCurrentCheckBox : public iggWidget
	{

	public:

		WindowUnderFocusCurrentCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Make window under focus current",1);
			this->SetBaloonHelp("Toggles whether the visualization window with the keyboard focus becomes the current one","When this box is checked, moving the leyboard focus into a visualization window will make that window the current one.");
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			mSubject->SetDown(this->GetMainWindow()->GetWindowUnderFocusCurrent());
			return true;
		}

		void OnVoid1Body()
		{
			this->GetMainWindow()->SetWindowUnderFocusCurrent(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};


	class WidgetsControlAllWindowsCheckBox : public iggWidget
	{

	public:

		WidgetsControlAllWindowsCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Widgets control all windows",1);
			this->SetBaloonHelp("Toggles whether widgets control all visualization windows or only the current one","When this box is checked, widget controls in the gui window will affect all visualization windows, not just the current one. This option should be used with care, since it only makes sense if different visualization windows display similar scenes (for example, the same visualization object for different data files). If the scenes in different visualization windows are unlike each other, the effects of this option may be quite unexpected.");
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}

		void OnVoid1Body()
		{
			if(mSubject->IsDown())
			{
				iggWidgetKeyHandlerBase::SetGlobalExecuteFlags(_ObjectModeCurrent|_ModuleModeAll|_RenderModeAuto);
			}
			else
			{
				//
				//  Go back to default ones
				//
				iggWidgetKeyHandlerBase::SetGlobalExecuteFlags(0);
			}
		}

		ibgWidgetButtonSubject *mSubject;
	};


	class WindowStatusIcon : public iggFrame
	{

	public:

		WindowStatusIcon(iggFrame *parent) : iggFrame("Status",parent,1)
		{
			mIcon1 = iImageFactory::FindIcon("winindiv.png");
			mIcon2 = iImageFactory::FindIcon("winclone.png");

			mText1 = "Individual";
			mText2 = "Clone of #";

			mIconView = new iggWidgetImageArea(*mIcon1,false,this);
			mTextView = new iggWidgetTextArea(mText1,this);
			this->AddLine(mIconView);
			this->AddLine(mTextView);
//			this->AddSpace(10);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->IsClone())
			{
				mIconView->SetImage(*mIcon2,false);
				mTextView->SetText(mText2+iString::FromNumber(this->GetShell()->GetControlModule()->GetViewModule()->GetCloneOfWindow()+1));
			}
			else
			{
				mIconView->SetImage(*mIcon1,false);
				mTextView->SetText(mText1);
			}
			return iggFrame::UpdateWidgetBody();
		}

		const iImage *mIcon1, *mIcon2;
		iString mText1, mText2;
		iggWidgetImageArea *mIconView;
		iggWidgetTextArea *mTextView;
	};


	//
	//  Marker widgets
	//
	class MarkerFrameBox : public iggFrame
	{

	public:

		MarkerFrameBox(iggFrame *parent) : iggFrame("Marker",parent,2)
		{
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()>0 || this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->IsVisible())
			{
				this->Enable(true);
				return iggFrame::UpdateWidgetBody();
			}
			else
			{
				this->Enable(false);
				return true;
			}
		}
	};


	class CurrentMarkerBox : public iggWidgetKeyComboBox
	{

	public:

		CurrentMarkerBox(iggFrame *parent) : iggWidgetKeyComboBox("",0,iViewModule::KeyMarkerCurrent(),parent)
		{
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			int i, n = this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex() + 1;

			if(n==1 && !this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->IsVisible())
			{
				mSubject->Clear();
			}
			else
			{
				if(n != mSubject->Count())
				{
					mSubject->Clear();
					for(i=0; i<n; i++)
					{
						mSubject->InsertItem("#"+iString::FromNumber(i+1)+" "+this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetTypeAsString());
					}
				}
				else
				{
					i = this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetCurrentMemberIndex();
					mSubject->SetItem("#"+iString::FromNumber(i+1)+" "+this->GetShell()->GetControlModule()->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetTypeAsString(),i,true);
				}
			}
			return iggWidgetKeyComboBox::UpdateWidgetBody();
		}
	};


	class MarkerMoveButton : public iggWidgetKeyToggleButton
	{

	public:

		MarkerMoveButton(iggFrameCreateDeleteButton *buddy1, iggWidgetKeyComboBox *buddy2, const iString &text, const iObjectKey &key, iggFrame *parent) : iggWidgetKeyToggleButton(0,text,key,parent,-1,0)
		{
			mBuddy1 = buddy1;
			mBuddy2 = buddy2;
		}

		void SetBuddy3(MarkerMoveButton *buddy3)
		{
			mBuddy3 = buddy3;
		}

		virtual void QueryValue(bool &val) const
		{
			iggWidgetKeyToggleButton::QueryValue(val);
			mBuddy1->SetActive(!val);
			mBuddy2->Enable(!val);
			mBuddy3->Enable(!val);
		}

	protected:

		iggFrameCreateDeleteButton *mBuddy1;
		iggWidgetKeyComboBox *mBuddy2;
		MarkerMoveButton *mBuddy3;
	};


	class MarkerPositionFrame : public iggFramePosition, public iggRenderWindowObserver
	{

	public:

		MarkerPositionFrame(iggFrame *parent) : iggFramePosition("Position",iMarkerViewObject::KeyPosition(true),parent,true), iggRenderWindowObserver(parent)
		{
		}

	protected:

		virtual void OnRenderWindowModified()
		{
			iggFramePosition::UpdateWidgetBody();
		}
	};


	//
	//  A small dialog with the line edit to set the ruler scale exactly
	//
	class RulerScaleDialog : public iggDialog
	{

	public:

		RulerScaleDialog(iggMainWindow *parent) : iggDialog(parent,0U,0,"Ruler Scale",0,1,false)
		{
			mFrame->AddLine(new iggWidgetKeyFloatLineEdit("Scale",iViewModule::KeyRulerScale(),_RenderModeUseGlobal,mFrame));
			mFrame->AddSpace(10);
		}
	};


	class ShowRulerCheckBox : public iggWidgetKeyCheckBox, public iggRenderWindowObserver
	{

	public:

		ShowRulerCheckBox(iggFrame *parent) : iggWidgetKeyCheckBox("Ruler",iViewModule::KeyRuler(),parent), iggRenderWindowObserver(parent)
		{
			mDialog = new RulerScaleDialog(this->GetMainWindow());
		}

		virtual ~ShowRulerCheckBox()
		{
			delete mDialog;
		}

	protected:

		virtual void OnRenderWindowModified()
		{
			mDialog->UpdateDialog();
		}

		void OnVoid1Body()
		{
			if(mDialog != 0) mDialog->Show(mSubject->IsDown());
			iggWidgetKeyCheckBox::OnVoid1Body();
		}

		RulerScaleDialog *mDialog;
	};


	//
	//  A small dialog with several widgets to set the record label parameters
	//
	class RecordLabelDialog : public iggDialog
	{

	public:

		RecordLabelDialog(iggMainWindow *parent) : iggDialog(parent,0U,0,"Record Label Properties",0,3,false)
		{
			mFrame->AddLine(new iggWidgetTextArea("Name",mFrame),new iggWidgetKeyTextLineEdit(false,"",iViewModule::KeyLabelName(),_RenderModeUseGlobal,mFrame),(iggWidget *)0);
			mFrame->AddLine(new iggWidgetTextArea("Unit",mFrame),new iggWidgetKeyTextLineEdit(false,"",iViewModule::KeyLabelUnit(),_RenderModeUseGlobal,mFrame),(iggWidget *)0);
			mFrame->AddLine(new iggWidgetTextArea("Scale",mFrame),new iggWidgetKeyFloatLineEdit("",iViewModule::KeyLabelScale(),_RenderModeUseGlobal,mFrame),(iggWidget *)0);
			mFrame->AddLine(new iggWidgetKeySpinBox(0,7,"Number of digits after the point",0,iViewModule::KeyLabelDigits(),mFrame),3);
			mFrame->AddSpace(10);
		}
	};


	class ShowRecordLabelCheckBox : public iggWidgetKeyCheckBox
	{

	public:

		ShowRecordLabelCheckBox(iggFrame *parent) : iggWidgetKeyCheckBox("Record label",iViewModule::KeyLabel(),parent)
		{
			mDialog = new RecordLabelDialog(this->GetMainWindow());
		}

		virtual ~ShowRecordLabelCheckBox()
		{
			delete mDialog;
		}

	protected:

		void OnVoid1Body()
		{
			if(mDialog != 0) mDialog->Show(mSubject->IsDown());
			iggWidgetKeyCheckBox::OnVoid1Body();
		}

		RecordLabelDialog *mDialog;
	};


	class ShowInStereoCheckBox : public iggWidgetKeyCheckBox, public iggRenderWindowObserver
	{

	public:

		ShowInStereoCheckBox(iggFrame *parent) : iggWidgetKeyCheckBox("Show in stereo",iViewModule::KeyStereo(),parent), iggRenderWindowObserver(parent)
		{
		}

	protected:

		virtual void OnRenderWindowModified()
		{
			this->UpdateWidget();
		}
	};


	class ClippingRangeButton : public iggWidgetSimpleButton
	{

	public:

		ClippingRangeButton(int type, iggWidgetKeyCheckBox *buddy, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mBuddy = buddy;
			mBuddy->AddDependent(this);
			mType = type;

			if(mType%2 == 0)
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("moveleft.png"));
			}
			else
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("moveright.png"));
			}
			this->SetBaloonHelp("Adjust camera clipping range","This button adjust the clipping range of the current camera.");
        }

	protected:

		virtual bool UpdateWidgetBody()
		{
			bool v;
			mBuddy->QueryValue(v);
			this->Enable(!v);
			return true;
		}

		virtual void Execute()
		{
			double cr[2];

			if(this->GetShell()->GetControlModule()->QueryValue(iCamera::KeyClippingRange(),cr,2))
			{
				switch(mType)
				{
				case 0:
					{
						cr[0] *= 0.5;
						break;
					}
				case 1:
					{
						cr[0] *= 2.0;
						break;
					}
				case 2:
					{
						cr[1] *= 0.5;
						break;
					}
				case 3:
					{
						cr[1] *= 2.0;
						break;
					}
				}
				iString ws;
				this->GetShell()->GetControlModule()->PackCommand(ws,iCamera::KeyClippingRange(),cr,2);
				this->GetShell()->GetControlModule()->Execute(ws,true);
			}
		}

		int mType;
		iggWidgetKeyCheckBox *mBuddy;
	};


	class ClippingRangeDisplay : public iggWidgetNumberDisplay, protected iggRenderWindowObserver
	{

	public:

		ClippingRangeDisplay(int type, iggFrame *parent) : iggWidgetNumberDisplay("",parent), iggRenderWindowObserver(parent)
		{
			mType = type;
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			double cr[2];

			if(this->GetShell()->GetControlModule()->QueryValue(iCamera::KeyClippingRange(),cr,2))
			{
				switch(mType)
				{
				case 0:
					{
						this->Display(cr[0]);
						return true;
					}
				case 1:
					{
						this->Display(cr[1]);
						return true;
					}
				default:
					{
						return false;
					}
				}
			}
			else return false;
		}

		virtual void OnRenderWindowModified()
		{
			this->UpdateWidget();
		}

		int mType;
	};


	class StepOnPath : public iggWidgetKeyIntSlider
	{

	public:

		StepOnPath(iggFrame *parent) : iggWidgetKeyIntSlider(0,100,5,"Step on path",iAnimator::KeyPositionOnPath(),_RenderModeUseGlobal,parent)
		{
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			iAnimator *an = this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator();
			this->Enable(an->GetMode() == 4);
			iPointerCast<iggAnimatorEventObserver,iAnimatorEventObserver>(an->GetObserver())->AddDependent(this);
			this->SetRange(0,this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetNumberOfCameraPathSteps()-1);
			return iggWidgetKeyIntSlider::UpdateWidgetBody();
		}
	};


	class PlayDemo : public iggWidgetKeyToolButton
	{
		
	public:
		
		PlayDemo(iggFrame *parent) : iggWidgetKeyToolButton("",iAnimator::KeyCameraPathDemo(),parent)
		{
			mSubject->SetIcon(*iImageFactory::FindIcon("moveright.png"));
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			this->Enable(this->GetShell()->GetControlModule()->GetViewModule()->GetAnimator()->GetMode() == 4);
			return true;
		}
	};

	//
	//  Window size dialog and its widgets
	//
	class WindowSizeDialog : public iggDialog
	{

	public:

		WindowSizeDialog(iggMainWindow *parent) : iggDialog(parent,_DialogModal,0,"Visualization Window Size",0,2,true,"Set")
		{
			mBox[0] = new iggWidgetSimpleSpinBox(120,32768,1,"",mFrame);
			mBox[1] = new iggWidgetSimpleSpinBox(120,32768,1,"",mFrame);
			mFrame->AddLine(new iggWidgetTextArea("%bWidth:",mFrame),mBox[0]);
			mFrame->AddLine(new iggWidgetTextArea("%bHeight:",mFrame),mBox[1]);
			mFrame->AddSpace(10);

			mBox[0]->SetBaloonHelp("Set window width","Set the width of the current visualization window.");
			mBox[1]->SetBaloonHelp("Set window height","Set the height of the current visualization window.");
		}

		virtual void Show(bool s)
		{
			if(s)
			{
                const int *size = this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindowSize();
				mBox[0]->SetValue(size[0]);
				mBox[1]->SetValue(size[1]);
			}
			iggDialog::Show(s);
		}

	protected:

		virtual bool CanBeClosed()
		{
			const int *s = this->GetShell()->GetControlModule()->GetViewModule()->GetRenderWindowSize();
			if(s[0]!=mBox[0]->GetValue() || s[1]!=mBox[1]->GetValue())
			{
				this->GetShell()->GetControlModule()->GetViewModule()->SetRenderWindowSize(mBox[0]->GetValue(),mBox[1]->GetValue());
			}
			return true;
		}

		iggWidgetSimpleSpinBox *mBox[2];
	};


	//
	//  Misc buttons
	//
	class ClearLogButton : public iggWidgetSimpleButton
	{

	public:

		ClearLogButton(iggFrame *parent) : iggWidgetSimpleButton("Clear log",parent)
		{
			this->SetBaloonHelp("Clear the log window","This button removes all the text in the log window.");
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->ClearLog();
		}
	};


	class ResetAnimationImageIndexButton : public iggWidgetSimpleButton
	{

	public:

		ResetAnimationImageIndexButton(iggFrame *parent) : iggWidgetSimpleButton("Reset image counter",parent)
		{
			this->SetBaloonHelp("Reset image counter back to 1","This button resets the counter of animation image files to 1. The animation files created so far will be overwritten in the next animation.");
        }

	protected:

		virtual void Execute()
		{
			if(this->GetMainWindow()->PopupWindow("Please confirm: reseting the counter will overwrite the animation files created so far.",_PopupWindowMessage,"Reset","Cancel") == 0)
			{
				this->GetShell()->GetControlModule()->GetViewModule()->GetWriter()->ResetAnimationImageIndex();
			}
		}
	};
};


using namespace iggPageView_Private;


iggPageView::iggPageView(iggFrameBase *parent) : iggPageMain(parent)
{
	int i;
	const iImage *icon = iImageFactory::FindIcon("view.png");

	mWindowSizeDialog = new WindowSizeDialog(this->GetMainWindow());

	//
	//  Main page
	// ************************************************
	//
	{
		iggFrame *page0 = new iggFrame(mBook,3);
		mBook->AddPage("Main",icon,page0);

		page0->AddSpace(2);

		iggWidgetKeyTextComboBox *ur = new iggWidgetKeyTextComboBox(0,"Update rate",0,iViewModule::KeyUpdateRate(),page0);
		ur->InsertItem("Non-interactive");
		ur->InsertItem(" 3 frames/sec");
		ur->InsertItem("10 frames/sec");
		ur->InsertItem("30 frames/sec");
		ur->SetInvalidValue(0);
		page0->AddLine(ur,2);
		page0->AddSpace(2);

		iggFrame *pf = new iggFrame("Projection",page0,1);
		pf->AddLine(new iggWidgetKeyCheckBox("Parallel",iCamera::KeyParallelProjection(),pf));
		iggWidgetKeyCheckBox *pp = new iggWidgetKeyCheckBox("Perspective",iCamera::KeyParallelProjection(),pf);
		pf->AddLine(pp);
		pp->SetReverse(true);
		page0->AddLine(pf);
		page0->AddLine(new iggWidgetKeyExecButton("Reset view",iCamera::KeyReset(),page0),new iggWidgetLaunchButton(mWindowSizeDialog,"Set window size",page0));
		page0->AddSpace(2);

		iggFrame *fb = new iggFrame("Features",page0,1);
		fb->AddLine(new iggWidgetKeyCheckBox("Bounding box",iViewModule::KeyBoundingBox(),fb));
		fb->AddLine(new ShowRecordLabelCheckBox(fb));
		fb->AddLine(new ShowRulerCheckBox(fb));
		fb->AddLine(new iggWidgetKeyCheckBox("Color bars",iViewModule::KeyColorBars(),fb));
		fb->AddLine(new iggWidgetKeyCheckBox("Measuring box",iViewModule::KeyMeasuringBox(),fb));

		iggFrame *fb2 = new iggFrame(page0,1);
		iggWidgetKeyRadioBox *isb = new iggWidgetKeyRadioBox(1,"Interactor style",0,iViewModule::KeyInteractorStyle(),fb2);
		isb->InsertItem("Trackball");
		isb->InsertItem("Joystick");
		isb->InsertItem("Flight");
		isb->InsertItem("Keyboard");
		iggWidget *wid = new iggWidgetKeyColorSelection(iViewModule::KeyBackgroundColor(),fb2,false);
		if(this->GetMainWindow()->GetDialogImageComposer() != 0) wid->AddDependent(this->GetMainWindow()->GetDialogImageComposer()->GetArea());
		fb2->AddLine(isb);
		fb2->AddLine(wid);
		page0->AddLine(fb,fb2);

		page0->AddSpace(10);
		page0->SetColStretch(2,10);
	}

	//
	//  Scene page
	// ************************************************
	//
	{
		iggFrame *page1 = new iggFrame(mBook,1);
		mBook->AddPage("Scene",icon,page1);
		//
		//  Book
		//
		iggFrameBook *sb = new iggFrameBook(page1);
		page1->AddLine(sb);
		//
		//  Camera page
		//
		{
			iggFrame *sbpage0 = new iggFrame(sb,2);
			sb->AddPage("Camera",icon,sbpage0);

			iggFrame *vf = new iggFrame("View angle",sbpage0,1);
			vf->AddSpace(1);
			vf->AddLine(new iggWidgetKeyFloatSlider(10.0,120.0,110,0,3,"",iCamera::KeyViewAngle(),_RenderModeUseGlobal,vf));
			vf->AddLine(new iggWidgetKeyCheckBox("Vertical",iCamera::KeyViewAngleVertical(),vf));
			vf->AddSpace(1);
			sbpage0->AddLine(vf);
			sbpage0->AddSpace(2);

			iggFrame *crf = new iggFrame("Camera clipping range",sbpage0,5);
			iggWidgetKeyCheckBox *cfa = new iggWidgetKeyCheckBox("Automatic",iCamera::KeyClippingRangeAuto(),crf);
			crf->AddLine(cfa,3);
			crf->AddLine(new ClippingRangeButton(0,cfa,crf),new ClippingRangeButton(1,cfa,crf),new iggWidgetTextArea("Adjust clipping range",crf),new ClippingRangeButton(2,cfa,crf),new ClippingRangeButton(3,cfa,crf));
			crf->AddLine(new ClippingRangeDisplay(0,crf),2,new iggWidgetTextArea("(in units of focal distance)",crf),1,new ClippingRangeDisplay(1,crf),2);
			sbpage0->AddLine(crf);
			sbpage0->AddSpace(2);

			sbpage0->AddSpace(4);
			sbpage0->SetColStretch(1,10);
		}
		//
		//  Lights page
		//
		{
			iggFrame *sbpage1 = new iggFrame(sb,3);
			sb->AddPage("Lights",icon,sbpage1);

			iggFrame *db = new iggFrame("Main light direction",sbpage1,1);
			db->AddLine(new iggWidgetKeyTrackBall(iViewModule::KeyLightAngles(),true,_RenderModeUseGlobal,db));
			sbpage1->AddLine(db);

			iggFrame *ib = new iggFrame("Intensity",sbpage1,1);
			iggWidgetKeyFloatSlider *mli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Main light",iViewModule::KeyLightIntensity(),_RenderModeUseGlobal,ib,0);
			iggWidgetKeyFloatSlider *fli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Fill light",iViewModule::KeyLightIntensity(),_RenderModeUseGlobal,ib,1);
			iggWidgetKeyFloatSlider *hli = new iggWidgetKeyFloatSlider(0.01f,10.0f,200,1,0,"Head light",iViewModule::KeyLightIntensity(),_RenderModeUseGlobal,ib,2);
			hli->AddBuddy(mli);
			fli->AddBuddy(mli);
			mli->SetStretch(3,10);
			hli->SetStretch(3,10);
			fli->SetStretch(3,10);
			ib->AddLine(mli);
			ib->AddLine(fli);
			ib->AddLine(hli);
			sbpage1->AddLine(ib,2);

			sbpage1->AddSpace(10);
			sbpage1->SetColStretch(1,10);
			sbpage1->SetColStretch(2,3);
		}
		//
		//  Special modes page
		//
		{
			iggFrame *sbpage2 = new iggFrame(sb,3);
			sb->AddPage("Special",icon,sbpage2);
			//
			//  Book
			//
			iggFrameBook *vb = new iggFrameBook(sbpage2);
			sbpage2->AddLine(vb);
			{
				//
				//  Stereo mode page
				//
				iggFrame *vbpage0 = new iggFrame(vb,3);
				vb->AddPage("Stereo",icon,vbpage0);

				vbpage0->AddLine(new ShowInStereoCheckBox(vbpage0));
				iggWidgetKeyRadioBox *sm = new iggWidgetKeyRadioBox(1,"Method",0,iViewModule::KeyStereoType(),vbpage0);
				sm->InsertItem("Both eyes in 2 windows");
				sm->InsertItem("Crystal eyes");
				sm->InsertItem("Blue-red");
				sm->InsertItem("Interlaced");
				sm->InsertItem("Left eye only");
				sm->InsertItem("Right eye only");
				sm->InsertItem("Dresden display");
				vbpage0->AddLine(sm,new iggWidgetTextArea("To toggle in and out\n of stereo mode,\n press 3 within\n the visualization window",vbpage0));
				vbpage0->AddLine(new iggWidgetKeyFloatSlider(0.1f,10.0f,20,1,4,"Angle between eyes",iCamera::KeyEyeAngle(),_RenderModeUseGlobal,vbpage0),2);
				vbpage0->AddLine(new iggWidgetKeyCheckBox("Show alighnment markers in a two-window mode",iViewModule::KeyStereoAlignmentMarkers(),vbpage0),2);

				vbpage0->AddSpace(10);
				vbpage0->SetColStretch(1,10);
				vbpage0->SetColStretch(2,3);
			}
			//
			//  Multi view mode page
			//
			{
				iggFrame *vbpage1 = iggExtensionFactory::CreateMultiViewFrame(vb);
				if(vbpage1 != 0) vb->AddPage("Multi view",icon,vbpage1);
			}
		}
		//
		//  Color bars page
		//
		{
			iggFrame *sbpage3 = new iggFrame(sb,3);
			sb->AddPage("Color bars",icon,sbpage3);

			iggWidgetKeyCheckBox *acb = new iggWidgetKeyCheckBox("Automatic",iColorBars::KeyAutomatic(),sbpage3);

			sbpage3->AddLine(acb,2);
			sbpage3->AddLine(new ColorBarFrame(acb,"Left bar",iColorBars::KeyBarLeft(),sbpage3),new ColorBarFrame(acb,"Right bar",iColorBars::KeyBarRight(),sbpage3));

			sbpage3->AddSpace(10);
			sbpage3->AddLine(new iggWidgetKeyFloatSlider(0.01,0.2,19,0,0,"Side offset",iColorBars::KeySideOffset(),_RenderModeUseGlobal,sbpage3),2);

			sbpage3->SetColStretch(2,10);
		}
		//
		//  Clip plane page
		//
		{
			iggFrame *sbpage4 = new iggFrame(sb,3);
			sb->AddPage("Clip plane",icon,sbpage4);

			sbpage4->AddLine(new iggWidgetKeyCheckBox("Use clip plane",iViewModule::KeyClipPlane(),sbpage4));
			sbpage4->AddSpace(2);

			iggFrame *cb = new iggFrame("Clip plane direction",sbpage4,1);
			cb->AddLine(new iggWidgetKeyTrackBall(iViewModule::KeyClipPlaneDirection(),false,_RenderModeUseGlobal,cb));
			sbpage4->AddLine(cb);
			sbpage4->AddLine(new iggWidgetKeyFloatSlider(-2.0f,2.0f,400,0,0,"Clip plane distance",iViewModule::KeyClipPlaneDistance(),_RenderModeUseGlobal,sbpage4),2);
			sbpage4->AddLine(new iggWidgetKeyCheckBox("Show as glass plane",iViewModule::KeyGlassClipPlane(),sbpage4));

			sbpage4->AddSpace(10);
			sbpage4->SetColStretch(1,10);
			sbpage4->SetColStretch(2,3);
		}
	}

	//
	//  Animation page
	// ************************************************
	//
	{
		iggFrame *page2 = new iggFrame(mBook,1);
		mBook->AddPage("Animation",icon,page2);
		//
		//  Book
		//
		iggFrameBook *ab = new iggFrameBook(page2);
		page2->AddLine(ab);
		{
			//
			//  Style page
			//
			iggFrame *abpage0 = new iggFrame(ab,3);
			ab->AddPage("Style",icon,abpage0);

			iggWidgetKeyRadioBox *as = new iggWidgetKeyRadioBox(1,"Style",0,iAnimator::KeyMode(),abpage0);
			as->InsertItem("Static");
			as->InsertItem("Rotate/scale");
			as->InsertItem("Tumble");
			as->InsertItem("Fly-by");
			as->InsertItem("Camera path");

			iggFrame *tmp = new iggFrame(abpage0);
			iggWidgetKeyRadioBox *ao = new iggWidgetKeyRadioBox(1,"Animation output",0,iViewModule::KeyAnimationOutput(),tmp);
			ao->InsertItem("Series of images");
#ifndef IVTK_4
			ao->InsertItem("MPEG2 movie");
#ifdef IVTK_SUPPORTS_AVI
			ao->InsertItem("AVI movie");
#endif
#endif
			tmp->AddLine(ao);
			tmp->AddSpace(10);
			abpage0->AddLine(as,tmp);
			abpage0->AddLine(new iggWidgetKeyCheckBox("Restore camera after animation",iAnimator::KeyRestoreCamera(),abpage0),2);

			abpage0->AddSpace(10);

			iggWidgetKeyLargeIntSlider *af = new iggWidgetKeyLargeIntSlider(1,150,5,"Frames per file",iAnimator::KeyNumberOfFrames(),_RenderModeNoRender,abpage0);
			af->SetRange(1,150,1);
			abpage0->AddLine(af,2);

			abpage0->AddSpace(10);

			iggFrame *abut = new iggFrame(abpage0,3);
			abut->AddLine(0,new AnimateButton(abut));
			abut->SetColStretch(0,10);
			abut->SetColStretch(2,10);
			abpage0->AddLine(abut,3);

			abpage0->SetColStretch(1,10);
			abpage0->SetColStretch(2,3);
		}
		//
		//  Settings page
		//
		{
			iggFrame *abpage1 = new iggFrame(ab,2);
			ab->AddPage("Settings",icon,abpage1);

			iggWidgetKeyFloatSlider *tmpfs = new iggWidgetKeyFloatSlider(1.0e-4,1.0,40,1,0,"Fly-by speed",iAnimator::KeyFlybySpeed(),_RenderModeNoRender,abpage1);
			tmpfs->SetStretch(2,10);
			abpage1->AddLine(tmpfs);
			tmpfs = new iggWidgetKeyFloatSlider(0.0,0.1,30,0,0,"Cross section speed",iAnimator::KeySlideSpeed(),_RenderModeNoRender,abpage1);
			tmpfs->SetStretch(2,10);
			abpage1->AddLine(tmpfs);

			abpage1->AddSpace(2);

			iggFrame *ar = new iggFrame("Rotations",abpage1,1);
			tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Azimuth",iAnimator::KeyPhi(),_RenderModeNoRender,ar);
			tmpfs->SetStretch(3,10,0);
			ar->AddLine(tmpfs);
			tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Elevation",iAnimator::KeyTheta(),_RenderModeNoRender,ar);
			tmpfs->SetStretch(3,10,0);
			ar->AddLine(tmpfs);
			tmpfs = new iggWidgetKeyFloatSlider(-10.0,10.0,200,0,4,"Roll angle",iAnimator::KeyRoll(),_RenderModeNoRender,ar);
			tmpfs->SetStretch(3,10,0);
			ar->AddLine(tmpfs);
			abpage1->AddLine(ar);
			abpage1->AddLine(new iggWidgetKeyFloatSlider(0.98,1.02,400,1,5,"Scale",iAnimator::KeyZoom(),_RenderModeNoRender,abpage1));

			abpage1->AddSpace(10);
			abpage1->SetColStretch(0,10);
			abpage1->SetColStretch(1,3);
		}
		//
		//  Scripting page
		//
		{
			iggFrame *abpage2 = new iggFrame(ab,2);
			ab->AddPage("Scripting",icon,abpage2);

			abpage2->AddLine(new iggWidgetKeyCheckBox("Use script",iAnimator::KeyUseScript(),abpage2));
			abpage2->AddLine(new iggWidgetKeyCheckBox("Inherit settings from widgets",iAnimator::KeyInheritSettings(),abpage2));
			abpage2->AddLine(new iggWidgetKeyTextLineEdit(false,"Current script",iAnimator::KeyScriptFileName(),_RenderModeNoRender,abpage2));
			abpage2->AddLine(new iggWidgetLaunchButton(this->GetMainWindow()->GetDialogScriptDebugger(),"Launch script debugger",abpage2));

			abpage2->AddSpace(10);
			abpage2->SetColStretch(0,10);
			abpage2->SetColStretch(1,3);
		}
		//
		//  Advanced page
		//
		{
			iggFrame *abpage3 = new iggFrame(ab,2);
			ab->AddPage("Advanced",icon,abpage3);

			iggWidgetKeySpinBox *tmpsb = new iggWidgetKeySpinBox(0,999,"Number of blended frames",0,iAnimator::KeyNumberOfBlendedFrames(),abpage3);
			tmpsb->SetStretch(10,0);
			abpage3->AddLine(tmpsb);
			tmpsb = new iggWidgetKeySpinBox(0,999,"Number of transition frames",0,iAnimator::KeyNumberOfTransitionFrames(),abpage3);
			tmpsb->SetStretch(10,0);
			abpage3->AddLine(tmpsb);

			abpage3->AddSpace(5);

			iggFrame *acp = new iggFrame("Camera path",abpage3,2);
			tmpsb = new iggWidgetKeySpinBox(2,9999,"Number of steps",0,iAnimator::KeyNumberOfCameraPathSteps(),acp);
			tmpsb->SetStretch(10,0);
			StepOnPath *sop = new StepOnPath(acp);
			tmpsb->AddDependent(sop);
			acp->AddLine(tmpsb);
			tmpsb = new iggWidgetKeySpinBox(2,1000,"Number of handles",0,iAnimator::KeyNumberOfCameraPathHandles(),acp);
			tmpsb->SetStretch(10,0);
			acp->AddLine(tmpsb);
			acp->AddLine(new iggWidgetKeyCheckBox("Make a loop",iAnimator::KeyCameraPathClosed(),acp));
			acp->AddLine(new iggWidgetKeyCheckBox("Use focal point path",iAnimator::KeyFocalPathEnabled(),acp));
			acp->AddLine(new iggWidgetKeyCheckBox("Collapse focal path into a point",iAnimator::KeyFocalPathToPoint(),acp));
			acp->AddLine(new iggWidgetKeyCheckBox("Stop on path",iAnimator::KeyStopOnPath(),acp));
			acp->AddLine(sop,new PlayDemo(acp));

			abpage3 ->AddLine(acp);

			abpage3->AddSpace(5);

			abpage3->AddLine(new ResetAnimationImageIndexButton(abpage3));

			abpage3->AddSpace(10);
			abpage3->SetColStretch(0,10);
			abpage3->SetColStretch(1,3);
		}
		//
		//  Title/logo page
		//
		{
			iggFrame *abpage4 = new iggFrame(ab,2);
			ab->AddPage("Title/logo",icon,abpage4);

			iggFrame *atbox = new iggFrame("Title",abpage4,3);
			iggFrame *atf = new iggFrame(atbox,3);
			iggWidgetKeyTextLineEdit *atle = new iggWidgetKeyTextLineEdit(true,"",iAnimator::KeyTitlePageFile(),_RenderModeNoRender,atf);
			atf->AddLine(atle,3);
			ViewImageArea *atia = new ViewImageArea(atbox);
			LoadButton *alb = new LoadButton(atia,"Open title page file",iAnimator::KeyTitlePageFile(),iAnimator::KeyTitlePageImage(),atf);
			RemoveButton *arb = new RemoveButton(atia,iAnimator::KeyTitlePageFile(),atf);
			ViewButton *avb = new ViewButton(atia,atf);
			alb->AddDependent(avb);
			arb->AddDependent(avb);
			atf->AddLine(alb,arb,avb);

			atbox->AddLine(atf,2,atia);

			iggWidgetKeySpinBox *tmpsb = new iggWidgetKeySpinBox(0,1000,"Frames to show title",0,iAnimator::KeyNumberOfTitlePageFrames(),atbox);
			tmpsb->SetStretch(10,0);
			atbox->AddLine(tmpsb,2);
			tmpsb = new iggWidgetKeySpinBox(0,1000,"Frames to dissolve title",0,iAnimator::KeyNumberOfTitlePageBlendedFrames(),atbox);
			tmpsb->SetStretch(10,0);
			atbox->AddLine(tmpsb,2);

			atbox->SetColStretch(1,1);
			atbox->SetColStretch(2,10);
			abpage4->AddLine(atbox);

			iggFrame *albox = new iggFrame("Logo",abpage4,3);
			iggFrame *alf = new iggFrame(albox,3);
			iggWidgetKeyTextLineEdit *alle = new iggWidgetKeyTextLineEdit(true,"",iAnimator::KeyLogoFile(),_RenderModeNoRender,alf);
			alf->AddLine(alle,3);
			ViewImageArea *alia = new ViewImageArea(albox);
			alb = new LoadButton(alia,"Open title page file",iAnimator::KeyLogoFile(),iAnimator::KeyLogoImage(),alf);
			arb = new RemoveButton(alia,iAnimator::KeyLogoFile(),alf);
			avb = new ViewButton(alia,alf);
			alb->AddDependent(avb);
			arb->AddDependent(avb);
			alf->AddLine(alb,arb,avb);

			albox->AddLine(alf,2,alia);

			iggWidgetKeyComboBox *tmpcb = new iggWidgetKeyComboBox("Position",0,iAnimator::KeyLogoPosition(),albox);
			tmpcb->InsertItem("Upper left");
			tmpcb->InsertItem("Upper right");
			tmpcb->InsertItem("Lower left");
			tmpcb->InsertItem("Lower rigt");
			tmpcb->SetStretch(10,0);
			albox->AddLine(tmpcb,2);
			iggWidgetKeyFloatSlider *tmpfs = new iggWidgetKeyFloatSlider(0,1.0,10,0,4,"Opacity",iAnimator::KeyLogoOpacity(),_RenderModeNoRender,albox);
			tmpfs->SetStretch(10,0);
			albox->AddLine(tmpfs,2);

			albox->SetColStretch(1,1);
			albox->SetColStretch(2,10);
			abpage4->AddLine(albox);
			abpage4->AddSpace(10);
			abpage4->SetColStretch(0,10);
			abpage4->SetColStretch(1,3);
		}
	}

	//
	//  Markers page
	// ************************************************
	//
	{
		iggFrame *page3 = new iggFrame(mBook,3);
		mBook->AddPage("Markers",icon,page3);
		//
		//  Book
		//
		iggFrameBook *mb = new iggFrameBook(page3);
		page3->AddLine(mb);
		{
			//
			//  Main page
			//
			iggFrame *mbpage0 = new iggFrame(mb,4);
			mb->AddPage("Main",icon,mbpage0);

			iggFrame *msb = new MarkerFrameBox(mbpage0);
			CurrentMarkerBox *mcm = new CurrentMarkerBox(msb);
			msb->AddLine(new iggWidgetTextArea("Current",msb),mcm);
			iggWidgetKeyComboBox *mcs = new iggWidgetKeyComboBox("",0,iMarkerViewObject::KeyType(),msb);
			mcs->AddDependent(mcm);
			mcm->AddDependent(mbpage0);
			for(i=0; i<__NumPointTypes; i++)
			{
				mcs->InsertItem(iPointGlyph::GetName(i));
			}
			msb->AddLine(new iggWidgetTextArea("Type",msb),mcs);
			msb->AddLine(new iggWidgetTextArea("Caption",msb),new iggWidgetKeyTextLineEdit(false,"",iMarkerViewObject::KeyCaptionText(),_RenderModeUseGlobal,msb));
			msb->AddLine(new iggWidgetTextArea("Marker size",msb),new iggWidgetKeySizeSlider(1.0e-5,0,"",iMarkerViewObject::KeySize(true),_RenderModeUseGlobal,msb));

			iggFrame *mcb = new iggFrame("Controls",mbpage0,1);
			iggFrameCreateDeleteButton *mcd = new iggFrameCreateDeleteButton(page3,iMarkerViewObject::Type(),false,"marker",mcb);
			mcd->AddDependent(mbpage0);
			mcb->AddLine(mcd);

			MarkerMoveButton *mmm = new MarkerMoveButton(mcd,mcm,"Move marker",iViewModule::KeyPlaceMarker(),mcb);
			MarkerMoveButton *mmc = new MarkerMoveButton(mcd,mcm,"Move caption",iViewModule::KeyMoveMarkerCaption(),mcb);
			mmm->SetBuddy3(mmc);
			mmc->SetBuddy3(mmm);

			mcb->AddLine(mmm);
			mcb->AddLine(mmc);

			mcb->AddSpace(10);

			mbpage0->AddLine(msb,(iggWidget *)0,mcb);

			iggFrame *mfc = new iggFrame("Colors",mbpage0,1);
			iggWidgetKeyColorSelection *mcc = new iggWidgetKeyColorSelection(iMarkerViewObject::KeyColor(),mfc,false);
			mcc->SetText("Marker");
			mfc->AddLine(mcc);
			mcc = new iggWidgetKeyColorSelection(iMarkerViewObject::KeyTextColor(),mfc,false);
			mcc->SetText("Caption");
			mfc->AddLine(mcc);

			iggFramePosition *mpos = new MarkerPositionFrame(mbpage0);
			mbpage0->AddLine(mpos,2,mfc,1);
			mmm->AddDependent(mpos);

			mbpage0->AddLine(new iggWidgetKeyCheckBox("Move objects interactively",iMarkerViewObject::KeyInteractiveMove(),mbpage0),3);
			mbpage0->AddLine(new iggWidgetKeyCheckBox("Scale marker to keep size fixed",iMarkerViewObject::KeyScaled(),mbpage0),3);

			mbpage0->AddSpace(10);
			mbpage0->SetColStretch(1,10);
			mbpage0->SetColStretch(3,3);
		}
		//
		//  Material page
		//
		{
			iggFrame *mbpage1 = new iggFrame(mb,2);
			mb->AddPage("Material ",icon,mbpage1);

			mbpage1->AddLine(new iggFrameMaterialProperties(false,false,iMarkerViewObject::Type(),mbpage1));
			mbpage1->AddLine(new iggWidgetKeyFloatSlider(0.0f,1.0f,10,0,0,"Opacity",iMarkerViewObject::KeyOpacity(),_RenderModeUseGlobal,mbpage1));

			mbpage1->AddSpace(10);
			mbpage1->SetColStretch(0,10);
			mbpage1->SetColStretch(1,3);
		}
		//
		//  Legend page
		//
		{
			iggFrame *mbpage2 = new iggFrame(mb,2);
			mb->AddPage("Legend",icon,mbpage2);

			mbpage2->AddLine(new iggWidgetKeyCheckBox("Show legend",iViewModule::KeyMarkerLegend(),mbpage2));
			iggWidgetKeyRadioBox *mlp = new iggWidgetKeyRadioBox(1,"Legend position",0,iViewModule::KeyMarkerLegendPosition(),mbpage2);
			mlp->InsertItem("Lower right corner");
			mlp->InsertItem("Lower left corner");

			mbpage2->AddLine(mlp);

			mbpage2->AddSpace(10);
			mbpage2->SetColStretch(1,10);
		}
		//
		//  Transform page
		//
		{
			iggFrame *mbpage3 = new iggFrame(mb,2);
			mb->AddPage("Transform",icon,mbpage3);

			iggFrame *mtb = new iggFrame("Scale",mbpage3,1);
			mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"X",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,3));
			mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"Y",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,4));
			mtb->AddLine(new iggWidgetKeyFloatSlider(0.01f,100.0f,200,1,5,"Z",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,5));
			mbpage3->AddLine(mtb);

			mtb = new iggFrame("Rotate",mbpage3,1);
			mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"X",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,0));
			mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"Y",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,1));
			mtb->AddLine(new iggWidgetKeyFloatSlider(-180.0,180.0,360,0,5,"Z",iMarkerViewObject::KeyTransform(),_RenderModeUseGlobal,mtb,2));
			mbpage3->AddLine(mtb);

			mbpage3->AddSpace(10);
			mbpage3->SetColStretch(0,10);
			mbpage3->SetColStretch(1,3);
		}
	}

	//
	//  Windows page
	// ************************************************
	//
	{
		iggFrame *page4 = new iggFrame(mBook,3);
		mBook->AddPage("Windows",icon,page4);

		iggFrame *wfb = new iggFrame("Windows",page4,1);
		WindowNumberComboBox *wnb = new WindowNumberComboBox(wfb);
		wfb->AddLine(wnb);
		CreateWindowButton *wcb1 = new CreateWindowButton(_ModuleTypeNew,wfb);
		wcb1->AddDependent(wnb);
		wfb->AddLine(wcb1);
		CreateWindowButton *wcb2 = new CreateWindowButton(_ModuleTypeCopy,wfb);
		wcb2->AddDependent(wnb);
		wfb->AddLine(wcb2);
		CreateWindowButton *wcb3 = new CreateWindowButton(_ModuleTypeClone,wfb);
		wcb3->AddDependent(wnb);
		wfb->AddLine(wcb3);
		DeleteWindowButton *wdb = new DeleteWindowButton(wfb);
		wfb->AddLine(wdb);
		wcb1->AddDependent(wdb);
		wcb2->AddDependent(wdb);
		wcb3->AddDependent(wdb);

		WindowStatusIcon *wsi = new WindowStatusIcon(page4);
		wnb->AddDependent(wsi);
		wcb1->AddDependent(wsi);
		wcb2->AddDependent(wsi);
		wcb3->AddDependent(wsi);
		wdb->AddDependent(wsi);

		page4->AddLine(wfb,wsi);

		page4->AddLine(new WidgetsControlAllWindowsCheckBox(page4));
		page4->AddLine(new WindowUnderFocusCurrentCheckBox(page4));
		page4->AddLine(new iggWidgetKeyCheckBox("Synchronize window interactors",iControlModule::KeySynchronizeInteractors(),page4));
		page4->AddLine(new iggWidgetKeyExecButton("Synchronize cameras",iControlModule::KeySynchronizeCameras(),page4));

		page4->AddSpace(10);
		page4->SetColStretch(2,10);
	}
}


iggPageView::~iggPageView()
{
	delete mWindowSizeDialog;
}


void iggPageView::UpdateMarkerWidgets()
{
	if(mBook->GetPage(3) != 0) mBook->GetPage(3)->UpdateWidget();
}


iggFrame* iggPageView::CreateFlipFrame(iggFrameBase *parent) const
{
	iggFrame *tmp = new iggFrame(parent,2);
	tmp->AddSpace(1);
	tmp->AddLine(0,new ClearLogButton(tmp));
	tmp->SetColStretch(0,10);
	return tmp;
}

