/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggwidgetarea.h"


#include "ihistogram.h"
#include "imath.h"

#include "iggframehistogramareaextra.h"

#include "ibgwidgetareasubject.h"

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


//
//  TextArea widget
//
iggWidgetTextArea::iggWidgetTextArea(const iString &text, iggFrame *parent) : iggWidget(parent)
{
	mSubject = iggSubjectFactory::CreateWidgetDisplayAreaSubject(this,text);

	mNeedsBaloonHelp = false;
}


void iggWidgetTextArea::SetText(const iString &text)
{
	mSubject->SetText(text);
}


void iggWidgetTextArea::AlignLeft(bool s)
{
	mSubject->AlignLeft(s);
}


bool iggWidgetTextArea::UpdateWidgetBody()
{
	//
	//  Nothing to do here
	//
	return true;
}


//
//  DrawArea widget - abstract base class for all drawing widgets
//
iggWidgetDrawArea::iggWidgetDrawArea(iggFrame *parent, bool interactive) : iggWidget(parent)
{
	mSubject = iggSubjectFactory::CreateWidgetDrawAreaSubject(this,interactive);
}


bool iggWidgetDrawArea::UpdateWidgetBody()
{
	mSubject->RequestPainting();
	return true;
}


void iggWidgetDrawArea::DrawBackground()
{
	this->DrawBackgroundBody();
}


void iggWidgetDrawArea::DrawForeground()
{
	this->DrawForegroundBody();
}


//
//  ImageArea widget
//
iggWidgetImageArea::iggWidgetImageArea(iggFrame *parent) : iggWidgetDrawArea(parent,false)
{
	mNeedsBaloonHelp = false;
}


iggWidgetImageArea::iggWidgetImageArea(const iImage &image, bool scaled, iggFrame *parent) : iggWidgetDrawArea(parent,false)
{
	mNeedsBaloonHelp = false;

	this->SetImage(image,scaled);
}


void iggWidgetImageArea::DrawBackgroundBody()
{
	if(mScaled)
	{
		mSubject->DrawImage(0,0,mSubject->Width()-1,mSubject->Height()-1,mImage,iImage::_Both);
	}
	else
	{
		mSubject->DrawImage(0,0,mSubject->Width()-1,mSubject->Height()-1,mImage,iImage::_FitMin);
	}
}


void iggWidgetImageArea::SetImage(const iImage &image, bool scaled)
{
	mImage = image;
	mScaled = scaled;
}


void iggWidgetImageArea::SetFixedSize(int w, int h)
{
	mSubject->SetFixedSize(w,h);
}


//
//  HistogramArea widget (with a histogram as a background)
//
iggWidgetHistogramArea::iggWidgetHistogramArea(bool inlog, bool withticks, iggFrame *parent) : iggWidgetDrawArea(parent,true)
{
	mInLog = inlog;
	mWithTicks = withticks;
	mHistogram = 0;
	mRange = 0;
	mExtra = 0;
}


void iggWidgetHistogramArea::SetHistogram(const iHistogram *h, float *range)
{
	mHistogram = h;
	mRange = range;
}
	

void iggWidgetHistogramArea::DrawHistogram(const iColor &drawColor)
{
	static float topOffset = 0.95f;
	static const iColor tickColor = iColor(60,60,60);    // tick mark color
	static int tickWidth = 3;
	int i, ixmax, iymax, ix1, ix2, iy1, iytick;
	float x1, x2, dx, yoffset;

	if(mHistogram == 0)	return;
	
	float ys, ymax, ymin;
	ymax = ymin = mHistogram->Y(1);
	for(i=2; i<mHistogram->N()-1; i++)
	{
		if(ymax < mHistogram->Y(i)) ymax = mHistogram->Y(i);
		if(ymin > mHistogram->Y(i)) ymin = mHistogram->Y(i);
	}

	if(ymax < 1) return;

	float ynorm = float(1.0/log(2*ymax));

	ixmax = mSubject->Width() - 1;
	iymax = mSubject->Height() - 1;
	
	yoffset = topOffset*iymax;
	iytick = round(0.05*iymax);
	
	dx = float(1.0+(float)ixmax)/(mHistogram->N()-2);
	x2 = 0.0;
	if(mRange != 0)
	{
		dx *= (mRange[1]-mRange[0]);
		x2 += mRange[0]*(1.0f+ixmax);
	}
	ix2 = round(x2);

	for(i=1; i<mHistogram->N()-1; i++)  
	{
		x1 = x2;
		ix1 = ix2;
		x2 += dx;
		ix2 = round(x2);
		if(ix2 > ix1)
		{
			if(mInLog)
			{
				ys = ynorm*float(log(1.0e-30f+2*mHistogram->Y(i)));
				if(ys < 0.0) ys = 0.0;
			}
			else
			{
				ys = mHistogram->Y(i)/ymax;
			}
			iy1 = round(yoffset*ys);
			if(ix2 > ix1+1)
			{
				mSubject->DrawRectangle(ix1,0,ix2-1,iy1,drawColor,drawColor);
			}
			else
			{
				mSubject->DrawLine(ix1,0,ix1,iy1,drawColor);
			}
			if(mWithTicks && mHistogram->N()<20) 
			{
				mSubject->DrawLine((ix1+ix2)/2,0,(ix1+ix2)/2,iytick,tickColor,tickWidth);
			}
		}
	}

	if(mExtra != 0) mExtra->UpdateWidget();
}


void iggWidgetHistogramArea::DrawBackgroundBody()
{
	this->DrawHistogram();
}


void iggWidgetHistogramArea::SetLogScale(bool s)
{
	mInLog = s;
	this->UpdateWidget();
}


void iggWidgetHistogramArea::GetHistogramRange(float &min, float &max) const
{
	if(mHistogram == 0) return;
	min = mHistogram->GetMinValue();
	max = mHistogram->GetMaxValue();
}


void iggWidgetHistogramArea::AddExtra(iggFrameHistogramAreaExtra *extra)
{
	mExtra = extra;
}


