/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
  
//
//  This class is based on original vtkQGLRenderWindow class written by Manish P. Pagey [pagey@drcsdca.com ]
//

#include "iqtrenderwindowsubject.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "iviewmodule.h"

#include "iggmainwindow.h"
#include "iggrenderwindow.h"
#include "iggshell.h"

#include "iqtassistant.h"
#include "iqtmainwindowsubject.h"
#include "iqtwindowhelper.h"

#ifdef IQT_3
#include <qapplication.h>
#include <qimage.h> 
#include <qlayout.h>
#include <qtimer.h>
#else
#include <QtCore/QTimer>
#include <QtGui/QApplication>
#include <QtGui/QImage> 
#include <QtGui/QLayout>
#include <QtGui/QMouseEvent>
#endif


iqtRenderWindowSubject::iqtRenderWindowSubject(iggRenderWindow *owner, QWidget* parent) : 
#ifdef IQT_3
	QGLWidget(parent,0,0,parent?Qt::WFlags(0):iqtAssistant::GetFlags(iqtAssistant::_VisualizationWindow)), 
#else
	QGLWidget(parent,0,parent?Qt::WFlags(0):iqtAssistant::GetFlags(iqtAssistant::_VisualizationWindow)), 
#endif
	ibgRenderWindowSubject(owner)
{
	mWindowHelper = new iqtWindowHelper(this,iImageFactory::FindIcon("genie1vtk.png"),""); IERROR_ASSERT_NULL_POINTER(mWindowHelper);

	this->setFocusPolicy(QFocus::StrongFocus);

	//
	//  Window stuff
	//
	if(!this->doubleBuffer())
	{
		IERROR_REPORT_FATAL_ERROR("IFrIT only works in double buffering mode.");
	}
	this->setMinimumWidth(120);
	this->setMinimumHeight(120);
	this->resize(320,240);
	this->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
	this->setAutoBufferSwap(false);

	if(parent!=0 && parent->layout()!=0) iPointerCast<QGridLayout,QLayout>(parent->layout())->addWidget(this,0,0);

	mTimer = new QTimer(this); IERROR_ASSERT_NULL_POINTER(mTimer);
	if(!connect(mTimer,SIGNAL(timeout()),this,SLOT(OnTimer()))) IERROR_REPORT_FATAL_ERROR("Missed connection.");
}


iqtRenderWindowSubject::~iqtRenderWindowSubject() 
{
	if(this->parentWidget() != 0)
	{
		this->AttachToParent(0);
		delete this->parentWidget();
	}
}


void iqtRenderWindowSubject::AttachToParent(QWidget *parent)
{
	if(parent == 0)
	{
		iqtAssistant::SetParent(this,parent,iqtAssistant::GetFlags(iqtAssistant::_VisualizationWindow));
#ifndef IQT_3
		this->setFormat(this->format()); // is this Qt 4 bug?
#endif
	}
	else
	{
		iqtAssistant::SetParent(this,parent,0);
	}
#ifdef I_DEBUG
	if(!this->context()->isValid())
	{
		IERROR_REPORT_FATAL_ERROR("Invalid context in iqtRenderWindowSubject.");
	}
#endif
}


//
// Initialize the window for rendering.
//
void iqtRenderWindowSubject::Initialize(bool show) 
{
	//
	// Enable mouse tracking. It is required by trackball interactor style. 
	//
	this->setMouseTracking(true);

	//
	//  show the window
	//
	if(show)
	{
		if(this->parentWidget() != 0) this->parentWidget()->show(); else this->show();
	}
}


//
//  Render the scene
//
void iqtRenderWindowSubject::Render()  
{
	this->updateGL();
}


//
// End the rendering process and display the image. 
//
void iqtRenderWindowSubject::SwapBuffers() 
{
	this->swapBuffers();
}


void iqtRenderWindowSubject::SetFullScreen(bool s) 
{
	if(s) this->showFullScreen(); else this->showNormal();
}


void iqtRenderWindowSubject::MakeCurrent() 
{
	this->makeCurrent();
}


void* iqtRenderWindowSubject::GetGenericWindowId() const
{
	return (void *)this->winId();
}


void iqtRenderWindowSubject::SetWindowName(const iString &s)
{
#ifdef IQT_3
	this->setCaption(iqtAssistant::Convert(s));
#else
	this->setWindowTitle(iqtAssistant::Convert(s));
#endif
} 


void iqtRenderWindowSubject::RenderIntoMemory(iImage &image)
{
#ifdef IQT_3
	QImage q = this->renderPixmap(image.Width(),image.Height(),false).convertToImage();
#else
	QImage q = this->renderPixmap(image.Width(),image.Height(),false).toImage();
#endif
	image = iqtAssistant::Convert(q);
}


bool iqtRenderWindowSubject::IsReadyForDumpingImage()
{
	if(this->isMinimized()) 
	{
		this->showNormal();
		if(this->isMinimized()) 
		{
			const iggShell *s = iPointerCast<const iggShell,iShell>(this->GetShell());
			if(s != 0) s->GetMainWindow()->PopupWindow("The visualization window must be visible.\n Please, restore the visualization window before creating an image.");
			return false;
		}
	}
	//
	//  Make sure we are ready for the action...
	//
	this->raise();
	qApp->flush();
//	qApp->processEvents();

	if(this->isVisible()) return true; else return false;
}


//
//  RenderWindowInteractor methods
//
int iqtRenderWindowSubject::CreateTimer(int) 
{
	//
	// Start a one-shot timer for 10ms. 
	//
#ifdef IQT_3
	mTimer->start(10,true);
#else
	mTimer->setSingleShot(true);
	mTimer->start(10);
#endif
	return 1;
}


int iqtRenderWindowSubject::DestroyTimer() 
{
	//
	// QTimer will automatically expire after 10ms. So 
	// we do not need to do anything here.
	//
	return 1;
}


// ------------------------------------------------------------
// Methods from QGLWidget class. 
//
void iqtRenderWindowSubject::initializeGL() 
{
	//
	//  I don't think we need this - it crashes X-server over the remote connection
	//
	mOwner->OpenGLInit();
}


void iqtRenderWindowSubject::paintGL() 
{
	if(!mOwner->IsInitialized()) return;

	//
	//  iqtRenderWindowSubject only works with AutoBufferSwap turned off
	//
	if(this->autoBufferSwap())
	{
		IERROR_REPORT_ERROR("iqtRenderWindowSubject only works with AutoBufferSwap turned off.");
		return;
	}

	//
	//  Must do it this way so that child Render operations are not called from here
	//
	mOwner->vtkOpenGLRenderWindow::Render();
}


void iqtRenderWindowSubject::resizeGL(int w, int h) 
{
	//
	//  I do not see why we need it
	//
}


void iqtRenderWindowSubject::mouseMoveEvent(QMouseEvent *event) 
{
#ifdef IQT_3
	this->OnMouseMoveBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->state()));
#else
	this->OnMouseMoveBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->buttons() | event->modifiers()));
#endif
}


void iqtRenderWindowSubject::mousePressEvent(QMouseEvent *event) 
{
#ifdef IQT_3
	this->OnMousePressBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->stateAfter()));
#else
	this->OnMousePressBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->buttons() | event->modifiers()));
#endif
}


void iqtRenderWindowSubject::mouseReleaseEvent(QMouseEvent *event) 
{
#ifdef IQT_3
	this->OnMouseReleaseBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->state()));
#else
	this->OnMouseReleaseBody(event->x(),this->height()-event->y()-1,iqtAssistant::Convert(event->buttons() | event->button() | event->modifiers()));
#endif
}


void iqtRenderWindowSubject::keyPressEvent(QKeyEvent * event)
{
#ifdef IQT_3
	this->OnKeyPressBody(event->key(),iqtAssistant::Convert(event->state()));
#else
	this->OnKeyPressBody(event->key(),iqtAssistant::Convert(event->modifiers()));
#endif
}


void iqtRenderWindowSubject::OnTimer() 
{
	this->OnTimerBody();
}


bool iqtRenderWindowSubject::event(QEvent *e)
{
	if(e!=0 && e->spontaneous() && this->IsBlocked()) 
	{
		return false;
	}
	else
	{
		return QGLWidget::event(e);
	}
}

//
//  Event handles
//
void iqtRenderWindowSubject::closeEvent(QCloseEvent *)
{
}


void iqtRenderWindowSubject::focusInEvent(QFocusEvent *e)
{ 
    QGLWidget::focusInEvent(e);
//	repaint(false);  not sure why we need it
	this->OnWindowFocusInBody();
}


void iqtRenderWindowSubject::focusOutEvent(QFocusEvent *e)
{
    QGLWidget::focusOutEvent(e);
//	repaint(false);
	this->OnWindowFocusOutBody();
}
	

void iqtRenderWindowSubject::enterEvent(QEvent *e)
{ 
    QGLWidget::enterEvent(e);
	if(this->hasFocus()) this->OnWindowEnterBody();
}


void iqtRenderWindowSubject::leaveEvent(QEvent *e)
{
    QGLWidget::leaveEvent(e);
	if(this->hasFocus()) this->OnWindowLeaveBody();
}
	

void iqtRenderWindowSubject::resizeEvent(QResizeEvent *e)
{
    QGLWidget::resizeEvent(e);
	this->OnWindowResizeBody();
}


void iqtRenderWindowSubject::moveEvent(QMoveEvent *e)
{
    QGLWidget::moveEvent(e);
    this->OnWindowMoveBody();
}


