/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  A base class for all DataSubjects. One DataSubject provides data for a single unique DataType.
//
#ifndef IDATASUBJECT_H
#define IDATASUBJECT_H


#include "iobject.h"


class iDataLimits;
class iDataSyncRequest;
class iDataType;
class iFileLoader;
class iHistogramMaker;
class iProbeFilter;
class iViewSubject;

class vtkDataSet;


//
//  Useful macro to declare all keys that have to be present in children too
//
#define IDATASUBJECT_DECLARE_INHERITED_KEYS \
	static const iObjectKey& KeyResizeLimits(); \
	static const iObjectKey& KeyFixedLimits(); \
	static const iObjectKey& KeyDataPresent(); \
	static const iObjectKey& KeyFileName(); \
	static const iObjectKey& KeyNumVars(); \
	static const iObjectKey& KeyStretch(); \
	static const iObjectKey& KeyMin(); \
	static const iObjectKey& KeyMax(); \
	static const iObjectKey& KeyLowerLimit(); \
	static const iObjectKey& KeyUpperLimit(); \
	static const iObjectKey& KeyFixedStretch(); \
	static const iObjectKey& KeyName()


class iDataSubject : public iObject
{

	IPOINTER_AS_PART(ViewModule);

	friend class iFileLoader;

public:

	vtkTypeMacro(iDataSubject,iObject);
	static const iObjectType& Type();

	static bool IsTypeMatch(const iString &prefix);
	static iString GetDataSubjectName(const iString &prefix);

	IDATASUBJECT_DECLARE_INHERITED_KEYS;

	void SetFixedLimits(bool s);
	inline bool GetFixedLimits() const { return mFixedLimits; }

	virtual bool IsThereData() const;

	//
	//  Access to components
	//
	inline iFileLoader* GetLoader() const { return mLoader; }
	iDataLimits* GetLimits() const;
	vtkDataSet* GetData() const;

	const iDataSyncRequest& Request(int var = -1) const;

	virtual const iObjectType& GetObjectType() const;
	virtual const iDataType& GetDataType() const = 0;

	virtual void PackCompleteState(iString &s) const;
	virtual void UnPackCompleteState(const iString &s);

	//
	//  For picking and data exploring, we need to know what ProbeFilter and HistogramMaker would work with this Subject
	//
	virtual iProbeFilter* CreateProbeFilter(iViewSubject *vo) const;
	virtual iHistogramMaker* CreateHistogramMaker() const;

protected:

	iDataSubject(iFileLoader *fl, const iString &name);
	virtual ~iDataSubject();

	virtual void DataSubjectPackStateBody(iString &) const {}
	virtual void DataSubjectUnPackStateBody(const iString &){}

	virtual iDataLimits* CreateLimits() const;
	virtual void NotifyDependencies();

	void ConfigureLimits(int num, const iString &name);

	//
	//  Components
	//
	const int mId;
	iFileLoader *mLoader;

	bool mFixedLimits;

private:

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

	int mLimitsNum;
	iString mLimitsName;
	mutable iDataLimits *mLimits; // must be accessed via GetLimits() only
};


//
//  Useful macros to declare all members that have to be overwritten in children
//
#define IDATASUBJECT_DECLARE_INHERITED_MEMBERS \
	static const iObjectType& Type(); \
	static const iDataType& DataType(); \
	virtual const iObjectType& GetObjectType() const; \
	virtual const iDataType& GetDataType() const

#define IDATASUBJECT_DEFINE_TYPE(_object_,_id_,_fname_,_sname_,_keywords_,_environment_) \
	const iDataType& _object_::DataType() \
	{ \
		static const iDataType tmp(_id_,_fname_,_sname_,_keywords_,_environment_); \
		return tmp; \
	} \
	const iObjectType& _object_::Type() \
	{ \
		static const iString fname = "Data-" + _object_::DataType().GetName(); \
		static const iString sname = "d-" + _object_::DataType().GetShortName(); \
		static const iObjectType *pval = iObjectTypeRegistry::CreateType(iObjectType::_Data,fname.ToCharPointer(),sname.ToCharPointer()); \
		return *pval; \
	} \
	const iObjectType& _object_::GetObjectType() const { return _object_::Type(); } \
	const iDataType& _object_::GetDataType() const { return _object_::DataType(); }

#define IDATASUBJECT_DEFINE_INHERITED_KEYS(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,ResizeLimits,-rl,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FixedLimits,fl,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,DataPresent,dp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FileName,fn,String,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,NumVars,n,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Stretch,s,Int,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Min,min,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Max,max,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,LowerLimit,lo,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,UpperLimit,up,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FixedStretch,fs,Bool,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Name,na,String,0)

#define IDATASUBJECT_DECLARE_CLASS(_prefix_,_name_) \
class _prefix_##_name_##DataSubject : public iDataSubject \
{ \
public: \
	vtkTypeMacro(_prefix_##_name_##DataSubject,iDataSubject); \
	_prefix_##_name_##DataSubject(iFileLoader *fl); \
	IDATASUBJECT_DECLARE_INHERITED_KEYS; \
	IDATASUBJECT_DECLARE_INHERITED_MEMBERS; \
}

#define IDATASUBJECT_DEFINE_CLASS(_prefix_,_name_,_fname_,_sname_,_keywords_,_environment_) \
IDATASUBJECT_DEFINE_TYPE(_prefix_##_name_##DataSubject,_prefix_##Extension::SubjectId(),_fname_,_sname_,_keywords_,_environment_); \
IDATASUBJECT_DEFINE_INHERITED_KEYS(_prefix_##_name_##DataSubject); \
_prefix_##_name_##DataSubject::_prefix_##_name_##DataSubject(iFileLoader *fl) : iDataSubject(fl,_prefix_##_name_##DataSubject::Type().FullName()){}

#endif

