/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticlesplitter.h"


#include "idatalimits.h"
#include "idatastretch.h"
#include "ierror.h"
#include "ihistogrammaker.h"
#include "irangecollection.h"
#include "irangemapping.h"
#include "iviewmodule.h"
#include "iviewsubject.h"

#include <vtkCellArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

#include <limits.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"


float iParticleSplitter::mAttRange = iMath::_LargeFloat;


namespace iParticleSplitter_Private
{
//
//  QuickFind
//
#define ARR(i)	Atts[ncom*(i)+attSort]

bool Find(vtkIdType n, int ncom, int attSort, float *Atts, vtkIdType iStart, float aval, vtkIdType &iLo, vtkIdType &iHi)
{
	vtkIdType i1 = iStart;
	vtkIdType i2 = n - 1;
	vtkIdType ic;

	if(aval < ARR(i1))
	{
		iLo = i1;
		iHi = i1;
		return true;
	}
	
	if(aval > ARR(i2))
	{
		iLo = i2;
		iHi = i2;
		return true;
	}

	while(i2-i1 > 1)
	{
		ic = (i1+i2)/2;
		if(aval >= ARR(ic)) i1 = ic;
		if(aval <= ARR(ic)) i2 = ic;
	}

	iLo = i1;
	iHi = i2;

	return true;

}
//
//  QuickSort
//
#define SAVE(CELL,i1)    { for(ii=0; ii<3; ii++) poi##CELL[ii] = Points[3*(i1)+ii]; for(ii=0; ii<ncom; ii++) att##CELL[ii] = Atts[ncom*(i1)+ii]; if(Nrms != 0) for(ii=0; ii<3; ii++) nrm##CELL[ii] = Nrms[3*(i1)+ii]; }
#define MOVE(i1,i2)      { for(ii=0; ii<3; ii++) Points[3*(i1)+ii] = Points[3*(i2)+ii]; for(ii=0; ii<ncom; ii++) Atts[ncom*(i1)+ii] = Atts[ncom*(i2)+ii]; if(Nrms != 0) for(ii=0; ii<3; ii++) Nrms[3*(i1)+ii] = Nrms[3*(i2)+ii]; }
#define RESTORE(i2,CELL) { for(ii=0; ii<3; ii++) Points[3*(i2)+ii] = poi##CELL[ii]; for(ii=0; ii<ncom; ii++) Atts[ncom*(i2)+ii] = att##CELL[ii]; if(Nrms != 0) for(ii=0; ii<3; ii++) Nrms[3*(i2)+ii] = nrm##CELL[ii]; }
#define SWAP(i1,i2)      { SAVE(1,i1); MOVE(i1,i2); RESTORE(i2,1); }

//
// Do our own quick sort for efficiency reason (based on a Java code by Denis Ahrens)
//

template<class T>
void Sort(iParticleSplitter *me, vtkIdType n, int ncom, int attSort, T *Points, float *Atts, float* Nrms)
{
	T poi1[3];
	float nrm1[3];
	float *att1; att1 = new float[ncom]; IERROR_ASSERT(att1);

	SortWorker(me,0,n-1,ncom,attSort,Points,Atts,Nrms,poi1,att1,nrm1);
	
	delete [] att1;
}

//
//  Recursive worker
//
template<class T>
void SortWorker(iParticleSplitter *me, vtkIdType l, vtkIdType r, int ncom, int attSort, T *Points, float *Atts, float *Nrms, T *poi1, float *att1, float *nrm1)
{
	const int M = 8;
	vtkIdType i, j;
	float v;
	int ii;

	if(me->GetAbortExecute()) return;

	if ((r-l)>M)
	{
		//
		// Use quicksort
		//
		i = (r+l)/2;
		if (ARR(l)>ARR(i)) SWAP(l,i);     // Tri-Median Method!
		if (ARR(l)>ARR(r)) SWAP(l,r);
		if (ARR(i)>ARR(r)) SWAP(i,r);
		
		j = r-1;
		SWAP(i,j);
		i = l;
		v = ARR(j);
		for(;;)
		{
			do i++; while(ARR(i) < v); // no ++i/--j in macro expansion!
			do j--; while(ARR(j) > v);
			if (j<i) break;
			SWAP(i,j);
		}
		SWAP(i,r-1);
		SortWorker(me,l,j,ncom,attSort,Points,Atts,Nrms,poi1,att1,nrm1);
		SortWorker(me,i+1,r,ncom,attSort,Points,Atts,Nrms,poi1,att1,nrm1);
	}
	else 
	{
		//
		// Array is small, use insertion sort. 
		//
		for (i=l+1; i<=r; i++)
		{
			SAVE(1,i);
			v = ARR(i);
			j = i;
			while (j>l && ARR(j-1)>v)
			{
				MOVE(j,j-1);
				j--;
			}
			RESTORE(j,1);
		}
    }
}

};


using namespace iParticleSplitter_Private;


iParticleSplitter::iParticleSplitter(iViewSubject *vo) : iGenericPolyDataToPolyDataFilter<vtkPolyDataToPolyDataFilter>(vo,1,true,true)
{
	mRangeSet = mOwnsData = false;

	mSavedPoints = 0;
	mSavedVerts = 0;
	mSavedNorms = 0;
	mSavedAtts = 0;

	//
	//  Splitter functionality
	//
	mAtt = -1;

	mSorted = false;

	mRanges = new iRangeCollection;  IERROR_ASSERT(mRanges);
	mHistogramMaker = iHistogramMaker::New(this->GetViewSubject()->GetViewModule()); IERROR_ASSERT(mHistogramMaker);
	mMapping = new iRangeMapping(mRanges,mHistogramMaker,this);  IERROR_ASSERT(mRanges);

	mRanges->SetGlobalRange(-mAttRange,mAttRange);
	mRanges->SetRange(0,-mAttRange,mAttRange);

	mSortingMTime = 0;
}


iParticleSplitter::~iParticleSplitter()
{
	mHistogramMaker->Delete();
	delete mRanges;
	delete mMapping;
	if(mOwnsData)
	{
		mSavedPoints->Delete();
		mSavedVerts->Delete();
		mSavedNorms->Delete();
		mSavedAtts->Delete();
	}
}


int iParticleSplitter::GetNumberOfPieces() const
{
	return mRanges->GetN();
}


float iParticleSplitter::GetMemorySize() const
{
	float s = 0.0;
	if(mOwnsData) 
	{
		s += mSavedPoints->GetActualMemorySize();
		s += mSavedVerts->GetActualMemorySize();
		s += mSavedNorms->GetActualMemorySize();
		s += mSavedAtts->GetActualMemorySize();
	}
	return s;
}


iRangeMapping* iParticleSplitter::GetSplitRanges()
{
	this->Modified();
	return mMapping;
}


bool iParticleSplitter::CreateGroup()
{ 
	if(mAtt == -1) return false;

	//
	//  Create a new piece
	//
	vtkPolyData *newOutput;
	newOutput = vtkPolyData::New(); 
	if(newOutput == 0)
	{
		return false;
	}
	this->AddOutput(newOutput);
	newOutput->Delete();

	mRanges->AddRange();
	this->Modified();

	return true;
}


bool iParticleSplitter::DeleteGroup(int n)
{
	if(mRanges->RemoveRange(n))
	{
		this->RemoveOutput(Outputs[n]);
		int i;
		for(i=n; i<mRanges->GetN(); i++)
		{
			this->SetNthOutput(i,this->GetOutput(i+1));
		}
		return true;
	}
	else return false;
}


void iParticleSplitter::SetSplitRangesStretch(int s)
{ 
	mRanges->SetStretch(s);
	this->Modified();
}


int iParticleSplitter::GetSplitRangesStretch() const
{ 
	return mRanges->GetStretch();
}


void iParticleSplitter::SetSplitRangesTiled(bool s)
{ 
	mRanges->SetTiled(s);
	this->Modified();
}


bool iParticleSplitter::GetSplitRangesTiled() const
{ 
	return mRanges->GetTiled();
}


void iParticleSplitter::SetSplitRangesMinMax(float amin, float amax)
{
	mRanges->SetGlobalRange(amin,amax);
	this->Modified();
}


void iParticleSplitter::GetSplitRangesMinMax(float &amin, float &amax) const
{
	amin = mRanges->GetGlobalMin();
	amax = mRanges->GetGlobalMax();
}


void iParticleSplitter::SetSplitRangeLimits(int n, float amin, float amax)
{
	mRanges->SetRange(n,amin,amax);
	this->Modified();
}


void iParticleSplitter::GetSplitRangeLimits(int n, float &amin, float &amax) const
{
	mRanges->GetRange(n,amin,amax);
}


void iParticleSplitter::TakeOverData(bool s)
{
	if(mOwnsData == s) return;
	mOwnsData = s;
	if(mOwnsData)
	{
		mSavedPoints = vtkPoints::New(); IERROR_ASSERT(mSavedPoints);
		mSavedVerts = vtkCellArray::New(); IERROR_ASSERT(mSavedVerts);
		mSavedNorms = vtkFloatArray::New(); IERROR_ASSERT(mSavedNorms);
		mSavedAtts = vtkFloatArray::New(); IERROR_ASSERT(mSavedAtts);
	}
	else
	{
		mSavedPoints->Delete(); mSavedPoints = 0;
		mSavedVerts->Delete(); mSavedVerts = 0;
		mSavedNorms->Delete(); mSavedNorms = 0;
		mSavedAtts->Delete(); mSavedAtts = 0;
	}
	this->Modified();
}


void iParticleSplitter::SetAttributeToSplit(int a)
{
	//
	//  Reset to a single piece if the attribute changes
	//
	if(a!=mAtt && a>=-1 && a<this->GetLimits()->GetNumVars())
	{
		mRanges->SetGlobalRange(-mAttRange,mAttRange); 
		mRanges->SetRange(0,-mAttRange,mAttRange); 

		mAtt = a;
		mSorted = false;

		if(a == -1)
		{
			mRangeSet = true;
			mHistogramMaker->SetInput(0,0);
		}
		else
		{
			mRangeSet = false;
			mHistogramMaker->SetInput(this->GetInput(),mAtt);
			this->UpdateRange();
		}
		this->Modified();
	}
}


void iParticleSplitter::UpdateRange()
{
	if(mRangeSet) return;

	mRangeSet = false;
	//
	//  Single piece mode
	//
	if(mAtt == -1)
	{
		mRangeSet = true;
		return;
	}

	vtkPolyData *input = this->GetInput();
	if(input == 0) return;

	vtkFloatArray *inatt = vtkFloatArray::SafeDownCast(input->GetPointData()->GetScalars());
	if(inatt == 0) return;
	float *data = inatt->GetPointer(0);
	if(data == 0) return;

	int ncom = inatt->GetNumberOfComponents();
	vtkIdType ntup = inatt->GetNumberOfTuples();

	if(mAtt >= ncom) return;

	vtkIdType l;
	float d;
	//
	//  Find attribute limits
	//
	data += mAtt;  // the first component is the number in file
	float attMin, attMax;
	attMin = attMax = *data;
	for(l=1; l<ntup; l++)
	{
		d = *(data+l*ncom);
		if(d < attMin) attMin = d;
		if(d > attMax) attMax = d; 
	}
	//
	//  Spread limits a little bit so that even the limiting values are inside the interval
	//
	if(attMin < attMax)
	{
		//
		//  Find second lower and upper values
		//
		float attMin2, attMax2;
		attMin2 = attMax;
		attMax2 = attMin;
		for(l=0; l<ntup; l++)
		{
			d = *(data+l*ncom);
			if(d>attMin && d<attMin2) attMin2 = d;
			if(d<attMax && d>attMax2) attMax2 = d;
		}
		attMin -= 0.01*(attMin2-attMin);
		attMax += 0.01*(attMax-attMax2);
	}
	else
	{
		if(fabs(attMin) > 1.0) d = 0.0001*fabs(attMin); else d = 0.0001;
		attMin -= d;
		attMax += d;
	}

	mRanges->SetGlobalRange(attMin,attMax);

	this->Modified();

	mRangeSet = true;
}

	
void iParticleSplitter::ProduceOutput()
{
	vtkPolyData *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	output->Initialize();
	//
	//  Do it again in case the attribute was Set before the data were loaded, like in
	//  restoring from a state file. 
	//
	if(!mRangeSet) this->UpdateRange();
	//
	//  If updateRange did not Set the range, then the data are still not available, so exit without
	//  doing anything
	//
	if(!mRangeSet) return;

	vtkPoints *newPoints, *oldPoints;
	vtkCellArray *newVerts, *oldVerts;
	vtkFloatArray *newNorms, *oldNorms;
	vtkFloatArray *newAtts, *oldAtts;

	if(mAtt==-1 || mAtt>=this->GetLimits()->GetNumVars())
	{
		output->ShallowCopy(input);
		return;
	}

	if(mOwnsData)
	{
		oldPoints = mSavedPoints;
		oldVerts = mSavedVerts;
		oldAtts = mSavedAtts;
		oldPoints->SetDataType(input->GetPoints()->GetDataType());
		oldPoints->DeepCopy(input->GetPoints());
		oldVerts->DeepCopy(input->GetVerts());
		if(input->GetPointData()->GetNormals() != 0)
		{
			oldNorms = mSavedNorms;
			oldNorms->DeepCopy(input->GetPointData()->GetNormals());
		}
		else
		{
			oldNorms = 0;
		}
		oldAtts->DeepCopy(input->GetPointData()->GetScalars());
	}
	else
	{
		oldPoints = input->GetPoints();
		oldVerts = input->GetVerts();
		if(input->GetPointData()->GetNormals() != 0)
		{
			oldNorms = vtkFloatArray::SafeDownCast(input->GetPointData()->GetNormals());
		}
		else
		{
			oldNorms = 0;
		}
		oldAtts = vtkFloatArray::SafeDownCast(input->GetPointData()->GetScalars());
	}

	float *pPointsF = 0;
	double *pPointsD = 0;

	bool pointsAreFloat;
	switch(oldPoints->GetDataType())
	{
	case VTK_FLOAT:
		{
			pointsAreFloat = true;
			pPointsF = (float *)oldPoints->GetVoidPointer(0);
			break;
		}
	case VTK_DOUBLE:
		{
			pointsAreFloat = false;
			pPointsD = (double *)oldPoints->GetVoidPointer(0);
			break;
		}
	default: 
		{
			output->ShallowCopy(input);
			vtkErrorMacro("Incorrect Points type");
			return;
		}
	}

	vtkIdType  *pVerts = (vtkIdType *)oldVerts->GetPointer();
	float *pNorms = 0;
	if(oldNorms != 0) pNorms = (float *)oldNorms->GetPointer(0);
	float *pAtts = (float *)oldAtts->GetPointer(0);

	int ncom = oldAtts->GetNumberOfComponents();
	vtkIdType ntup = oldAtts->GetNumberOfTuples();

	if(!mSorted || mSortingMTime<input->GetMTime())
	{
		mSorted = true;
		mSortingMTime = input->GetMTime();
		if(pointsAreFloat) 
		{
			Sort(this,ntup,ncom,mAtt,pPointsF,pAtts,pNorms); 
		}
		else
		{
			Sort(this,ntup,ncom,mAtt,pPointsD,pAtts,pNorms);
		}
#ifdef I_CHECK2
		for(vtkIdType l=0; l<ntup-1; l++) if(pAtts[mAtt+ncom*l] > pAtts[mAtt+ncom*(l+1)])
		{
			IERROR_LOW("Sorted incorrectly.");
			break;
		}
#endif
	}

	int n, nmax = mRanges->GetN();
	vtkIdType iLo, iHi;
	vtkIdType iStart, iTotal;

	vtkFloatArray  *fArray;
	vtkDoubleArray *dArray;
	vtkIdTypeArray *jArray;

	for(n=0; n<nmax; n++)
	{
		if(!Find(ntup,ncom,mAtt,pAtts,0,mRanges->GetMin(n),iLo,iHi))
		{
			vtkErrorMacro("Error #1 in iGroupSplitter");
			iLo = iHi = 0;
		}
		iStart = iHi;

		if(!Find(ntup,ncom,mAtt,pAtts,iLo,mRanges->GetMax(n),iLo,iHi))
		{
			vtkErrorMacro("Error #2 in iGroupSplitter");
			iLo = iHi = ntup - 1;
		}
		iTotal = iLo - iStart + 1;

		if(iTotal<0 || iStart<0 || iLo>=ntup)
		{
			vtkErrorMacro("Error #3 in iGroupSplitter");
		}
		else
		{
			output = this->GetOutput(n);
			output->Initialize();
			
			if(iTotal > 0)
			{
				newPoints = vtkPoints::New(oldPoints->GetDataType()); IERROR_ASSERT(newPoints);

				if(pointsAreFloat)
				{
					fArray = vtkFloatArray::New(); IERROR_ASSERT(fArray);
					fArray->SetNumberOfComponents(3);
					fArray->SetArray(pPointsF+3*iStart,3*iTotal,1);
					newPoints->SetData(fArray);
					fArray->Delete();
				}
				else
				{
					dArray = vtkDoubleArray::New(); IERROR_ASSERT(dArray);
					dArray->SetNumberOfComponents(3);
					dArray->SetArray(pPointsD+3*iStart,3*iTotal,1);
					newPoints->SetData(dArray);
					dArray->Delete();
				}
				output->SetPoints(newPoints);
				newPoints->Delete();

				newAtts = vtkFloatArray::New(); IERROR_ASSERT(newAtts);
				newAtts->SetNumberOfComponents(ncom);
				newAtts->SetArray(pAtts+ncom*iStart,ncom*iTotal,1);
				output->GetPointData()->SetScalars(newAtts);
				newAtts->Delete();

				if(oldNorms != 0) 
				{
					newNorms = vtkFloatArray::New(); IERROR_ASSERT(newNorms);
					newNorms->SetNumberOfComponents(3);
					newNorms->SetArray(pNorms+3*iStart,3*iTotal,1);
					output->GetPointData()->SetNormals(newNorms);
					newNorms->Delete();
				}

				newVerts = vtkCellArray::New(); IERROR_ASSERT(newVerts);
				jArray = vtkIdTypeArray::New(); IERROR_ASSERT(jArray);
				jArray->SetNumberOfComponents(1);
				jArray->SetArray(pVerts,2*iTotal,1);
				newVerts->SetCells(iTotal,jArray);
				jArray->Delete();
				output->SetVerts(newVerts);
				newVerts->Delete();
			}
		}
	}
}


void iParticleSplitter::SyncWithData(const iDataSyncRequest &r)
{
	if(r.Var()==-1 || r.Var()==mAtt)
	{
		this->Modified();
		mRangeSet = false;
	}
}

