/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogcommandline.h"


#include "icontrolmodule.h"
#include "icommandinterpreter.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iimagefactory.h"
#include "ishell.h"

#include "iggcontrolscript.h"
#include "iggframe.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgettext.h"

#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgettexteditorsubject.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


namespace iggDialogCommandLine_Private
{
	static iString Help1 = 
		"The multi-line and the single-line input windows behave somewhat differently. The <b>single-line mode</b> is equivalent to the <a href=\"sr.cl.i\">command-line</a> shell interface. "
		"Querying commands <code><b>help</b></code>, <code><b>list</b></code>, and <code><b>print</b></code> are executed immediately, other commands are executed when a line "
		"terminated by a semi-colon (;) is typed. In the <b>multi-line mode</b>, the whole piece of the code must be typed together. The script is executed when the command terminating with "
		"the semi-colon is entered. The entered piece of code is first compiled to check for syntax errors, and is only then executed.";
	//
	//  Command Interpreter
	//
	class CommandInterpreter : public iCommandInterpreter
	{

	public:

		CommandInterpreter(iggWidgetTextBrowser *browser) : iCommandInterpreter(browser->GetShell())
		{
			mBrowser = browser;
			mCurLine = 0;
		}

		void Compile(const iString &text0, iString &msg, int &para, int &index)
		{
			iString text(text0);
			int io = text.Length() - 1;
			while(io>=0 && text.IsWhiteSpace(io)) io--;
			if(io>=0 && text[io]==';') text.Replace(io,' ');

			this->Script()->SetText(text);
			if(!this->Script()->Compile())
			{
				msg = this->Script()->GetErrorStatus()->Message();
				para = this->Script()->GetThisLine();
				index = this->Script()->GetErrorPosition();
			}
			else 
			{
				msg.Clear();
				para = -1;
				index = 0;
			}
		}

		void RunMultiLine(const iString &text)
		{
			int i;

			mCurCode = text;
			mNumLines = text.Contains('\n');
			if(mNumLines == 0)
			{
				mCurCode += "\n";
				mNumLines++;
			}

			mCurLine = 0;
			iRequiredCast<iggControlScript>(INFO,this->Script())->AttachCommandInterpreter(this);

			for(i=0; i<mNumLines; i++)
			{
				iString w(mCurCode.Section("\n",i,i));
				w.Replace("<","&lt;");
				w.Replace(">","&gt;");
				this->DislayLineOfText("<nobr><b>"+w+"</b></nobr>",_CommandInterpreterTextTypeCode);
			}

			this->Exec();

			iRequiredCast<iggControlScript>(INFO,this->Script())->AttachCommandInterpreter(0);
			//
			//  After execution
			//
			this->ResetAfterExec();
		}

		void RunSingleLine(const iString &line)
		{
			mCurCode = line;
			mNumLines = 999;  // disables CheckBreak()

			this->Script()->GetErrorStatus()->Clear(); // need to do this explicitly, otherwise it is never cleared after the first error

			mCurLine = 0;
			iRequiredCast<iggControlScript>(INFO,this->Script())->AttachCommandInterpreter(this);

			iString w(mCurCode);
			w.Replace("<","&lt;");
			w.Replace(">","&gt;");
			this->DislayLineOfText("<nobr><b>"+w+"</b></nobr>",_CommandInterpreterTextTypeCode);

			this->ExecLine(mCurCode);

			iRequiredCast<iggControlScript>(INFO,this->Script())->AttachCommandInterpreter(0);
			//
			//  After execution
			//
			this->ResetAfterExec();
		}

		virtual void DislayLineOfText(const iString &text, int type) const
		{
			static iColor error(255,0,0);
			static iColor result(0,0,0);
			static iColor code(0,0,255);

			switch(type)
			{
			case _CommandInterpreterTextTypeError:
				{
					mBrowser->AppendTextLine(text,error);
					break;
				}
			case _CommandInterpreterTextTypeResult:
				{
					mBrowser->AppendTextLine(text,result);
					break;
				}
			case _CommandInterpreterTextTypeCode:
				{
					mBrowser->AppendTextLine(text,code);
					break;
				}
			default:
				{
					IERROR_LOW("Ivalid text type.");
				}
			}
		}

		void ResetScript()
		{
			this->Script()->SetText("");
			mFullCode.Clear();
			mBrowser->Clear();
		}

		inline const iString& GetCode() const { return mFullCode; }

	protected:

		void ResetAfterExec()
		{
			static iColor c(0,120,0);

			if(this->Script()->GetErrorStatus()->NoError() && this->GetExecFlag())
			{
				mFullCode += this->GetLastExecutedScriptText();
				mBrowser->GetMainWindow()->UpdateAll();
				if(mBrowser->SupportsHTML())
				{
					mBrowser->AppendTextLine("&lt;--- Commands below this line have not been executed yet.",c);
				}
				else
				{
					mBrowser->AppendTextLine("<--- Commands below this line have not been executed yet.",c);
				}
			}
		}

		virtual void Initialize()
		{
		}

		virtual void DisplayPrompt() const
		{
		}

		virtual void ReadLineOfText(iString &line) const
		{
			if(mCurLine < mNumLines)
			{
				line = mCurCode.Section("\n",mCurLine,mCurLine); 
				mCurLine++;
			}
		}

		virtual void AnalyseSpecialCommands(const iString &line)
		{
		}

		virtual void CheckBreak()
		{
			if(mCurLine == mNumLines) this->Break(0);
		}

		virtual void Finalize()
		{
		}

		mutable int mCurLine, mNumLines;
		iString mFullCode, mCurCode;
		iggWidgetTextBrowser *mBrowser;
	};


	//
	//  Editor with button update
	//
	class CommandMultiLineEditor : public iggWidgetTextEditor
	{

	public:

		CommandMultiLineEditor(CommandInterpreter *ci, iggDialogCommandLine *dialog, iggFrame *parent) : iggWidgetTextEditor(0U,parent)
		{
			mInterpreter = ci;
			mDialog = dialog;

			this->SetBaloonHelp("Type several script commands","Type a piece of Control Script to be executed. Add a semicolon (;) at the end of the last line to execute the piece.<p>"+Help1);
		}

	protected:

		void OnReturnPressed()
		{
			int para, index;

			mSubject->GetCursorPosition(para,index);

			if(para > 0)
			{
				iString s = this->GetLine(para-1);
				s.ReduceWhiteSpace();
				if(s[s.Length()-1] == ';')
				{
					s.Clear();
					int i, p;
					for(i=0; i<para; i++)
					{
						s += this->GetLine(i) + "\n";
					}

					iString msg;
					mInterpreter->Compile(s,msg,p,i);
					if(msg.IsEmpty())
					{
						mInterpreter->RunMultiLine(s);
						if(mInterpreter->Script()->GetErrorStatus()->NoError())
						{
							for(i=0; i<para; i++) this->RemoveLine(0);
						}
					}
					else
					{
						this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),msg,_PopupWindowError);
						mSubject->Select(p,i,p,999);
					}
				}
			}
			else
			{
#ifdef I_CHECK1
				IERROR_REPORT_BUG;
#endif
			}
		}

		CommandInterpreter *mInterpreter;
		iggDialogCommandLine *mDialog;
	};

	//
	//  Single line command editor
	//
	class CommandSingleLineEditor : public iggWidget
	{

	public:

		CommandSingleLineEditor(CommandInterpreter *ci, iggDialogCommandLine *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mInterpreter = ci;
			mDialog = dialog;
			mCurHistoryLine = -1;

			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,false,0,"",0);
			this->SetBaloonHelp("Type a script command","Type a Control Script command to be executed. Add a semicolon (;) at the end of the command to execute this and all previous unexecuted commands.<p>"+Help1);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		virtual void OnVoid1Body()
		{
			mHistory.Add(mSubject->GetText());
			mInterpreter->RunSingleLine(mSubject->GetText());
			if(mInterpreter->Script()->GetErrorStatus()->NoError())
			{
				mSubject->SetText("");
				mCurHistoryLine = -1;
			}
		}

		virtual void OnBool1Body(bool up)
		{
			if(up)
			{
				if(mCurHistoryLine == -1)
				{
					mCurLine = mSubject->GetText();
				}
				if(mCurHistoryLine < mHistory.MaxIndex())
				{
					mCurHistoryLine++;
					mSubject->SetText(mHistory[mHistory.MaxIndex()-mCurHistoryLine]);
				}
			}
			else
			{
				if(mCurHistoryLine >= 0)
				{
					mCurHistoryLine--;
					if(mCurHistoryLine == -1)
					{
						mSubject->SetText(mCurLine);
					}
					else
					{
						mSubject->SetText(mHistory[mHistory.MaxIndex()-mCurHistoryLine]);
					}
				}
			}
		}

		int mCurHistoryLine;
		iString mCurLine;
		iArray<iString> mHistory;
		ibgWidgetEntrySubject *mSubject;
		CommandInterpreter *mInterpreter;
		iggDialogCommandLine *mDialog;
	};

	//
	//  LineModeSwitchButton
	//
	class LineModeSwitchButton : public iggWidgetSimpleButton
	{

	public:

		LineModeSwitchButton(bool type, iggDialogCommandLine *dialog, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mDialog = dialog;
			mType = type;
			
			if(mType)
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("multiline.png"));
				this->SetBaloonHelp("Switch to multi-line mode","Switch the input window into the multi-line mode.");
			}
			else
			{
				mSubject->SetIcon(*iImageFactory::FindIcon("singleline.png"));
				this->SetBaloonHelp("Switch to single-line mode","Switch the input window into the single-line mode.");
			}
		}

	protected:

		virtual void Execute()
		{
			mDialog->SetMultiLine(mType);
		}

		bool mType;
		iggDialogCommandLine *mDialog;
	};


	//
	//  ResetScriptButton
	//
	class ResetScriptButton : public iggWidgetSimpleButton
	{

	public:

		ResetScriptButton(CommandInterpreter *ci, iggFrame *parent) : iggWidgetSimpleButton("Reset script",parent)
		{
			mInterpreter = ci;

			this->SetBaloonHelp("Reset the script","This button resets the current script by removing all defined variables and their values and clearing the history.");
		}

	protected:

		virtual void Execute()
		{
			mInterpreter->ResetScript();
		}

		CommandInterpreter *mInterpreter;
	};

	//
	//  A dialog to show the full script text and its helper buttons
	//
	class SaveScriptButton : public iggWidgetSimpleButton
	{

	public:

		SaveScriptButton(CommandInterpreter *ci, iggDialog *dialog, iggFrame *parent) : iggWidgetSimpleButton("Save script to file",parent)
		{
			mInterpreter = ci;
			mDialog = dialog;

			this->SetBaloonHelp("Save the script","This button opens up a file loading dialog which allows to select a file to save the script into.");
		}

	protected:

		virtual void Execute()
		{
			int i, n;
            iString fn;

			if(mOldFileName.IsEmpty()) mOldFileName = this->GetShell()->GetEnvironment(_EnvironmentScript);
			fn = this->GetMainWindow()->GetFileName("Save script file",mOldFileName,"Script file (*.ics)",false);
			if(!fn.IsEmpty() && fn.Part(-4).Lower()!=".ics") fn += ".ics";

			iFile f(fn);
			if(!f.Open(iFile::_Write,iFile::_Text))
			{
				this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Failed to create a file.",_PopupWindowError);
				return;
			}

			iString ws = mInterpreter->GetCode();
			n = ws.Contains('\n');
			if(n == 0) n++;

			for(i=0; i<n; i++)
			{
				if(!f.WriteLine(ws.Section("\n",i,i)))
				{
					this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Unable to write into a file.",_PopupWindowError);
					break;
				}
			}
			f.Close();

			if(i == n) mOldFileName = fn;
		}

		iString mOldFileName;
		iggDialog *mDialog;
		CommandInterpreter *mInterpreter;
	};


	class ViewScriptDialog : public iggDialog
	{

	public:

		ViewScriptDialog(CommandInterpreter *ci, iggDialogCommandLine *parent) : iggDialog(parent,_DialogModal,0,"View Script Text",0,3)
		{
			mInterpreter = ci;
			mBrowser = new iggWidgetTextBrowser(false,true,mFrame);

			mFrame->AddLine(mBrowser,3);
			mFrame->AddLine(new SaveScriptButton(ci,this,mFrame),static_cast<iggWidget*>(0),new iggWidgetDialogCloseButton(this,mFrame));
			mFrame->SetRowStretch(0,10);
			mFrame->SetColStretch(1,10);

			this->ResizeContents(300,400);
		}

		virtual const iString& GetToolTip() const 
		{
			static const iString tmp = "Shows the script text typed in so far";
			return tmp; 
		}

	protected:

		virtual void Show(bool s)
		{
			if(s)
			{
				iString text = mInterpreter->GetCode();
				text.Replace("\n","<br>");
				mBrowser->SetText(text);
			}
			iggDialog::Show(s);
		}

		CommandInterpreter *mInterpreter;
		iggWidgetTextBrowser *mBrowser;
	};

};


using namespace iggDialogCommandLine_Private;


iggDialogCommandLine::iggDialogCommandLine(iggMainWindow *parent) : iggDialog(parent,0U,iImageFactory::FindIcon("comline.png"),"Command Line","sr.gg.dc",1,0)
{
	iggWidgetTextBrowser *b = new iggWidgetTextBrowser(true,false,mFrame);
	CommandInterpreter *ci = new CommandInterpreter(b);
	ci->Start();

	mMultiLineFrame = new iggFrame(mFrame,2);
	CommandMultiLineEditor *ml = new CommandMultiLineEditor(ci,this,mMultiLineFrame);

	mSingleLineFrame = new iggFrame(mFrame,2);
	CommandSingleLineEditor *sl = new CommandSingleLineEditor(ci,this,mSingleLineFrame);

	mFrame->AddLine(new iggWidgetTextArea("%bHistory & Results",mFrame));
	mFrame->AddLine(b);
	mFrame->AddLine(new iggWidgetTextArea("%bInput Commands",mFrame));

	iggFrame *tmp;
	tmp = new iggFrame(mMultiLineFrame);
	tmp->AddSpace(10);
	tmp->AddLine(new LineModeSwitchButton(false,this,tmp));
	tmp->AddSpace(10);
	mMultiLineFrame->AddLine(ml,tmp);
	mMultiLineFrame->SetColStretch(0,10);

	tmp = new iggFrame(mSingleLineFrame);
	tmp->AddSpace(10);
	tmp->AddLine(new LineModeSwitchButton(true,this,tmp));
	tmp->AddSpace(10);
	mSingleLineFrame->AddLine(sl,tmp);
	mSingleLineFrame->SetColStretch(0,10);

	mFrame->AddLine(mMultiLineFrame);
	mFrame->AddLine(mSingleLineFrame);

	mFrame->AddLine(new iggWidgetTextArea("(add semicolon at the end of the last command to execute script)",mFrame));

	mFrame->SetColStretch(2,10);
	mFrame->SetRowStretch(1,10);
	mFrame->SetRowStretch(3,5);

	mViewScriptDialog = new ViewScriptDialog(ci,this);

	//
	//  Bottom line of buttons
	//
	tmp = new iggFrame(mFrame,4);
	tmp->AddLine(new iggWidgetLaunchButton(mViewScriptDialog,"Show script",tmp),new ResetScriptButton(ci,tmp),static_cast<iggWidget*>(0),new iggWidgetDialogCloseButton(this,tmp));
	tmp->SetColStretch(2,10);

	mFrame->AddLine(tmp);
	this->SetMultiLine(false);

	mInterpreter = ci;
	this->ResizeContents(700,500);
}


iggDialogCommandLine::~iggDialogCommandLine()
{
	delete mViewScriptDialog;
	mInterpreter->Delete();
}


void iggDialogCommandLine::SetMultiLine(bool s)
{
	mSingleLineFrame->Show(!s);
	mMultiLineFrame->Show(s);
	mFrame->SetRowStretch(3,s?5:0);
}


bool iggDialogCommandLine::IsMultiLine() const
{
	return mMultiLineFrame->IsVisible();
}

#endif
