/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ibasicparticlesfileloader.h"


#include "icommoneventobservers.h"
#include "ifile.h"
#include "ierrorstatus.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


iBasicParticlesFileLoader::iBasicParticlesFileLoader(iDataReader *r) : iParticleFileLoader(r)
{
}


void iBasicParticlesFileLoader::ReadFileBody(const iString &suffix, const iString &fname)
{
	//
	//  is the suffix valid?
	//
	if(suffix.Lower()!="txt" && suffix.Lower()!="bin")
	{
		this->GetErrorStatus()->Set("Incorrect file suffix.");
		return;
	}

	bool err, isBin = (suffix.Lower() == "bin");

	//
	//  Open the file
	//
	iFile F(fname);
	if(!F.Open(iFile::_Read,isBin?iFile::_Binary:iFile::_Text))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);

	//
	//  Read the header
	//
	int i, natt;
	bool paf;
	long ntot0;
	float ll[3], ur[3];
	iString buffer;
	if(isBin)
	{
		err = !this->ReadBinFileHeader(F,ntot0,ll,ur,paf,natt);
	}
	else
	{
		err = !this->ReadTxtFileHeader(F,ntot0,ll,ur,paf,natt,buffer);
	}
	if(ntot0 <= 0) err = true;

	if(err)
	{
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	//
	//  Compute scale and offset
	//
	float offsetF[3], scaleF[3];
	double offsetD[3], scaleD[3];
	if(paf)
	{
		for(i=0; i<3; i++)
		{
			offsetF[i] = ll[i];
			scaleF[i] = 2.0/(ur[i]-ll[i]);
		}
	}
	else
	{
		for(i=0; i<3; i++)
		{
			offsetD[i] = ll[i];
			scaleD[i] = 2.0/(ur[i]-ll[i]);
		}
	}

	//
	//  Configure streams
	//
	this->ConfigureStreams(&ntot0,&natt,paf);
	mObserver->SetProgress(0.01);

	//
	//  Read actual data
	//
	if(isBin)
	{
		err = !this->ReadBinFileContents(F,paf,scaleF,offsetF,scaleD,offsetD);
	}
	else
	{
		err = !this->ReadTxtFileContents(F,paf,scaleF,offsetF,scaleD,offsetD,buffer);
	}

	if(err || mObserver->IsAborted())
	{
		if(!err) this->GetErrorStatus()->Set("Aborted."); else this->GetErrorStatus()->Set("Corrupted data.");
		mObserver->Finished();
		this->ReleaseStreams();
		F.Close();
		return;
	}
	
	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();
}


bool iBasicParticlesFileLoader::ReadBinFileHeader(iFile &F, long &ntot0, float *ll, float *ur, bool &paf, int &natt)
{
	//
	//  Read the header
	//
	if(!this->DetectFortranFileStructure(F,sizeof(int))) return false;

	int ntot1;
	if(!this->ReadFortranRecord(F,Buffer(ntot1),0.0f,0.0f)) return false;
	ntot0 = ntot1;

	float bounds[6];
	if(!this->ReadFortranRecord(F,Buffer(bounds,6),0.0f,0.0f)) return false;

	int i;
	for(i=0; i<3; i++)
	{
		ll[i] = bounds[i];
		ur[i] = bounds[3+i];
	}

	//
	//  Auto-detect whether points are float or double
	//
	long nrec = sizeof(float)*ntot0;
	int mar = F.SetMarker();
	if(!this->SkipFortranRecord(F,nrec))  // X
	{
		//
		//  not float - try double
		//
		paf = false;
		nrec = sizeof(double)*ntot0;
		F.ReturnToMarker(mar);
		if(!this->SkipFortranRecord(F,nrec)) return false;  // X
		if(!this->SkipFortranRecord(F,nrec)) return false;  // Y
		if(!this->SkipFortranRecord(F,nrec)) return false;  // Z
	}
	else
	{
		if(!this->SkipFortranRecord(F,nrec)) return false;  // Y
		if(!this->SkipFortranRecord(F,nrec)) return false;  // Z
		paf = true;
	}
	//
	//  Measure the file size to find out the number of attributes
	//
	nrec = sizeof(float)*ntot0;
	for(i=0; i<999 && this->SkipFortranRecord(F,nrec); i++);
	F.ReturnToMarker(mar,true);
	natt = i;

	return true;
}



bool iBasicParticlesFileLoader::ReadTxtFileHeader(iFile &F, long &ntot0, float *ll, float *ur, bool &paf, int &natt, iString &buffer)
{
	iString s;

	//
	//  TxT file coordinates are float - do we need to check whether they are long enough to be double?
	//
	paf = true;

	//
	// First line
	//
	if(!F.ReadLine(s)) return false;
	int ret = sscanf(s.ToCharPointer(),"%ld",&ntot0);
	if(ret != 1) return false;
	if(ntot0 <= 0) return false;

	//
	//  Second line
	//
	if(!F.ReadLine(s)) return false;

	char *axisName[3];
	int i;
	for(i=0; i<3; i++) 
	{
		axisName[i] = 0;
		axisName[i] = new char[8192]; if(axisName[i] == 0) break;
	}
	if(i < 3 )
	{
		for(i=0; i<3; i++) if(axisName[i] != 0) delete [] axisName[i];
		return false;
	}

	ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %s %s %s",&ll[0],&ll[1],&ll[2],&ur[0],&ur[1],&ur[2],axisName[0],axisName[1],axisName[2]);
	if(ret == 9) this->GetViewModule()->SetAxesBox(axisName[0],axisName[1],axisName[2],ll[0],ur[0],ll[1],ur[1],ll[2],ur[2]);
	for(i=0; i<3; i++) if(axisName[i] != 0) delete [] axisName[i];

	if(ret!=6 && ret!=9) return false;

	//
	//  Find out the number of attributes
	//
	if(!F.ReadLine(s)) return false;

	double xyz[3];
	float f[10];
	ret = sscanf(s.ToCharPointer(),"%lg %lg %lg %g %g %g %g %g %g %g %g %g %g",&xyz[0],&xyz[1],&xyz[2],&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<3 || ret>12) return false;

	natt = ret - 3;
	buffer = s;

	return true;
}


bool iBasicParticlesFileLoader::ReadBinFileContents(iFile &F, bool paf, float *scaleF, float *offsetF, double *scaleD, double *offsetD)
{
	ParticleStream *s = this->GetStream(0);

	//
	//  parameters for the Progress Bar
	//
	float updateStart, updateDuration = 0.99/(s->NumAttributesInFile+3);

	//
	//  Read coordinates
	//
	int n;
	bool err = false;
	for(n=0; !err && n<3; n++)
	{
		updateStart = 0.01 + updateDuration*n;
		if(paf)
		{
			err = !this->ReadPositions(F,1,n,updateStart,updateDuration,scaleF+n,offsetF+n);
		}
		else
		{
			err = !this->ReadPositions(F,1,n,updateStart,updateDuration,scaleD+n,offsetD+n);
		}
	}

	if(err || mObserver->IsAborted())
	{
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return false;
	}
	

	//
	//  Read attributes
	//
	for(n=0; !err && n<s->NumAttributesInFile; n++) 
	{
		updateStart = 0.01 + updateDuration*(n+3);
		err = !this->ReadAttributes(F,1,n,updateStart,updateDuration);
	}

	if(err || mObserver->IsAborted())
	{
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return false;
	}

	return true;
}


bool iBasicParticlesFileLoader::ReadTxtFileContents(iFile &F, bool paf, float *scaleF, float *offsetF, double *scaleD, double *offsetD, iString buffer)
{
	ParticleStream *s = this->GetStream(0);

	//
	//  Use the buffer
	//
	double xyz[3], xyzD[3];
	float xyzF[3], f[10];

	int i, ret;
	vtkIdType l;

	mIterator.Start();
	for(l=0; l<s->NumTotal; l++) 
	{
		if(l > 0)
		{
			if(!F.ReadLine(buffer))
			{
				this->GetErrorStatus()->Set("Truncated file.");
				return false;
			}
		}
		ret = sscanf(buffer.ToCharPointer(),"%lg %lg %lg %g %g %g %g %g %g %g %g %g %g",&xyz[0],&xyz[1],&xyz[2],&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);

		if(ret < s->NumAttributesInFile+3)
		{
			this->GetErrorStatus()->Set("Corrupted data.");
			return false;
		}
		if((100*l)/s->NumTotal < (100*(l+1))/s->NumTotal) 
		{
			mObserver->SetProgress(0.01+(float)l/s->NumTotal);
			if(mObserver->IsAborted())
			{
				this->GetErrorStatus()->SetAbort();
				return false;
			}
		}
		if(mIterator.IsSelected())
		{
			if(paf)
			{
				for(i=0; i<3; i++) xyzF[i] = -1.0 + scaleF[i]*(xyz[i]-offsetF[i]);
				s->Points->SetPoint(mIterator.GlobalIndex(),xyzF);
			}
			else
			{
				for(i=0; i<3; i++) xyzD[i] = -1.0 + scaleD[i]*(xyz[i]-offsetD[i]);
				s->Points->SetPoint(mIterator.GlobalIndex(),xyzD);
			}
			if(s->Scalars != 0) s->Scalars->SetTuple(mIterator.GlobalIndex(),f);
		}
	}
	mIterator.Stop();

	return true;
}

