/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ifileloader.h"


#include "icommoneventobservers.h"
#include "idata.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "ishellfactory.h"
#include "isystem.h"
#include "iviewmodule.h"

#include <vtkDataSet.h>

//
//  Templates
//
#include "iarraytemplate.h"


#define	IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(_type_) \
void iFileLoader::SwapBytes(_type_ &p) \
{ \
	iFileLoader_Private::SwapBytes(p); \
} \
void iFileLoader::SwapBytesRange(_type_ *p, long count) \
{ \
	iFileLoader_Private::SwapBytesRange(p,count); \
}

#define	IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(_type_) \
bool iFileLoader::ReadBlock(iFile& F, _type_ *p, long len, float updateStart, float updateDuration) \
{ \
	return iFileLoader_Private::ReadBlockToArray(F,p,len,updateStart,updateDuration,mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver); \
} \
iFileLoader::ReadingBuffer iFileLoader::Buffer(_type_ &d) const \
{ \
	return ReadingBuffer(&d,1UL,iFileLoader_Private::TypeIndex(&d),sizeof(_type_)); \
} \
iFileLoader::ReadingBuffer iFileLoader::Buffer(_type_ *d, long l) const \
{ \
	return ReadingBuffer(d,l,iFileLoader_Private::TypeIndex(d),sizeof(_type_)); \
}


namespace iFileLoader_Private
{
	//
	//  Byte swap helpers
	//
	inline void Swap4Bytes(char* data)
	{ 
		char one_byte; 
		one_byte = data[0]; data[0] = data[3]; data[3] = one_byte;
		one_byte = data[1]; data[1] = data[2]; data[2] = one_byte; 
	}


	inline void Swap8Bytes(char* data)
	{ 
		char one_byte;
		one_byte = data[0]; data[0] = data[7]; data[7] = one_byte;
		one_byte = data[1]; data[1] = data[6]; data[6] = one_byte;
		one_byte = data[2]; data[2] = data[5]; data[5] = one_byte;
		one_byte = data[3]; data[3] = data[4]; data[4] = one_byte; 
	}

	
	void Swap4BytesRange(char *data, long count)
	{
		char *pos;
		long i;

		pos = data;
		for(i = 0; i<count; i++)
		{
			Swap4Bytes(pos);
			pos += 4;
		}
	}


	void Swap8BytesRange(char *data, long count)
	{
		char *pos;
		long i;

		pos = data;
		for(i = 0; i<count; i++)
		{
			Swap8Bytes(pos);
			pos += 8;
		}
	}


	template<class T>
	inline void SwapBytes(T &p)
	{
		switch(sizeof(T))
		{
		case 4:
			{
				Swap4Bytes((char *)&p);
				break;
			}
		case 8:
			{
				Swap8Bytes((char *)&p);
				break;
			}
		default:
			{
				IERROR_FATAL("IFrIT was not ported to that platform.");
			}
		}
	}


	template<class T>
	inline void SwapBytesRange(T *p, long count)
	{
		switch(sizeof(T))
		{
		case 4:
			{
				Swap4BytesRange((char *)p,count);
				break;
			}
		case 8:
			{
				Swap8BytesRange((char *)p,count);
				break;
			}
		default:
			{
				IERROR_FATAL("IFrIT was not ported to that platform.");
			}
		}
	}


	//
	//  Check validity of numbers
	//
	bool IsNumberValid(int v){ return true; }
	bool IsNumberValid(long v){ return true; }
	bool IsNumberValid(float v){ return (-iMath::_FloatMax<=v && v<=iMath::_FloatMax); }
	bool IsNumberValid(double v){ return (-iMath::_DoubleMax<=v && v<=iMath::_DoubleMax); }


	//
	//  Type indices
	//
	const int _Int = 1;
	const int _Long = 2;
	const int _Float = 3;
	const int _Double = 4;

	int TypeIndex(int *){ return _Int; }
	int TypeIndex(long *){ return _Long; }
	int TypeIndex(float *){ return _Float; }
	int TypeIndex(double *){ return _Double; }


	//
	//  Read blocks of data from a file
	//
	template<class T>
	bool ReadBlockToArray(iFile& F, T *p, long len, float updateStart, float updateDuration, bool swapbytes, iProgressEventObserver *observer)
	{
		const long nline = 16384L;
		long nread, l, nstep = (len+nline-1)/nline;
		float ustep = updateDuration/nstep;

		for(l=0; l<nstep; l++) 
		{
			if(observer->IsAborted()) return true;
			if(l < nstep-1) nread = nline; else nread  = len - nline*l;
			if(!F.ReadBlock(p+l*nline,nread*sizeof(T))) return false;
			observer->SetProgress(updateStart+ustep*l);
		}

		if(swapbytes) 
		{
			iFileLoader_Private::SwapBytesRange(p,len);
		}

		bool ok = true;
		for(l=0; ok && l<len; l++)
		{
			ok = iFileLoader_Private::IsNumberValid(p[l]);
		}

		return ok;
	}


	bool ReadBlockToBuffer(iFile& F, const iFileLoader::ReadingBuffer &b, float updateStart, float updateDuration, bool swapbytes, iProgressEventObserver *observer)
	{
		switch(b.Type)
		{
		case _Int:
			{
				int *d = (int *)b.Data;
				return iFileLoader_Private::ReadBlockToArray(F,d,b.Length,updateStart,updateDuration,swapbytes,observer);
			}
		case _Long:
			{
				long *d = (long *)b.Data;
				return iFileLoader_Private::ReadBlockToArray(F,d,b.Length,updateStart,updateDuration,swapbytes,observer);
			}
		case _Float:
			{
				float *d = (float *)b.Data;
				return iFileLoader_Private::ReadBlockToArray(F,d,b.Length,updateStart,updateDuration,swapbytes,observer);
			}
		case _Double:
			{
				double *d = (double *)b.Data;
				return iFileLoader_Private::ReadBlockToArray(F,d,b.Length,updateStart,updateDuration,swapbytes,observer);
			}
		default:
			{
				return false;
			}
		}
	}


	template<class T>
	bool AnalyzeFirstRecordForType(iFile &F, unsigned int len, bool &isBig)
	{
		T lrec1, lrec2;

		int m = F.SetMarker();
		if(!F.ReadBlock(&lrec1,sizeof(T)) || !F.SkipBlock(len) || !F.ReadBlock(&lrec2,sizeof(T)))
		{
			F.ReturnToMarker(m,true);
			return false; // unable even to read
		}
		F.ReturnToMarker(m,true);

		//
		//  Is the header/footer length ok?
		//
		if(lrec1 == lrec2) // succeeded
		{
			//
			// auto-detect data endiness
			//
			if(lrec1 != len)
			{
				SwapBytes(lrec1);
				if(lrec1 != len) return false; else isBig = !iSystem::IsBigEndianMachine();
			}
			else isBig = iSystem::IsBigEndianMachine();
			return true;
		}
		else return false;
	}
};


using namespace iFileLoader_Private;
using namespace iParameter;


iFileLoader::iFileLoader(iDataReader *r, int priority) : mViewModule(r?r->GetViewModule():0), mErrorStatus("Data Subject"), mReader(r), mReaderExtension(0), mPriority(priority)
{
	this->Define();
}


iFileLoader::iFileLoader(iDataReaderExtension *ext, int priority) : mViewModule(ext?ext->GetReader()->GetViewModule():0), mErrorStatus("Data Subject"), mReader(ext->GetReader()), mReaderExtension(ext), mPriority(priority)
{
	this->Define();
}


void iFileLoader::Define()
{
	IERROR_ASSERT(mReader);

	mFortranHeaderFooterLength = -1;

	// mFileRoot = mFileSuffix = mLastFileName = "";
	mUpdated = true;
	mIsBigEndian = true;
	mTwoCopies = false;
	mRecord = -1;
	mShift[0] = mShift[1] = mShift[2] = 0.0f;

	mPeriodic[0] = mPeriodic[1] = mPeriodic[2] = false;
	mBoundaryConditions = _BoundaryConditionsWall;

	//
	//  Event observers are driven directly (otherwise we would have to create at least 16 new events, which is
	//  cumbersome), so adding with AddObserver is not necessarily
	//
	mObserver = dynamic_cast<iProgressEventObserver *>(iShellFactory::CreateEventObserver("Progress",mReader->GetViewModule())); IERROR_ASSERT(mObserver);
}


iFileLoader::~iFileLoader()
{
	mObserver->Delete();
	while(mStreams.Size() > 0) delete mStreams.RemoveLast();
}


bool iFileLoader::IsUsingData(const iDataType &type) const
{
	int i;
	for(i=0; i<mStreams.Size(); i++)
	{
		if(mStreams[i]->Subject->GetDataType() == type) return true;
	}
	return false;
}


const iDataType& iFileLoader::GetDataType(int n) const
{
	if(n>=0 && n<mStreams.Size()) return mStreams[n]->Subject->GetDataType(); else return iDataType::Null();
}


vtkDataSet* iFileLoader::GetData(int n) const
{
	if(n>=0 && n<mStreams.Size()) return mStreams[n]->ReleasedData; else return 0;
}


vtkDataSet* iFileLoader::GetData(const iDataType &type) const
{
	int i;
	for(i=0; i<mStreams.Size(); i++)
	{
		if(mStreams[i]->Subject->GetDataType() == type) return this->GetData(i);
	}
	return 0;
}


iDataSubject* iFileLoader::GetSubject(int n) const
{
	if(n>=0 && n<mStreams.Size()) return mStreams[n]->Subject; else return 0;
}


iDataSubject* iFileLoader::GetSubject(const iDataType &type) const
{
	int i;
	for(i=0; i<mStreams.Size(); i++)
	{
		if(mStreams[i]->Subject->GetDataType() == type) return mStreams[i]->Subject;
	}
	return 0;
}


void iFileLoader::SetBoundaryConditions(int s)
{
	if(s>=0 && s<__NumBoundaryConditions)
	{
		mBoundaryConditions = s;
	}
}


void iFileLoader::SetDirectionPeriodic(int d, bool s)
{
	if(d>=0 && d<3)
	{
		mPeriodic[d] = s;
	}
}


void iFileLoader::ReadFile(const iString &fname)
{
	int i;

	this->GetErrorStatus()->Clear();
	mOverflow = false;
	mFortranHeaderFooterLength = -1; // must be set by a child class

	iString root, suffix;
	int record = this->DissectFileName(fname,root,suffix);

	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->GetLimits()->BlockNotifications(true);
		if(mStreams[i]->ReleasedData!=0 && !mTwoCopies)
		{
			//
			//  The data can be referenced by other objects, so delete will not delete it. Thus, we first erase
			//  the actual data by initializing the data object
			//
			mStreams[i]->ReleasedData->Initialize();
			mStreams[i]->ReleasedData->Delete();
			mStreams[i]->ReleasedData = 0;
		}
	}
	this->ReadFileBody(suffix,fname);
	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->GetLimits()->BlockNotifications(false);
	}

	if(this->GetErrorStatus()->NoError())
	{
		mLastFileName = fname;
		mFileSuffix = suffix;
		mFileRoot = root;
		mUpdated = false;
		mRecord = record;
		mShift[0] = mShift[1] = mShift[2] = 0.0f;
	}
}


void iFileLoader::Finalize()
{
	int i;

	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->GetLimits()->BlockNotifications(true);
	}

	this->FinalizeBody();

	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->GetLimits()->BlockNotifications(false);
	}

	if(!mUpdated)
	{
		mUpdated = true;

		if(this->GetErrorStatus()->NoError() && mOverflow)
		{
			this->GetErrorStatus()->Set("The data are read correctly, but some values in the file were outside of range ("+iString::FromNumber(-iMath::_LargeFloat)+","+iString::FromNumber(iMath::_LargeFloat)+").\n These values were clamped to be within the required range.",-1);
		}
	}

	this->NotifyDependencies();
}


void iFileLoader::NotifyDependencies()
{
	int i;

	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->NotifyDependencies();
		mStreams[i]->Subject->ClearCache();
	}
}


void iFileLoader::AttachDataToStream(int i, vtkDataSet *ds)
{
	if(i>=0 && i<mStreams.Size())
	{
		if(ds == 0)
		{
			IERROR_FATAL("Cannot attach a null data set.");
		}
		else
		{
			if(mStreams[i]->ReleasedData != 0)
			{
				mStreams[i]->ReleasedData->Delete();
			}
			mStreams[i]->ReleasedData = ds->NewInstance(); IERROR_ASSERT(mStreams[i]->ReleasedData);
			if(mTwoCopies)
			{
				mStreams[i]->ReleasedData->DeepCopy(ds);
			}
			else
			{
				mStreams[i]->ReleasedData->ShallowCopy(ds);
			}
			this->Polish(mStreams[i]->ReleasedData);
		}
	}
}


bool iFileLoader::IsAborted() const
{
	return mObserver->IsAborted();
}


bool iFileLoader::IsThereData(int n) const
{
	if(n>=0 && n<mStreams.Size())
	{
		iDataHelper h(this->GetData(n));
		return h.IsThereData();
	}
	else return false;
}


bool iFileLoader::IsThereData() const
{
	int i;
	for(i=0; i<mStreams.Size(); i++)
	{
		if(this->IsThereData(i)) return true;
	}
	return false;
}


bool iFileLoader::IsThereData(const iDataType &type) const
{
	int i;
	for(i=0; i<mStreams.Size(); i++)
	{
		if(mStreams[i]->Subject->GetDataType() == type) return this->IsThereData(i);
	}
	return false;
}


void iFileLoader::EraseData()
{
	int i;
	for(i=0; i<mStreams.Size(); i++) if(mStreams[i]->ReleasedData != 0)
	{
		mStreams[i]->ReleasedData->Initialize();
	}

	mRecord = -1;
	mShift[0] = mShift[1] = mShift[2] = 0.0f;

	for(i=0; i<mStreams.Size(); i++)
	{
		mStreams[i]->Subject->GetLimits()->AssignVars(); // if no data, use all vars
		mStreams[i]->Subject->ClearCache();
	}
}


void iFileLoader::SetTwoCopies(bool s)
{
	if(mTwoCopies != s)
	{
		mTwoCopies = s;
	}
}


void iFileLoader::Reset()
{
	int i;
	for(i=0; i<mStreams.Size(); i++) if(mStreams[i]->ReleasedData != 0)
	{
		mStreams[i]->ReleasedData->Modified();
		this->GetViewModule()->NotifyDataConsumers(mStreams[i]->Subject->Request());
	}
}


void iFileLoader::ShiftData(double dr[3])
{
	int i;
	double dxmax, dx[3];

	mObserver->Started(iProgressEventObserver::_Shifting);
	mObserver->SetProgress(0.0);

	dxmax = 0.0;
	for(i=0; i<3; i++)
	{
		dx[i] = dr[i] - mShift[i];
		if(fabs(dx[i]) > dxmax) dxmax = fabs(dx[i]);
	}

	if(dxmax > 1.0e-30)
	{
		for(i=0; i<this->NumStreams(); i++) if(mStreams[i]->ReleasedData != 0)
		{
			this->ShiftDataBody(mStreams[i]->ReleasedData,dx);
		}
		for(i=0; i<3; i++) mShift[i] = dr[i];
	}

	mObserver->Finished();
}


float iFileLoader::GetMemorySize() const
{
	int i;
	float s = 0.0f;

	for(i=0; i<mStreams.Size(); i++) if(mStreams[i]->ReleasedData != 0)
	{
		s += mStreams[i]->ReleasedData->GetActualMemorySize();
	}
	if(mTwoCopies) s *= 2;
	return s;
}


iFileLoader::Stream* iFileLoader::CreateNewStream() const
{
	return new Stream();
}


//
//  Data reading helpers
//
bool iFileLoader::SkipFortranRecord(iFile& F, long len)
{
	long lrec1, lrec2;

	if(!this->ReadFortranHeaderFooter(F,lrec1)) return false;
	if(!F.SkipBlock(len)) return false;
	if(!this->ReadFortranHeaderFooter(F,lrec2)) return false;
	if(lrec1!=lrec2 || lrec1!=len) return false;

	return true;
}


bool iFileLoader::ReadFortranRecord(iFile& F, const ReadingBuffer &b, float updateStart, float updateDuration, bool autoswap)
{
	long lrec1, lrec2;

	if(!this->ReadFortranHeaderFooter(F,lrec1)) return false;

	if(!ReadBlockToBuffer(F,b,updateStart,updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;

	if(!this->ReadFortranHeaderFooter(F,lrec2)) return false;
	if(lrec1!=lrec2 || lrec1!=b.Length*b.Size) return false;

	return true;
}


bool iFileLoader::ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, float updateStart, float updateDuration, bool autoswap)
{
	long lrec1, lrec2;
	long len = b1.Length*b1.Size + b2.Length*b2.Size;
	float f1 = float(b1.Length*b1.Size)/len;
	float f2 = float(b2.Length*b2.Size)/len;

	if(!this->ReadFortranHeaderFooter(F,lrec1)) return false;

	if(!ReadBlockToBuffer(F,b1,updateStart,f1*updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;
	if(!ReadBlockToBuffer(F,b2,updateStart+f1*updateDuration,f2*updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;

	if(!this->ReadFortranHeaderFooter(F,lrec2)) return false;
	if(lrec1!=lrec2 || lrec1!=len) return false;

	return true;
}


bool iFileLoader::ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, const ReadingBuffer &b3, float updateStart, float updateDuration, bool autoswap)
{
	long lrec1, lrec2;
	long len = b1.Length*b1.Size + b2.Length*b2.Size + b3.Length*b3.Size;
	float f1 = float(b1.Length*b1.Size)/len;
	float f2 = float(b2.Length*b2.Size)/len;
	float f3 = float(b3.Length*b3.Size)/len;

	if(!this->ReadFortranHeaderFooter(F,lrec1)) return false;

	if(!ReadBlockToBuffer(F,b1,updateStart,f1*updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;
	if(!ReadBlockToBuffer(F,b2,updateStart+f1*updateDuration,f2*updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;
	if(!ReadBlockToBuffer(F,b3,updateStart+(f1+f2)*updateDuration,f3*updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
	if(mObserver->IsAborted()) return true;

	if(!this->ReadFortranHeaderFooter(F,lrec2)) return false;
	if(lrec1!=lrec2 || lrec1!=len) return false;

	return true;
}

	
bool iFileLoader::ReadFortranRecord(iFile& F, int nbufs, const ReadingBuffer *b, float updateStart, float updateDuration, bool autoswap) // read any number of buffers
{
	long lrec1, lrec2;

	if(b==0 || nbufs<=0 || !this->ReadFortranHeaderFooter(F,lrec1)) return false;

	int i;
	long len = 0L;
	for(i=0; i<nbufs; i++)
	{
		len += b[i].Length*b[i].Size;
		if(!ReadBlockToBuffer(F,b[i],updateStart+i*updateDuration,updateDuration,autoswap && mIsBigEndian!=iSystem::IsBigEndianMachine(),mObserver)) return false;
		if(mObserver->IsAborted()) return true;
	}

	if(!this->ReadFortranHeaderFooter(F,lrec2)) return false;
	if(lrec1!=lrec2 || lrec1!=len) return false;

	return true;
}


bool iFileLoader::ReadFortranHeaderFooter(iFile& F, long &lrec)
{
	bool ret = false;

	if(mFortranHeaderFooterLength == sizeof(long))
	{
		ret = F.ReadBlock(&lrec,sizeof(long));
		if(ret && mIsBigEndian!=iSystem::IsBigEndianMachine()) this->SwapBytes(lrec);
	}
	else if(mFortranHeaderFooterLength == sizeof(int))
	{
		int ltmp;
		ret = F.ReadBlock(&ltmp,sizeof(int));
		if(ret)
		{
			if(mIsBigEndian != iSystem::IsBigEndianMachine()) this->SwapBytes(ltmp);
			lrec = (long)ltmp;
		}
	}
	else
	{
		IERROR_LOW("Invalid Fortran header/footer length.");
	}

	return ret;
}


bool iFileLoader::DetectFortranFileStructure(iFile &F, long len)
{
	//
	//  Try int-sized records
	//
	if(AnalyzeFirstRecordForType<int>(F,len,mIsBigEndian))
	{
		mFortranHeaderFooterLength = sizeof(int);
		return true;
	}
	//
	//  Try long-sized records
	//
	if(AnalyzeFirstRecordForType<long>(F,len,mIsBigEndian))
	{
		mFortranHeaderFooterLength = sizeof(long);
		return true;
	}
	//
	//  Failed to figure it out
	//
	return false;
}


iString iFileLoader::GetRecordAsString(int rec) const
{
	const iString format = "%0" + iString::FromNumber(mReader->GetRecordLength()) + "d";
	return iString::FromNumber(rec,format.ToCharPointer());
}


int iFileLoader::DissectFileName(const iString &fname, iString &root, iString &suffix) const
{
	bool ok;
	int rec;

	if(fname.Contains('.') > 0)
	{
		suffix = fname.Section(".",-1);
		root = fname.Part(0,fname.Length()-suffix.Length()-1);
	}
	else
	{
		suffix.Clear();
		root = fname;
	}
	iString record = root.Part(-mReader->GetRecordLength());
	rec = record.ToInt(ok);
	if(!ok || record!=this->GetRecordAsString(rec) || rec<0) 
	{
		return -1;
	}
	else
	{
		root = root.Part(0,root.Length()-mReader->GetRecordLength());
		return rec;
	}
}


const iString iFileLoader::GetFileName(int rec) const
{
	if(mFileSuffix.IsEmpty())
	{
		return mFileRoot + this->GetRecordAsString(rec);
	}
	else
	{
		return mFileRoot + this->GetRecordAsString(rec) + "." + mFileSuffix;
	}
}


bool iFileLoader::IsSeriesFileName(const iString &fname) const
{
	iString root, suffix;
	int	rec = this->DissectFileName(fname,root,suffix);
	return (rec>=0 && root==mFileRoot && suffix==mFileSuffix);
}


//
//  Overloads
//
IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(int);
IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(long);
IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(float);
IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(double);
//IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(unsigned int);
//IDATASUBJECT_DEFINE_SWAP_BYTES_FUNCTIONS(unsigned long);

IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(int);
IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(long);
IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(float);
IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(double);
//IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(unsigned int);
//IDATASUBJECT_DEFINE_READ_BLOCK_FUNCTIONS(unsigned long);


//
//  Helper class
//
iFileLoader::Stream::Stream()
{
	Subject = 0;
	ReleasedData = 0;
}


iFileLoader::Stream::~Stream()
{
	if(ReleasedData != 0) ReleasedData->Delete();
	if(Subject != 0) Subject->Delete();
}

