/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef ICONTROLMODULE_H
#define ICONTROLMODULE_H


#include "iobject.h"


#include "iarray.h"
#include "iimage.h"

#include "ipointermacro.h"

class iAnimatorScript;
class iControlScript;
class iDataType;
class iDataSubject;
class iEventObserver;
class iFile;
template<class O, class C>class iFamily;
class iFunctionMapping;
class iHistogram;
class iImageComposer;
class iInteractorEventObserver;
class iLimits;
struct iLimitsDataHolder;
class iParallelManager;
class iParallelUpdateEventObserver;
class iRangeMapping;
class iShell;
class iViewModule;
class iViewObject;
class iViewObjectFamily;
class iViewSubject;


namespace iParameter
{
	namespace ModuleType
	{
		const int New	= 0;
		const int Copy	= 1;
		const int Clone	= 2;
	};

	namespace ObjectOption
	{
		const int Mask		= 0x00F;
		const int One		= 0x1;
		const int All		= 0x2;
	};

	namespace ModuleOption
	{
		const int Mask		= 0x0F0;
		const int One		= 0x10;
		const int All		= 0x20;
		const int Clones	= 0x30;
	};

	namespace RenderOption
	{
		const int Mask		= 0xF00;
		const int One		= 0x100;
		const int All		= 0x200;
		const int Clones	= 0x300;
		const int Auto		= 0xF00;
	}

	const int DefaultOption = ObjectOption::One | ModuleOption::One | RenderOption::Auto;
};


#define ICONTROLMODULE_QUERYVALUE_DECLARATION(_type_) \
	bool QueryValue(const iObjectKey &key, _type_ *val, int n, int mod = -1, int obj = -1); \
	bool QueryValue(const iObjectKey &key, int index, _type_ &val, int n, int mod = -1, int obj = -1); \
	inline bool QueryValue(const iObjectKey &key, _type_ &val, int mod = -1, int obj = -1){ return this->QueryValue(key,&val,1,mod,obj); }

#define ICONTROLMODULE_PACKCOMMAND_DECLARATION(_type1_,_type2_) \
	void PackCommand(iString &s, const iObjectKey &key, _type1_ *val, int n) const; \
	void PackCommand(iString &s, const iObjectKey &key, _type2_ val) const; \
	void PackCommand(iString &s, const iObjectKey &key, int index, _type2_ val) const;


class iControlModule : public iObject
{

	friend class iExtensionFactory;

	IPOINTER_AS_PART(Shell);

public:

	vtkTypeMacro(iControlModule,iObject);
	static iControlModule* New(iShell *s = 0);
	static const iObjectType& Type();
		
	IOBJECT_DECLARE_GETSET1(AutoRender,bool);
	//void SetAutoRender(bool s);
	//inline bool GetAutoRender() const { return mAutoRender; }

	IOBJECT_DECLARE_GETSET(SynchronizeInteractors,mSyncInteractors,bool);
	//void SetSynchronizeInteractors(int m);
	//inline int GetSynchronizeInteractors() const { return mSyncInteractors; }

	IOBJECT_DECLARE_GETSET2(OptimizationMode,int);
	//void SetOptimizationMode(int m);
	//int GetOptimizationMode() const;

	//
	//  Action keys
	//
	static const iObjectKey& KeySynchronizeCameras();
	void SynchronizeCameras(int mod = -1);

	virtual bool LoadStateFromFile(const iString &filename);
	virtual bool SaveStateToFile(const iString &filename) const;

	//
	//  Execute a packed command
	//
	bool Execute(const iString &command, bool render, int option = iParameter::DefaultOption, int mod = -1);
	
	//
	//  A simple wrapper
	//
	inline void Render(int renMode = iParameter::RenderOption::Auto) { this->Execute("",true, iParameter::ObjectOption::One | iParameter::ModuleOption::One | renMode); }
	
	//
	//  Manipulating objects
	//
	virtual bool CreateObject(const iObjectType &type, bool render, int mod = -1, int vmtype = iParameter::ModuleType::New);
	virtual bool DeleteObject(const iObjectType &type, bool render, int mod = -1, int obj = -1);

//	virtual bool SetCurrentData(const iObjectType &type, int obj = -1, int mod = -1);
//	virtual bool SetCurrentData(int dc, int mod = -1);

//	virtual bool SetCurrentDataType(const iDataType &type, int mod = -1);

	virtual void Show(const iObjectType &type, bool show, bool render, int mod = -1);

	//
	//  Get the value corresponding to the key - shortcuts for Getting and unpacking the state
	//
	ICONTROLMODULE_QUERYVALUE_DECLARATION(int)
	ICONTROLMODULE_QUERYVALUE_DECLARATION(bool)
	ICONTROLMODULE_QUERYVALUE_DECLARATION(float)
	ICONTROLMODULE_QUERYVALUE_DECLARATION(double)
	ICONTROLMODULE_QUERYVALUE_DECLARATION(iColor)
	ICONTROLMODULE_QUERYVALUE_DECLARATION(iString)

	//
	//  Special query function
	//
	bool QueryValue(const iObjectKey &key, iFunctionMapping* &f, int mod = -1, int obj = -1) const;
	bool QueryValue(const iObjectKey &key, iRangeMapping* &f, int mod = -1, int obj = -1) const;
	bool QueryValue(const iObjectKey &key, iImage &im, int mod = -1, int obj = -1) const;
	bool QueryValueAsString(const iObjectKey &key, iString &s, int mod = -1, int obj = -1);

	//
	//  Query the whole state of the object
	//
	void QueryState(const iObjectType &type, iString &state, int mod = -1, int obj = -1);

	//
	//  Packing helpers - mostly moved iObject helpers into public interface
	//
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(int,int)
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(bool,bool)
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(float,float)
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(double,double)
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(iColor,iColor)
	ICONTROLMODULE_PACKCOMMAND_DECLARATION(iString,const iString&)

	//
	//  Additional commands
	//
	void FlyTo(const iObjectType &type, float time, int mod = -1, int obj = -1) const;
	void FlyTo(const double *pos, float time, int mod = -1) const;

	//
	//  mod = -2: sum up all view modules
	//  mod = -1: use the current one
	//
	float GetUpdateTime(int mod = -1) const;
	float GetMemorySize(int mod = -2) const;

	//
	//  Access to components
	//
	int GetNumberOfViewModules() const;
	int GetCurrentViewModuleIndex() const;
	iViewModule* GetViewModule() const;
	iViewModule* GetViewModule(int mod) const;
	bool SetCurrentViewModuleIndex(int mod);

	iAnimatorScript* GetAnimatorScript(int mod = -1) const;
	inline iImageComposer* GetImageComposer() const { return mComposer; }
	inline iControlScript* GetControlScript() const { return mControlScript; }
	inline iParallelManager* GetParallelManager() const { return mParallelManager; } 
	iEventObserver* GetInteractorObserver() const;

protected:

	virtual ~iControlModule();

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

private:

    iControlModule(iShell *s);
	void FinishInitialization();

	void ExecuteForViewModule(iViewModule *module, const iString &command, bool render, int objMode, int &renMode);
	iObject* FindSingleObject(const iObjectType &type, iViewModule *module);

	iObject* FindObject(const iObjectKey &key, int mod = -1, int obj = -1);
	iObject* FindObject(const iObjectType &type, int mod = -1, int obj = -1);
	iViewSubject* FindViewSubject(const iObjectType &type, iViewModule *module, int obj = -1) const;
	iViewObjectFamily* FindViewObjectFamily(const iObjectType &type, iViewModule *module) const;

	iFamily<iViewModule,iControlModule> *mViewModules;
	bool mAutoRender, mSyncInteractors, mBlockBroadcasts, mExecuteOk, mInExecute;
	mutable iString mDataSubjectName;

	iDataSubject *mCurrentSubject;
	iImageComposer *mComposer;
	iInteractorEventObserver *mInteractorObserver;

	static int mIdCounter;
	int mId;

	//
	//  Control script
	//
	iControlScript *mControlScript;

	//
	//  Parallel objects
	//
	iParallelUpdateEventObserver *mParallelObserver;
	iParallelManager *mParallelManager;
};


inline void iControlModule::SetAutoRender(bool s)
{ 
	mAutoRender = s; 
	this->ClearCache();
}


inline iObject* iControlModule::FindObject(const iObjectKey &key, int mod, int obj)
{
	return this->FindObject(key.Type(),mod,obj);
}

#endif  // ICONTROLMODULE_H
