/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipalette.h"


#include "ierror.h"
#include "iimage.h"
#include "imath.h"
#include "ipiecewisefunction.h"

#include <vtkColorTransferFunction.h> 
#include <vtkLookupTable.h>

//
//  Templates
//
#include "iarraytemplate.h"


//
// Implementation of iPalette class
//
iPalette::iPalette()
{
	mName = "empty";

	mCTF[0] = vtkColorTransferFunction::New(); IERROR_ASSERT(mCTF[0]);
	mCTF[1] = vtkColorTransferFunction::New(); IERROR_ASSERT(mCTF[1]);
	mLT[0] = vtkLookupTable::New(); IERROR_ASSERT(mLT[0]);
	mLT[1] = vtkLookupTable::New(); IERROR_ASSERT(mLT[1]);
	
	mFun[0] = iPiecewiseFunction::New(); IERROR_ASSERT(mFun[0]);
	mFun[1] = iPiecewiseFunction::New(); IERROR_ASSERT(mFun[1]);
	mFun[2] = iPiecewiseFunction::New(); IERROR_ASSERT(mFun[2]);

	mCTF[0]->SetColorSpaceToRGB();
	mCTF[1]->SetColorSpaceToRGB();
	mLT[0]->SetNumberOfTableValues(256);
	mLT[1]->SetNumberOfTableValues(256);

	//
	//  Image representation
	//
	mImage = new iImage(3); IERROR_ASSERT(mImage);
	mImageNeedsUpdate = true;
	mImageWidth = 256;
	mImageHeight = 32;
}


iPalette::~iPalette()
{
	delete mImage;

	mCTF[0]->Delete(); 
	mCTF[1]->Delete(); 
	mLT[0]->Delete(); 
	mLT[1]->Delete(); 

	mFun[0]->Delete();
	mFun[1]->Delete();
	mFun[2]->Delete();
}


void iPalette::SetComponents(const iPiecewiseFunction *r, const iPiecewiseFunction *g, const iPiecewiseFunction *b)
{
	if(r != 0) mFun[0]->Copy(r);
	if(g != 0) mFun[1]->Copy(g);
	if(b != 0) mFun[2]->Copy(b);

	this->Update();
}

	
iColor iPalette::GetColor(int n) const
{
	n = ( n < 0 ) ? 0 : n;
	n = ( n > 255 ) ? 255 : n;
	float x = (float)n/255.0;
	float r = mFun[0]->GetValue(x);
	float g = mFun[1]->GetValue(x);
	float b = mFun[2]->GetValue(x);
	return iColor(round(255.0*r),round(255.0*g),round(255.0*b));
}


void iPalette::Update()
{
	int i, ir, ig, ib;
	float x;

	//
	//  Update vtkColorTransferFunction
	//
	mCTF[0]->RemoveAllPoints();
	mCTF[1]->RemoveAllPoints();
	ir = ig = ib = 0;
	while(ir<mFun[0]->N() && ig<mFun[1]->N() && ib<mFun[2]->N())
	{
		x = mFun[0]->X(ir);
		if(x > mFun[1]->X(ig)) x = mFun[1]->X(ig);
		if(x > mFun[2]->X(ib)) x = mFun[2]->X(ib);
		
		mCTF[0]->AddRGBPoint(255.0*x,mFun[0]->GetValue(x),mFun[1]->GetValue(x),mFun[2]->GetValue(x));
		mCTF[1]->AddRGBPoint(255.0*(1.0-x),mFun[0]->GetValue(x),mFun[1]->GetValue(x),mFun[2]->GetValue(x));
		
		if(fabs(x-mFun[0]->X(ir)) < 1.0e-4) ir++;
		if(fabs(x-mFun[1]->X(ig)) < 1.0e-4) ig++;
		if(fabs(x-mFun[2]->X(ib)) < 1.0e-4) ib++;
	}

	//
	//  Update vtkLookupTable
	//
	for(i=0; i<256; i++)
	{
		x = (float)i/255.0;
		mLT[0]->SetTableValue(i,mFun[0]->GetValue(x),mFun[1]->GetValue(x),mFun[2]->GetValue(x));
		mLT[1]->SetTableValue(255-i,mFun[0]->GetValue(x),mFun[1]->GetValue(x),mFun[2]->GetValue(x));
	}

	mImageNeedsUpdate = true;
}


void iPalette::Copy(iPalette *p)
{
	mFun[0]->Copy(p->mFun[0]);
	mFun[1]->Copy(p->mFun[1]);
	mFun[2]->Copy(p->mFun[2]);
	mName = p->mName;

	this->Update();
}


const iImage* iPalette::GetImage()
{
	if(mImageNeedsUpdate)
	{
		mImageNeedsUpdate = false;
		//
		//  Update the image
		//
		mImage->Scale(mImageWidth,mImageHeight);
		unsigned char *dPtr = mImage->DataPointer();

		int i;
		float x;
		unsigned char r, g, b;

		//
		//  Fill first row
		//
		for(i=0; i<mImageWidth; i++)
		{
			x = (float)i/(mImageWidth-1);
			r = char(round(255.0*mFun[0]->GetValue(x)));
			g = char(round(255.0*mFun[1]->GetValue(x)));
			b = char(round(255.0*mFun[2]->GetValue(x)));
			dPtr[3*i+0] = r;
			dPtr[3*i+1] = g;
			dPtr[3*i+2] = b;
		}

		//
		//  Copy to the rest of rows
		//
		for(i=1; i<mImageHeight; i++) memcpy(dPtr+3*i*mImageWidth,dPtr,3*mImageWidth);
	}
	return mImage;
}
