/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogeventrecorder.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "ifile.h"
#include "iimagefactory.h"
#include "iinteractoreventrecorder.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggframe.h"
#include "iggmainwindow.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetbuttonsubject.h"

#include "iggsubjectfactory.h"


using namespace iParameter;
using namespace iParameter;


namespace iggDialogEventRecorder_Private
{
	//
	//  Load and display the file name
	//
	class FileButton : public iggWidgetSimpleButton
	{

	public:

		FileButton(iggDialogEventRecorder *dialog, iggWidgetTextDisplay *display, iggFrame *parent) : iggWidgetSimpleButton("File",parent)
		{
			mDialog = dialog;
			mDisplay = display;
		
			this->SetBaloonHelp("Open the record file","Open a file to serve as record media. A new file can be opened for recording only, or an existing file can be opened for recording or playback.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			iString fn = mDialog->GetEventRecorder()->GetFileName();
			mDisplay->Display(fn);
		}

		virtual void Execute()
		{
			iString fn, oldfn = mDialog->GetEventRecorder()->GetFileName();
			if(oldfn.IsEmpty()) oldfn = this->GetShell()->GetEnvironment(Environment::Base);
			fn = this->GetMainWindow()->GetFileName("Open record file",oldfn,"Event record (*.ier)",false);
			if(!fn.IsEmpty() && fn.Part(-4).Lower()!=".ier") fn += ".ier";

			if(!mDialog->SetFile(fn)) this->GetMainWindow()->PopupWindow(mDialog->GetFrame(),"Failed to open a record file.",PopupWindow::Error);
		}

		iggWidgetTextDisplay *mDisplay;
		iggDialogEventRecorder *mDialog;
	};

	//
	//  Control button
	//
	namespace ControlMode
	{
		const int Record =	0;
		const int Play =	1;
		const int Rewind =	2;
		const int Stop =	3;
	};


	class ControlButton : public iggWidget
	{

	public:

		ControlButton(int type, iggDialogEventRecorder *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mType = type;
			mDialog = dialog;

			switch(type)
			{
			case ControlMode::Record:
				{
					mButton = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::ToggleButton,"",1);
					mIconOn = iImageFactory::FindIcon("er_recon.png");
					mIconOff = iImageFactory::FindIcon("er_rec.png");
					this->SetBaloonHelp("Record");
					break;
				}
			case ControlMode::Play:
				{
					mButton = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::ToggleButton,"",1);
					mIconOn = iImageFactory::FindIcon("er_playon.png");
					mIconOff = iImageFactory::FindIcon("moveright.png");
					this->SetBaloonHelp("Play");
					break;
				}
			case ControlMode::Rewind:
				{
					mButton = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::ToolButton,"",1);
					mIconOn = 0;
					mIconOff = iImageFactory::FindIcon("moveleft2.png");
					this->SetBaloonHelp("Rewind");
					break;
				}
			case ControlMode::Stop:
				{
					mButton = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::ToolButton,"",1);
					mIconOn = 0;
					mIconOff = iImageFactory::FindIcon("er_stop.png");
					this->SetBaloonHelp("Stop");
					break;
				}
			default:
				{
					IERROR_FATAL("Incorrect button type.");
				}
			}
			mButton->SetIcon(*mIconOff);
			this->Enable(false);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			switch(mType)
			{
			case ControlMode::Record:
				{
					if(mDialog->GetEventRecorder()->IsRecording())
					{
						mButton->SetDown(true);
						mButton->SetIcon(*mIconOn);
					}
					else
					{
						mButton->SetDown(false);
						mButton->SetIcon(*mIconOff);
					}
					this->Enable(mDialog->IsReady());
					break;
				}
			case ControlMode::Play:
				{
					if(mDialog->GetEventRecorder()->IsPlaying())
					{
						mButton->SetDown(true);
						mButton->SetIcon(*mIconOn);
					}
					else
					{
						mButton->SetDown(false);
						mButton->SetIcon(*mIconOff);
					}
					this->Enable(mDialog->IsReady() && !mDialog->GetEventRecorder()->IsAtEnd());
					break;
				}
			case ControlMode::Rewind:
				{
					this->Enable(mDialog->GetCurrentFrame()>0 && mDialog->GetEventRecorder()->IsStopped());
					break;
				}
			case ControlMode::Stop:
				{
					this->Enable(!mDialog->GetEventRecorder()->IsStopped());
					break;
				}
			}
		}

		virtual void OnVoid1Body()
		{
			mDialog->SetStateChanged();
			switch(mType)
			{
			case ControlMode::Record:
				{
					if(mButton->IsDown())
					{
						mDialog->GetEventRecorder()->Record();
						mDialog->UpdateDialog();
					}
					else mButton->SetDown(true);
					break;
				}
			case ControlMode::Play:
				{
					if(mButton->IsDown())
					{
						mDialog->GetEventRecorder()->Play();
						mDialog->UpdateDialog();
					}
					else mButton->SetDown(true);
					break;
				}
			case ControlMode::Rewind:
				{
					mDialog->GetEventRecorder()->Rewind();
					mDialog->ResetCurrentFrame();
					mDialog->UpdateDialog();
					break;
				}
			case ControlMode::Stop:
				{
					mDialog->GetEventRecorder()->Stop();
					mDialog->UpdateDialog();
					break;
				}
			}
		}

		int mType;
		const iImage *mIconOn, *mIconOff;
		ibgWidgetButtonSubject *mButton;
		iggDialogEventRecorder *mDialog;
	};

	//
	//  Dump images check box
	//
	class DumpImagesCheckBox : public iggWidget
	{

	public:

		DumpImagesCheckBox(iggDialogEventRecorder *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mDialog = dialog;
			mButton = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Dump images while playing",1);

			this->SetBaloonHelp("Dump images while playing","If this box is checked, IFrIT will create images for each frame during the playback. This feature can be used to create animations by first recording all events while interactively changing the visualizatin scene, and then playing the recorded sequence of events back into a series of images.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mButton->SetDown(mDialog->GetDumpImages());
		}

		virtual void OnVoid1Body()
		{
			mDialog->SetDumpImages(mButton->IsDown());
		}

		ibgWidgetButtonSubject *mButton;
		iggDialogEventRecorder *mDialog;
	};
};


using namespace iggDialogEventRecorder_Private;


iggDialogEventRecorder::iggDialogEventRecorder(iggMainWindow *parent) : iggDialog(parent,0U,iImageFactory::FindIcon("recorder.png"),"Event Recorder","sr.gg.dr",1)
{
	mCurrentFrame = 0;
	mDumpImages = mReady = false;
	mModule = 0;
	mStateChanged = true;

	iggFrame *tmp = new iggFrame(mFrame,2);
	iggWidgetTextDisplay *td = new iggWidgetTextDisplay("",tmp);
	tmp->AddLine(new FileButton(this,td,tmp),td);
	mFrame->AddLine(tmp);

	tmp = new iggFrame(mFrame,5);
	mFrameDisplay = new iggWidgetNumberDisplay("Current frame",tmp);
	tmp->AddLine(new ControlButton(ControlMode::Record,this,tmp),new ControlButton(ControlMode::Play,this,tmp),new ControlButton(ControlMode::Rewind,this,tmp),new ControlButton(ControlMode::Stop,this,tmp),mFrameDisplay);
	mFrame->AddLine(tmp);

	mFrame->AddLine(new DumpImagesCheckBox(this,mFrame));

	this->ResizeContents(300,100);
}


bool iggDialogEventRecorder::SetFile(const iString &fn)
{
	if(fn.IsEmpty()) return false;

	iFile f(fn);
	if(f.Open(iFile::_ReadWrite,iFile::_Text))
	{
		//
		//  Opened for playing/recording
		//
		f.Close();
		this->GetEventRecorder()->SetFileName(fn.ToCharPointer());
		mReady = true;
	} 
	else if(f.Open(iFile::_Write,iFile::_Text))
	{
		//
		//  Opened for recording only (a new file)
		//
		f.Close();
		this->GetEventRecorder()->SetFileName(fn.ToCharPointer());		
		mReady = true;
	}
	else mReady = false;
	mCurrentFrame = 0;
	this->UpdateDialog();

	return mReady;
}


void iggDialogEventRecorder::SetStateChanged()
{
	mStateChanged = true;
}


void iggDialogEventRecorder::ResetCurrentFrame()
{
	mCurrentFrame = 0;
	mFrameDisplay->Display(mCurrentFrame);
}


void iggDialogEventRecorder::UpdateDialog()
{
	if(this->GetEventRecorder()->IsStopped())
	{
		if(mModule != this->GetShell()->GetControlModule()->GetViewModule())
		{
			mModule = this->GetShell()->GetControlModule()->GetViewModule();
            this->SetFile(this->GetEventRecorder()->GetFileName());
		}
	}
	iggDialog::UpdateDialog();
}


void iggDialogEventRecorder::UpdateInteraction()
{
	if(this->GetEventRecorder()->IsStopped()) return;

	mCurrentFrame++;
	mFrameDisplay->Display(mCurrentFrame);
	if(mStateChanged)
	{
		mStateChanged = false;
		this->UpdateDialog();
	}
	if(mDumpImages && this->GetEventRecorder()->IsPlaying()) this->GetShell()->GetControlModule()->GetViewModule()->DumpImages(ImageType::AnimationFrame);
	this->GetMainWindow()->ProcessEvents();
}


iInteractorEventRecorder* iggDialogEventRecorder::GetEventRecorder()
{
	return this->GetShell()->GetControlModule()->GetViewModule()->GetEventRecorder();
}

#endif
