package org.broad.igv.sam.mods;

import org.junit.Test;

import java.util.*;

import static org.junit.Assert.*;

public class BaseModificationsTest {

    @Test
    public void testOrientTopFwd() {

        //top-fwd	0	*	0	0	*	*	0	0	AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT	*	Mm:Z:C+m,1,3,0;	Ml:B:C,128,153,179
        byte[] sequence = "AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT".getBytes();
        String mm = "C+m,1,3,0";
        byte[] ml = {(byte) 128, (byte) 153, (byte) 179};
        boolean isNegativeStrand = false;
        int[] expectedPositions = {7, 30, 31};

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('C', sequence[pos]);
        }

        int idx = 0;
        assertEquals(ml[idx++], (byte) likelihoods.get(7));
        assertEquals(ml[idx++], (byte) likelihoods.get(30));
        assertEquals(ml[idx++], (byte) likelihoods.get(31));
    }

    @Test
    public void testOrientTopRev() {

        //                                          ....45..........6.......4...8.0.....
        //top-rev	16	*	0	0	*	*	0	0	ATATGGCATATCCCCCGCCGATCCGCTAGAGATCCT	*	Mm:Z:C+m,1,3,0;	Ml:B:C,128,153,179
        byte[] sequence = "ATATGGCATATCCCCCGCCGATCCGCTAGAGATCCT".getBytes();
        String mm = "C+m,1,3,0";
        byte[] ml = {(byte) 128, (byte) 153, (byte) 179};
        boolean isNegativeStrand = true;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('G', sequence[pos]);
        }

        int idx = 0;
        assertEquals(ml[idx++], (byte) likelihoods.get(28));
        assertEquals(ml[idx++], (byte) likelihoods.get(5));
        assertEquals(ml[idx++], (byte) likelihoods.get(4));
    }


    @Test
    public void testOrientBotFwd() {
        //                                                    1         2
        //                                          .12.......0.12...78.01234....9......
        //bot-fwd	0	*	0	0	*	*	0	0	AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT	*	Mm:Z:G-m,0,0,4,3;	Ml:B:C,115,141,166,192

        byte[] sequence = "AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT".getBytes();
        String mm = "G-m,0,0,4,3;";
        byte[] ml = {(byte) 115, (byte) 141, (byte) 166, (byte) 192};
        boolean isNegativeStrand = false;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('G', sequence[pos]);
        }

        int idx = 0;
        assertEquals(ml[idx++], (byte) likelihoods.get(1));
        assertEquals(ml[idx++], (byte) likelihoods.get(2));
        assertEquals(ml[idx++], (byte) likelihoods.get(18));
        assertEquals(ml[idx++], (byte) likelihoods.get(23));
    }

    @Test
    public void testOrientBotRev() {

        //                                          ......6....12345.78...23.5.......34.
        //bot-rev	16	*	0	0	*	*	0	0	ATATGGCATATCCCCCGCCGATCCGCTAGAGATCCT	*	Mm:Z:G-m,0,0,4,3;	Ml:B:C,115,141,166,192
        byte[] sequence = "ATATGGCATATCCCCCGCCGATCCGCTAGAGATCCT".getBytes();
        String mm = "G-m,0,0,4,3;";
        byte[] ml = {(byte) 115, (byte) 141, (byte) 166, (byte) 192};
        boolean isNegativeStrand = true;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('C', sequence[pos]);
        }

        int idx = 0;
        assertEquals(ml[idx++], (byte) likelihoods.get(34));
        assertEquals(ml[idx++], (byte) likelihoods.get(33));
        assertEquals(ml[idx++], (byte) likelihoods.get(17));
        assertEquals(ml[idx++], (byte) likelihoods.get(12));

    }


    @Test
    public void testChebi() {
        // @CO	Separate m, h and N modifications
        //                                                    1         2         3
        //                                          ..2.4.67.....3...7.90...4.5.8.01.34.
        //        *	0	*	0	0	*	*	0	0	AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA	*	Mm:Z:C+m,2,2,1,4,1;C+76792,6,7;N+n,15;	Ml:B:C,102,128,153,179,204,161,187,212,169
        byte[] sequence = "AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA".getBytes();
        String mm = "C+m,2,2,1,4,1;C+76792,6,7;N+n,15;";
        byte[] ml = {(byte) 102, (byte) 128, (byte) 153, (byte) 179, (byte) 204, (byte) 161, (byte) 187, (byte) 212, (byte) 169};
        boolean isNegativeStrand = false;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(3, modificationSets.size());

        for (BaseModificationSet bms : modificationSets) {
            String mod = bms.getModification();
            if (mod.equals("m")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                int idx = 0;
                assertEquals(ml[idx++], (byte) likelihoods.get(6));
                assertEquals(ml[idx++], (byte) likelihoods.get(17));
                assertEquals(ml[idx++], (byte) likelihoods.get(20));
                assertEquals(ml[idx++], (byte) likelihoods.get(31));
                assertEquals(ml[idx++], (byte) likelihoods.get(34));

            } else if (mod.equals("76792")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                int idx = 5;
                assertEquals(ml[idx++], (byte) likelihoods.get(19));
                assertEquals(ml[idx++], (byte) likelihoods.get(34));

            } else if (mod.equals("n")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                assertEquals(ml[7], (byte) likelihoods.get(15));
            }
        }
    }

    @Test
    public void testDouble() {
        //public static void main(String [] args) {
        // @CO	Modifications called on both strands of the same record,
        // @CO	including potentially at the same location simultaneously.
        //                                                1         2         3
        //                                      .....5.7...1....6..9..........01....
        // *	0	*	0	0	*	*	0	0	AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT	*	Mm:Z:C+m,1,3,0;G-m,0,2,0,4;G+o,4;	Ml:B:C,128,153,179,115,141,166,192,102
        //                                      .12.......0.23...78.01234....9......

        byte[] sequence = "AGGATCTCTAGCGGATCGGCGGGGGATATGCCATAT".getBytes();
        String mm = "C+m,1,3,0;G-m,0,2,0,4;G+o,4";
        byte[] ml = {(byte) 128, (byte) 153, (byte) 179, (byte) 115, (byte) 141, (byte) 166, (byte) 192, (byte) 102};
        boolean isNegativeStrand = false;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(3, modificationSets.size());

        for (BaseModificationSet bms : modificationSets) {
            String mod = bms.getModification();
            if (mod.equals("m") && bms.getBase() == 'C') {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                int idx = 0;
                assertEquals(ml[idx++], (byte) likelihoods.get(7));
                assertEquals(ml[idx++], (byte) likelihoods.get(30));
                assertEquals(ml[idx++], (byte) likelihoods.get(31));

            } else if (mod.equals("m") && bms.getBase() == 'G') {
                int idx = 3;
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('G', sequence[pos]);
                }
                assertEquals(ml[idx++], (byte) likelihoods.get(1));
                assertEquals(ml[idx++], (byte) likelihoods.get(12));
                assertEquals(ml[idx++], (byte) likelihoods.get(13));
                assertEquals(ml[idx++], (byte) likelihoods.get(22));

            } else if (mod.equals("o")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                assertEquals(ml[7], (byte) likelihoods.get(13));
            }
        }
    }

    @Test
    public void testMulti1() {
        // @CO	Testing multiple m, h and N modifications on the same read.
        // @CO	r1 has them separated out.
        // @CO	r2 has them combined together, for example as produced by
        // @CO	a joint basecaller which assigns probabilities to all
        //  @CO	trained events simultaneously.
        //                                                1         2          3
        //                                      ..2.4.67.....3...7.90...4.6.8.01.34.
        // r1	0	*	0	0	*	*	0	0	AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA	*	Mm:Z:C+m,2,2,1,4,1;C+h,6,7;N+n,15,2;	Ml:B:C,128,153,179,204,230,159,6,215,240

        byte[] sequence = "AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA".getBytes();
        String mm = "C+m,2,2,1,4,1;C+h,6,7;N+n,15,2";
        byte[] ml = {(byte) 128, (byte) 153, (byte) 179, (byte) 204, (byte) 230, (byte) 159, (byte) 6, (byte) 215, (byte) 240};
        boolean isNegativeStrand = false;


        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(3, modificationSets.size());

        for (BaseModificationSet bms : modificationSets) {
            String mod = bms.getModification();
            if (mod.equals("m")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                int idx = 0;
                assertEquals(ml[idx++], (byte) likelihoods.get(6));
                assertEquals(ml[idx++], (byte) likelihoods.get(17));
                assertEquals(ml[idx++], (byte) likelihoods.get(20));
                assertEquals(ml[idx++], (byte) likelihoods.get(31));
                assertEquals(ml[idx++], (byte) likelihoods.get(34));

            } else if (mod.equals("h")) {
                int idx = 5;
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                assertEquals(ml[idx++], (byte) likelihoods.get(19));
                assertEquals(ml[idx++], (byte) likelihoods.get(34));

            } else if (mod.equals("n")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                int idx = 7;
                assertEquals(ml[idx++], (byte) likelihoods.get(15));
                assertEquals(ml[idx++], (byte) likelihoods.get(18));

            }
        }
    }

    @Test
    public void testMulti2() {
        // @CO	Testing multiple m, h and N modifications on the same read.
        // @CO	r1 has them separated out.
        // @CO	r2 has them combined together, for example as produced by
        // @CO	a joint basecaller which assigns probabilities to all
        //  @CO	trained events simultaneously.
        //                                                1         2          3
        //                                      ..2.4.67.....3...7.90...4.6.8.01.34.
        // r2	0	*	0	0	*	*	0	0	AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA	*	Mm:Z:C+mh,2,2,0,0,4,1;N+n,15;	Ml:B:C,77,159,103,133,128,108,154,82,179,57,204,31,240

        byte[] sequence = "AGCTCTCCAGAGTCGNACGCCATYCGCGCGCCACCA".getBytes();
        String mm = "C+mh,2,2,0,0,4,1;N+n,15";
        byte[] ml = {(byte) 77, (byte) 159, (byte) 103, (byte) 133, (byte) 128, (byte) 108, (byte) 154, (byte) 82, (byte) 179, (byte) 57, (byte) 204, (byte) 31, (byte) 240};
        boolean isNegativeStrand = false;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(3, modificationSets.size());

        for (BaseModificationSet bms : modificationSets) {
            String mod = bms.getModification();
            if (mod.equals("m")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                assertEquals((byte) 77, (byte) likelihoods.get(6));
                assertEquals((byte) 103, (byte) likelihoods.get(17));
                assertEquals((byte) 128, (byte) likelihoods.get(19));
                assertEquals((byte) 154, (byte) likelihoods.get(20));
                assertEquals((byte) 179, (byte) likelihoods.get(31));
                assertEquals((byte) 204, (byte) likelihoods.get(34));

            } else if (mod.equals("h")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                for (Integer pos : likelihoods.keySet()) {
                    assertEquals('C', sequence[pos]);
                }
                assertEquals((byte) 159, (byte) likelihoods.get(6));
                assertEquals((byte) 133, (byte) likelihoods.get(17));
                assertEquals((byte) 108, (byte) likelihoods.get(19));
                assertEquals((byte) 82, (byte) likelihoods.get(20));
                assertEquals((byte) 57, (byte) likelihoods.get(31));
                assertEquals((byte) 31, (byte) likelihoods.get(34));

            } else if (mod.equals("n")) {
                Map<Integer, Byte> likelihoods = bms.getLikelihoods();
                assertEquals((byte) 240, (byte) likelihoods.get(15));

            }
        }


    }

    @Test
    public void testONTForwardAlignment() {
        byte[] sequence = "GTGTAGCTGGTTCAGTTACATTATTGCTCACTGCAGCCTCCGCCTCCTGGGTTCAAGTGATTCTTCTTGCCTCAGCCTCCCAAGTAGCTGGGATTACAGGCGCCCGCCACTACCCGGCTAATTTTTGTATTTTTAGTAGAGACAGGGTTTCACCATGTTAGCCAGGATGGTCATGATCTCTTCCTGACCTCATGATCCGCCCGCCTCGGCCTCCCAAAGTGCTGGGATTACAGGTGTGAGCCACTGCACCAGAAAACAAGAGTGAATTTTAAAAAACAGTGCAGCAGGGCCGGGCTCCGTGGCTCACATCTGTAATCCCAGCACTTTGGGAGGCAAAGCCGGGCAGTTCGTTGGAGGTCGGAGTTCAGGAGACCAGCCCGGGCAACATGGTGAAACCCCGTCTCGACTAAAAATACTAAAAATTAGGCCAGGTGCAGTGGCTCACTGCCTGTAATCCCAGCACTTTGGGAGGCCGAGGCGAGGGCGCTTGAGGTCAGGAGATGAGACCATCCTGACCAACATGGCAAGACCCATCTTAAAAACACACACTCTTAAAGCAGGTTTGTGTTTGGCACCAGTAGATCAGTCCTCAGGAGGCTGAGGCGAAAGTCAACAGGTGAGGGAGTTACGGTGTCGAGACAGGCCACTGCACTTCCAGCCTGGGTGAGTTAGAAGACCTGTCTCAACAAAAATTAATAAAAATTAGCTGGGTGTGAGTGCACGTACACTTGGGAGGCTGAGGCAGCAGAATCATTTAAACCTGGGAGGCAGGGGGTTGTAGTAGAAGCCAAGATCGTGACACCGTACCAGCCTGGGTGACAGAGAAAGACCCCATCTCAAAAAAAACAGAAAACAAAAAACAAAACAAACAAAAAAACTAAATATCTTTGTAGCCCACAGGCTATCCATTTTATTCAAAGCATCAAGAAGTAAGAAAATGAAGAAAGGGCTCTGTGTTGCCCATAGGTGGGTGGGATTGCAATAACTCGAGAAGCATCAGTGAGGCTTTCAGGCAGGAGGCTCTCACTCCCTCTGTCAGTCATCTTGGAGATGTCCAGAGGCTAAGATGCCCAGCCATACAGAGGTGGTGAGAACCATCGTGGCGGCTCCAGGAGAAGAGGCCCAGAAATGGACATTTGAATGATTGAAACTGCCTAAAGCACTCCTGCAACAAAGCACTCCATGTCTCACATTGACATGGAACAATTAACTGAAATGGGAAACATATCAACTCAGGCTGCCTCCCTCTGTTTCAGACCAGTGTTCTACTCTACACAGCAAACCAGGTCTCCCTTTCTTTCCTGCCTGCAGTCATTCCCAGAAGTTTACCTCAAAGGAACTCATGTTTCTGAAAATCCAGTTTGGGCATGGGCTCTCAATGCCTTCTGGCTCCCTCTCTGAATTTCATAATCTGATATAGAAAATAGTAGCTCACATTTCTGACACTCTGTATGGAGAGATGTGTAGCATGGCTCGCCTCTCCATGAACGTCAGTGACCCTGAGATTTGGATCTTTGCCCTTCACATTTCTATATTCAGGAGCTGGCTCCATGCAATTTGGACTCTACCCCTAATAAGTCAGGGACATGAACCCCAAATTCTAAACACTTGGACCTCCAGAATCACCACCTGGATTCATTCCTGTCCAAATAAATGACTGTGGCCAGTATGTGTGTGTGCTATATTTATAAAGCACTTTTGATGTTAGGTTTATTTAAGCTTCACAATTCTTCAGTGAGGTAGGAAATAATCCCATTTTATGGAGGGAAAAGCTGAGATTTGGAAAAGTTATCCCCCAAAGTTACACAGCTACTAAGGGACAGAGCTGGACTTCAAATCAAGTTGTTTGATCTAAACTCAGTGCATGTCCACCATTTCATCCTACCTTTTGATTACGGTTGACATTATAAATTAAATACATGTCCCTAAGTGGATTTTTAAGTAAGCAAATAAGATCTCTTCCTACCTAAGCACCTAGCACATTAGCTGGAGAACTAGCACAAAGCGAAGCCACACAGATTGTTACCAACTCAGCTCTCTCCCAATGGATGCATGAGTTTAAAGGATATTAAAGTTACCAATTATGGAACTCATTCTAGGTGCCAGGCCCTGTGCCAGTCTCTTTACATCCATTATCACTTAATCTTAATATCAGTTATTTCAAGATAGGCATTATTCTTCCACTGGGAAACTGAGGCTCAAGAAGTCAAATATTGGCTGGGTGTGGTGGCTCATGCCTGCAATCCCAGCACTTTGGGAGGATCACTTGAGGTCAGGAGTTTGAGACCAGCCTGGCCAACATGGTGAAACCCCATCTCTACTAAAAATACAAAAATTAGCCGGGCGTGGTGGCAGGCATCTGTAATCCCAGTTATTCAGGAGGCTGAGGCAGGAGAATCACCTGAACCCGGGAGGCAGAGGTTCAGTGAGTCGAGATCTTGCCACTGCACTCCAGCCTGGGAGACAGAGGGAAATTCGTCTCAAAAAAAAAAAAGAAAAGAAAAGTAGAAGACGAAGATGAAGAAGAATAAGAAGAGAGAAAGAAGAGAGGTTAGAGAAAAGAAGGAGAAGGAGAAGGAGGAGGAGAAGGAGAAGGAGAAGAAGAAGAAGAAGAAGAAGAAGGAAGAAGAAGAAGAAGAAGAGTCAATATCTGGAGAAGAAGAAAGAGAGGAAGAAGAAGAAGAGGAAGAAGAAGAAGAAGGGAAGGAAGAAAGAGAAGAAGAACAGAAGAAAGAGGGAGTCAAATATCTGGAGAAGGGAGGAAGAAAGAAGAAAGAAGAAAGAGGAGGAGGAAGAAGAAGAAGAAGAAGAAGAAGCAGGAGCAGGAGAAGAGAAGAAGAAGAAGAGAGAGGGAAGAGGGAAGGAAGAGGAGAGGAAGAGAGAGAAAGAAGAAGAAGAAAGAAGAAGAAAGAAGAAAGAAAGAAAGAAAGAAGAAGAAAGAAGAAAGAAGAAGAAGAAAGAAGAAAAGACAAATATCTGGAGCCAGGATTTGAGCCTGGCCCTTGACATATAATGCAAGAGCCTCACCACACTCTCCTTACCAGCCCCTGTCCTTAGGCCTGGGCCCCTGATTGCCTTGTCAGAGAAGCCAGAGAATCTGGGCCTTTTTAGCTTTTATCCTAGGAGGCTGGAATAATCTCCTTCTAAGGGGTACTAAGGGGACGCCCAGGGAGAACAGGGGTTCTGATGAGGGTGGAGGGAGAGGGGTGTGATGCTGTACTGGCTTTGTTCACTAACACAGCTACTAAGGGAGCTTTGAGAGAGAGGCCTCTCAGTGAAGAGCCCGGGTCAACCCCCATTTCCCTCGTTCAGGTACGACATACCCTGGGAAGCCTTGACTCATTAGGGCTCCAGAGTCTCTAACTGTGCAGCAGGG".getBytes();
        String mm = "C+m,10,17,2,5,16,2,2,18,2,11,1,0,5,5,18,0,0,27,0,2,2,15,11,2,35,27,95,5,88,21,92,17,59,21,9,1;C+h;";
        byte[] ml = null;  // not testing for this
        final boolean isNegativeStrand = false;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('C', sequence[pos]);
        }
    }

    @Test
    public void testONTNegativeAlignment() {
        byte[] sequence = "TGTACTTCGTTCAGTTACGTATTGCTAAAGCTTTCTGGAGATTGTGTTAGTTTTCTATACTTGCTGTAACAAATCTACAAATTGGGGACCACAAAACAGCACAGAAAAAACATTATCTGCGGTTTCTGTAGGCCAAAAGCCCAGGTGATCTGACTGGGTCTTCAGTTTGGGGTTTCACAGGGCTAAAATCCCAAGGCTGTTCTTTGTGGGGGCTCTGGGGAAGAATCCACTGCAGGGCTCATTTCAGTTGTTTGGTGAATTCAGTTAATTGAGGTTGTAGGACCAGGTTCTTGTTTCCTTATTGGCTGTCCAAATTCCTTCTCATTCCCATGTGGCTGCTTCAGCTGTGGCAAGTTAGGTCCTCATGCTGAGAATCAATCCAAGTCTCCCTCCACCACATCTCTCTTTCTTCCATCCCCAGCCCTGAGATAACTCTGTTTAAGGTTCATTTGATTATCCTGAGGTCCACTTGGATAATCCCATGATAATCTCCCTATCTTAAGACCTGTAACCTTAATTACATTTGATATTGTGAAGACAATGGTGTCTGTTTACTTACCCTACAGAATCTGATGGAATTGGGGCTATTCTTCTCCTTCTAGGACATTCACTTGGCCAGAGGTGTACGACATGGCTGCTCAAACGAGTCCAGATGCCCACATACCTGTGCTCTTTGCCGTCATAAAGCTGGAAACCTACACGCATTTGCTCAGGATATCCTAATTTTTTAATGCAAGTTTGTCTTGAACAGGTCCTGAATCTCTGAAAGTCTCAGTGAGCAACTCTTAGGGACAAAGTGACAAAGACCATCAGGAGTCTCTTGTGGACTGGCTTTCCTTTGAAAGTAAAATTCAGAGTCTTACACTGCCATGCAGAGTAATTCTGCATGTTCTGTCAGTAATGTGTTGACACTTCCAGAGTATAAGCAATTAATTCATTCAGCAGATGTTCTTTTCCAGTACACACTACATTCAGCCTTAAGCACAGTGACATGTACTTTATATATCTTCTTCAAAAGCAGTTCTTACCATCAATAGCCGAGACTGGTAGGCAGGCGGCAGGGTGGCAAGAGATACAAACAGCTACAGTTTGAAAAAAAGTAAAGTGTCACAAATCTTTGGGGTTATTCAAAGGTGCTATTTAAGCCTTGTGGAAGATGTAAATAACTGAGTGTGTTTTCCAAGACTGAAGCAAGTGGGTGGGAGAGAAGGGTTGTGTGTGAGTGGGTGATGGGTTTGGGAAGAGATGGGTAGCTCTGAGTGGAGGTGGGAAGCTTGGATTGTTTGAGAGAATGTCTCTTGCCAGCAGTTCACAACGAGATGGAACTGGCTGAAACTGGACAACACCATTGGACCACACTGGGACTTGTTGTGATTGGCCTCATTGTTCTGGGCTGATTTTGTTGCTTAGAAATCACCATGGGTAGGATCACGAGATCACAGGAAAAACCTGCTCAGGGAATCAGTTCCTCAAGGTATCCCCACTAAATAAACAGTTTCCCATCAAATCAGAGTCCGCTGCGGTTTCCCGCTCAGGGCTGTGATGGGAAACAGGCAAGACTTGGGGACCATCTGAAACAACTTCAGACTTCCTGCTTGAAGTGAAAGCCCTTTGGGATGCTGGGGAACTGAGTGCCTAAGACCTTGTTACCAATCTGTCATCGGTGACCTAGTCCCACGTTATGAATAAAGACCACCCATTTCTGTTGTTAGAACAACCTCCCCTCTGTCAAGAGGGCAGCACTGGGTTTCTGACTTCCTACAGACAAACCCAAGGTGAAGAAGGGGGAGTTTGGAGAAGGCAGAAATCCAGATCCCTCGAAATATGATAAAGACACAAATATACAAATAGACCTTGTACAAGCACATCCTATGAATGACCTGTGAGAGACTCTGGAGCCCTAATGAGTCTGGCTTCCCAGGGCATGTGGGTACCTGAACTTGAGGAATGGGGGTTGGCCCGGGCTCTTCACTGAAGGCCTCTCTCAAAGCTCCCTTGGTAGCTGTGGGTAGTGAACAAGCCAGTGCAACATCACCCCTCTCCTCCACCTCATCGGAGACCCTGTTTCTCCCTGGGGCGTCCCCCTTGATGCCCCCTTAGAAGGGAGATTATTCCAGCCTCCTAGGATAAAGCTAAAAGAGCCCAGATTCTCTGGCTTCTCCGACAGGCAAATCAGAGGCCCAGGCCTAAGGGACAGGGGCTGGTGGGGAGGTGTAAGAGGCCTAGGACATATGTCAAGGGGGCCAGGCTCAAATCCGACTCAGATATTTGTCTTTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCCTCTTCTCTTCCTCTTCCTCTTCTCTTCTTCTTCTTCTTCTTCTTCTCCTGCTCCTGCTTCTTCTTCTTCTTCTTCTTCTTCCTCCTCCCCTCTTCTTCTTCTTCTTCCTCCTCTTCTTCTTCTTCTCAGATATTTGACTTCTTCTTCTTCTTGTTCTTCTTCTTTCTTCCTCTTCCTCTTCTTCTTCTTCTTCCTCTTCTTCTTCTTCCTCTTCTTCTTCTTCTTCTCCAGATATTTGACTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTCTTTCTTCTCCTTCTCCTTCTCCTCCTCCTTCTCCCTTCTCCTTCTCTTTCTTTCTCTAACCTTCTCTTCTTCTTTCTCTCTTCTTATTCTTCTTCATCTTCGTCTTCTACTTTTCTTTTTTCTTTTTTTTTTTGAGACAATTCCCTCTGTCCCCCAGGCTGGAGTGCAGTGGCAAGATCTCATGACTCACTGGAACCTCTGCCTCCCGGGTTCAGGTGGATTCTCCTGCCTCAGCCTCCTGAATAGCTGGGATTACAGATGCCTGCCACCACGCCCGGCTAATTTTTGTATTTTTAGTAGAGATGGGGTTTCACCATGTTGGCCAGGCTGGTCTCAAACTCCTGACCTCAGTCCTCCCAAAGTGCTGGGATTGCAGGCATGAGCCACCACACCCAGCCAATATTTGACTTCTTGAGCCTCAGTTTTCCCTGGTATTGCAAAAAAGGAAGAATAATGCCTATCTTAGACAACTGATATTAAGATTAAGTGGGATAATGGATGTAAAGAGACTGGCACAGGGCCTGGCACCTAGAATGAGTTCCCATAATTGGTAACTTTTAATATCCTTTAAAAAACTCATGGCAAATCCATTGGGAGAGCTGAGTTGGTAACAATCTGTGTGGCTTCAGCGCTTTACCGAGTTAGTTCTCAGCTAATGTATAGGTGCCAGAGTAGGAAGAGATCTTGCTTGCTTGCTTAAAAATCGAATTTAGGGACATGAAAATTTAATTTATAATGTCAGCAATCAAAAGGTAGGATGAAATGGTGGACATACTGAGTTTGGGATCAAAACAGCTTGGATTTGATCCAGCTCTGTCTGCTCTAGTAGCTGTGTAACTTTGGGGATAACACACTTTTCCCAAATCAGCTTTTCCTCCATAAAATGAGGTCATTTTCCTACCTCACTGAAAGAAGTAAACCAAGCTTAAATAAACCTAACATCAAAAGTGCTTTATAAATATAGCACACACATATGACCACAGTCATTTATTTGTGAGACGAATCAGGTGGTGGATTCTGAGGTCAAGTGTTTAGAATTTGGGGTTCAATGTCCCTGACATTGGGGTTCAGTTCCCAGGGATGCAGCCTGGTTTGTAGAAATGTGCCAGGAAGGGCCAGCATCCAAATCTCAGGTCAGGACGATCATGGAGAGGGCAGCCATGCTATGACATCTCACAGGTGTCAGGGTGTGAGCACATTAGGACCAGGATGAGAACAGAGGGGAGGCGCAGAAGGCAGCATTGGTGAGCCCTGCCTCAAGATCTGGATGAGGAACATGGCTATTAGAGTGAAACTTCTGGGGAATGACTGCAGCGGAAGAAAGGGAGACCTGTTTGGGGACTTGCTGTAGTGAGTAGAACACTGTACCTCGAAATATAAGGAAAGACGGCCTGAGTTGATATATTTCCATTTCAGCTAATTGTTCCATGTCAATGTGAGACATGGAGCAAGCTTGAAAGCAGGAGTGCTTTAGGCAGTTTCAATCATTCAAATGTCCATTTTCTAGGCCTCTTCTCCTGGAGCCGCCACGGTAGTTCTCACCACCTCTGTGCATGGTTGGGCATCTTGCAGCCTCTGGACATCTCCAGATGACTGACAGAGGGAGGGAGAGAGCCTCCTGCCTGAAAGCCACTGATGCTTCTCGGTTGTGCAAAAATCCCACCCACTCTATGGGCACACAGAGCCCTTCTTCATTTTCTTACTTCACCGATGCTTTGAATAAAAATGGATAGCCTGTGGGCTACAAGATATCTAGTTTTTGTTTGTTTTGTTTTTATTTTCGTTTTTTTGAGATGGAGTCTTTCTCTGTCACCCAGGCTGGAGTGCGGTGTCGATCTTGGCTTACTACAACCCCCGTCTCCCAGGTTTAAATGATTCTGCTGCCTCAGCCTCCCAAAGTGTACGCCACCACACCCAGCTAATTTTTTGTTGTTTTTTTTTATGAGACAGTCTCACTCTGTCGCCAGGCTGGAATGCAGTGGCCCAGTCTTGGCTCACCGCAACCTCCACCTCCCAGGTTCAAGTGATTCTCCTGCCTCAGCCTCCTGAGTAACTGGGACTACCGGTGCGCACCACCACACCCAGCTTTTTGTGTGTGTATTTTTAGTAAAGACGGGGTTTCACCATGTTGGTCAGGATGGTCTCGATCTCCTGACTTCAGTGACCACCCACCTCAGCCTCCCCAGTGCTGGGATTACAGGCGTGAGCCACTGCACCTGGCTAATTTAGTATTTTTAGTCCGAGACAGGGTTTCACCATGTTGCCCAGGCTGGTCTCGAACTTTTGACCTCAAGAGCACCCGCCTCGGCTTGGGGCTCCCAAAGTGCTGGGATTACAGATGTGAGCCACTGAGCCTGGCCAGTATCAACTTTTTAAAATCTGAATTCTTGTTTTGTGCAGTGGCTCACACCTGTAATCCCAGCACTTGGGAGGCCGAGGCAGGCGGATCATGAGGTCAGAGATCGAGACCATCCTGGCCAACATGGTGAAACCCTGTCTCTACTAAAAATACAAATCGCTGGGCATGGTGGCGGGCGCCTATAGTCCCAGCTACTCGGGAGGCTGAGGCAGAAGAATCTTGGAACCCAGGAGGCAGAGGCTGCAGTGAGCCAGAAATCACACCACTGCACTCTAGCCTGGGTGACAGAGTGAGACTCTGTCTCAGGAAAAAAAAAAAAAAAAAAAAATTATAATTTTCTTTTGTGACTCATGATTCCTCCCCAATTCTTCTGATTAAGTGGAGTAGCTCTTCAAATGCAAATCATTTTTTTCTAAAAAGTCAAGTCAGATGAAATCAATTTTTGTTTCTCTGGAGTGGGGGTCCAGGAAAGTCTTTTTATCTGAAGTAACCTTTCTCTGGAATTCTTTTTCTCAAATCAGTAGTGGAGGTGCTTCTATATAGTTTTGCTGCATTCCCCTCACTGGGTGGGCATCCCAGCCACAAGAAAGGAGGGGCATGGTGAGGCTGAGAAGAGAGAAGAAACATTTTGAGCACTCCTGCTATGAGCCAGGCTCTGTTTCCATCTGTTTCTCATTTCATCTTCTCATGGTGAGGCTAGAGGTAGCTAGAATGACCTCTGTTTAACACCAGGAGAAACTGAGGCCAGAAGAAACTGAAAAGATCTTGTGTGATAAAATAATGGGCTGTGATTTGAGTCCAGATCTAACACCACAGATGTTCTTTCCTCTATGTTAGGTTCAGTTATTGCTTGCTTCTGTAACAAGCTTCTCTCCAAAGTTAGGGTCTTAAAACAACAATCACCCTTCTATTTTATCTCAAGATTTTGTGGGTCAGAATTTGGGCAGGGTGTGGTCAGGTGATTCTTCTGGTCACTTAGTGGATGGAGCTGTTCTAGAATCAAGACAGGTTTACTCACTCAGCAGTGACTGACCAGTATACCCAGCATCCCAGATGCGGCCTCTCTTGGCATGGTGGACAAACATTTGGGCTTTACACTGTAGCTCAGCATTCCCACAGAGAATGTTCACGGAGACACAGACAAAATACAGCTTTGCTAGACCAAGCCTCAGAAGTTCCAGAACATCCCTTCCACCATGTTCCGTTAGTCAGGCATTCTCCGTCTCTTCAGTGAGAGAAGTAGCAAGGAATTTGCAGCCATCATTTGTCCTACTATAAAAGATACTGTTTAATCACCAATGGTTACATAAATTACAGTACAAAGGTGGCAGAAAGAAATGCATGGCCTAGGCTGCCTGTTTTGGAAAACAAGAGTTAGACAGCCCAGTTGAGCCAGCGTGCTAATCTTCATCTTCCCTGCCCTAGACTGAAGAAATGCAGATGTGTGAAAGGTAAGTGGAGGGGGAAGAGAGATGTCCTCCTCTGTTCCTTAGCTAAACACATACACCCTCACGTGGCCGCTACACACACGCACACACACACACACTCTTGAATAATGCTTAGTTCTTTGCTTTGGCTCAAGAGACATGAGCCAAGAAGTCTAGAGGTCCAGCATGGTTGGGACTCTCTGTCTGGGGCACCCTAGGTGCAGACTCCACTCTGTGTCTCCCAGATTCATTCCTGACACGCAGTGGGTGGGAACCACACAGACCCTCTCAACCACCTCCCATCCTCATTCATTTCCGTTCCCGAACCAGTCCCCATAAGCCTAGCTCCTCTGCCTCAGCTCCTGGCTGAGGTCCTCCTGTGGGGGATGACCTTGCGCATGGGTTGCTGTCAGCCATCCTCTCTGCCCACTAGGGTTCTGCTGCTGGATGGGAACACAGGAAATTGTGCATCCTTGGGTGGAGAGTGAGAAAAGCAGCCCTCACTATGGACTGGCCTCTAGATAGGAATGTATCAAATAAATAAATTGAACTTCTCGTGAACTGTGATGGTCGATACTATGCTCATGTGTTCCCTAAGCCTTATTATGCACAAAGAGCATCTACTTCTTCCCAGTTTTTTTCTTTTTTTCTTTAAGAGATGGGGTCTCACTATGTTGGCCCAGGCTGGATTGGAATTTCTGTGTTCAAGTGACTCTCCTGCCTCAGCCTCTTGAGTAGCTGGGACTACAGGCACGCACTGGTGTGTCCAGCTCCCAGCTTTGGGTTTTAATATCAAAGTTACCCCCTTATTAGTTTAGCACTGGGTGTATGGGCTAGGTATTAAAAGTGTGGGAAAAACCCTTCCAAACCATCTGCTAGTCAGTGGTGACTGGAGACTCATGGTCTGAAAATTCATGGTTTTGAATTTTCAAATTTTAATTTTATTACATTTATCCTATCTTCGAGTAGTGATTAGAAAACCCAGAACTCCCATTATTAAAAATAAACCTTCACAAATTGTCTGGCCCACAGATAACTCACCTTGACATTGTTTCATTTTCAGCAACTACAATGAGCTTTACCTGAAAATTACTCCAAGTTCTTCAGGCTCTCCCACTGTCTTGTGCTGTTTACTATCTTTATGACACAAGAACACATGGACCCATGGCGGGGGCAGTGTCTCACACCTGTAATTCTAGCACTTGGGAGGTCGAGGCGGGTGGATCACCTGAGGTCAGGAGTTCAAGACCAGCCTGGCCAACATAGTGAAAACCCCATCTCTACTAAAAATACAAAAATTAGCTGGGCATGGTGGCAGATGCCTGCAATCCCAGCTACTCGGGAAGGCTGAGACAGGAGAATTGCTTGAATCCGGGAGGTGGAGGTTGCAGTGAGGCGAGATCATGCCATTGCACTCCAGCCTGGGTGACAAAAGTGAAACTCTGTCTCAAAAGAAAAAAAAAAGAACACACTGACCCACAGCAGATGCACTGGGAGAGAGAGTCTCCAAAGATTCACTCCTATTAAGAGAACCAGGAAAAGGTAGCAATACTCAACCACTGTCTTCACAATAGCAGGCAAATCAGACCAGGACAATGACTCCTCAGCAATATGAGGACCTGTATTCTTCCAGCAGCTGTGAGGCCACCTTGAGGCCTGGCCCCAGCAAACTGGGAGTAGACACCTGCCCCAGCCAGCACAGAGCAGATGTGGTGGCATGTACAGGCCTGGAATAGATCTGCGGTAGGCCAATGGTACTGGGAAGGAAACCAAAATATTTCTCCCCTAAAATATTGAGGATTGGTGAGCTAAAGACAACTGAGACAGTACGGAGACTCTCTGCCTCTCCTTTGTTTGCCTAAAGGCAGGACATAAATCCCCGTCTTACTGGAGATAGCCACTTTATCAGCCTAGAGACGGCACTAGCAGACACCAGAGAAACCTGCGAGCAGACCTTACTTCATGAGTATCTTCCCCACACAGCTTTCCCCCCTTTGAGGCTGAGCTGTTCTCTCCTTTGTCTTGTCACTGCACTCTAAAATTCATTGCTTTTTGTTAAATGCTACTTAAACAAAGCTTAAAGCCTCAGGTTTCCAGCCAATGCTTTGAGAGTCACTTTGCAGAGTTACTTTCTCCCCATGATGGACACGGCATATACATTATTAAGCATACCTGTTTTCCTTGTTGATCTGTCTTTGTTTCAGAAGGGGTGACTCAGTATGAATTTTATGAGAGTGGCAAAAAGATTATATAATTTTCTCCTTCCACATTATACCAGAAATCTGAAACCAAGCGGAAGGAGTCAATGTCATAGTTCGAAGAAAAGGAGCCTCCCTTAAAAGAAAGCCTCATATACATTTCAGAAATTTCAGCCATGTGGTTGTCTTCATTTGGGTTCCCCCAGAAGCTGACCATGAGCAGGGATTTGCATCCCCTTAGTAGTTTATTTAGTAGGATTATATTCAAGAAATATAAATAAGGAAGGAAACCAGTACAGAGAGTTGTATCAAAGTTTGCCAGGCTGGGCGCAGTGGCTCACACCTACAATCTCAACAATTTGGGAAGCTGAGGCAGGAGGATCACTTGAGCTCAGGAGTTCAAGACCAGCCTGGGCAATGTAAAGACCTTGTCTCTACAAAAAATAAAAATACAAAACTAGCCAAGCGTGGTGGTGCATGCCTGTAGTCCCAGCTACTTGGGAGGCTGAGGTGGGAGGATCACTTGAGTCCAGAGGTTGGTGCAGTGAGCTGAGATCACGCCACTGCACTCCAGCCTGGGTGACAAAGTGAGACCCTAGTCTCACAAAAAAAGGTACCACTGTGAGTCACCAGAGCTCAGAATCCACTGGGGAACTCTGGTATAGAACACGCACCTCATGGTTATCCCACCCAAAAGGGACATGAGGGCAGGATGTTTACATTCCCCAACTGCTCACAGTCATTGGTTGAGCAGTGCTTTAGTGGAACATTAGTGCCATTACTTCCACCCCCATGTAGGAAGATTAGGAGCCAACAGGAAAGCCCTTAGGAAATGGATGCAGGTGTAGTTACAAGCAGCTTTACTCAACAATAGAAAGCCGAGGGGATGGAGACCCAAATTTTCTTGCCTCTGGGGCAATGATTCTGGCTGCCACTTTCCTGACCCTCCCACCCCTTTCCTTCAATAAACAATTCCCTGACACTTACAGTGTCTGGCTTGCACCTCTGCCAAGGACCCAGGAAGGAAGAGCTGCCCTCGGAAGCTTTACAATGTGCTGTGAAGAGAAGCCTCCATTATTAGGGCATCTACTCTGTGCCAGTCACTCCCTGGGCTAGCTCCTTTCCATCCACCAGGCATTCTGTGAGTAAGATTCTTCTTCTCATTTTATAGACATGCCATGCAACCATATCAATTCTCTTCAGAACAGTGTAACCCAGTGTGAGGATGTTAAAAGCAGGATCGTGTGTCTGATGCTAAGCTGATGAACAGTTTCCGCCTCTAAAAGAAGCTTCAGATGCAGGCAGAGTAAATGGGCTGATGAATGCATGCCATGTGTTGAAAAGAACTGGAATGCAGTAAGTAGGCTGCAGCCGGGACCTGCTTCAGCCTGGGACCCTGAGTGATGGGGACAGGAGTGTGGGACATGGTTAGTTCTGCCCATTTCCTCCCCTCTTCCCACCCTACCTGCCTCTGCCTCCTCCATGGTTGAGAGAAAGTGAGGGTGGTGAGAGAGGTGAGTTTAGGGAATAGGAAAAGATTTCTCACTTTATTCTTTATAAGTTATCAATAAAGGTGCCCGCTGAGCATAGAAGTTCACTGTGATCGACTCCCATGAAGCACTTTGCCTCTTTCTTGGGGATTCCATACTGAGGAACCTTATGTAACATGGGATCTGCAAGCACAACTGAAACAAATTTGGGTATAGGTAGCTCATTTAGGAAGTGTCCTTAGGGCTCAGAATTGAGGAGAGAAAGGATGAAAAGCCATAAAAAAGCATGCGGTTTAAGATTCTGCTGAAACTAATGAAGGGGTCCTATTCTACCAGGATCTGAAGGTTTGCAGGAAAGAAGCCTCCCAGAATTGTCTGCCTGACCAGTAGCCTGAAAGACAGTAGGCTGGGTGCTTCACTGCTCCCTTCCTGTCATTTAGGATTGCTCCAGGGTGCACATTAACTTTACCTCAAATGCTCTCAGCTGCTTCTGAGGGGGTGAGTATGTCTGAGGGCCAGAACCGTACCTTCCCTTAGTGTTGACACAGTTAGCAGGAGTCTGCCTTACACAGAAAGGTCCTCTGCAACTGCAGTAACAATCACTGGGCAGTCAAGCATCTGCTTCCATGAGTGAGTGTTCTGAGCCTTGAGGTCACCCACTGGTCATAGACAGACTCTCTGGATGGGACTCACCTGCAGTAGGCTGAATAATGGCCTCAAAAGGCATCCACATCTAATCCTGAAACTGTGAATGTTCATGTCAGAAAGGGACTCTGCAGAAGTGATTAAGATATCTTGAGAGGGAGATTATCCTGGATTATCCAGGCAGGCTTTAAATGCAATCACAGGTGTCCTTATAAAGAAGAGGCAGAGGGTTGTATAGACAGAGGAGGTACCTCCTGGATTATCAGGCAGGCTTTCAAATGTAATCTAAGTGTCTTAGAAAAGGGAGGCAGAGATTTGACACAGACAGAAGGAGGAGGCAATGTCACTAGGCAGAGAGAGGAGTGATGTCCACAGGGAATGCCCGCTCTTACTTAGAAGCCCAAAGAGGCAAGGGATGGAGTTCCGCCTTAGAGCCTCTGAGAGAGTGTCTCAGCTTCTGGCCTCCAGAACTATGAGGGAATAAATGCCTGTGGTTTAAGCCTAAATTTCATTAATTGTAAATTTTTTTATAGCAGCATTAGGAAATGAATACACCATGCACTGGCTCCTCTATGTTCTTTCCTGGTTCCTGAGGAACACACACTCTTGCTGCTGCTGGTGGTGGAGAACAGTTAGCTCCCAGAACTGAAAGCCCACGAGTTTCTCCTGCTGTCTTGCTTCTAGTCTCACTCTCCTCAGGCAAGTAGCCTCCCAGCCTCTCCGCATTTGGCATGTGTCACACCCCAGTCCCTCTGTCCCCACACTCTGGGACCACCAGAGGAGGCTCTCCAGTGGCTCTCTTATAGCCCCTCTCCTTTGGCCAAAAATTGGGGGCAGGCTCTCCCCATCCCCAGTGGGAAATGCAACAGCCTTGACTCAAAACCTCAGGAAAGTCTGGCCTCTGCCCTCCTACTTCCTTCAAATGCCCTGCCCCTCACCCCCATCCATTTGCAGTTCCAACTCAGAACATCAGCACCTCATTTTATAAGTGGGGCCACTTGGCTCTCTTTCGGTAACTCTCAGCCTTTGGGAACGGACAGCTTTG".getBytes();
        String mm = "C+m,139,16,5,60,507,349,11,24,0,10,5,18,2,8,9,7,2,30,1,2,3,10,2,0,7,59,4,96,14,15,8,8,55,56,88,100,49,5,16,46,4,31,53,101,8,52,24,104,5;C+h;";
        byte[] ml = null; // not testing for this
        final boolean isNegativeStrand = true;

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, sequence, isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        for (Integer pos : likelihoods.keySet()) {
            assertEquals('G', sequence[pos]);
        }
    }


    @Test
    public void testHiFINegativeAlignment() {
        String mm = "C+m,36,27,1,0,52,20,5,2,28,10,5,7,15,12,3,2,10,11,11,7,13,24,3,6,12,1,4,0,12,12,6,63,0,16,46,54,5,1,1,31,4,2,10,0,1,7,9,5,5,3,5,1,8,72,47,2,2,0,16,1,2,0,8,12,30,3,7,5,3,37,24,38,34,38,1,3,6,3,0,14,22,7,11,5,54,61,43,1,6,29,28,19,29,15,3,2,2,1,1,0,0,0,1,1,1,0,4,0,1,3,6,3,1,2,2,1,5,1,9,3,1,8,1,0,2,0,3,2,6,4,2,0,1,0,1,4,9,1,4,3,5,0,2,7,6,6,7,29,0,1,5,21,12,9,0,7,1,1,0,0,4,4,2,2,2,3,6,16,8,2,7,0,17,5,6,5,3,1,6,1,14,7,17,7,0,14,4,5,2,1,1,4,4,0,2,0,3,5,1,8,0,5,17,1,4,3,5,1,2,1,2,9,3,1,8,0,3,4,1,5,0,7,1,86,5,6,11,25,2,9,4,9,3,0,8,0,18,8,0,2,7,0,1,6,0,3,6,0,7,2,4,13,2,16,3,7,3,1,0,1,1,23,6,48,20,1,0,2,11,6,3,5,1,0,1,5,4,0,3,10,4,3,2,9,6,26,19,6,19,40,11,33,1,2,10,7,53,12,41,71,6,1,8,7,5,0,14,0,3,6,2,7,27,26,33,4,3,0,1,10,4,3,1,4,2,0,1,1,1,0,1,10,1,2,0,3,11,0,2,1,4,1,2,17,1,1,3,0,0,6,2,1,0,3,5,6,0,14,7,2,42,7,1,15,10,12,1,1,3,5,5,0,17,1,5,9,1,11,1,4,7,1,0,0,3,0,5,3,4,2,1,4,1,1,7,4,0,0,0,0,0,5,0,0,1,6,0,0,10,2,7,8,0,7,1,2,1,1,8,1,5,2,1,4,2,0,9,2,3,3,1,3,0,4,7,2,0,2,9,2,5,1,1,1,0,3,0,0,1,3,17,6,8,1,1,6,8,0,5,5,1,0,6,5,1,2,0,10,3,5,2,2,1,0,2,3,1,16,5,3,16,3,0,16,0,0,1,2,1,3,0,0,1,2,0,0,1,3,0,1,1,2,0,4,2,4,11,0,0,3,0,0,0,9,2,1,2,0,2,3,2,6,3,8,4,7,1,1,0,0,2,6,0,4,0,1,0,7,2,3,2,8,2,0,2,4,0,7,2,4,0,0,1,2,0,0,4,11,1,0,0,1,0,1,0,0,0,0,0,1,0,0,7,1,0,0,0,0,1,1,0,0,1,1,3,3,0,3,0,5,1,2,1,3,3,2,2,1,0,1,0,4,2,1,1,2,3,6,1,0,2,1,2,0,0,0,0,7,0,0,1,3,2,1,7,1,5,10,0,0,0,4,0,2,0,1,0,0,1,2,1,3,7,0,5,3,0,1,6,0,2,5,1,0,0,1,8,2,2,6,0,1,0,1,1,1,0,0,0,6,0,0,7,3,12,6,0,0,1,2,4,6,3,0,0,0,1,4,2,7,1,0,6,4,9,3,13,0,15,7,11,2,0,0,1,4,7,18,1,8,3,0,2,2,13,3,2,16,1,11,6,8,8,7,16,2,4,1,3,4,5,1,1,11,5,0,4,9,6,10,3,3,5,8,5,7,10,13,3,12,21,0,13,13,10,3,51,22,0,8,23,3,32,2,27,1,26,0,1,9,12,7,6,10,4,6,15,18,1,36,0,11,0,1,7,4,12,7,10,1,10,12,5,2,6,23,1,36,19,9,25,7,5,0,17,2,11,9,17,14,16,8,12,18,7,22,7,4,46,7,0,2,0,1,6,1,4,5,10,3,4,0,1,3,3,8,4,15,6,3,11,14,0,2,4,3,0,32,3,19,16,1,9,8,11,38,14,39,20,0,4,18,20,22,5,15,0,15,28,9,5,1,2,4,1,10,9,3,6,26,11,3,3,3,0;";
        byte[] ml = new byte[]{64, -79, -1, -7, -4, -33, -5, -1, -84, 106, -4, -64, -5, 21, 18, -11, -31, -6, -1, 27, -2, 41, -59, -11, -37, -85, -1, -1, 114, -125, 56, -110, -8, -63, -37, -33, -95, 83, 112, 90, -20, -17, -4, -42, -78, -31, -7, -29, 73, -40, -3, -1, 102, -34, -79, 57, -3, -6, -3, -2, -4, -1, -102, -107, 95, 2, -17, -18, 106, -4, 20, -66, -95, 15, -8, 94, -90, 37, 11, -50, -24, -22, -10, -2, -13, 2, 99, 5, 122, -18, -31, -119, 2, 9, 4, 5, -38, 53, 11, 47, 2, 44, 2, 5, 0, 0, 61, 2, 73, 3, 38, 46, 10, 92, 42, 4, 21, 6, 19, 30, 100, 104, 25, 45, 2, 17, 43, 2, -94, 3, 6, 58, 5, 6, 10, 126, 120, 6, -100, 40, 2, 118, 28, 40, 12, 62, -46, 82, -49, -114, 18, -45, 103, -16, -14, -73, 7, 6, -4, -1, -45, -10, -22, 87, -60, -22, 67, 106, -37, -100, -2, -4, -60, -115, -82, -21, 84, -7, -12, -19, 38, -22, 81, -3, -1, -16, -6, -43, -56, -2, -32, 37, 32, 8, 90, 32, 46, -30, -21, -13, -94, -64, -3, 49, -4, -7, 95, -47, -69, -89, -5, -70, -82, -9, -82, -60, 22, -12, -38, -103, -81, -43, -78, -44, -60, -14, 55, -106, -126, -29, -7, 28, 25, 47, -2, -39, -9, -1, -1, -95, 36, -42, -39, -63, -11, -31, -20, 32, -20, 21, 18, -7, -75, 56, 120, -120, -5, 102, -12, 102, 111, -14, 4, -32, 10, 15, 48, -40, 121, 94, 104, -14, -67, -24, -24, -90, -7, -9, -17, -94, -9, -127, -93, -1, -42, 32, 51, -30, -33, -100, -90, 8, 13, 92, -59, -8, -10, -3, -2, 0, 113, -34, 63, 52, 62, 9, 11, -44, 32, -85, -18, -48, 14, -98, -1, -62, -22, -3, 118, 67, -18, -6, -1, -3, -5, 50, 36, -63, 121, -13, -2, -2, -1, -3, -5, -13, 92, 18, -55, -18, -6, -1, -48, -34, -9, -75, -59, -42, -42, -4, -69, 103, -7, -9, 113, -1, -4, -9, -51, -9, 112, 111, 103, 3, -94, -88, -41, -17, -2, 54, 58, 11, 0, 0, 12, 15, -50, 77, 8, 96, -95, 6, 1, 24, 21, 8, 0, 18, 1, 30, 16, 70, 64, 109, 29, 10, 0, 0, -73, 11, 20, 29, 29, 71, 70, 15, 41, 16, 34, 21, -115, 10, 12, 0, 53, 0, 0, 1, 2, 0, 88, 3, 3, 0, 0, 0, -97, 122, 15, 1, 4, 1, 10, 9, -105, 65, -34, 40, 0, 7, 1, 11, 45, 68, -126, 4, 33, 4, 30, 59, 19, -50, 2, 0, 0, 120, 8, 69, 45, 10, 7, 0, 28, 13, 1, 0, 0, 82, 7, 31, 0, 0, 1, -128, 38, 14, 105, 37, 35, 6, 29, 5, 5, 1, 1, 1, 65, 2, 4, 8, 24, 19, 12, -120, 125, 0, 0, 6, 9, 3, 0, 0, 0, 26, 1, 2, 0, 2, 2, 0, 0, 0, 0, -109, 123, 22, 0, 0, 56, 103, 3, 32, 12, 2, 11, 12, 8, -104, 7, 7, 6, 2, 9, 92, 9, 0, 0, 9, 27, 19, 87, 54, 2, 9, 121, 97, 19, -109, 118, 57, 7, 3, 0, -36, -89, 2, 20, 44, 0, 0, 0, 0, 22, 4, 8, 2, 18, 0, 0, 0, 0, 0, 0, 0, 1, 0, 6, 0, 1, 52, 0, 0, 35, 106, 35, 0, 0, 12, 7, 50, 17, 8, 2, 36, 5, 0, 0, 115, 23, 10, 81, -110, 76, -122, 11, 30, 44, 3, 0, 5, -58, -34, 55, 0, 0, 0, 0, 1, 118, -125, -89, 126, -24, 75, 52, 5, 8, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2, 69, 51, 4, -51, 18, 0, 2, 0, 2, -126, 54, -115, 16, 0, 0, 7, 52, 7, 0, 72, 27, -26, 2, 0, 1, 40, -31, 53, 112, 101, 0, 0, 18, 9, 88, 4, 2, 11, 16, 0, 18, -70, 106, 2, 25, -91, -38, -63, 0, 0, -9, -3, -1, 5, -14, 67, 124, -8, -60, -18, -2, 118, -34, -1, -1, 82, 95, 94, 17, 36, -11, 59, 50, 100, -57, -9, -15, 69, 27, 38, -5, 45, -30, -5, -125, -3, -16, 118, 45, -125, -10, -48, -80, 62, -2, -5, -16, -5, -8, 36, -22, -102, -27, -31, -5, -106, -3, -3, -5, -1, -6, 98, 126, -103, -61, 96, -49, 24, -6, -6, -5, -14, -15, -6, -1, -1, -7, -105, 93, -33, -86, -95, -121, -47, 30, -1, 77, 33, -93, -68, 81, -115, 116, -84, -1, -124, -117, -119, 5, -52, -5, -14, -18, -28, -53, -29, -49, 92, 14, -2, 13, -127, 75, 79, -24, 42, -105, -128, 1, 108, -1, -61, -19, 44, 8, -69, 1, -100, -21, -28, -20, -10, -45, -44, -5, -10, -28, -3, -10, -1, -95, -4, -4, 13, -3, -89, 41, -35, -18, 104, -12, 18, -19, -22, -120, -7, -10, -23, -27, -1, -49, -89, -16, 117, -4, -10, -3, -34, 7, -64, -7, -124, -29, -52, 71, -63, 22, 6, 71, -51, -10, -29, 94, 3, -124, -102, -83, -93, -73, 71, 4, -5, -101, -71, -111, 121};
        String seq = "ATGCAGGCCAACCCTGTGGGCACGCCGGGCTCCAGCGAAGAGGACGAAGACACCACATGGACCCCCACCCGGCTGGCCTCACCCCTGCTGGCAGCTGAGAAAAGGCCACGAAGGGCCAAGTAGCCAGGGCCCCTGTGAAGCCCAAGGAGAAGAAGAAAGGCCCCTGCCCACCCCAGATGAAGAAGAAGTGTGTCAACGGCTTCATCATGTTCTGCAGGATGAACCGGAAGCAGTACATCCGGTGGGTAGGGGGTCGTCTTTGGCCCTGGGGAAGGCCCTGGCTTCGTTCTCTGCCCGCAGCCTGGCCATGCCGCTCCTCACAACTGTCTGACACTTTTTCCCCCTCCGAGACCCCACAAACCCTTCGTGGACCACAGTTCTTCCCCAGAACTGAACCCGGCCCCATTGTATTGGACAGGGTCCATCCATTGTCAGCGCTGAAACCCAATTCAGGCTTTAGACTTCAGAAAAAAAGGAGTTTTTTTTGTTTTGTTTTGTTTTTTGAGACAGAGTCTCACTCTATTGCCAGGCTGTAGTGCAGTGGCACAATCACAGCTCACTGCAATGTCTGCCTCCCGGGCTCAAATCACTGCAACCTCTGCCTCTCAGATTCTCCTGCCTCAGCCTCCTGAGTAGCTGGGATAACAGGCGCGCCATCAGCTAATTTTTGTATTTTTAGTAGAGACAGGGTTTCACCATGTTGGCCAGGCTGGTCTCGAATGCCTCACCTCAAGTGATCTGCCTGCCTCGGCCTCTCAAATTGCTGGGATTACAGGTGTGAGCCATCAGGCCCAGCAAGAAAAGAAAATGAATTGGCTTATATAACCTAGAAAACAAGGGCGGGTATAGACTGCCTTGAGACATAGCTGGTTTTAGGGCTCACACAGGCTGTCTGGAATCTTTGTCCTTTTCACCCAGTTCCTCTGTCTTCCGGGCTGGCTTCATTCTCTGGCAGGTGGGGTCACCCCAGGAGGCACAGGCTCGGGTCCTGTCCCCCACCCAAGCGACCCCGGGAGGAGGAGAGGTCTCTAAGGTTTTAAGCAAAAGTCCTTGGGACTGACTTTGGTAGATCCGGTGCCAACCCTGAACCAGTCACTGCCAGGCCTGGAGCCAGGGGATGGGGAGGCCCCCCCCAAGCCACAGTGGAAAACAGCTAGTTCCTTTACAGGAAGATCCAGATGTGGGGGCCAGGAGCGGTGCCCACAGGTCCCCTGCACTCTCCTGGGACACCAGGTGCCCCCTTGTGAACAGCCGCTGAGTCCAGCTCCCTCTGCCTCCTCAGCTCCAGAGGGTGGTCTTGGAGGCCTTGTAGACCCACAAGGCTAGGAAGGGTCTGAGTTCTCAGTCAGGGTGTCCTGCAGACCTCTCCTTACCTCCTCCTCCTTCCTGTCCCAGATCCTGCCCTGGAACCGCTTCCACAGCTGCCACCAAGGAGCTGGCCCAGCTGTGGCGGGTGATGACCCAGCAGGAGCGGAGGCCATACTGGTGAGAGGCTCCGGCCCCGAGGTGTGGGTGGGGGGTGGAAGAGCCTCCCTACCCCGGAACCCTGCTCTGTCTCTGGAGCCTTGGGTGTCATGGCATGGGAGACTGGGCCGAGGAATTTGCGTGTTGCCCCTTCTTATGAAACCCTTGGGGGTGACACAGGTGTAGGGGTACCCCACCACAGCTAAGCTGGGGAACAGCTGAGCCCAGACATGGACCCCTGCCTCAATTCTTTTGCTGTCTCTGCCACACGACTTTCTCGGGGCGTGGTCTGATTTTCCTGTCGCTTGGCCACCTCGACTCCCCCGAATCCTGTCCCCTCCACCTTCAAGGAGCTGCCACCTGCCCTGGCTCCTGACTTTGAACACATCCCAGAGCCCTTGGCCGAGTCCCATCTTACTGACTGGGACTTGGTGGACCCAAGGCTCGCCCAATAGAGGCTAAAACGGTGGCCCAGGCCCCTGCGGGGACACAGGATTGGGAGTGGTACAGGGATGCCCGGGAGGCGAGTGAGGAAGCCAGGCCATCCTGCCACCGTCTCCCCAGCACCAAGGCTCGCAGGTTCAGCCGCCAGCACAACCGCATCGTGAAGCAGGACGGCTCCAGCAGCGAGGCTGAGGGACTGGGAGACGCCAAAGCCCTTCTACCAGCTGCTGGCCGAGAAGGCCTTGCCGCTGCCCCCGCACCTCCAGTGAGAGGGCGGCCCCTCGCCTCGCTCCCCTCACCCCTCATGGCAACCGCGGCTCTTGCTGGAAGCCAGGACCCATCGATGAACTTGTCCCTCCTGGGCCTCCAGCCCCTGAGAATGCAGGTCCCATGGGACTGGGGAGGGGGGCACTGATTAGCCCCAACTGCTGGGCACATTTGCCTGGAGCACCAGAGTCACCCACAGCTGCCTTGATTCTCCCCCAGCTTAGGAAGGAAACCCAAAATGAAATGCGGGGTGTCGGAAGGTGAAGTTTACCCACCCCTCTCCCCTTCCCTTCCCCACCCCAGAGCCACTCGGTTGCAACCCTGTTCATGCTCACCTCACCCTACTCCTCCCTCTCCTGTTCTATTTTTAGACTATTTATTGTTTTAAATAAAATAAAGCAAGTGGAACCTTTGTTACCAGCAAGAGAGACAAGGGCAGGCCCTCCTGCGTCCCCCTGCTGCCCCACTGGGGTGCTCCGTCCACCCTCATCTGGAGGCCTGTGGCTTGCAAGAGGCCTCAGTGCCACCCCCTCCCCAGCCACTCTGTGCATCTGAATGCAAACCCCTCTTCCCCTCGGGAGGCAGCAATGGGAAGGCCAGGTCTCGAGGTGGGCTCTGTGTTCAGCGCTGCTTGGACAGACACTGGAAGGCACCCCAGACCCCTGGTGGTGGTGTAATCCGATAACTTTATTTTAATTGAAAATGGAGGCATAACATGTCCTAGAAAAATAAAGATTTAGGATACAAAGGAGACACAGCGGCAGGGCTGCCCCCAGGGATGAGGGAATCTTTGGTCTGGGCCGGATAATGAGGACAAGGGCTTGAGTCGAGGGGAGCTGGTGAAGGAAGACCCCCGTCTCCCCACAGCTGCCCCACCCCCCGCCCCTGGCAGGAAATGCCACACTGGGGAGGGTCTCCAGTCTCAGGGGCGCGGGGCTGCCTGTCCTCCACCTTCGGATTCCAGGCAGTTGTGACAACACCAGCTATCGGCAGAGCTATTAATAGTGTTTCAGGGAGTGACAGGGAGAGGGCTCTGGGGGCTGGAAGTCCGGCCCTGGGGCAGGGTGTTCCGCTTACAGCTAGTACCAAGTGGAGGGGGCAGGGCCCCTGAGTCAGGACCCTGAGTCCCCCACGTATATGGCAGGGCACAGCATGGGGGAGGGCTGTAACAGAGAGGCCTCCCATCCAAAGGGGGGATGGAGACCTGGACAGGAGGTGGCCGGCGATACAACCCCCAGCACCACCAGGGCTTGGAGAGGCCACCCAGGCAGAAGGATGTGGTGACTGGGGTCTTCAGCAACCGCATTTCTGGGGCTCCCCCTCCCATTCCTGTCCCTGCGTCTTCAGCACCAATGTCGGGAGAGGCCACGGGGCCACACTGGGTCACAGTTCCAAGGGCTCCTCCACAGGCACCGACTGGAGCTGGGTCAGAACCTTGGCCTCAGCTTCCAACCCCTCGTCAACCTCACCCCCTGCGGTGGCCCCCAGGAGCAGCCCCTCAGGGTCGGGGTCTGGCAGCCTCAGCACGGTGTGGGGGGGCCTGTGTCCCCAGCCCCTTTTCCGCTTCCAGCAGCCCCTCTGTTCCTGCGCTTAGTTCCAGCCCTGCTGACCAGACAGGCACGGCCGCGGGTGACAACATCAGCCCCTCTGGTGGGGGCGCCGGGAAGTTGGGCAGGAGGCCAGGGGAGTCAGGGGAGAAGGACAGGCTGGTGCTGGAGGTGGTGGCAGCGGCGGGGGGTGGCTGCTGTGCAGAAAGGGTGCCTAGGGCGTGAGCCTCTGGGAGAGCAGGGCTGGGGGCCACCGGGAGGCCTCCCTCAGGCACGGAGATGGCCGTCTCTGGCTTCAGTGGCAGGGCCAGGCCGGGGGCTGGCGGCAGGACCTGGGAGACGAGCATGCTGGTGGGGAAGGTGGCGGGTGAGGATGATCTTGCCCTGCTGCACGGCCACACCCGTCACGATGGGGCTGCCAGACACAGGGTTGGCCAGGAGGAAGTTTCCTGTGGGGAGAGAGGGACCGAGCGCAGGTGAGAGCCAGGAGAGCAGGCCAAGCCAGAGAGAAGTGGAGACTGTGCAGCTGGAGGGCGGGCGGAGGGAGCCTTGTGGTGGGCCTTGGGGCACTGGGACTTGCCAGGGATGAGATGCCCTCCAGGAGCCCAGGGACAGCCCCTCCCTCTCCGAGATGACTGCACCCCTCCCCGGCTCTCACCTAGGCCTGAGGGAGGGAGATGGGGGTACCTGGGGCAGTGGCCGAAGGCAGCTGCAGGGCAGTCACGCCCACCCCGGAGTTGATGAGGTGCACATTGGCAGGGCCTGCTGCAGCTGCCACCTTCACAGGGCTGCCTGGCCCGGGTGGGTGTGGTCTGGCCAACAGCTGCAGGGGCCCCACAGCCTGGGGCAGGGGTCACCACCTGTGAGGTGGGTACTACCTGGGGCAGGTTCAATAGTGGGGAGGTGGGGCTCAGGCCCGTGGGATACCCCGGGGGTGGGGAGAGCGGGTACCACTTGTGGGGCAGCCACAGCAGGCACTGGCCCCGGGGGGCAGAGGAAAGGTGGCAGCCGTCGGGGGGCCAGGCACCACTTGGGCCAGGGGCCCCAGGACCTCCTCTCCAAGGGCTGGTCCCGGAGCAGCCACCTGGGCCCCTTTGGTCTCAGGGGCCTCCGACTGAGCCTCCTCCAGCCGCACCTCCCCTGTCTGAGGGTCCAGGACCAGAGAGGTCTTGGTCTCGCTGGCCCCCTGAGGGCTGGGCTGCGGTGGAGGGGCACCCCCGCCCCCAGTGAGCAGCAGCGGGCCCAGGCTGGAGGCCTCGCCCAGGGCCAGGCCGTTGATGATGACGGGGCCCCCGTTGAGGAGCACTGCTGGGGAGCCGCTGGCTGCCAGGAAGCTCCCGTTCACCAGGATGGAGGAGGAAGCCGGGCAAGGCGCGGGAGGGCCGGTCCCTGCCAGGAATATGGAGCCCTGGGCAGCGGCCTCGGCGGACACTGGGGCCGCCCCTCTCTCCAGGTCCTCAGGACTTCGGCTGGACTCGTCCTCAGTCGTGGGATTCCCATCAGACTCGCTGGCCGGAGAAATGGCTGTCAGCTGGGGTCCCTTAGCCTGTGGGGCCTCCCGCATAGCCAGCCAGCTGCTCCCCCACCTTTCCCTGGCCCAAGTTTCGGTGGATCATTCCAGGGGTTATGACGCCTCTCTGAGACCACTCCAAACTCCAGGGCCTTTTGCAAGCCCTGCGTGGACACCACTGCAGAGAAGCGCTGGGGAGGGAAGAGGCCCTTGCTCTCCCCCACCTCGGCCTCGGGTCCCCAGAGGCTGAGCTCCCAGGTTGCCCTAAAAGTTCCTTGGTTGGGCGGAAAGCGGGGCGAGTGGCCTGGAGAGAGGGAGCAGCAGCCGCCAGCCCAGTTCCCGGTGCCCTCCCCGCGGGGCGTCAGAGAAGGGAGAGGCCGGCGCTCAGGGTGGGAGGAGAAGGGTTTGGGTTACAGGGAAACCGGAGCTGGGAAAGGTTCACGTTTCACAACAAAGGCAGAAGACGGACCACGCCGTCCGGGCCCGGAGGGAGTGTGGGGGCGGGTGGGTAAGAGTAACGGTCAGTGAAGAAAGGGGGCTGGGAGGCAGCCCCTACGCGGAGTGGAGTGGCCACAGGCCCTGTCCTTTTTCCTCAGTCCCTCTAGTGCCCCCCGCAGTGTGATTCCCCAACACCGATGGGATTTGGGAAGGAGCTCGGAATGGAGCCGCTGGAAGAGGAGACGCGTGCGGGGAGAGGGCCCGGGCGGGTGCCTGTCCCGTCCACACTTAGTCCCCGCGCCCCGCGGGGCGCCTGAGATTGTGATCTGGTCCCGGGAGATGTCCGAGGACCTCGGCGCGCCGGCCCCAGCAGTGGGCACGGGGGAAGAGGGCTGGTGGACGGGATGTCCCCGGGAGAGCTGGACTTGCGCCGCCCGAGGCCCCTCACCTCTTGCAGGGCGCGCCGCCTCCGGCCCCCGGTCCGGTCGCGCTGTCGCCGGTTCTTGAACCAGTTGCTGACCTGCGTGAGCGACAGGCCGGTGAGTGTGGCCAGGCGGCGCTTCTCGTCCGGCGTGGGGTAGCGGTTGCCGCGGTAGCAGGCCTTGAGCGCTGCGCGGGAGCGCTCCTTGAAGCAGTAGACTGTCTCCTCGCCGTCCCAGATGGTCTTGGGCAGCGGGAACTTCTTGCGCAGTCGATACTTGTCCACTGCGCCAAGCGCGCGGCCGCGGGCCGCTCGGCCTCATGGTAGCGCGCGCGCAGGTAGAGGTCCTGCAGGAAGGCGTGGTGGGCGGCGGGGAAGGGGCGGCTCTCGAGTAGCCGGTAGAGCTCGGCGTACTCGCCCCGCTGGAAGGCCACCAGGGCCCGCGCGCGCAACACCGGGTCGCTGCCACGTAGGCGCTCGGCCGGGGGCAGTGCGCCCAGGAAGCGGCTCAAGCGGCCGGCGTGGCCCGCCTGGAGCAGCGCCTCGCAGACGCACGCCACCTGCTCGGGCGAGAAGCGGAGGCCCGTGGGCGGTTCGGAAGCGGCCTCGGGGGGCGACCCGGGGACGCCCGGGGATCCCGGGCCCTCAGCTCCCGCAGCCGCTGCGCCCGCCCCGGCCCCGGCCGCCGCCGCCGCCTCACCCTCGGCCGCCTGCAAAGTCTGCAAGAGCTGGCGCGCTTCCTCCTCCTCCTCTTCGGTCGCCGCCGCCGCCGCCACCGCCTCCCCCCCAGCCGCCGGCCCCGCGCTCGGCTCCGCAGGCAAGGTAGCCATGTTTTGCAACTTTGGGAAGTTCCTCCCTCCCTCTCTTCCTCCCTCGGGCTTTCCCCAGCCTCCTCCCCCACCTGTCCCCCCTTTTCGCCCCCACTCCCCGCTCTTCTCGATCTTCTTTCTGGCCGACCCTGCGCCCCACGCCGGGAAGGCGAGATCCAGCTCTCCACTCGGGTCTCTGTCCCCTTGTGTGTGTCCGTCCCCCCTCCCGTCTGTCTGTGATTCTCCCTTTGTTTTCCCTCCGCCTCTGGCCGCGCTTTCTGCCTCCCCCCAGCGTGTGCTTCTGGCTCAGGGCCTCAGTTTCCCCATCGGGACAACGCAGAAGGTAACGGGCCGTCCAGGAGGACTAAGGGCGCGAAGCCTCCGCCCCGAGACTGAGCTTCTGCACGCCTCCGTCTCCAGGGTCCTCTGCAGGCCCCCACATTCCCCATCTCGGCCTGCGCTCCGCCCCTCGGAATTCCCGGCTCCGCAGGGGGGGCGGGTCTGGCCGGGAGGAGGGGCGGGGAACGGGCTAGAAAGTTTGCAGCAACTTTTCTCGAGCTTGCGTCCCAGGAGCGGATGCGCGTGGCGTGCGCAGGCGCAGTGGAAGGAGGATGGCCGCGCGCGCTGCCAGCCCAGCCCCCTCTTCTCGACGCTCGGTGGCACAGCTGGGCCACAGCTGGGCGGGGGCGGTGCCTCCGGGTGGCCCGCTCGCCCTCCTATTGGCCGGACGCCAAAGCCCGCCCCGTGGCTTTTCCTCCCCCAACCCTGATTCGGCCGCTTCGCATCCCGCTAGCTCCTCCCAGACCTTCGGCCGCCTCCACACGCCTCCGGATTGGCCCGCTGCGGAGCCTCCGGCCCACAACGCAAACCGCGGACACTGTGGAGTCCAGAGCTTTGGGCAGATGGAGGGCCTTTTATTCGCGAGGGTCGGGGGTGGGGGTCCTAGGTGGGGACAGACAATAAATACCGAGGAATGTCGGGGTCTCAGTGCATCCAAAACGTGGATTGGGGTTGTTGGGGGTCCTGTAGCCTGTCAGCGAGTCGGAGGACGAGGTCAATAAATATCCAAACCGCCGAAGCGGGCGGAGCCGGCTGGGGCTCCGAGAGCAGCGCAAGTGAGGAGGGGGGCGCGGGATCCCCGAAAAAGCGGGTTTGGCAAAAGCAAATTTCCCGAGTAAGCAGGCAGAGATCGCGCCAGACGCTCCCCAGAGCAGGGCGTCATGCACAAGAAAGCTTTGCACTTTGCGAACCAACGATAGGTGGGGGTGCGTGGAGGATGGAACACGGACGGCCCGGCTTGCTGCCTTCCCAGGCCTGCAGTTTGCCCATCCACGTCAGGGCCTCAGCCTGGCCGAAAGAAAGAAATGGTCTGTGATCCCCCCAGCAGCAGCAGCAGCAGCAGCAGCAGCAGCAGCATTCCCGGCTACAAGGACCCTTCGAGCCCCGTTCGCCGGGGCCGCGGACCCGGCCCCCTCCCTCCCCGGCCGCTAGGGGGCGGGCCCGGATCACAGGACTGGAGCTGGGCGGAGACCCACGCTCGGAGCGGTTGTGAACTGGCAGGCGGTGGGCGCGGCTTCTGTGCCGTGCCCCGGGCACTCAGTCTTCCAACGGGGCCCCGGAGTCGAAGACAGTTCTAGGGTTCAGGGAGCGCGGGCGGCTCCTGGGCGGCGCCAGACTGCGGTGAGTTGGCCGGCGTGGGCCACCAACCCAATGCAGCCCAGGGCGGCGGCACGAGACAGAACAACGGCGAACAGGAGCAGGGAAAGCGCCTCCGATAGGCCAGGCCTAGGGACCTGCGGGGAGAGGGCGAGGTCAACACCCGGCATGGGCCTCTGATTGGCTCCTGGGACTCGCCCCGCCTACGCCCATAGGTGGGGCCCGCACTCTTCCCTGCGCCCCGCCCCCGCCCCAACAGCCTACAGCTGTTGTTAGTCCACTCGCACGCCTCGAATCCCGTCCGAACTCGTCATTGGCTGCTTCCTAGCGGCCTGTGTTGATTGGCTGCCCGAAGATCCGCCCTCCTGCCGTGGGCCCAGCCCCGCAAATGCGCAGCTAAGCGGGTGGCACGGGGCGGGTGGAGCGCGGGGCGCGACGGCGGAGGGGGGCGTGGGCAGCCGGACGTACCCTGGCAGGGAGCAGCAGGTGGCGGCGGTGCATGGGGCCTGGCCCCACCAGCGGGCACTGGCCCACAGCCACGGCCGGGGGGCCATCTAGCTGGAGAGAGAAGGGACAGGTGACCCGATCGGAGCCCAGCCCAGCCCTCAGCGGTGGGGCGAGAGACAGCGAGCGGAATCGAGGTTGGGGAGGTTATCTAGGGAGATCCCGGAGGGAATCTGGTGAGGCCTGAACGGAGGGAGATCTGGGGCTGAATAAAGGGCTTCTGCCCTCTAAAGTCGCAAAGACGTAGGGTGAGCCCTATATCTGGACGGGGAGACCAGGAGCCAGGGAGGGGATCTGCAGAATGGGCAGCAGGTCTGAGGCAGGGGAAAGAGAGGGGTCTTACATGGGAAGGTGGATCCGTGGCCCGGGGACTGGGGACCCCCGTGACAGCTGGAAGGAGAAGAAAGAGGCATAGGGCGCGTGGAGGGGCGAAGGAGGGCGGTGGCGCGGCGTGCCCCAGCGTGGGTCCCTTCCTCCTCCAGGTGTCTATACACGCCCCGCGGAGCAGACGGCCCACCTCCTCCCGGTCCTCCGGGGAAGGGGACACATGAGGGACTCACCTGTGGCTCCCTCTGCCTGCAGCAACTCCATCCGCTCCTGCAACTGCCGGACGTGTGCCTCTAGGTCCCGGTTCCGAGCCTCTGCCTCGCGTAGTTGACTGTGGGGAGGTAAGGACGGTGAGTCCGTCCGGGCCGGACGAGAGGGGCTGCCAAGGGTTGCCACCGGCCCGCATCCCGGCCCCGGCCCCGGCCCCGATCCCGACCTGGCGAAGTTCTGGTTGTCCGTGCGGATGGCCTCCATCTCCCGGCTCAGGCTCTGCCGGGTGAGCACCTCCTCCTCCAGGGCTTCCTGGAGCTCCCGCAGCGTCACCTCGGCCTCAGCCTCTGCCGCAGGGACAGCCGCTGGAACTGCCACTTCAGCCTGTGTATGGGGACCAGGCTTAAGGCTGCCTGTGGCTCCTGGAAGACTCAGGACTTGGGCACTGGTTCCAGGCTAGGAATCTTGTTTATCCCCTACTCCTCCGTCCCCTCAACATTTCTGGAATCCCCATAGCTCCTGCAATGATCCAAGCCCCCTCCCTTCCCTACCTCCCTCAGCCCCATCCCTGAGTCTGGTCCTCTAAATCTACACAGGGACCAGAGGGCTGGTGCTCAAACACTAACACAACCTATGTCCCTCTGCTGCTCAAAATCCCTCCAGCTCCCTAATGCCCTCACGACAAAAGGCCTTGCTGGGTTTTGTTTCCTGCTGGCCTCTCCAGCCTTCTCAGGAATGATTCAGCCAAACTACCTTTCAGGAACAAATCAGGATTCCCACCTGCCCAAGGCCTTTGCCCTGGAGGCTCTCGCCTGACCACTTGGCACCTTTCCTTCCATGGCTTGTTTCTCCTTCACCAGCGCTCAGCTCAAAGATCGACTTCTCAGATCCCCAGCAACACACACACCCTTACTTAATGCCCCACTGTAACTACAGAGACCGTCCCATGTCTGAAGTAACCTCGTCTCTCCGTGGTTTCTGTCTGCTTCTGTTCAGGAAGTCCCTAGAGGGCTCTACAGTGCACGCCACCGGGAAGCCCTCACCTTTTCTCTCCCAATTGTCCCTGACGGACCCCCTCCCCTCCACCAACTTACTGTTTCATCCTGTGGGGACACCGAGGGCTCCAGGCTGGGCGCTTGCACGTGTGGCTCAAGCAGCTGCTCGGCCTCCACTTCCATGGGTGTGGGGCCTGGGACCTCACTGTCCCTGGGGGAGAGGAGGAGGGGAGGTGGGGAGGGAGACAGAATGCTGATTATCTGGTGGAGAACCAGAACTTCTGGCCTGTGGGTAGGGGCAGCTGCTTCCAAGACCTCCTGATTTGAGGAAGGGGAGCAGCAGAGCGAAGAGAACAGAGTCCCAGGTAGGCCAATAGCCCCTCCTCTGCTTAGGAAAAGCCCTGCCCCTCTCCTGGCCTTGGGCCCCTGGGAGCTGCTCTTTCTAAACCTTCTGCAGGGACTTGGGGTCCAGACCCCCATGTTCTAGGGTCAGCTCATCTCCTCCCTTGACATGTGACCGCTGCAGACCCCTCTTCTCAGGGCCTCAGGACAGGGAGGCCGACCTGATGGGGAGACTGCTTGGCTCTGGCCACCTCCACAGATGTGGTCCTAAGACTGGGCATGGGACCCCAAGATGGCAACTTGAGGTCAGCACACCCCACTCAGCTGTGGGAAGGGGACAGCTCCTCCCTGGGGTATGAAGTGGCTGTCCCCTCCAACCCAGCAGTCCCTCCTGTGGAACACGGCTCAGGACCTAGCTATCCTGGACATGCGCCTGGGTTCTAATAGAGGATGGGTTCAATCCTGACCCACCGTTCAGGCCCCGCAGCGGTTCAGAATCAAGCTGGAGTTCATTAATGATAAGGTATAGTATGTGTAATGTTGTCCAGTAATAAAAGCTGACTAGAAAACCAACTGTTCTCTTAGACAAAGTAGCATGAGGAAGGGGAAAAAAACAAACCAAAAAAACACTGTTCTCCACTATTTAAACATGTGTCAGTACACGTGTGAGCCAGGGACCACTGCCATCTGGTGAGCTCACGAGATATCAACTTCCTTTTCCTTATCTGTATTTTCTAATTTACTTGTGATAAGCAATGCATTATTTGTGTAATTAATACTTTTGTTGAGACCAGGTCTCACTTTGTTGCCTAGGCTGGAATCTATCATGGCTCACTGAAGCCTCAAACTCCAGGCTCAAGTGATTCTCCTACTTCAGACTCCTGAGTAGCTAGGACTGCAGGCACAGACCACCATGCCTGGCACGATTTTTTCAATTTTTTGTAGAGACAGGGTCTCATTATGTTGCTCAGGCTGGTCTTGAACTCCTGGGGCTGAAGCAATCCGTCCACTTCAGCCTCCCAAAGTGCTGGATTCCAAGCGTGAGCCACCATGCCCAGCCTGTAAATTAGTACTTTTTTTTTTTTTTTTTTGAGATGAAGTTTCACTCTTGTTGCCCAGGCTGGAATGCAATGGCACGATCTCAGCTAACCACAACCTCCCCCTCCTGGGTTCTGATTCTCCTGCCTCAGCCTCCTGAGTAGCTGGGATACAGGTATATGCCACCATGCCTGGCTAATTTTGTTTTTTAGTAGAGATGAGGTTTCTCCACGTTGGTCAGGCTGGTCTCGAACTCCTGACCTCAGGTGATCCACCCACCTCATCCTCCCAAAGCACTGGAATTACAGGCGTGAGCACCACACCCGGCCTAAATAAATACTTTTTTTTTTTTTTTTTTTTGAGACGGAGTCTTGCTCTGTCGCCCAGGCTGGAGTGCAGTGGCATGATCTCGGCTCACTGCAAGCTCCGCCTCCCGGGTTCATGCCATTCTCCTGCCTCGCCTCCCAAGTAGCTGGGACCGCAAGCGCCCGCCACCATGCCCGGCTAATTTTTTTGTATTTTTAGTAGAGACGGGGTTTCACCGTGTTAGCCAGGATGGTCTCGATCTCCTGACCTCATGATCCACCTGCCTTGGCCTCCCAAAGTGCTAGGATTACAGGCGTGAGCCACCGCGGCCCGCCTAAATTAATACTTTTAAATGGCCTTATTGTTATATGGCTGATTCAAAGAATTTGCATTCTTTTACAACTGATTGTAAAAGTCTAAACTTCATATTTTCCAAGTTCTGAAGTCCTGTGGCTCTGTGTACTACAGACGTAAAAGTTAATGGTTACAAGATTCTGGGAAGCCCAGTCTGTGACTCTGAGGTTTTAAAATTCAATCTAACATTTTGAAGTACATAGATCTGAAAATTTAAGGTCCTCCAACTCTGGCCTCTTAGGAGTCTTTGGGCCCAAAACAGTAAGGTTCCAAGACTGATCCTGCAACTCCAGTGGCTGCCAAGGAGCAGGACCCAGGAAGCTCAAGCAGCCTCGTGGCCCCTGGAGGCCGTCCAGGGCAGTGCTTACCTGAGGGCCATGCAGGAGTAGGAGTAGCCCACAAAAGGCAGGTGGACCCGCTAGCGGCGCACCTTCCCGAATGTCCGACAGTGTCTCCTGCGCAAGACACACAGATGTGAGCAGCAGTCGTCAGGGCGGGCCCCTCATGCTTTTTGATCTTGGCTTACATGTTCCCCCCAAACCAGCCCCAACTCAGGGCTTATCTAAAGTGGCCCCTCCAGCATTTTTTCTTTTTTCTTTTTCTTTGATACGGAGTCTCGCTCTGTTACCCAGGCTGGAGTGCAGTGGTGCAATCTCAGCTCACTGCAACCTCCGCCTCCCAGGTTCAGGCGATTCTCCTGCCTCAGCGTCCTGAGTAGCTGAGATTACAGGCGCCCTCTACCACGCCTGGCTAATTTTTTATATTTTTGGTAGAGACGGGGTTTCGCCACGCTGCCAGGCTGGTCTCCATCTTCTGACCTCGTGATCCACCCGCCTCGGCCTCCCAAAGTGCTGAGATTACAGGCGTGAGCCACCACGCCCGGCCAGCCCAGCCTATTTTTTAATTTCAGTTTATCTTATTTTTGAGACAGAGTCTCTCTCCGTTGCCCAGGCTGGAGTGCAGTGGCACAATCACAGCTCACTGCAGCCTTGACCTCCTGGGCTCAAGCCCACGTCGGCCACCCTAGTAGCTGAGTATACAGGCATGCGCCACCACGCCCAACCAATTTTTAAAACATTTTTGTAGAGACGGGGGTCTCACCACATTGCCCAGGCTGGTCTCGAACTCCTGGGCTCAAGCAATCCACCCACTTCGGCCTGCCAAATTGCTGGGGTTACAGGCGTGAGCCACCGTGCCTGGCCCCATCATTTTTTCTTGTATCCTGTTGCTTCCCTTCAGGGCACCTGCCACACTCTCCATAATTCTCTAATTCTCTAGACTGCAACTGGATGAGGGCAGGGACCAGAGCTCATCTGGTCCGTGCTGGATCCCCAGTATTGTTCAGCAAAAGGGCACCCAGAGCCGAGCAAGCATGCCATGACAGCTTGTTACACGGTGAAGAGAGACGGCCTAAGGGAGGAGGAGTCCTCTCCTGCTTCTCATCCCTCCAAGTCTCAGCTCAGATAGCTCCCCACTCCAGAAAGCCCTCCAGGACCTTCCTTGCTGAGTCAGGAGTCCCCCACCCCCTGCACTCCATTGTCTCAGCCCTGATCACTCTGGGGCCTTACTGTCTGAAGACTGCTCTGTGTTCCCCACTGGACTGTAAGTCTAGGTCACTGCTGGGTCCTCAGTAGTAGATGGGCACAGAGCAGGTGCTCTGGGGAATGAGTGATTCAGGACCCCAGAAGGTAGGCACTGTCCTTACTCCAACTTTATGGAGGGAGCATGGGGAGGTTCCCGCAGCCGAGCAGGGGCCACAGGTACCTACCCCGCCCCCGCTCACCATGGCAGTGAGCCCGTCCTCCACCAAGTCGAAGTTGCATGTGTCGGTGGCACCTTCGAAATCCGGTGTAAAGGGGGGCACGCTGTCCCGGAGACCATCCCAGTCGAGGCCAAAGAAGAAGGGATGTGTCCGGAGTCGCCTGCTCCACCCCGGCCCAGCCGTGTCTCCGGGGGACACAGCAACCGCTGAATGAAGTCTCGAGCCTCCTCAGGGACCCCTTCGTCCACCAGCGGCAGAGAGAGGTGCTCCTGCTCAGAGGGAGAGGAGGCGATAGCCTGGGAGCGCCTACCGGGAGAGGCCAGGTCTCCCTGCGGCCGTGCTCACCTTGTAGTGGACGATCTTGCCATAGGTCCTCCGCCGTGGAATCCGCGTAGAAGGGCGTCTGCCATAGAACATTTCATAGGCGAATACACCCAGCGCCCACCAGTCACACTCGGGCCCGTAGCTGCCTGTCCCAGGCCCACCGCCCACAGCCTGCAGGATCTCGGGGGACAGGTAGTCTGGGGTGCCCACAGCCACCAGCGACCGCACCTGGACCAAAAGGAGCAGAGCGAGGCTTGGGCCCACCCCTCTGGGCCCACCAGCTCTGGGCCCTCCTTCCAACCACTCCCCAAATGCTTAGCCCCTCCCTCTGCCTGGTCTAATACTCCGCCACACAGATGCACACTTAAGCCTGGGTCACACCACCTCTTTTCCCCTCCAAATCCAGTCCCGCTCCCACACTCTGTGCCTTCCATCCCTCATCAGCAGCCCCAGTTGCTCTGTGGCCAGGGCACTGGCTCACCGTTCCATCTGCCCGCAGCTTGAGGCAAGAGCCGAAGTCGGCCAGGCGGATGTGGCCACAGCGGTCCAGCAGGATGTTGTCGGGTTTGATGTCCCTGCACGGAGGAAAAGAGAAGGGTGGGATAAATGAACCTCCCTTCTGTGGTCCCACCAGGCTCCGCCCCTCCGGGCAGTGCCACCTGGGACCCCACTTCCCTCGGGTTCCGCCCCTGTAGGGCTTCTACAGTTCTGACCCCTACTCCCAGGCACCCCCCGGTGGGCCCCAACCAAGAAGGTCCCTCTCCAGGCCCTGGCCTCTCTGGGCTCTGACCTTCCAAGAACCCGGCCAGGGAACAAGCTTGCTATCCCCTCGGCATGCTGCGCCCACCTGTGCACGTAGCCAAGCCGGTGCACCGAGTCTATGGCCATGACAATCTCCGCCAGGTAGAAGCGCGCCATCTCGGCCGGAATCCGCTCCCCAAACTTGCTCAGCAGTGTCAGCAGGTCCCCGCCACGTAATACTCCATGACCAGGTACTGAGAAGGGGTTCGTCATGGGTGGTTGGTAGTCCCCTGAGGCCCTGATCCATCACGGATGGCTGGGACAACCCCTCCCAGAGACACCCCATCCTTGGGCAGAGACCTGCAGCCCCAGCCAGAGATAACCATAGAGATCTGCCGGGCCAAATCAGAGACCACCTGCGGCCCCCGCCCCCCGCCCCTCAAAAAAAAACACAAACCAAAAAAACTTTCCTGGGATGTCACTGGGCAGATTCACTCCCCCTGAGATGTTCTGGGAAAGAGAACCAGAGGCCACCCAACACCCAGAGAATAGGTCCCAGACACTCCTTCCCCTAGGCAGTGGCCCCGTTAGAATTCCTGAAGACCTATCCCATTCCAGGTAAGAGACCCCCCGCAACAGAGACATCTTTATAAGAGTCCCCCAGATCACCCCAGAAACGGGCAGTGAAGGACACCATCCCCGTCTCAGATAGGGAAGGCCCCTCATCCACCTGACACACCCTCTTACCGCACACAGACTTTCCCACAGACGTTTCCGGGCAGCACCCCCAATCCTAGAGCTTCCTCTCCCCACCTCCTCGTCAGTGACAGCACGGGCTCTTGTCCCTCTTCCTAGTCCCCCCGGCCCGGAGCTCACCAGGTAGTTCTCATCCTGGAAGGCGAAGTGCAGCTGCGTGATCCACCGCGGTCCCCATTCACCAACACGTCCCTCTCCTCACGGAAGCACGACACCTGCAGGGCACCCGGAGGAGCTGCAGCCGGAGACGGGGCGCGGATCCTCAAAGCCCCCCACGTCCGCCCAGCCCCTCACCTCGCCCCTCTTCAGCATGTCCCACTTGTTCATGATCTTCATGGCATACACCTGGCCCGTCTGCTTCATCTTCACTACCGCTACCTGAGGTCGAGATAGTGAGACAGAGTGGAGACGGCGGGAAAACAAAAGGGCTCGCCCAGACCCAACTCCACCCGCTTCTGCACCCAGCCGTGGCCCCGCCCCACCTGCCACACCACGCCCATTGGTCCCAAGCCCCCGCCTCCAGCCCAGCCCTAGGTTCTAAGGCTCGGTCATTCATCAATTTCTAAGGCCCCGCCCCAACCCCTATGCCCCGCCCACCACGAGTCAAGTCAGGCTCCCGCCCGGTTCGGCTTACCTCGCTGAACGCCCCGCGTCCGATCACCTTCAGAATCTCGAAGTCGTCCCTCTGCAGTCGGACCTCCTTAAGCCTCACCACGATGGGCTCCGCTGGGGGGGTGGTGGGGGAAAAGAACCGAGGGTCACCAGAAAGGGCACTGGAGACAAGGGGGAAAGCCCCACCCTCTGTCTGTCTCCCCTTCTCTGCCTCTCAGCTTCACCCTAGGGACTGTCTGCTTCCCAGGGGCTTCCCCACATAAACACCGTGTAAGGTTCTGGGGGCCAAAAATGCCCTCCCATAGAGGTGAGATGCCTGAGAAGCCCTTTGCGGGACAGGGCATTGGCCCAGAGGGCTCCAAGGGTGTGCAGGATGGTTAGGGTGGGGGTAACGGAGTCTGCAGAAGGACAGACCCTAAAGAACAAGGGGAGAAGGAAATAAGACCCAGTTCTTCCACCTTCCACCCTGACTCCAGGTGACAGTTCAGGTGCAGCCAGGGCCCCACCCCCACGTTCTCATGTAGAATGTCCTGGGTAACGCCCAGACGTGGGGTTGTATCCAGTACCTCTAGATTCAGATGCAGGTGGTTCTTGAACCACACTTTGGAAAACCCTGGATTTGGGGCTGGGGGAAGGAACAGGACAGGGAGACAAAGGTAGCATGGTCACATATCCCAGACTCAAGTGCTCCGGTCCAGCCCATCTCTCAGTCCTCCAGGAGGAGTGCTTTAGTCCTACCCCTTATTTACAGATGACAATCAGGCCTCTCAGAAGTCAAGGTCCTATGACTAGGAATGTTAAACTGGGCAGCCTGGGTGGGGGACCAGCTGGGGTGGAGGCTTGTGGCTCAGGGACTGTGGGGACAGGGAGGCCTGGACTCCGGGGGTGGAAGAAGTGGAGGCAGGATGGAGAGAGGGATGGGTATGGGAGGGGAGGGAAGGAGAGGAGACAGTGGGTTCTGGAGGAAAGAGAGGGGTTCCGGGGGGTTCCGGGAGAGTGGGCCAGCAGGGACGAACCAGCCAGGGCTGAGGACGGGGGAGGGAGGAGAGAGCCAAGACATTGAGCCCTTTTAAGGCAGCAGGAACCCTGGCCCCCTCCCCCGCCCCAGGCCTGGCAGGGGGGAGGGGCCGCGAGATGCAGCTCGGGCCACAAAAGGAGTGCTCCTCCCGGGAGCGCCAGACCCCTACCCCACCCGGTCTCAGTTTACCCTGCCAACCCAACTTCATCCTCTACTAAAGGGAGGCCAGGAGAGTCATTAGGGGCTGTGGGAGGTTGGGAGAAGGTTTTTCCAGAGGCTGAATGGCCTGGCCCTTCTTGGCCGCCACCTTCCCATCTGTGAAATGGGAGGAGGAGGGAAAGGAACCAGGCCTGGGTCAGAAAGAGATGGACACAAGAGGAAGCAGGGGAAGGCGGAGAAACAGGAAACAAGTGTCACACACAGTTTGTTACACACAGGCCAAAAACCAAACACCAGTCCACTGAGGGCCAGACATATGAGGGCCAGAGGAGCAGGGTGCTGGGAATGAAAGAAACCAGTGACCAGTGAGCCCGAGTCCTGGGCTTGGGGAGGAGGGTGGGCAGACAAGGCAGCTGGCAGAAGCGGAAGCATCCTCCTTCCAGCCCTGGCCCAGGCCCTGGAAAGCCCTTGCAAGAAGGGGTTTTGTGGGACAGCTGGAAGGTTGGAGGCCTGGGCCAGGAGAACTAAAGGACGCAGGGACCCGGGGTGGCGGGAATGGTGGCTGACCCACACGGCTCATAGGAACCGAGACTTTGGGGCTGGGGGTGGGGGTGGGTGCAGAACCTACAAAAGAAGGAGACACACCCAGAAGAACCCAAAGTTGTCCCTCCTGGATCGCAGAGGAGGGGGCACTGGTGGCGCCTGTCTGCAAAGCTGGTTCTCCCACAGGGCCCGCGTGAGTCACCGCCCTCCCAGCGCCTGGGCACCTGTTGGGGCAGCTGGAGAGGCTGGCGCCGAACACCTGCCTGTCGGCTGCGCCCCTGGCAGCTGCCCCATGCTGGCCCTCCTGGCTTGCCCCAGGCCACAGGAAACCAGGGGAGAGGGCCATAGAGCCCACTTTTGGGGGGAGGGTCCTAGGAGAGGGAGCTCCCAGCATGACCCTGACCCTGACCCTGAGGCGGGGTGCAGAGAGGCCATGGGAGGCTGTTGGCAGCCTAGCCTGCTGTGACTCCTAAGCCATGGCTCTGAGTCACAGCCACCCTAGCCTCTGCTGACCTTACTCTGCCCCTCCAGAGTGGTGGCATAGGACAGAGAAATGTTGCCCCAGGGCCCAGGTCCACACTCTGAGCCCCAGGATGGGAGGATCTCCCCATGCCCATCCTGCCATCCTGCCCCCCCAACAGCCAGGCCTCTGTGCCCAGACCTGTCTCCTGCCCCACCCCCCACCCCATCTGCAGGAGCCCCGAGGGTAGGCACTCACCCCACTGCAAGAAGTCGGCCACGTACTTGTCCTGGGCCAGTTCGGAGGCGCCCAGCTCCTGGTGGACGCCCAGGAGAAGGTCGAGCAGGGGCTCCAGCCCCAGGAAGCCCGGGTCCAACACCAGCTGCTGGAGCCGCCTCAGCCGCACCTCGGCTGACATGTTGGACAGGCAGCACCATGGCCCCTCCCCGGGCCGGGGGCTCGGGGTCCTCCTGTCACAGGGCCTGGCAGCCCCTGTCCAGGCCCTGGAGCCCTGGCTGCATGTCTGCCTGTCCCTGGCTGTCCCCTGGGCCTCTCTGGCCACTTCTCTCTGCGGCCGGCCGAGGCCTCCTCCCCTTCTCCCCACCCCTTGGTCCAGCCCCCCTCCCGCCTCCCAGCCTTAACCCCTCACAAGCCAGCCCCCCAAGCTCCCTCCTTCCAGGGCCCTTCAGAACCCTTCAGTGCTAGAGTAGGGGGAATGCATGGAGAATCTCAGCTGTCACACTGTCACCCCAATAAAAAGGTAGAGCACAGAGGAAGCCACAGGTGTGATGAATTTCAAGAAGAACCTGCCCATTGCTGAACAGGGCCAATTGAGGCTGAGTGACCCAGTAGCAACCACCCCCTTGCTCCCGACAAGCTCCAGAACTGGGGGCTAGGGGGTGAGGGCTGGGGGTTGGGGCTGGGTGGGAGAAAGGGGTTGAGGCCTGGGAAGGGGACGGGAGGCCAGGGCAGCTTCTTGGGTGACTCAGAGATGGATTCCGACTCGGACAGTTAAATTTAGCCCTCAGGCTCTCTGCTTTATACCAGCTTTTTTTTTTTTTTTTTTTTTTTTCCCAGGAGGTGGGGAAGGGGGTGGTGAGGACAGGACCAGGAGGGGGCACCCTCAGAGCCTGAACAGGGGTGACCTGCTGCCCAAAGGCTGGGGGAGGAAAGTGGTGGAGCCAAGCCCTCCCTGGGCAAACGAGACCCCAAAATAGCTCCTTGGCCCCTGGCCGGGGAGCCACATCCTGCCCAGGCTGGGATTAGAAACAGAAACATTTCGGGGGGTGGAGGGTGGCGCGGGAAGACACACTCCCTGCGGCTCCGAGAGCCAAGAGGAATCTGAGCTCTTCTTTCCAGGGTGGACGGTTCTCCAGAGTGGAAGTCGGGGCCTCCAACCGGCCTGGGGTGCCTGGGCCTTGAGAGGGCCAGGCCTGAGAAACTAGGAGGCAAGGACCGAGGAGTCCCAGCTGGGCAGGGCCTGGGAAACGTGGTCCTGGGCAGTTCTGATAATTTAAAAAACACCGAGGACTTTGATGAGGGGCCCTGGCTGCGGGCAGGATGCTCTTCTCCCCAAGAGGGTCCTGCTCCAGCCGCTGGTGTGGGTCACCAGTTGTGTGACTCGCAGGTCCGTTCCCTCCCTGGGCTCCGGCTTTTCCTGCTATGAAAAGGGGTGGGAGGCGCTGGGCTGGGAGCAGGCCTGCACTCCTCCTCTGGGGAAAGCGGACTGCCTAGAACCACAGAGTACACACACAGGTCCTCTACATCATCTCCTCCGAAGGGGACAGACCCGCTGAGAAAACAGGGAACTTTAGTATAAATAAGAGGTCCTGCGGGACAGGGAGGAACCTGAGGGGCTTCCATAATTTAACACTCTTCAAAAGCACAGACAATAGCAAGGGCAGCTGGGGTGGGGGCCGGGCGAGGGCTGGACCACCCCCGGCCCTGCCCACTCAACTGAGGGGCACACTGCAGACCTGGGGCTCTTGTGGCAGGGGCGGGGGAGTGTCTCCCCAGGAGTGACCAGTCACATGCTGGGGACAGGGATGAGGGTAACACTGATGTTTCAGAGGAAGAGGTCATTGTACTTGGCAGTGGGGTGGGGGACAAGGCTGAGGCCACAAGGGCTATTACATCGCCCGTGTCCGGGCCAGTCTGGGGGGCCCCCAAATTTTGTGCAGGTGGGGGGACTGGAGCAGTCCATCCATCATGGTTAGTCTTGTTAATACGTTGATCTGTGAGGTCAGCAGGGAGGGTTATGGCTAGGAGGCTGGGGGCATGGGGTTGGGGGGGCCCGATATCCATG";
        final boolean isNegativeStrand = true;

        Set<Integer> expectedPositions = new HashSet<>(Arrays.asList(23, 26, 36, 45, 70, 109, 197, 225, 240, 255, 285, 296, 312, 347, 366, 398, 436, 577, 650, 652, 717, 749, 841, 932, 983, 1005, 1011, 1073, 1195, 1253, 1411, 1451, 1472, 1497, 1503, 1540, 1594, 1605, 1734, 1743, 1748, 1767, 1780, 1789, 1868, 1910, 1929, 1947, 1982, 1989, 2018, 2039, 2051, 2063, 2068, 2080, 2092, 2113, 2141, 2155, 2164, 2183, 2191, 2196, 2221, 2223, 2250, 2422, 2430, 2486, 2624, 2653, 2751, 2780, 2801, 2855, 2934, 2977, 3003, 3030, 3056, 3105, 3107, 3130, 3163, 3225, 3246, 3308, 3396, 3399, 3477, 3514, 3532, 3544, 3589, 3633, 3651, 3681, 3702, 3737, 3763, 3796, 3800, 3802, 3833, 3836, 3904, 3907, 3943, 3977, 3997, 4007, 4036, 4046, 4063, 4087, 4115, 4125, 4130, 4190, 4194, 4257, 4261, 4350, 4371, 4423, 4446, 4455, 4521, 4640, 4652, 4667, 4707, 4732, 4735, 4795, 4834, 4853, 4899, 4924, 4941, 4960, 4980, 4995, 5007, 5016, 5040, 5061, 5086, 5095, 5097, 5106, 5139, 5145, 5148, 5161, 5189, 5199, 5209, 5229, 5236, 5282, 5328, 5353, 5400, 5422, 5459, 5465, 5517, 5524, 5529, 5558, 5573, 5585, 5587, 5592, 5611, 5614, 5655, 5674, 5697, 5704, 5707, 5711, 5717, 5734, 5751, 5788, 5790, 5845, 5866, 5888, 5899, 5914, 5916, 5920, 5933, 5937, 5950, 5967, 5969, 5974, 5976, 5981, 6004, 6015, 6024, 6027, 6029, 6032, 6050, 6071, 6082, 6099, 6102, 6106, 6130, 6132, 6135, 6141, 6148, 6153, 6157, 6159, 6165, 6168, 6194, 6200, 6208, 6225, 6228, 6235, 6239, 6242, 6252, 6259, 6261, 6278, 6283, 6285, 6291, 6319, 6322, 6344, 6357, 6363, 6379, 6386, 6388, 6390, 6394, 6396, 6401, 6405, 6419, 6421, 6423, 6425, 6450, 6459, 6462, 6473, 6480, 6488, 6498, 6501, 6507, 6512, 6533, 6535, 6537, 6539, 6547, 6552, 6560, 6566, 6570, 6574, 6585, 6596, 6605, 6609, 6612, 6619, 6631, 6636, 6642, 6646, 6657, 6661, 6668, 6676, 6682, 6687, 6693, 6699, 6706, 6711, 6717, 6721, 6730, 6745, 6751, 6756, 6760, 6765, 6771, 6775, 6778, 6781, 6784, 6795, 6799, 6824, 6826, 6846, 6850, 6853, 6856, 6859, 6862, 6868, 6883, 6886, 6892, 6894, 6898, 6904, 6966, 7007, 7020, 7029, 7045, 7053, 7060, 7063, 7071, 7090, 7116, 7128, 7161, 7171, 7173, 7193, 7228, 7236, 7248, 7253, 7272, 7274, 7283, 7288, 7306, 7312, 7352, 7359, 7364, 7371, 7380, 7386, 7397, 7407, 7419, 7426, 7455, 7463, 7474, 7480, 7482, 7487, 7491, 7497, 7517, 7519, 7521, 7523, 7548, 7551, 7555, 7582, 7588, 7597, 7606, 7610, 7625, 7629, 7639, 7644, 7672, 7676, 7681, 7688, 7709, 7713, 7723, 7729, 7739, 7744, 7753, 7763, 7770, 7772, 7820, 7822, 7829, 7868, 7878, 7900, 7938, 7943, 7950, 7972, 7975, 7980, 7984, 7990, 8002, 8011, 8029, 8031, 8040, 8048, 8072, 8091, 8093, 8100, 8117, 8146, 8154, 8169, 8185, 8189, 8194, 8233, 8253, 8321, 8338, 8345, 8349, 8352, 8358, 8360, 8366, 8382, 8386, 8396, 8402, 8425, 8435, 8439, 8444, 8462, 8469, 8471, 8483, 8490, 8509, 8517, 8523, 8549, 8551, 8555, 8566, 8569, 8579, 8591, 8594, 8624, 8627, 8632, 8645, 8648, 8667, 8673, 8697, 8708, 8721, 8752, 8757, 8763, 8780, 8794, 8799, 8805, 8839, 8843, 8848, 8855, 8859, 8865, 8885, 8907, 8915, 8926, 8940, 8948, 8958, 8967, 8972, 8981, 8983, 8988, 8990, 8993, 8996, 9006, 9016, 9020, 9047, 9050, 9076, 9096, 9100, 9140, 9144, 9166, 9174, 9184, 9188, 9194, 9224, 9249, 9295, 9303, 9327, 9419, 9426, 9443, 9478, 9480, 9490, 9500, 9506, 9508, 9511, 9521, 9554, 9559, 9561, 9570, 9585, 9593, 9653, 9668, 9672, 9689, 9695, 9708, 9710, 9736, 9745, 9749, 9754, 9758, 9784, 9789, 9796, 9802, 9808, 9814, 9820, 9828, 9844, 9848, 9866, 9881, 9920, 9925, 9933, 9949, 9961, 10083, 10277, 10407, 10458, 10474, 10538, 10560, 10568, 10621, 10627, 10665, 10713, 10731, 10739, 10760, 10938, 11111, 11152, 11336, 11365, 11406, 11417, 11422, 11597, 11634, 11858, 11938, 11974, 12071, 12204, 12222, 12282, 12297, 12337, 12353, 12383, 12400, 12407, 12431, 12452, 12458, 12462, 12474, 12504, 12515, 12535, 12593, 12603, 12605, 12610, 12748, 12966, 12982, 13049, 13055, 13058, 13068, 13076, 13091, 13119, 13127, 13242, 13250, 13305, 13322, 13339, 13364, 13376, 13409, 13417, 13422, 13453, 13464, 13469, 13497, 13509, 13513, 13574, 13645, 13648, 13680, 13688, 13722, 13754, 13786, 13814, 13824, 13952, 13996, 14026, 14039, 14371, 14377, 14403, 14409, 14430, 14445, 14460, 14472, 14479, 14496, 14504, 14520, 14546, 14552, 14566, 14575, 14583, 14599, 14614, 14636, 14646, 14684, 14698, 14705, 14727, 14731, 14751, 14771, 14774, 14783, 14785, 14795, 14820, 14833, 14850, 14856, 14880, 14901, 14938, 14942, 14967, 15065, 15128, 15200, 15213, 15232, 15238, 15246, 15261, 15280, 15299, 15357, 15365, 15396, 15403, 15452, 15521, 15549, 15559, 15573, 15584, 15592, 15616, 15629, 15631, 15639, 15643, 15650, 15687, 15692, 15727, 15769, 15856, 15879, 15886, 15893, 16045, 16091, 16137, 16160, 16207, 16229, 16235, 16278, 16292, 16321, 16326, 16359, 16372, 16382, 16384, 16403, 16417, 16425, 16443, 16458, 16464, 16469, 16471, 16491, 16495, 16512, 16567, 16588, 16601, 16625, 16628, 16646, 16666, 16682, 16690, 16709, 16730, 16761, 16787, 16806, 16815, 16833, 16837, 16842, 16852, 16859, 16864, 16866, 16870, 16888, 16894, 16908, 16930, 16940, 16968, 17095, 17159, 17220, 17339, 17366, 17374, 17518, 17708, 17807, 17817, 17839, 17860, 17928, 17956, 17958, 17970, 17995, 18001, 18021, 18147, 18238, 18377, 18428, 18537, 18547, 18555, 18577, 18591, 18681, 18704, 18738, 18740, 18750, 18761, 18797, 18800, 18815, 18821, 18964, 19249, 19281, 19287, 19308, 19314, 19332, 19347, 19375, 19400, 19409, 19416, 19455, 19460, 19468, 19579, 19583, 19587, 19633, 19878, 19964, 20007, 20012, 20209, 20241, 20288, 20305, 20307, 20326, 20332, 20373, 20393, 20406, 20464, 20496, 20532, 20560, 20601, 20630, 20638, 20656, 20688, 20730, 20783, 20798, 20840, 20925, 20929, 20947, 21005, 21143, 21147, 21153, 21235));

        List<BaseModificationSet> modificationSets = BaseModificationUtils.getBaseModificationSets(mm, ml, seq.getBytes(), isNegativeStrand);
        assertEquals(1, modificationSets.size());

        BaseModificationSet bmSet = modificationSets.get(0);
        Map<Integer, Byte> likelihoods = bmSet.getLikelihoods();
        assertEquals(expectedPositions.size(), likelihoods.size());
        for (Integer pos : likelihoods.keySet()) {
            assertTrue(expectedPositions.contains(pos));
            assertEquals('G', seq.charAt(pos));
        }
    }
}
