#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>

#include "imapfilter.h"
#include "session.h"
#include "buffer.h"


buffer ibuf;			/* Input buffer. */


int receive_response(session * ssn, char *buf);
int tagged_response(session * ssn, int tag);

int check_bye(void);
int check_continuation(void);
int check_trycreate(void);


/*
 * Read data the server sent.
 */
int
receive_response(session * ssn, char *buf)
{

	if (socket_read(ssn, buf, RESPONSE_BUF) == -1)
		return -1;

	debug("getting response (%d):\n\n%s\n", ssn->socket, buf);

	return 0;
}


/*
 * Search for tagged response in the data that the server sent.
 */
int
tagged_response(session * ssn, int tag)
{
	int r;
	char s[5];
	char *c;

	r = RESPONSE_NONE;

	snprintf(s, sizeof(s), "%04X", tag);

	c = ibuf.data;
	while (r == RESPONSE_NONE && (c = xstrcasestr(c, s))) {
		c += 4;
		if (!strncasecmp(c, " OK ", 4))
			r = RESPONSE_OK;
		else if (!strncasecmp(c, " NO ", 4))
			r = RESPONSE_NO;
		else if (!strncasecmp(c, " BAD ", 4))
			r = RESPONSE_BAD;

	}
	if (r != RESPONSE_NONE) {
		c -= 4;
		verbose("S (%d): ", ssn->socket);
		do {
			verbose("%c", *c);
		} while (*c++ != '\n');
	}
	return r;
}


/*
 * Check if server sent a BYE response (connection is closed immediately).
 */
int
check_bye(void)
{

	if (xstrcasestr(ibuf.data, "* BYE ") &&
	    !xstrcasestr(ibuf.data, " LOGOUT "))
		return 1;
	else
		return 0;
}


/*
 * Check if the server sent a continuation request.
 */
int
check_continuation(void)
{
	if (ibuf.data[0] == '+' && ibuf.data[1] == ' ')
		return 1;
	else
		return 0;
}


/*
 * Check if the server sent a TRYCREATE response.
 */
int
check_trycreate(void)
{
	if (xstrcasestr(ibuf.data, "[TRYCREATE]"))
		return 1;
	else
		return 0;
}


/*
 * Get server data and make sure there is a tagged response inside them.
 */
int
response_generic(session * ssn, int tag)
{
	int r;

	if (tag == -1)
		return -1;

	buffer_reset(&ibuf);

	do {
		buffer_check(&ibuf, strlen(ibuf.data) + RESPONSE_BUF);
		if (receive_response(ssn, ibuf.data + strlen(ibuf.data)) == -1)
			return -1;
		if (check_bye())
			return -1;
		if (check_continuation())
			return RESPONSE_CONTINUE;
	} while ((r = tagged_response(ssn, tag)) == RESPONSE_NONE);

	if (r == RESPONSE_NO &&
	    (check_trycreate() || get_option_boolean("create")))
		return RESPONSE_TRYCREATE;

	return r;
}


/*
 * Process the greeting that server sends during connection.
 */
int
response_greeting(session * ssn)
{

	buffer_reset(&ibuf);

	if (receive_response(ssn, ibuf.data) == -1)
		return -1;

	if (check_bye())
		return -1;

	verbose("S (%d): %s", ssn->socket, ibuf);

	if (xstrcasestr(ibuf.data, "* PREAUTH "))
		return RESPONSE_PREAUTH;

	return RESPONSE_NONE;
}


/*
 * Process the data that server sent due to IMAP CAPABILITY client request.
 */
int
response_capability(session * ssn, int tag)
{
	int r;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	ssn->protocol = PROTOCOL_NONE;

	if (xstrcasestr(ibuf.data, "IMAP4rev1"))
		ssn->protocol = PROTOCOL_IMAP4REV1;
	else if (xstrcasestr(ibuf.data, "IMAP4"))
		ssn->protocol = PROTOCOL_IMAP4;
	else {
		error("server supports neither the IMAP4rev1 nor the "
		    "IMAP4 protocol\n");
		return -1;
	}

	ssn->capabilities = CAPABILITY_NONE;

	if (xstrcasestr(ibuf.data, "NAMESPACE"))
		ssn->capabilities |= CAPABILITY_NAMESPACE;
#ifndef NO_CRAMMD5
	if (xstrcasestr(ibuf.data, "AUTH=CRAM-MD5"))
		ssn->capabilities |= CAPABILITY_CRAMMD5;
#endif
#ifndef NO_SSLTLS
	if (xstrcasestr(ibuf.data, "STARTTLS"))
		ssn->capabilities |= CAPABILITY_STARTTLS;
#endif
	return r;
}


#ifndef NO_CRAMMD5
/*
 * Process the data that server sent due to IMAP AUTHENTICATE client request.
 */
int
response_authenticate(session * ssn, int tag, unsigned char **cont)
{
	int i, r;
	char *c;

	if ((r = response_generic(ssn, tag)) == RESPONSE_CONTINUE) {
		c = *cont = (unsigned char *)xmalloc(strlen(ibuf.data) + 1);

		for (i = 2; ibuf.data[i] != '\r'; i++)
			*c++ = ibuf.data[i];

		*c = '\0';
	}
	return r;
}
#endif


/*
 * Process the data that server sent due to IMAP NAMESPACE client request.
 */
int
response_namespace(session * ssn, int tag)
{
	int r;
	char *c, *e;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	ssn->ns.prefix = NULL;
	ssn->ns.delim = '\0';

	if ((c = xstrcasestr(ibuf.data, "* NAMESPACE "))) {
		c += 12;
		if (*c++ == '(' && *c++ == '(' && *c++ == '"') {
			e = strchr(c, '"');
			if (e - c > 0) {
				ssn->ns.prefix = (char *)xmalloc(e - c + 1);
				ssn->ns.prefix[0] = '\0';
				strncat(ssn->ns.prefix, c, e - c);
			}
			ssn->ns.delim = *(e + 3);
		}
	}
	debug("namespace (%d): '%s' '%c'\n", ssn->socket, ssn->ns.prefix,
	    ssn->ns.delim);

	return r;
}


/*
 * Process the data that server sent due to IMAP STATUS client request.
 */
int
response_status(session * ssn, int tag, int *exist, int *recent,
    int *unseen)
{
	int r;
	char *c;

	*exist = *recent = *unseen = -1;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	if ((c = xstrcasestr(ibuf.data, "MESSAGES "))) {
		c += 9;
		*exist = strtoul(c, NULL, 10);
	}
	if ((c = xstrcasestr(ibuf.data, "RECENT "))) {
		c += 7;
		*recent = strtoul(c, NULL, 10);
	}
	if ((c = xstrcasestr(ibuf.data, "UNSEEN "))) {
		c += 7;
		*unseen = strtoul(c, NULL, 10);
	}
	return r;
}


/*
 * Process the data that server sent due to IMAP EXAMINE client request.
 */
int
response_examine(session * ssn, int tag, int *exist, int *recent,
    int *unseen)
{
	int r;
	char *c;

	*exist = *recent = *unseen = -1;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	if ((c = xstrcasestr(ibuf.data, "EXISTS"))) {
		while (--c >= ibuf.data && *c != '*');
		c++;
		*exist = strtoul(c, NULL, 10);
	}
	if ((c = xstrcasestr(ibuf.data, "RECENT"))) {
		while (--c >= ibuf.data && *c != '*');
		c++;
		*recent = strtoul(c, NULL, 10);
	}
	if ((c = xstrcasestr(ibuf.data, "UNSEEN"))) {
		c += 7;
		*unseen = strtoul(c, NULL, 10);
	}
	return r;
}


/*
 * Process the data that server sent due to IMAP SELECT client request.
 */
int
response_select(session * ssn, int tag)
{
	int r;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	if (xstrcasestr(ibuf.data, "[READ-ONLY]"))
		return RESPONSE_READONLY;

	return r;
}


/*
 * Process the data that server sent due to IMAP SEARCH client request.
 */
int
response_search(session * ssn, int tag, char **mesgs)
{
	int r;
	char *c, *m;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	if ((c = xstrcasestr(ibuf.data, "* SEARCH "))) {
		m = *mesgs = (char *)xmalloc(strlen(ibuf.data) + 1);

		c += 9;
		while (*c != '\0' && (isdigit((int)*c) || *c == ' '))
			*(m++) = *(c++);
		*m = 0;
	} else
		*mesgs = NULL;

	return r;
}


/*
 * Process the data that server sent due to IMAP FETCH client request.
 */
int
response_fetch(session * ssn, int tag, char **fetch)
{
	int r;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	*fetch = ibuf.data;

	return r;
}
