<?php

require_once 'Horde/IMAP/Search.php';

/**
 * The IMP_VFolder:: class provides the code necessary to handle virtual
 * folders in IMP.
 *
 * $Horde: imp/lib/VFolder.php,v 1.6.6.2 2005/03/07 06:52:57 slusarz Exp $
 *
 * Copyright 2004-2005 Michael Slusarz <slusarz@curecanti.org>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@curecanti.org>
 * @version $Revision: 1.6.6.2 $
 * @since   IMP 4.0
 * @package IMP
 */
class IMP_VFolder {

    /**
     * Obtains the list of virtual folders for the current user.
     *
     * @access public
     *
     * @return mixed
     */
    function getVFolderList()
    {
        $vfolder = $GLOBALS['prefs']->getValue('vfolder');
        return (empty($vfolder)) ? $vfolder : unserialize($vfolder);
    }

    /**
     * Saves the list of virtual folders for the current user.
     *
     * @access private
     */
    function _saveVFolderList()
    {
        $vfolder = empty($_SESSION['imp']['vfolder']) ? '' : serialize($_SESSION['imp']['vfolder']);
        $GLOBALS['prefs']->setValue('vfolder', $vfolder);
    }

    /**
     * Add a virtual folder for the current user.
     *
     * @access public
     *
     * @param object &$ob     A IMAP_Search_Query object.
     * @param array &$search  The array of information used to build the
     *                        search UI page.
     *
     * @return string  The virtual folder ID.
     */
    function addVFolder(&$ob, $search)
    {
        $id = base_convert(microtime() . mt_rand(), 10, 36);
        $_SESSION['imp']['vfolder'][$id] = array(
            'ob' => serialize($ob),
            'search' => $search
        );
        IMP_VFolder::_saveVFolderList();

        return $id;
    }

    /**
     * Remove a virtual folder for the current user.
     *
     * @access public
     *
     * @param string $id  The virtual folder ID to delete.
     */
    function deleteVFolder($id)
    {
        $id = IMP_VFolder::_stripIMPLabel($id);
        if (!empty($_SESSION['imp']['vfolder'][$id])) {
            unset($_SESSION['imp']['vfolder'][$id]);
            IMP_VFolder::_saveVFolderList();
        }
    }

    /**
     * Return a list of IDs and VFolder labels, sorted by the label.
     *
     * @access public
     *
     * @return array  An array with the folder IDs as the key and the labels
     *                as the value.
     */
    function listVFolders()
    {
        $vfolders = array();

        if (!empty($_SESSION['imp']['vfolder'])) {
            foreach ($_SESSION['imp']['vfolder'] as $key => $val) {
                $vfolders[$key] = $val['search']['vfolder_label'];
            }
            natcasesort($vfolders);
        }

        return $vfolders;
    }

    /**
     * Set the necessary search parameters in the session given a virtual
     * folder ID.
     *
     * @access public
     *
     * @param string $id  The virtual folder ID.
     */
    function changeToVFolder($id)
    {
        $id = IMP_VFolder::_stripIMPLabel($id);
        $query = unserialize($_SESSION['imp']['vfolder'][$id]['ob']);
        $search = $_SESSION['imp']['vfolder'][$id]['search'];

        require_once IMP_BASE . '/lib/Search.php';
        IMP_Search::createSearchQuery($query, $search['folders'], $search, $search['vfolder_label'], $id);
    }

    /**
     * Change the label on a vfolder.
     *
     * @access public
     *
     * @param string $id     The virtual folder ID.
     * @param string $label  The new label.
     */
    function relabelVFolder($id, $label)
    {
        $id = IMP_VFolder::_stripIMPLabel($id);
        if (isset($_SESSION['imp']['vfolder'][$id]['search']['vfolder_label'])) {
            $_SESSION['imp']['vfolder'][$id]['search']['vfolder_label'] = $label;
            IMP_VFolder::_saveVFolderList();
        }
    }

    /**
     * Retrieve the UI query object for a given virtual folder ID.
     *
     * @access public
     *
     * @param string $id  The virtual folder ID.
     */
    function retrieveUIQuery($id)
    {
        $id = IMP_VFolder::_stripIMPLabel($id);
        if (isset($_SESSION['imp']['vfolder'][$id]['search'])) {
            return $_SESSION['imp']['vfolder'][$id]['search'];
        }
    }

    /**
     * Returns a link to edit a given virtual folder ID.
     *
     * @access public
     *
     * @param string $id  The virtual folder ID.
     *
     * @return string  The URL to the search page.
     */
    function editVFolderURL($id)
    {
        $id = IMP_VFOLDER_PREFIX . IMP_VFolder::_stripIMPLabel($id);
        return Util::addParameter(Horde::applicationUrl('search.php'), array('edit_vfolder' => 1, 'vfolder' => $id));
    }

    /**
     * Strip the IMP identifying label from a virtual folder ID.
     *
     * @access private
     *
     * @param string $id  The virtual folder ID.
     *
     * @return string  The virtual folder ID, with any IMP specific identifying
     *                 information stripped off.
     */
    function _stripIMPLabel($id)
    {
        return (strpos($id, IMP_VFOLDER_PREFIX) === 0) ? substr($id, strlen(IMP_VFOLDER_PREFIX)) : $id;
    }

}
