/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "indicator-network-service-dbus.h"

#include <dbus/dbus-glib.h>

#include "dbus-shared-names.h"
#include "marshal.h"
#include "indicator-network-service-server.h"
#include "connman-manager.h"

typedef struct _NetworkServiceDbusPrivate NetworkServiceDbusPrivate;

struct _NetworkServiceDbusPrivate
{
  DBusGConnection *session_bus;
  char *icon_name;
};


/* Signals */
enum {
  ICON_CHANGED,
  SCAN_REQUESTED,
  DEBUG_CHANGED,
  LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

#define NETWORK_SERVICE_DBUS_GET_PRIVATE(o)			\
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), NETWORK_SERVICE_DBUS_TYPE, \
				NetworkServiceDbusPrivate))

static void network_service_dbus_class_init (NetworkServiceDbusClass *klass);
static void network_service_dbus_init       (NetworkServiceDbus *self);
static void network_service_dbus_dispose    (GObject *object);
static void network_service_dbus_finalize   (GObject *object);

G_DEFINE_TYPE (NetworkServiceDbus, network_service_dbus, G_TYPE_OBJECT);

void network_service_dbus_set_icon(NetworkServiceDbus* self,
				   const gchar *icon_name)
{
  NetworkServiceDbusPrivate *priv = NETWORK_SERVICE_DBUS_GET_PRIVATE(self);

  if (priv->icon_name && g_str_equal(icon_name, priv->icon_name))
    return;

  g_free(priv->icon_name);

  priv->icon_name = g_strdup(icon_name);
  if (!priv->icon_name)
    return;

  g_signal_emit(self, signals[ICON_CHANGED], 0, priv->icon_name);
}

void network_service_dbus_get_icon(NetworkServiceDbus* self,
				   gchar **icon_name)
{
  NetworkServiceDbusPrivate *priv = NETWORK_SERVICE_DBUS_GET_PRIVATE(self);

  *icon_name = g_strdup(priv->icon_name);
}

void network_service_dbus_request_scan (NetworkServiceDbus* self)
{
	g_signal_emit (self, signals[SCAN_REQUESTED], 0, "wifi");
}

gboolean network_service_dbus_set_debug(NetworkServiceDbus* self, gint level,
					GError **error)
{
  g_signal_emit(self, signals[DEBUG_CHANGED], 0, level);

  return TRUE;
}

static void network_service_dbus_class_init(NetworkServiceDbusClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (object_class, sizeof(NetworkServiceDbusPrivate));

  object_class->dispose = network_service_dbus_dispose;
  object_class->finalize = network_service_dbus_finalize;

  g_assert(klass != NULL);
  dbus_g_object_type_install_info(NETWORK_SERVICE_DBUS_TYPE,
				  &dbus_glib__network_service_server_object_info
);

  dbus_g_object_register_marshaller (_marshal_VOID__STRING,
				     G_TYPE_NONE,
				     G_TYPE_STRING,
				     G_TYPE_INVALID);

  signals[ICON_CHANGED] = g_signal_new("icon-changed",
				       G_TYPE_FROM_CLASS(klass),
				       G_SIGNAL_RUN_LAST,
				       0, NULL, NULL,
				       _marshal_VOID__STRING,
				       G_TYPE_NONE, 1,
				       G_TYPE_STRING);

  signals[SCAN_REQUESTED] = g_signal_new("scan-requested",
										 G_TYPE_FROM_CLASS(klass),
										 G_SIGNAL_RUN_LAST,
										 0, NULL, NULL,
										 _marshal_VOID__STRING,
										 G_TYPE_NONE, 1,
										 G_TYPE_STRING);

  signals[DEBUG_CHANGED] = g_signal_new("debug-changed",
				       G_TYPE_FROM_CLASS(klass),
				       G_SIGNAL_RUN_LAST,
				       0, NULL, NULL,
				       _marshal_VOID__INT,
				       G_TYPE_NONE, 1,
				       G_TYPE_INT);
}

static void network_service_dbus_init (NetworkServiceDbus *self)
{
  NetworkServiceDbusPrivate *priv = NETWORK_SERVICE_DBUS_GET_PRIVATE(self);
  GError *error = NULL;

  priv->session_bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  if (error != NULL) {
    g_critical("Unable to connect to the dbus session bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  priv->icon_name = g_strdup("");

  dbus_g_connection_register_g_object(priv->session_bus,
				      INDICATOR_NETWORK_SERVICE_DBUS_OBJECT,
				      G_OBJECT(self));

  return;
}

static void network_service_dbus_dispose(GObject *object)
{
  G_OBJECT_CLASS(network_service_dbus_parent_class)->dispose(object);
}

static void network_service_dbus_finalize(GObject *object)
{
  NetworkServiceDbus *self = NETWORK_SERVICE_DBUS(object);
  NetworkServiceDbusPrivate *priv = NETWORK_SERVICE_DBUS_GET_PRIVATE(self);

  g_free(priv->icon_name);
  priv->icon_name = NULL;

  G_OBJECT_CLASS(network_service_dbus_parent_class)->finalize(object);
}
