/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "indicator-network-service.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <locale.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <glib/gi18n.h>

#include "marshal.h"
#include "connman-manager.h"
#include "connman-service.h"
#include "service.h"
#include "connman.h"
#include "dbus-shared-names.h"
#include "services.h"
#include "network-menu.h"
#include "indicator-network-agent-client.h"
#include "log.h"

#define NETWORK_AGENT LIBEXECDIR "/indicator-network-agent"

#define CONNECTING_ANIMATION_TIMEOUT 100

static void stop_connecting_animation(struct network_service *self);
static void start_agent(struct network_service *self);

const gchar *network_service_icon_name(const gchar *technology, gint signal)
{
  if (g_str_equal(technology, CONNMAN_TECHNOLOGY_ETHERNET)) {
    return ICON_CONNECTED_WIRED;
  }
  else if (g_str_equal(technology, CONNMAN_TECHNOLOGY_WIFI)) {
    if (signal > 75)
       return ICON_CONNECTED_WIFI_100;
    else if (signal > 50)
      return ICON_CONNECTED_WIFI_75;
    else if (signal > 25)
      return ICON_CONNECTED_WIFI_50;
    else if (signal > 0)
      return ICON_CONNECTED_WIFI_25;
    else
      return ICON_CONNECTED_WIFI_0;
  }
  else if (g_str_equal(technology, CONNMAN_TECHNOLOGY_CELLULAR)) {
    return ICON_CONNECTED_CELLULAR;
  }
  else if (g_str_equal(technology, CONNMAN_TECHNOLOGY_BLUETOOTH)) {
    return ICON_CONNECTED_BLUETOOTH;
  }

  return ICON_CONNECTED_DEFAULT;
}

static void set_animation_icon(struct network_service *self)
{
  gchar *icon;

  g_return_if_fail(self != NULL);
  g_return_if_fail(self->ns_dbus != NULL);
  g_return_if_fail(self->connecting_stage < CONNECTING_ICON_STAGES);
  g_return_if_fail(self->connecting_state < CONNECTING_ICON_STATES);

  icon = self->connecting_icons[self->connecting_stage][self->connecting_state];
  network_service_dbus_set_icon(self->ns_dbus, icon);
}

static gboolean connecting_animation_timeout(gpointer user_data)
{
  struct network_service *self = user_data;

  g_return_val_if_fail(self != NULL, FALSE);

  self->connecting_state++;
  if (self->connecting_state >= CONNECTING_ICON_STATES)
    self->connecting_state = 0;

  set_animation_icon(self);

  self->animation_counter++;

  /* fail safe in case animation is not stopped properly */
  if (self->animation_counter > 3000) {
    g_warning("connecting animation running for too long!");
    stop_connecting_animation(self);
    return FALSE;
  }

  return TRUE;
}

static void start_connecting_animation(struct network_service *self)
{
  g_return_if_fail(self != NULL);

  if (self->animation_timer != 0)
    return;

  self->connecting_stage = 0;
  self->connecting_state = 0;
  self->animation_counter = 0;

  set_animation_icon(self);

  self->animation_timer = g_timeout_add(CONNECTING_ANIMATION_TIMEOUT,
					connecting_animation_timeout,
					self);

  if (self->animation_timer == 0)
    g_warning("failed to add timeout for icon animation");
}

static void stop_connecting_animation(struct network_service *self)
{
  if (self->animation_timer == 0)
	  return;

  g_source_remove(self->animation_timer);
  self->animation_timer = 0;
}

void network_service_notify(struct network_service *self,
			    const gchar *summary, const gchar *body,
			    const gchar *icon)
{
  GError *error = NULL;
  gboolean result;

  g_return_if_fail(self);

  if (self->notification == NULL) {
    self->notification = notify_notification_new(summary, body, icon, NULL);
  } else {
    notify_notification_update(self->notification, summary, body, icon);
  }

  result = notify_notification_show(self->notification, &error);
  if (!result) {
    g_warning("Failed to show notification '%s/%s': %s", summary, body,
	      error->message);
    g_error_free(error);
  }
}

static void update_icon(struct network_service *self)
{
  const gchar *technology, *icon_name;

  g_return_if_fail(self);

  if (services_is_connecting(self->services)) {
    start_connecting_animation(self);
    return;
  }

  stop_connecting_animation(self);

  if (!services_is_connected(self->services)) {
    icon_name = ICON_DISCONNECTED;
    goto done;
  }

  technology = services_get_default_technology(self->services);

  if (technology == NULL || strlen(technology) == 0) {
    icon_name = ICON_DISCONNECTED;
    goto done;
  }

  icon_name = network_service_icon_name(technology, self->signal_strength);

 done:
  network_service_dbus_set_icon(self->ns_dbus, icon_name);
}

void network_service_state_changed(struct network_service *self,
				   const struct service *service)
{
  g_return_if_fail(self != NULL);
  g_return_if_fail(service != NULL);

  update_icon(self);

  network_menu_update_state(self->network_menu);
}

void network_service_default_service_changed(struct network_service *self,
					     const struct service *service)
{
  g_return_if_fail(self != NULL);

  g_free(self->default_service_path);
  self->default_service_path = NULL;

  if (service == NULL)
    goto done;

  self->default_service_path = g_strdup(service->path);
  self->signal_strength = service->signal_strength;

 done:
  update_icon(self);
}

void network_service_update_service_strength(struct network_service *self,
					     const struct service *service)
{
  /* only follow the signal of the default service */
  if (g_strcmp0(self->default_service_path, service->path) != 0)
    return;

  self->signal_strength = service->signal_strength;
  update_icon(self);
}

static void update_enabled_technologies(struct network_service *self,
					GValue *value)
{
  gboolean wifi, ethernet, cellular, bluetooth;
  gchar **technologies;
  int i;

  g_return_if_fail(self != NULL);
  g_return_if_fail(G_VALUE_HOLDS_BOXED(value));

  wifi = FALSE;
  ethernet = FALSE;
  cellular = FALSE;
  bluetooth = FALSE;

  technologies = g_value_get_boxed(value);

  for (i = 0; technologies[i] != NULL; i++) {
    if (g_strcmp0(technologies[i], CONNMAN_TECHNOLOGY_WIFI) == 0)
      wifi = TRUE;
    else if (g_strcmp0(technologies[i], CONNMAN_TECHNOLOGY_ETHERNET) == 0)
      ethernet = TRUE;
    else if (g_strcmp0(technologies[i], CONNMAN_TECHNOLOGY_CELLULAR) == 0)
      cellular = TRUE;
    else if (g_strcmp0(technologies[i], CONNMAN_TECHNOLOGY_BLUETOOTH) == 0)
      bluetooth = TRUE;
  }

  /* FIXME: how to handle bluetooth? */

  network_menu_update_wifi_state(self->network_menu, wifi);
  network_menu_update_wired_state(self->network_menu, ethernet);
  network_menu_update_cellular_state(self->network_menu, cellular);
}

static void connman_property_changed(DBusGProxy *proxy, const gchar *property,
				     GValue *value, gpointer user_data)
{
  struct network_service *self = user_data;

  if (g_str_equal(property, CONNMAN_PROPERTY_SERVICES)) {
    services_update(self->services, value);
  }
  else if (g_str_equal(property, CONNMAN_PROPERTY_ENABLED_TECHNOLOGIES)) {
    update_enabled_technologies(self, value);
  }

}

static void update_all_properties(struct network_service *self)
{
  GHashTable *properties;
  GError *error = NULL;
  GValue *value;

  /* FIXME: should be async */
  org_moblin_connman_Manager_get_properties(self->connman_proxy,
					    &properties, &error);

  if (error) {
    g_warning("%s(): org_moblin_connman_Manager_get_properties failed: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_SERVICES);
  if (value)
    services_update(self->services, value);

  value = g_hash_table_lookup(properties,
			      CONNMAN_PROPERTY_ENABLED_TECHNOLOGIES);
  if (value)
    update_enabled_technologies(self, value);

  network_menu_update_state(self->network_menu);

  /* FIXME: free properties */
}

static gboolean start_routine(gpointer user_data)
{
  struct network_service *self = user_data;

  network_menu_enabled(self->network_menu);
  update_all_properties(self);

  /* return false so that this callback isn't called anymore */
  return FALSE;
}

#ifndef TESTING
static
#endif
void connman_proxy_destroy(DBusGProxy *proxy, gpointer user_data)
{
  struct network_service *self = user_data;

  g_return_if_fail (self != NULL);

  if (self->connman_proxy != NULL) {
	  g_object_unref(self->connman_proxy);
	  self->connman_proxy = NULL;
  }

  network_menu_disabled(self->network_menu);
  services_remove_all(self->services);
  stop_connecting_animation(self);
  network_service_dbus_set_icon(self->ns_dbus, ICON_DISCONNECTED);
  stop_agent(self);

  g_debug("connman proxy destroyed");

  return;
}

static void create_connman_proxy(struct network_service *self)
{
  GError *error = NULL;

  g_return_if_fail(self->system_bus != NULL);
  g_return_if_fail(self->connman_proxy == NULL);

  self->connman_proxy = dbus_g_proxy_new_for_name_owner(self->system_bus,
							CONNMAN_SERVICE,
							CONNMAN_MANAGER_PATH,
							CONNMAN_MANAGER_INTERFACE,
							&error);

  if (error != NULL) {
    g_debug("unable to contact connmand, not running?");
    network_menu_disabled(self->network_menu);
    g_error_free(error);
    return;
  }

  g_signal_connect(G_OBJECT(self->connman_proxy), "destroy",
		   G_CALLBACK(connman_proxy_destroy), self);

  dbus_g_proxy_add_signal(self->connman_proxy, CONNMAN_SIGNAL_PROPERTY_CHANGED,
			  G_TYPE_STRING, G_TYPE_VALUE, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(self->connman_proxy,
			      CONNMAN_SIGNAL_PROPERTY_CHANGED,
			      G_CALLBACK(connman_property_changed),
			      self, NULL);

  g_debug("connman proxies created");

  /* connmand is now running */

  /* FIXME: start_routine should be deleted if connmand is gone again */
  g_idle_add(start_routine, self);
  start_agent(self);

  return;
}

static void agent_proxy_destroy(DBusGProxy *proxy, gpointer user_data)
{
  struct network_service *self = user_data;

  g_object_unref(self->agent_proxy);
  self->agent_proxy = NULL;

  g_debug("agent proxy destroyed");

  /* FIXME: start agent again if we didn't stop it */
  return;
}

void create_agent_proxy(struct network_service *self)
{
  g_return_if_fail(self->system_bus != NULL);
  g_return_if_fail(self->agent_proxy == NULL);

  self->agent_proxy = dbus_g_proxy_new_for_name(self->session_bus,
						INDICATOR_NETWORK_AGENT_NAME,
						INDICATOR_NETWORK_AGENT_OBJECT,
						INDICATOR_NETWORK_AGENT_INTERFACE);
  if (self->agent_proxy == NULL) {
    g_warning("failed to create agent proxy");
    return;
  }

  g_signal_connect(G_OBJECT(self->agent_proxy), "destroy",
		   G_CALLBACK(agent_proxy_destroy), self);

  g_debug("agent proxy created");
}

static void dbus_namechange(DBusGProxy *proxy, const gchar *name,
			    const gchar *prev, const gchar *new,
			    gpointer user_data)
{
  struct network_service *self = user_data;

  g_return_if_fail(self);
  g_return_if_fail(name);
  g_return_if_fail(new);

  if (g_strcmp0(name, CONNMAN_SERVICE) == 0 && strlen(new) > 0) {
    g_debug("connman manager interface visible");
    create_connman_proxy(self);
  }
  return;
}

void create_dbus_proxy(struct network_service *self)
{
  g_return_if_fail(self != NULL);
  g_return_if_fail(self->system_bus != NULL);

  GError *error = NULL;

  /* Set up the dbus Proxy */
  self->dbus_proxy = dbus_g_proxy_new_for_name_owner(self->system_bus,
						     DBUS_SERVICE_DBUS,
						     DBUS_PATH_DBUS,
						     DBUS_INTERFACE_DBUS,
						     &error);
  if (error != NULL) {
    g_warning("Unable to connect to DBus events: %s", error->message);
    g_error_free(error);
    return;
  }

  /* Configure the name owner changing */
  dbus_g_proxy_add_signal(self->dbus_proxy, "NameOwnerChanged",
			  G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
			  G_TYPE_INVALID);
  dbus_g_proxy_connect_signal(self->dbus_proxy, "NameOwnerChanged",
			      G_CALLBACK(dbus_namechange),
			      self, NULL);

}

static void start_agent_cb(DBusGProxy *proxy, GError *error,
			   gpointer user_data)
{
  if (error) {
    g_warning("failed to start agent: %s", error->message);
    g_error_free(error);
  }

}

void start_agent(struct network_service *self)
{
  if (self->agent_proxy == NULL)
    return;

  org_ayatana_indicator_network_agent_start_async(self->agent_proxy,
  						  start_agent_cb,
  						  self);
}

static void stop_agent_cb(DBusGProxy *proxy, GError *error,
			   gpointer user_data)
{
  if (error) {
    g_warning("failed to stop agent: %s", error->message);
    g_error_free(error);
  }

}

void stop_agent(struct network_service *self)
{
  if (self->agent_proxy == NULL)
    return;

  org_ayatana_indicator_network_agent_stop_async(self->agent_proxy,
						 stop_agent_cb,
						 self);
}

void request_scan (struct network_service *self)
{
	GError * error = NULL;

	g_debug ("request scan");
	if (self->connman_proxy == NULL) {
		g_warning ("but no connman instance available, so aborting");
		return;
	}

	/* FIXME: use async */
	org_moblin_connman_Manager_request_scan (self->connman_proxy,
											 "wifi", &error);
	if (error != NULL) {
		g_warning ("Error while trying to request scan: %s", error->message);
		g_error_free (error);
		return;
	}
}

static void debug_changed(struct network_service *self, gint level)
{
  log_set_debug(level > 0);
}

void network_service_set_system_bus(struct network_service *self,
									DBusGConnection *bus)
{
  g_return_if_fail (self != NULL);

  self->system_bus = bus;
}

void network_service_init (struct network_service *self)
{
  g_return_if_fail (self != NULL);
  g_return_if_fail (self->system_bus != NULL); /* set_system_bus should have been called before */

  gchar *icon;
  int i, j;
  GError *error = NULL;

  self->session_bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  if (error) {
    g_critical("Unable to connect to the dbus session bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  self->ns_dbus = g_object_new(NETWORK_SERVICE_DBUS_TYPE, NULL);
  g_signal_connect_swapped (self->ns_dbus, "scan-requested",
                                                        G_CALLBACK (request_scan), self);
  g_signal_connect_swapped(self->ns_dbus, "debug-changed",
			    G_CALLBACK(debug_changed), self);

  self->network_menu = network_menu_new(self);

  self->services = services_new(self);

  dbus_g_object_register_marshaller(_marshal_VOID__STRING_BOXED,
                                    G_TYPE_NONE,
                                    G_TYPE_STRING,
                                    G_TYPE_VALUE,
                                    G_TYPE_INVALID);

  dbus_g_object_register_marshaller(_marshal_VOID__INT,
                                    G_TYPE_NONE,
                                    G_TYPE_INT,
                                    G_TYPE_INVALID);

  create_agent_proxy(self);
  create_dbus_proxy(self);
  create_connman_proxy(self);

  /* create icon names */
  for (i = 0; i < CONNECTING_ICON_STAGES; i++) {
    for (j = 0; j < CONNECTING_ICON_STATES; j++) {
      icon = g_malloc(MAX_ICON_NAME_LEN);
      snprintf(icon, MAX_ICON_NAME_LEN,
               "nm-stage%02d-connecting%02d", i + 1, j + 1);
      self->connecting_icons[i][j] = icon;
    }
  }

}

void network_service_dispose (struct network_service *self)
{
  g_return_if_fail (self != NULL);

  int i, j;

  network_menu_free(self->network_menu);
  self->network_menu = NULL;

  stop_agent(self);
  notify_uninit();

  for (i = 0; i < CONNECTING_ICON_STAGES; i++) {
    for (j = 0; j < CONNECTING_ICON_STATES; j++) {
      g_free(self->connecting_icons[i][j]);
    }
  }

  g_free(self);
}
