/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libdbusmenu-gtk/menu.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <libindicator/indicator.h>
#include <libindicator/indicator-object.h>
#include <libindicator/indicator-service-manager.h>
#include <libindicator/indicator-image-helper.h>

#include "dbus-shared-names.h"
#include "indicator-network-service-client.h"
#include "marshal.h"

#define INDICATOR_DEFAULT_ICON		"nm-no-connection"

#define INDICATOR_NETWORK_TYPE            (indicator_network_get_type())
#define INDICATOR_NETWORK(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), INDICATOR_NETWORK_TYPE, IndicatorNetwork))
#define INDICATOR_NETWORK_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), INDICATOR_NETWORK_TYPE, IndicatorNetworkClass))
#define IS_INDICATOR_NETWORK(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), INDICATOR_NETWORK_TYPE))
#define IS_INDICATOR_NETWORK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), INDICATOR_NETWORK_TYPE))
#define INDICATOR_NETWORK_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), INDICATOR_NETWORK_TYPE, IndicatorNetworkClass))

typedef struct _IndicatorNetwork      IndicatorNetwork;
typedef struct _IndicatorNetworkClass IndicatorNetworkClass;

struct _IndicatorNetworkClass
{
  IndicatorObjectClass parent_class;
};


struct _IndicatorNetwork
{
  IndicatorObject parent;
  IndicatorServiceManager *service;
  DBusGProxy *network_dbus_proxy;
  GtkImage *indicator_image;
};

GType indicator_network_get_type(void);
INDICATOR_SET_VERSION
INDICATOR_SET_TYPE(INDICATOR_NETWORK_TYPE)

static void indicator_network_class_init(IndicatorNetworkClass *klass);
static void indicator_network_init(IndicatorNetwork *self);
static void indicator_network_dispose(GObject *object);
static void indicator_network_finalize(GObject *object);
G_DEFINE_TYPE(IndicatorNetwork, indicator_network, INDICATOR_OBJECT_TYPE);

static GtkLabel *get_label(IndicatorObject *io);
static GtkImage *get_icon(IndicatorObject *io);
static GtkMenu *get_menu(IndicatorObject *io);

static void connection_changed(IndicatorServiceManager * sm,
			       gboolean connected, gpointer user_data);

static void indicator_network_class_init(IndicatorNetworkClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);

  object_class->dispose = indicator_network_dispose;
  object_class->finalize = indicator_network_finalize;

  IndicatorObjectClass *io_class = INDICATOR_OBJECT_CLASS(klass);
  io_class->get_label = get_label;
  io_class->get_image = get_icon;
  io_class->get_menu = get_menu;

  dbus_g_object_register_marshaller(_marshal_VOID__STRING,
				    G_TYPE_NONE,
				    G_TYPE_STRING,
				    G_TYPE_INVALID);
}

static void indicator_network_init(IndicatorNetwork *self)
{
  self->service = NULL;


  self->service = indicator_service_manager_new_version(INDICATOR_NETWORK_DBUS_NAME,
							INDICATOR_NETWORK_DBUS_VERSION);
  g_signal_connect(G_OBJECT(self->service),
		   INDICATOR_SERVICE_MANAGER_SIGNAL_CONNECTION_CHANGE,
		   G_CALLBACK(connection_changed),
		   self);

  return;
}

static void set_icon_name(IndicatorNetwork *self, const gchar *name)
{
  if (name == NULL || strlen(name) == 0)
    return;

  g_debug("%s(%s)", __func__, name);

  indicator_image_helper_update(self->indicator_image, name);
}

static void icon_changed(DBusGProxy *proxy, const gchar *icon_name,
			 gpointer user_data)
{
  IndicatorNetwork *self = user_data;
  set_icon_name(self, icon_name);
}

static void get_icon_cb(DBusGProxy *proxy, gchar *icon_name, GError *error,
			gpointer user_data)
{
  IndicatorNetwork *self = user_data;

  if (error != NULL) {
    g_debug("%s() failed: %s", __func__, error->message);
    g_error_free(error);
    return;
  }

  set_icon_name(self, icon_name);

  g_free(icon_name);
}

static DBusGProxy *
setup_service_proxy (IndicatorNetwork *self)
{
  g_return_val_if_fail (IS_INDICATOR_NETWORK (self), NULL);

  DBusGConnection *bus;

  if (self->network_dbus_proxy != NULL)
    return self->network_dbus_proxy;

  bus = dbus_g_bus_get(DBUS_BUS_SESSION, NULL);
  self->network_dbus_proxy = dbus_g_proxy_new_for_name(bus,
						       INDICATOR_NETWORK_DBUS_NAME,
						       INDICATOR_NETWORK_SERVICE_DBUS_OBJECT,
						       INDICATOR_NETWORK_SERVICE_DBUS_INTERFACE);

  if (self->network_dbus_proxy == NULL) {
    g_warning("Unable to get network dbus proxy");
    return NULL;
  }

  g_debug("connecting to network service signals");

  dbus_g_proxy_add_signal(self->network_dbus_proxy,
			  INDICATOR_NETWORK_SIGNAL_ICON_CHANGED,
			  G_TYPE_STRING, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(self->network_dbus_proxy,
			      INDICATOR_NETWORK_SIGNAL_ICON_CHANGED,
			      G_CALLBACK(icon_changed),
			      self, NULL);

  return self->network_dbus_proxy;
}

static void indicator_connected(IndicatorNetwork *self)
{
  g_return_if_fail (IS_INDICATOR_NETWORK (self));

  setup_service_proxy (self);

  org_ayatana_indicator_network_get_icon_async(self->network_dbus_proxy,
					       get_icon_cb, self);
}

static void connection_changed(IndicatorServiceManager *sm,
			       gboolean connected,
			       gpointer user_data)
{
  IndicatorNetwork *self = user_data;

  if (connected) {
    indicator_connected(self);
  } else {
    //TODO : will need to handle this scenario
  }

  return;
}

static void indicator_network_dispose(GObject *object)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(object);

  if (self->service != NULL) {
    g_object_unref(G_OBJECT(self->service));
    self->service = NULL;
  }

  G_OBJECT_CLASS(indicator_network_parent_class)->dispose(object);

  return;
}

static void indicator_network_finalize(GObject *object)
{
  G_OBJECT_CLASS(indicator_network_parent_class)->finalize(object);
  return;
}

static GtkLabel *get_label(IndicatorObject * io)
{
  return NULL;
}

static GtkImage *get_icon(IndicatorObject * io)
{
  IndicatorNetwork *self = INDICATOR_NETWORK(io);

  self->indicator_image = indicator_image_helper(INDICATOR_DEFAULT_ICON);
  gtk_widget_show(GTK_WIDGET(self->indicator_image));

  return self->indicator_image;
}

static void dummy_cb () {}


static gboolean
confirm_menu_open (IndicatorObject *io)
{
	g_return_val_if_fail (IS_INDICATOR_NETWORK (io), FALSE);
	IndicatorNetwork *self = INDICATOR_NETWORK(io);

	if (setup_service_proxy (self) != NULL)
		org_ayatana_indicator_network_request_scan_async (self->network_dbus_proxy,
														  dummy_cb, NULL);
	return FALSE;
}

/* static */ void
menu_visibility_changed (GtkWidget *widget,
                         gpointer  *user_data)
{
	g_return_if_fail (GTK_IS_WIDGET (widget));

	static guint id = 0;

	/* confirm the menu is open after 2s, or cancel the timer
	   if its closed in the meantime */
	if (gtk_widget_get_visible (widget))
		id = g_timeout_add_seconds (2, (GSourceFunc) confirm_menu_open,
									user_data);
	else if (id > 0)
		g_source_remove (id);
}

/* Indicator based function to get the menu for the whole
   applet.  This starts up asking for the parts of the menu
   from the various services. */
static GtkMenu *get_menu(IndicatorObject *io)
{
  DbusmenuGtkMenu *menu = dbusmenu_gtkmenu_new(INDICATOR_NETWORK_DBUS_NAME,
											   INDICATOR_NETWORK_DBUS_OBJECT);

  g_signal_connect (GTK_MENU (menu),
					"map", G_CALLBACK (menu_visibility_changed), io);
  g_signal_connect (GTK_MENU (menu),
					"hide", G_CALLBACK (menu_visibility_changed), io);

  return GTK_MENU(menu);
}
