/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "network-menu.h"

#include <glib/gi18n.h>

#include <libdbusmenu-glib/client.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/menuitem.h>

#include "service.h"
#include "services.h"
#include "connman-manager.h"
#include "dbus-shared-names.h"
#include "connman.h"
#include "indicator-network-agent-client.h"

struct network_menu {
  DbusmenuMenuitem *root_menuitem;
  DbusmenuMenuitem *statusline;
  DbusmenuMenuitem *wired_header;
  DbusmenuMenuitem *wired_separator;
  DbusmenuMenuitem *wireless_header;
  DbusmenuMenuitem *wireless_list_top;
  DbusmenuMenuitem *wireless_other;
  DbusmenuMenuitem *wireless_separator;
  DbusmenuMenuitem *cellular_header;
  DbusmenuMenuitem *cellular_separator;
  DbusmenuMenuitem *network_settings;

  DbusmenuMenuitem *disabled_menuitem;

  GList *wired_services;
  GList *wireless_services;
  GList *cellular_services;

  gboolean wired_enabled;
  gboolean wireless_enabled;
  gboolean cellular_enabled;

  gboolean wired_shown;
  gboolean wireless_shown;
  gboolean cellular_shown;

  struct network_service *network_service;
};

void network_menu_enabled(struct network_menu *self)
{
  gboolean visible;

  visible = dbusmenu_menuitem_property_get_bool(self->disabled_menuitem,
						DBUSMENU_MENUITEM_PROP_VISIBLE);

  if (!visible)
    return;

  dbusmenu_menuitem_property_set_bool(self->statusline,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(self->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(self->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(self->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(self->network_settings,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

  dbusmenu_menuitem_property_set_bool(self->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
}

void network_menu_disabled(struct network_menu *self)
{
  struct service *service;
  gboolean visible;
  GList *iter;

  visible = dbusmenu_menuitem_property_get_bool(self->disabled_menuitem,
						DBUSMENU_MENUITEM_PROP_VISIBLE);

  if (!visible)
	  return;

  while ((iter = g_list_first(self->wired_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->wired_services = g_list_delete_link(self->wired_services,
						 iter);
  }

  while ((iter = g_list_first(self->wireless_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->wireless_services = g_list_delete_link(self->wireless_services,
						 iter);
  }

  while ((iter = g_list_first(self->cellular_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->cellular_services = g_list_delete_link(self->cellular_services,
						 iter);
  }

  dbusmenu_menuitem_property_set_bool(self->statusline,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(self->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(self->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(self->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(self->network_settings,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);

  dbusmenu_menuitem_property_set_bool(self->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
}

gboolean network_menu_is_disabled(struct network_menu *self)
{
	g_return_val_if_fail (self != NULL, FALSE);
	
	gboolean visible = dbusmenu_menuitem_property_get_bool (self->disabled_menuitem, DBUSMENU_MENUITEM_PROP_VISIBLE);

	return (visible == FALSE);
}

static void network_menu_update_wired(struct network_menu *self)
{
  struct services *services;
  struct service *service;
  GList *iter;
  guint pos;

  /* remove old services */
  while ((iter = g_list_first(self->wired_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->wired_services = g_list_delete_link(self->wired_services,
						 iter);
  }

  /* get new wired services */
  services = self->network_service->services;
  self->wired_services = services_get_wired(services);

  if (self->wired_services == NULL) {
    /* no wired services to show */
    self->wired_shown = FALSE;
    return;
  }

  pos = dbusmenu_menuitem_get_position(self->wired_header,
				       self->root_menuitem);
  pos++;

  for (iter = self->wired_services; iter != NULL; iter = iter->next) {
    service = iter->data;
    dbusmenu_menuitem_child_add_position(self->root_menuitem,
					 service->menuitem, pos);
    pos++;
  }

    self->wired_shown = TRUE;
}

static void network_menu_update_wireless(struct network_menu *self)
{
  struct services *services;
  struct service *service;
  GList *iter;
  guint pos;

  /* remove old services */
  while ((iter = g_list_first(self->wireless_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->wireless_services = g_list_delete_link(self->wireless_services,
						 iter);
  }

  /* get new wireless services */
  services = self->network_service->services;
  self->wireless_services = services_get_wireless(services);

  /*
   * Even there are now wireless services, there are other menuitems and
   * separator is still needed.
   */
  self->wireless_shown = TRUE;

  if (self->wireless_services == NULL) {
    /* no wireless services to show */
    dbusmenu_menuitem_property_set_bool(self->wireless_list_top,
					DBUSMENU_MENUITEM_PROP_VISIBLE,
					TRUE);
    return;
  }

  /* turn off the "No network detected" item */
  dbusmenu_menuitem_property_set_bool(self->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);


  pos = dbusmenu_menuitem_get_position(self->wireless_list_top,
					     self->root_menuitem);

  for (iter = self->wireless_services; iter != NULL; iter = iter->next) {
    service = iter->data;
    dbusmenu_menuitem_child_add_position(self->root_menuitem,
					 service->menuitem, pos);
    pos++;
  }
}

static void network_menu_update_cellular(struct network_menu *self)
{
  struct services *services;
  struct service *service;
  GList *iter;
  guint pos;

  /* remove old services */
  while ((iter = g_list_first(self->cellular_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(self->root_menuitem, service->menuitem);
    self->cellular_services = g_list_delete_link(self->cellular_services,
						 iter);
  }

  /* get new cellular services */
  services = self->network_service->services;
  self->cellular_services = services_get_cellular(services);

  if (self->cellular_services == NULL) {
    /* no cellular services to show */
    self->cellular_shown = FALSE;
    return;
  }

  pos = dbusmenu_menuitem_get_position(self->cellular_header,
				       self->root_menuitem);
  pos++;

  for (iter = self->cellular_services; iter != NULL; iter = iter->next) {
    service = iter->data;
    dbusmenu_menuitem_child_add_position(self->root_menuitem,
					 service->menuitem, pos);
    pos++;
  }

  self->cellular_shown = TRUE;
}

void network_menu_services_updated(struct network_menu *self)
{
  gboolean show;

  network_menu_update_wired(self);
  network_menu_update_wireless(self);
  network_menu_update_cellular(self);

  show = self->wired_shown && (self->wireless_shown || self->cellular_shown);
  dbusmenu_menuitem_property_set_bool(self->wired_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      show);

  show = self->wireless_shown && self->cellular_shown;
  dbusmenu_menuitem_property_set_bool(self->wireless_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      show);

  /* cellular_separator is always shown */
}

void network_menu_update_state(struct network_menu *self)
{
  struct services *services;
  gchar str[200];
  guint count;

  g_return_if_fail(self != NULL);
  g_return_if_fail(self->network_service != NULL);
  g_return_if_fail(self->network_service->services != NULL);
  g_return_if_fail(self->statusline != NULL);

  services = self->network_service->services;
  count = services_get_connected(services);

  if (count > 1)
    g_snprintf(str, sizeof(str), _("Connected (%d networks)"), count);
  else if (count == 1)
    g_strlcpy(str, _("Connected"), sizeof(str));
  else if (services_is_connecting(services))
    g_strlcpy(str, _("Connecting"), sizeof(str));
  else
    g_strlcpy(str, _("Disconnected"), sizeof(str));


  dbusmenu_menuitem_property_set(self->statusline,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 str);
}

void network_menu_update_wired_state(struct network_menu *self,
				     gboolean state)
{
  const gchar *str;

  g_return_if_fail(self->wireless_header != NULL);

  if (state)
    str = _("Wired: ON");
  else
    str = _("Wired: OFF");

  dbusmenu_menuitem_property_set(self->wired_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 str);
  self->wired_enabled = state;
}

void network_menu_update_wifi_state(struct network_menu *self, gboolean state)
{
  const gchar *str;

  g_return_if_fail(self->wireless_header != NULL);

  if (state)
    str = _("Wireless: ON");
  else
    str = _("Wireless: OFF");

  dbusmenu_menuitem_property_set(self->wireless_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 str);
  self->wireless_enabled = state;
}

void network_menu_update_cellular_state(struct network_menu *self,
				     gboolean state)
{
  const gchar *str;

  g_return_if_fail(self->cellular_header != NULL);

  if (state)
    str = _("Mobile: ON");
  else
    str = _("Mobile: OFF");

  dbusmenu_menuitem_property_set(self->cellular_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 str);
  self->cellular_enabled = state;
}

static void toggle_wired_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  struct network_menu *self = user_data;
  struct network_service *ns = self->network_service;
  GError *error = NULL;

  g_return_if_fail(ns != NULL);

  if (ns->connman_proxy == NULL)
    return;

  if (self->wired_enabled == FALSE)
    org_moblin_connman_Manager_enable_technology(ns->connman_proxy,
						 CONNMAN_TECHNOLOGY_ETHERNET,
						 &error);
  else
    org_moblin_connman_Manager_disable_technology(ns->connman_proxy,
						  CONNMAN_TECHNOLOGY_ETHERNET,
						  &error);

  if (error != NULL) {
    g_warning("Error while trying to %s wired state: %s",
	      self->wired_enabled ? "disable" : "enable", error->message);
    g_error_free(error);
    return;
  }
}

static void toggle_wireless_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  struct network_menu *self = user_data;
  struct network_service *ns = self->network_service;
  GError *error = NULL;

  g_return_if_fail(ns != NULL);

  if (ns->connman_proxy == NULL)
    return;

  if (self->wireless_enabled == FALSE)
    org_moblin_connman_Manager_enable_technology(ns->connman_proxy,
						 CONNMAN_TECHNOLOGY_WIFI,
						 &error);
  else
    org_moblin_connman_Manager_disable_technology(ns->connman_proxy,
						  CONNMAN_TECHNOLOGY_WIFI,
						  &error);

  if (error != NULL) {
    g_warning("Error while trying to %s wireless state: %s",
	      self->wireless_enabled ? "disable" : "enable", error->message);
    g_error_free(error);
    return;
  }
}

static void toggle_cellular_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  struct network_menu *self = user_data;
  struct network_service *ns = self->network_service;
  GError *error = NULL;

  g_return_if_fail(ns != NULL);

  if (ns->connman_proxy == NULL)
    return;

  if (self->cellular_enabled == FALSE)
    org_moblin_connman_Manager_enable_technology(ns->connman_proxy,
						 CONNMAN_TECHNOLOGY_CELLULAR,
						 &error);
  else
    org_moblin_connman_Manager_disable_technology(ns->connman_proxy,
						  CONNMAN_TECHNOLOGY_CELLULAR,
						  &error);

  /* FIXME: what to do with bluetooth? */

  if (error != NULL) {
    g_warning("Error while trying to %s cellular state: %s",
	      self->cellular_enabled ? "disable" : "enable", error->message);
    g_error_free(error);
    return;
  }
}

static void network_settings_activated(DbusmenuMenuitem *mi, guint timestamp,
				       gpointer user_data)
{
  gchar *argv[2];

  argv[0] = BINDIR "/indicator-network-settings";
  argv[1] = NULL;

  g_debug("%s(): starting %s", __func__, argv[0]);
  g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
}

static void show_wireless_connect_reply(DBusGProxy *proxy, GError *error,
					gpointer user_data)
{
  if (error) {
    g_critical("Unable to show wireless connect dialog: %s",
	       error->message);
    g_error_free(error);
    return;
  }
}

static void wireless_other_activated(DbusmenuMenuitem *mi, guint timestamp,
				       gpointer user_data)
{
  struct network_menu *self = user_data;
  struct network_service *ns = self->network_service;

  org_ayatana_indicator_network_agent_show_wireless_connect_async(ns->agent_proxy,
								  show_wireless_connect_reply,
								  self);

}

struct network_menu *network_menu_new(struct network_service *ns)
{
  struct network_menu *self;
  DbusmenuServer *server;

  self = g_malloc0(sizeof(*self));
  self->network_service = ns;
  self->root_menuitem = dbusmenu_menuitem_new();

  server = dbusmenu_server_new(INDICATOR_NETWORK_DBUS_OBJECT);
  dbusmenu_server_set_root(server, self->root_menuitem);
  dbusmenu_menuitem_property_set_bool(self->root_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      TRUE);

  self->statusline = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set_bool(self->statusline,
				      DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->statusline);
  network_menu_update_state(self);

  /* Wired section of the menu */
  self->wired_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->wired_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Wired: OFF"));
  dbusmenu_menuitem_property_set_bool(self->wired_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->wired_header);
  g_signal_connect(G_OBJECT(self->wired_header),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_wired_cb), self);


  self->wired_separator = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(self->wired_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->wired_separator);

  /* Wireless section of the menu */
  self->wireless_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->wireless_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Wireless: OFF"));
  dbusmenu_menuitem_child_append(self->root_menuitem, self->wireless_header);
  g_signal_connect(G_OBJECT(self->wireless_header),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_wireless_cb), self);

  self->wireless_list_top = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->wireless_list_top,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("No network detected"));
  dbusmenu_menuitem_property_set_bool(self->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->wireless_list_top);

  self->wireless_other = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->wireless_other,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Other Network..."));
  g_signal_connect(G_OBJECT(self->wireless_other),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(wireless_other_activated), self);
  dbusmenu_menuitem_child_append(self->root_menuitem,
				 self->wireless_other);

  dbusmenu_menuitem_child_append(self->root_menuitem,
				 self->wireless_separator);

  self->wireless_separator = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->wireless_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(self->root_menuitem,
				 self->wireless_separator);

  /* Mobile section */
  self->cellular_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->cellular_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Mobile: OFF"));
  dbusmenu_menuitem_property_set_bool(self->cellular_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->cellular_header);
  g_signal_connect(G_OBJECT(self->cellular_header),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_cellular_cb), self);

  self->cellular_separator = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(self->cellular_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->cellular_separator);

  /* Network settings */
  self->network_settings = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(self->network_settings,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Network Settings..."));
  g_signal_connect(G_OBJECT(self->network_settings),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(network_settings_activated), self);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->network_settings);

  /* "connmand not available" message */
  self->disabled_menuitem = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(self->disabled_menuitem,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("connmand not available"));
  dbusmenu_menuitem_property_set_bool(self->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_property_set_bool(self->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(self->root_menuitem, self->disabled_menuitem);

  return self;
}

void network_menu_free(struct network_menu *self)
{
  /* FIXME: free all menuitems */

  g_free(self);
}

