/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "service.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/menuitem.h>
#include <libdbusmenu-glib/client.h>

#include "indicator-network-service.h"
#include "connman-service.h"
#include "connman.h"
#include "network-menu.h"
#include "indicator-network-agent-client.h"

#define NETWORK_AGENT LIBEXECDIR"/indicator-network-agent"

struct service_state_string
{
  const gchar *str;
  ServiceState state;
};

static const struct service_state_string service_state_map[] = {
  { "unknown", SERVICE_STATE_UNKNOWN },
  { "idle", SERVICE_STATE_IDLE },
  { "failure", SERVICE_STATE_FAILURE },
  { "association", SERVICE_STATE_ASSOCIATION },
  { "configuration", SERVICE_STATE_CONFIGURATION },
  { "ready", SERVICE_STATE_READY },
  { "login", SERVICE_STATE_LOGIN },
  { "online", SERVICE_STATE_ONLINE },
  { "disconnect", SERVICE_STATE_DISCONNECT },
};

void service_connect_cb(DBusGProxy *proxy, GError *error, gpointer user_data)
{
  struct service *self = user_data;

  g_return_if_fail(self);

  if (error) {
    g_warning("failed to connect service %s: %s",
	      self->path, error->message);
    g_error_free(error);
    return;
  }
}

static void service_connect(struct service *self)
{
  g_return_if_fail(self);
  g_return_if_fail(self->proxy);

  g_debug("[%p] connect", self);

  /* FIXME: increase dbus timeout, this call might take a long time */
  org_moblin_connman_Service_connect_async(self->proxy, service_connect_cb,
					   self);
}

static void service_disconnect(struct service *self)
{
  GError *error = NULL;

  g_return_if_fail(self);
  g_return_if_fail(self->proxy);

  g_debug("[%p] disconnect", self);

  /* FIXME: use async */
  org_moblin_connman_Service_disconnect(self->proxy, &error);

  if (error) {
    g_warning("%s(): org_moblin_connman_Service_connect failed: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }
}

static void service_item_activated(DbusmenuMenuitem *mi, guint timestamp,
				   gpointer user_data)
{
  struct service *service = user_data;

  g_return_if_fail(service != NULL);
  g_return_if_fail(service->network_service != NULL);

  switch (service->state) {
  case SERVICE_STATE_IDLE:
  case SERVICE_STATE_FAILURE:
  case SERVICE_STATE_DISCONNECT:
    service_connect(service);
    break;
  case SERVICE_STATE_ASSOCIATION:
  case SERVICE_STATE_CONFIGURATION:
  case SERVICE_STATE_READY:
  case SERVICE_STATE_LOGIN:
  case SERVICE_STATE_ONLINE:
    service_disconnect(service);
    break;
  default:
    break;
  }
}

static ServiceState service_str2state(const gchar *state)
{
  const struct service_state_string *s;
  int i;

  for (i = 0; i < G_N_ELEMENTS(service_state_map); i++) {
    s = &service_state_map[i];
    if (g_str_equal(s->str, state))
      return s->state;
  }

  g_warning("%s(): unknown state %s", __func__, state);

  return SERVICE_STATE_IDLE;
}

static const gchar *service_state2str(ServiceState state)
{
  const struct service_state_string *s;
  int i;

  for (i = 0; i < G_N_ELEMENTS(service_state_map); i++) {
    s = &service_state_map[i];
    if (s->state == state)
      return s->str;
  }

  g_warning("%s(): unknown state %d", __func__, state);

  return "unknown";
}

static void service_update_label(struct service *service)
{
  g_return_if_fail(service);
  g_return_if_fail(service->menuitem);
  g_return_if_fail(service->name);
  g_return_if_fail(strlen(service->name) > 0);

  dbusmenu_menuitem_property_set(service->menuitem,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 service->name);
}

static void service_update_checkmark(struct service *service)
{
  gint state;

  g_return_if_fail(service);
  g_return_if_fail(service->menuitem);

  switch (service->state) {
  case SERVICE_STATE_READY:
  case SERVICE_STATE_LOGIN:
  case SERVICE_STATE_ONLINE:
    state = DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED;
    break;
  case SERVICE_STATE_IDLE:
  case SERVICE_STATE_FAILURE:
  case SERVICE_STATE_ASSOCIATION:
  case SERVICE_STATE_CONFIGURATION:
  default:
    state = DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED;
    break;
  }

  dbusmenu_menuitem_property_set_int(service->menuitem,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     state);
}

static void show_connect_error_cb(DBusGProxy *proxy, GError *error,
				     gpointer userdata)
{
  if (error) {
    g_warning("%s(): failed to show connection error: %s",
	      __func__, error->message);
    g_error_free(error);
  }
}

static void show_connect_error(struct service *service)
{
  DBusGProxy *proxy = service->network_service->agent_proxy;
  const gchar *error_type = "general-connect-error";

  if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_WIFI) == 0) {
    if (g_strcmp0(service->security, "none") == 0) {
      error_type = "wifi-open-connect-error";
    }
    else if (g_strcmp0(service->security, "wep") == 0) {
      if (service->state == SERVICE_STATE_CONFIGURATION)
	error_type = "wifi-wep-dhcp-error";
      else
	error_type = "wifi-wep-connect-error";
    }
    else if ((g_strcmp0(service->security, "wpa") == 0) ||
	     (g_strcmp0(service->security, "psk") == 0) ||
	     (g_strcmp0(service->security, "rsn") == 0)) {
      if (service->state == SERVICE_STATE_CONFIGURATION)
	error_type = "wifi-wpa-dhcp-error";
      else
	error_type = "wifi-wpa-connect-error";
    }
  }

  org_ayatana_indicator_network_agent_show_connect_error_async(proxy,
							       error_type,
							       show_connect_error_cb,
							       service);

}

static void service_update_state(struct service *service, const gchar *state)
{
  ServiceState new_state;
  const gchar *icon;

  g_return_if_fail(service);
  g_return_if_fail(service->type);
  g_return_if_fail(state);

  g_debug("[%p] %s -> %s", service, service_state2str(service->state),
	  state);

  new_state = service_str2state(state);

  /* only show state transitions */
  if (service->state == new_state)
    return;

  /*
   * if old state is unknown it means that we indicator-network-service just
   * started, so we should not send any notifications
   */
  if (service->state != SERVICE_STATE_UNKNOWN) {
    switch (new_state) {
    case SERVICE_STATE_READY:
      if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_WIFI))
	icon = ICON_NOTIFICATION_WIFI_FULL;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_ETHERNET))
	icon = ICON_NOTIFICATION_WIRED_CONNECTED;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_CELLULAR))
	icon = ICON_NOTIFICATION_CELLULAR_CONNECTED;
      else
	icon = NULL;
      network_service_notify(service->network_service, service->name,
			     _("Connection Established"), icon);
      break;
    case SERVICE_STATE_IDLE:
      if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_WIFI))
	icon = ICON_NOTIFICATION_WIFI_DISCONNECTED;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_ETHERNET))
	icon = ICON_NOTIFICATION_WIRED_DISCONNECTED;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_CELLULAR))
	icon = ICON_NOTIFICATION_CELLULAR_DISCONNECTED;
      else
	icon = ICON_NOTIFICATION_WIRED_DISCONNECTED;
      network_service_notify(service->network_service, service->name,
			     _("Disconnected"), icon);
      break;
    case SERVICE_STATE_FAILURE:
      if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_WIFI))
	icon = ICON_NOTIFICATION_WIFI_DISCONNECTED;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_ETHERNET))
	icon = ICON_NOTIFICATION_WIRED_DISCONNECTED;
      else if (g_str_equal(service->type, CONNMAN_TECHNOLOGY_CELLULAR))
	icon = ICON_NOTIFICATION_CELLULAR_DISCONNECTED;
      else
	icon = ICON_NOTIFICATION_WIRED_DISCONNECTED;
      network_service_notify(service->network_service, service->name,
			     _("Connection Failed"), icon);
      break;
    case SERVICE_STATE_ASSOCIATION:
    case SERVICE_STATE_CONFIGURATION:
    default:
      /* do nothing */
      break;
    }
  }

  if ((service->state == SERVICE_STATE_ASSOCIATION ||
       service->state == SERVICE_STATE_CONFIGURATION) &&
      new_state == SERVICE_STATE_FAILURE) {
    show_connect_error(service);
  }

  service->state = new_state;
  service_update_label(service);
  service_update_checkmark(service);

  network_service_state_changed(service->network_service, service);

  /*
   * network_service does not necessarily have the current signal strength,
   * so update it when the service has connected
   */
  if (service->state == SERVICE_STATE_READY)
    network_service_update_service_strength(service->network_service,
					    service);
}

static void service_update_icon(struct service *self)
{
  const gchar *icon;

  icon = network_service_icon_name(self->type, self->signal_strength);

  dbusmenu_menuitem_property_set(self->menuitem,
				 DBUSMENU_MENUITEM_PROP_ICON_NAME,
				 icon);
}

static void service_update_type(struct service *service, const gchar *type)
{
  g_return_if_fail(service);
  g_return_if_fail(type);

  g_free(service->type);
  service->type = g_strdup(type);

  g_debug("[%p] type %s", service, service->type);

  service_update_icon(service);
}

static void service_update_strength(struct service *service, GValue *value)
{
  guchar strength;

  g_return_if_fail(service);
  g_return_if_fail(value);

  strength = g_value_get_uchar(value);

  if (strength > 100)
    return;

  service->signal_strength = strength;
  service_update_icon(service);
  network_service_update_service_strength(service->network_service, service);
}

static void service_update_setup_required(struct service *service,
					  GValue *value)
{
  gboolean setup;
  gchar *argv[3];
  gchar *cmd = BINDIR "/indicator-network-mobile-wizard";

  g_return_if_fail(service);
  g_return_if_fail(value);
  g_return_if_fail(G_VALUE_HOLDS_BOOLEAN(value));

  setup = g_value_get_boolean(value);

  g_debug("[%p] setup required %d", service, setup);

  /* mobile wizard is only for cellular */
  if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_CELLULAR) != 0)
    return;

  if (!setup)
    return;

  argv[0] = cmd;
  argv[1] = service->path;
  argv[2] = 0;

  /*
   * FIXME: for the prototype just start the script, but with the proper UI
   * we use dbus
   */
  g_debug("%s(): starting %s %s", __func__, argv[0], argv[1]);
  g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
}

static void service_update_security(struct service *service, GValue *value)
{
  g_return_if_fail(service);
  g_return_if_fail(value);
  g_return_if_fail(G_VALUE_HOLDS_STRING(value));

  g_free(service->error);
  service->security = g_value_dup_string(value);
  if (service->security == NULL)
    g_warning("%s(): failed to allocate string for error", __func__);

  g_debug("[%p] security %s", service, service->security);
}

static void service_update_error(struct service *service, GValue *value)
{
  g_return_if_fail(service);
  g_return_if_fail(value);
  g_return_if_fail(G_VALUE_HOLDS_STRING(value));

  g_free(service->error);
  service->error = g_value_dup_string(value);
  if (service->error == NULL)
    g_warning("%s(): failed to allocate string for error", __func__);

  g_debug("[%p] error %s", service, service->error);
}

static void service_update_name(struct service *service, GValue *value)
{
  g_return_if_fail(service != NULL);
  g_return_if_fail(service->path != NULL);

  g_free(service->name);

  if (value && !G_VALUE_HOLDS_STRING(value))
    g_critical("%s(): value not a string", __func__);

  if (value && G_VALUE_HOLDS_STRING(value) &&
      strlen(g_value_get_string(value)) > 0)
    service->name = g_value_dup_string(value);
  else
    /* no valid name found for the service, use service path instead */
    service->name = g_strdup(service->path);

  g_debug("[%p] name %s", service, service->name);

  service_update_label(service);
}

static void service_get_properties(struct service *service)
{
  GHashTable *properties = NULL;
  GValue *value;
  GError *error = NULL;

  g_return_if_fail(service);
  g_return_if_fail(service->proxy);

  /* should this be async? */
  org_moblin_connman_Service_get_properties(service->proxy, &properties,
					    &error);

  if (error) {
    g_warning("%s(): org_moblin_connman_Service_get_properties failed: %s",
	      __func__, error->message);
    g_error_free(error);
    goto out;
  }

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_NAME);
  if (value)
    service_update_name(service, value);

  /* needs to be before service_update_state() */
  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_TYPE);
  if (value)
    service_update_type(service, g_value_get_string(value));

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_STATE);
  if (value)
    service_update_state(service, g_value_get_string(value));

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_STRENGTH);
  if (value)
    service_update_strength(service, value);

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_SETUP_REQUIRED);
  if (value)
    service_update_setup_required(service, value);

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_SECURITY);
  if (value)
    service_update_security(service, value);

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_ERROR);
  if (value)
    service_update_error(service, value);

 out:
  if (properties)
    g_hash_table_destroy(properties);
}

static void service_property_changed(DBusGProxy *proxy, const gchar *property,
				     GValue *value, gpointer user_data)
{
  struct service *service = user_data;

  if (g_str_equal(property, CONNMAN_PROPERTY_NAME)) {
    service_update_name(service, value);
  }

  if (g_str_equal(property, CONNMAN_PROPERTY_STATE)) {
    service_update_state(service, g_value_get_string(value));
  }

  if (g_str_equal(property, CONNMAN_PROPERTY_STRENGTH)) {
    service_update_strength(service, value);
  }

  if (g_strcmp0(property, CONNMAN_PROPERTY_SETUP_REQUIRED) == 0) {
    service_update_setup_required(service, value);
  }

  if (g_strcmp0(property, CONNMAN_PROPERTY_SECURITY) == 0) {
    service_update_security(service, value);
  }

  if (g_strcmp0(property, CONNMAN_PROPERTY_ERROR) == 0) {
    service_update_error(service, value);
  }
}

struct service *service_new(struct network_service *self, const gchar *path)
{
  struct service *service = NULL;
  GError *error = NULL;

  service = g_malloc0(sizeof(*service));
  if (!service)
    return NULL;

  service->path = g_strdup(path);
  if (!service->path) {
    goto err_service;
  }

  service->name = g_strdup(service->path);

  g_debug("[%p] service new %s", service, service->path);

  service->network_service = self;

  /* FIXME: follow also destroy signal */
  service->proxy = dbus_g_proxy_new_for_name_owner(self->system_bus,
						   CONNMAN_SERVICE,
						   path,
						   CONNMAN_SERVICE_INTERFACE,
						   &error);
  if (error != NULL) {
    g_warning("failed to create dbus proxy for service %s", path);
    goto err_service;
  }

  dbus_g_proxy_add_signal(service->proxy, CONNMAN_SIGNAL_PROPERTY_CHANGED,
			  G_TYPE_STRING, G_TYPE_VALUE, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(service->proxy,
			      CONNMAN_SIGNAL_PROPERTY_CHANGED,
			      G_CALLBACK(service_property_changed),
			      service, NULL);

  service->state = SERVICE_STATE_UNKNOWN;

  service->menuitem = dbusmenu_menuitem_new();
  if (!service->menuitem) {
    goto err_path;
  }

  dbusmenu_menuitem_property_set(service->menuitem,
				 DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,
				 DBUSMENU_MENUITEM_TOGGLE_CHECK);

  service_get_properties(service);

  g_signal_connect(G_OBJECT(service->menuitem),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(service_item_activated), service);

  return service;

 err_path:
  g_free(service->path);

 err_service:
  g_free(service);
  return NULL;
}

void service_free(struct service *service)
{
  g_debug("[%p] service free", service);

  g_free(service->path);
  g_free(service->name);
  g_free(service->type);
  g_free(service->error);
  g_free(service->security);

  if (service->menuitem) {
    g_object_unref(G_OBJECT(service->menuitem));
    service->menuitem = NULL;
  }

  g_object_unref(service->proxy);

  /* just a precaution */
  memset(service, 0, sizeof(*service));

  g_free(service);
}
