/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "services.h"

#include <glib.h>

#include "service.h"
#include "connman.h"
#include "network-menu.h"

struct services {
  GList *services;
  struct network_service *network_service;
};

static void services_free_all(GList *list)
{
  struct service *service;
  GList *iter;

  for (iter = list; iter != NULL; iter = iter->next) {
    service = iter->data;
    service_free(service);
  }

  g_list_free(list);
}

static GList *services_find(GList *list, const gchar *path)
{
  struct service *service;
  GList *iter;

  for (iter = list; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (g_strcmp0(path, service->path) == 0)
      return iter;
  }

  return NULL;
}

void services_free(struct services *self)
{
  services_free_all(self->services);
  self->services = NULL;
  g_free(self);
}

void services_remove_all(struct services *self)
{
  services_free_all(self->services);
  self->services = NULL;
}

struct services *services_new(struct network_service *ns)
{
  struct services *self;

  self = g_malloc0(sizeof(*self));
  self->services = NULL;
  self->network_service = ns;

  return self;
}

static void iterate_services(const GValue *value, gpointer user_data)
{
  GList **new_paths = user_data;
  gchar *path;

  path = g_value_dup_boxed (value);
  if (!path)
    return;

  *new_paths = g_list_append(*new_paths, path);
}

void services_update(struct services *self, GValue *value)
{
  GList *new_paths = NULL, *iter, *old_services, *link;
  struct service *service;
  gchar *path;
  gint count = 0;

  dbus_g_type_collection_value_iterate(value, iterate_services, &new_paths);

  old_services = self->services;
  self->services = NULL;

  for (iter = new_paths; iter != NULL; iter = iter->next) {
    path = iter->data;

    /* check if we already have this service and reuse it */
    link = services_find(old_services, path);
    if (link != NULL) {
      service = link->data;
      old_services = g_list_delete_link(old_services, link);
      link = NULL;
    } else {
      service = service_new(self->network_service, path);
      count++;
    }

    if (service == NULL)
      continue;

    self->services = g_list_append(self->services, service);
  }

  g_debug("services updated (total %d, new %d, removed %d)",
	  g_list_length(self->services), count, g_list_length(old_services));

  network_menu_services_updated(self->network_service->network_menu);

  link = g_list_first(self->services);
  if (link != NULL)
    service = link->data;
  else
    service = NULL;

  network_service_default_service_changed(self->network_service, service);

  /* remove unavailable services, we don't want to show them anymore */
  services_free_all(old_services);
  old_services = NULL;

  if (new_paths) {
    for (iter = new_paths; iter != NULL; iter = iter->next) {
      g_free(iter->data);
      iter->data = NULL;
    }

    g_list_free(new_paths);
    new_paths = NULL;
  }
}

GList *services_get_wired(struct services *self)
{
  GList *list = NULL, *iter;
  struct service *service;

  for (iter = self->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_ETHERNET) == 0) {
      list = g_list_append(list, service);
    }
  }

  return list;
}

GList *services_get_wireless(struct services *self)
{
  GList *list = NULL, *iter;
  struct service *service;

  for (iter = self->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_WIFI) == 0) {
      list = g_list_append(list, service);

      /* show only certain number of wireless networks */
      if (g_list_length(list) >= 5)
	break;
    }
  }

  return list;
}

GList *services_get_cellular(struct services *self)
{
  GList *list = NULL, *iter;
  struct service *service;

  for (iter = self->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_CELLULAR) == 0) {
      list = g_list_append(list, service);
    } else if (g_strcmp0(service->type, CONNMAN_TECHNOLOGY_BLUETOOTH) == 0) {
      list = g_list_append(list, service);
    }
  }

  return list;
}

gboolean services_is_connecting(struct services *self)
{
  struct service *service;
  GList *iter;

  for (iter = self->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (service->state == SERVICE_STATE_ASSOCIATION ||
	service->state == SERVICE_STATE_CONFIGURATION) {
      return TRUE;
    }
  }

  return FALSE;
}

gboolean services_is_connected(struct services *self)
{
  return services_get_connected(self) > 0;
}

guint services_get_connected(struct services *self)
{
  struct service *service;
  GList *iter;
  guint result = 0;

  for (iter = self->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (service->state == SERVICE_STATE_READY ||
	service->state == SERVICE_STATE_LOGIN ||
	service->state == SERVICE_STATE_ONLINE) {
      result++;
    }
  }

  return result;
}

const gchar *services_get_default_technology(struct services *self)
{
  struct service *service;
  GList *iter;

  iter = g_list_first(self->services);
  if (iter == NULL)
    return NULL;

  service = iter->data;
  return service->type;
}

