/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "wireless-connect-dialog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <locale.h>

#include "connman.h"
#include "connman-manager.h"
#include "connman-service.h"

G_DEFINE_TYPE(WirelessConnectDialog, wireless_connect_dialog, GTK_TYPE_DIALOG);

typedef struct _WirelessConnectDialogPriv WirelessConnectDialogPriv;

struct _WirelessConnectDialogPriv
{
  DBusGConnection *system_bus;
  DBusGProxy *connman_manager;
  DBusGProxy *service_proxy;
  GtkWidget *network_combo;
  GtkWidget *count_label;
  GtkWidget *security_combo;
  guint service_count;
  GtkListStore *store;
  DBusGProxyCall *services_call;
  DBusGProxyCall *connect_call;
  DBusGProxyCall *connect_service;
  gboolean service_selected;
  gboolean connecting;
};

#define WIRELESS_CONNECT_DIALOG_GET_PRIVATE(o)				\
  (G_TYPE_INSTANCE_GET_PRIVATE((o),					\
			       WIRELESS_CONNECT_DIALOG_TYPE,		\
			       WirelessConnectDialogPriv))

enum {
  COLUMN_PATH,
  COLUMN_NAME,
  COLUMN_SECURITY,
  N_COLUMNS,
};

static void wireless_connect_dialog_init(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GError *error = NULL;

  priv->system_bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);

  if (error) {
    g_critical("Unable to connect to the dbus system bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  if (priv->system_bus == NULL) {
    g_critical("%s(): failed to get system bus", __func__);
    return;
  }

  priv->connman_manager = NULL;
  priv->service_proxy = NULL;
  priv->network_combo = NULL;
  priv->security_combo = NULL;
  priv->count_label = NULL;
  priv->service_count = 0;
  priv->store = NULL;
  priv->services_call = NULL;
  priv->connect_call = NULL;
  priv->service_selected = FALSE;
  priv->connecting = FALSE;

  return;
}

static void wireless_connect_dialog_dispose(GObject *object)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(object);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  if (priv->service_proxy != NULL && priv->connect_call != NULL) {
    dbus_g_proxy_cancel_call(priv->service_proxy, priv->connect_call);
    priv->connect_call = NULL;
  }

  if (priv->connman_manager != NULL && priv->services_call != NULL) {
    dbus_g_proxy_cancel_call(priv->connman_manager, priv->services_call);
    priv->services_call = NULL;
  }

  if (priv->connman_manager != NULL && priv->connect_service != NULL) {
    dbus_g_proxy_cancel_call(priv->connman_manager, priv->connect_service);
    priv->connect_service = NULL;
  }

  if (priv->connman_manager != NULL) {
    g_object_unref(priv->connman_manager);
    priv->connman_manager = NULL;
  }

  if (priv->service_proxy != NULL) {
    g_object_unref(priv->service_proxy);
    priv->service_proxy = NULL;
  }

  G_OBJECT_CLASS(wireless_connect_dialog_parent_class)->dispose(object);
}

static void wireless_connect_dialog_finalize(GObject *object)
{
  G_OBJECT_CLASS(wireless_connect_dialog_parent_class)->finalize(object);
}

static void wireless_connect_dialog_class_init(WirelessConnectDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private(object_class, sizeof(WirelessConnectDialogPriv));

  object_class->dispose = wireless_connect_dialog_dispose;
  object_class->finalize = wireless_connect_dialog_finalize;

  g_assert(klass != NULL);
}

static void connect_service_cb(DBusGProxy *proxy, char *path,
			       GError *error, gpointer user_data)
{
  WirelessConnectDialog *self;
  WirelessConnectDialogPriv *priv;

  g_debug("%s", __func__);

  g_return_if_fail(user_data != NULL);

  self = WIRELESS_CONNECT_DIALOG(user_data);

  g_return_if_fail(self != NULL);
  g_return_if_fail(IS_WIRELESS_CONNECT_DIALOG(self));

  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  priv->connect_service = NULL;
  priv->connecting = FALSE;

  if (error) {
    /* FIXME: show warning to the user that the connection failed */
    g_warning("ConnectService() failed: %s",error->message);
    g_error_free(error);
  }

  g_free(path);
}

static void connect_service(WirelessConnectDialog *self, const gchar *name,
			    const gchar *security)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GHashTable *properties;
  DBusGProxyCall *call;
  GValue *value;

  g_debug("%s %s %s", __func__, name, security);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->system_bus != NULL);
  g_return_if_fail(priv->connman_manager != NULL);

  if (priv->connecting) {
    g_debug("%s(): already connecting", __func__);
    return;
  }

  if (name == NULL || strlen(name) == 0) {
    g_warning("%s(): invalid name", __func__);
    return;
  }

  if (security == NULL || strlen(security) == 0) {
    g_warning("%s(): invalid security", __func__);
    return;
  }

  properties = g_hash_table_new_full(g_str_hash, g_str_equal,
				     g_free, g_free);

  value = g_malloc0(sizeof(*value));
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, CONNMAN_TECHNOLOGY_WIFI);
  g_hash_table_insert(properties, g_strdup("Type"), value);

  value = g_malloc0(sizeof(*value));
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, "managed");
  g_hash_table_insert(properties, g_strdup("Mode"), value);

  value = g_malloc0(sizeof(*value));
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, name);
  g_hash_table_insert(properties, g_strdup("SSID"), value);

  value = g_malloc0(sizeof(*value));
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, security);
  g_hash_table_insert(properties, g_strdup("Security"), value);

  /* FIXME: how to ask passphrase? */
  /* g_hash_table_insert(properties, "Passphrase", value); */

  call = org_moblin_connman_Manager_connect_service_async(priv->connman_manager,
							  properties,
							  connect_service_cb,
							  self);
  priv->connect_service = call;

  g_hash_table_destroy(properties);
}

static void connect_cb(DBusGProxy *proxy, GError *error, gpointer user_data)
{
  WirelessConnectDialog *self;
  WirelessConnectDialogPriv *priv;

  self = WIRELESS_CONNECT_DIALOG(user_data);
  g_return_if_fail(self != NULL);
  g_return_if_fail(IS_WIRELESS_CONNECT_DIALOG(self));

  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  g_return_if_fail(priv != NULL);

  priv->connect_call = NULL;

  if (priv->service_proxy != NULL) {
    g_object_unref(priv->service_proxy);
    priv->service_proxy = NULL;
  }

  if (error) {
    /* FIXME: show warning to the user that the connection failed */
    g_warning("failed to connect service: %s",error->message);
    g_error_free(error);
    return;
  }
}

static void connect(WirelessConnectDialog *self, const gchar *path)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GError *error = NULL;
  DBusGProxy *proxy;

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->system_bus != NULL);

  if (priv->connecting) {
    g_debug("%s(): already connecting", __func__);
    return;
  }

  if (path == NULL || strlen(path) == 0) {
    g_warning("%s(): invalid path", __func__);
    return;
  }

  proxy = dbus_g_proxy_new_for_name_owner(priv->system_bus,
					  CONNMAN_SERVICE,
					  path,
					  CONNMAN_SERVICE_INTERFACE,
					  &error);

  if (error != NULL) {
    g_warning("failed to create dbus proxy for service %s: %s", path,
	      error->message);
    g_error_free(error);
    return;
  }

  if (proxy == NULL) {
    g_warning("failed to create dbus proxy for service %s", path);
    return;
  }

  priv->service_proxy = proxy;
  priv->connecting = TRUE;

  priv->connect_call = org_moblin_connman_Service_connect_async(proxy,
								connect_cb,
								self);
}

static void responded(GtkDialog *dialog, gint response_id, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GtkTreeIter iter;
  gboolean result;
  gchar *path, *name, *security;

  gtk_widget_hide_all(GTK_WIDGET(self));

  if (response_id != GTK_RESPONSE_ACCEPT)
    return;

  /* connect to the service */

  g_debug("%s(): %d", __func__, priv->service_selected);

  if (priv->service_selected) {
    result = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(priv->network_combo),
					   &iter);
    if (!result)
      return;

    gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		       COLUMN_PATH, &path, -1);
    connect(self, path);

    g_free(path);
    path = NULL;
  } else {
    name = gtk_combo_box_get_active_text(GTK_COMBO_BOX(priv->network_combo));
    security = gtk_combo_box_get_active_text(GTK_COMBO_BOX(priv->security_combo));
    connect_service(self, name, security);

    g_free(name);
    g_free(security);
    name = NULL;
    security = NULL;
  }
}

static int get_security_index(WirelessConnectDialog *self,
			       const gchar *security)
{
  if (g_strcmp0(security, CONNMAN_SECURITY_NONE) == 0)
    return 0;
  else if (g_strcmp0(security, CONNMAN_SECURITY_WEP) == 0)
    return 1;
  else if (g_strcmp0(security, CONNMAN_SECURITY_PSK) == 0)
    return 2;
  else if (g_strcmp0(security, CONNMAN_SECURITY_WPA) == 0)
    return 2;
  else if (g_strcmp0(security, CONNMAN_SECURITY_RSN) == 0)
    return 2;

  g_debug("unsupported security type: %s", security);
  return -1;
}


static void network_changed(GtkComboBox *combo, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GtkTreeIter iter;
  gboolean result;
  gchar *security;

  g_debug("%s()", __func__);

  result = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(priv->network_combo),
					 &iter);
  priv->service_selected = result;

  if (!result)
    return;

  gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		     COLUMN_SECURITY, &security, -1);

  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo),
			   get_security_index(self, security));

  g_free(security);
}

static void security_changed(GtkComboBox *combo, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GtkTreeIter iter;
  gboolean result;
  gchar *security;

  g_debug("%s()", __func__);

  result = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(priv->network_combo),
					 &iter);
  if (!result) {
    priv->service_selected = FALSE;
    return;
  }

  gtk_tree_model_get(GTK_TREE_MODEL(priv->store), &iter,
		     COLUMN_SECURITY, &security, -1);

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(priv->security_combo)) !=
      get_security_index(self, security))
    priv->service_selected = FALSE;
  else
    priv->service_selected = TRUE;

  g_free(security);
}

static void update_count_label(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  gchar buf[200];

  g_snprintf(buf, sizeof(buf), _("%d networks detected"), priv->service_count);
  gtk_label_set_text(GTK_LABEL(priv->count_label), buf);
}

static void iterate_array(gpointer data, gpointer user_data)
{
  WirelessConnectDialog *self = WIRELESS_CONNECT_DIALOG(user_data);
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  GValueArray *val_array = data;
  GHashTable *properties;
  const gchar *path, *name, *type, *security;
  GtkTreeIter iter;
  GValue *value;

  g_return_if_fail(val_array != NULL);

  if (val_array->n_values != 2)
    return;

  /* object path */
  value = g_value_array_get_nth(val_array, 0);
  g_return_if_fail(G_IS_VALUE(value));
  g_return_if_fail(G_VALUE_TYPE(value) == DBUS_TYPE_G_OBJECT_PATH);
  g_return_if_fail(G_VALUE_HOLDS_BOXED(value));

  path = g_value_get_boxed(value);
  if (!path)
    return;

  /* properties */
  value = g_value_array_get_nth(val_array, 1);
  g_return_if_fail(G_IS_VALUE(value));
  g_return_if_fail(G_VALUE_HOLDS_BOXED(value));

  properties = g_value_get_boxed(value);

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_TYPE);
  if (value == NULL)
    return;

  type = g_value_get_string(value);

  /* add only wifi services, drop others */
  if (g_strcmp0(type, CONNMAN_TECHNOLOGY_WIFI) != 0)
    return;

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_NAME);
  if (value == NULL)
    return;

  name = g_value_get_string(value);

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_SECURITY);
  if (value == NULL)
    return;

  security = g_value_get_string(value);

  gtk_list_store_append(priv->store, &iter);
  gtk_list_store_set(priv->store, &iter, COLUMN_PATH, path,
		     COLUMN_NAME, name, COLUMN_SECURITY, security, -1);

  priv->service_count++;
}

static void get_services_cb(DBusGProxy *proxy, GPtrArray *array, GError *error,
		     gpointer user_data)
{
  WirelessConnectDialog *self;
  WirelessConnectDialogPriv *priv;

  g_return_if_fail(user_data != NULL);
  self =  WIRELESS_CONNECT_DIALOG(user_data);

  g_return_if_fail(self != NULL);
  g_return_if_fail(IS_WIRELESS_CONNECT_DIALOG(self));

  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  g_return_if_fail(priv != NULL);

  priv->services_call = NULL;

  if (error) {
    g_warning("%s(): failed to get properties from ofono manager: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }

  gtk_list_store_clear(priv->store);
  priv->service_count = 0;

  g_ptr_array_foreach(array, iterate_array, self);
  g_ptr_array_free(array, TRUE);

  update_count_label(self);
}

static gboolean get_connman_manager(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  if (priv->connman_manager != NULL)
    return TRUE;

  /* FIXME: handle connman missing or crashing */
  priv->connman_manager = dbus_g_proxy_new_for_name(priv->system_bus,
						    CONNMAN_SERVICE,
						    CONNMAN_MANAGER_PATH,
						    CONNMAN_MANAGER_INTERFACE);
  if (priv->connman_manager == NULL)
    return FALSE;

  return TRUE;
}

static void get_services(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);
  DBusGProxyCall *call;

  if (priv->connman_manager == NULL)
    if (get_connman_manager(self) == FALSE)
      return;

  if (priv->services_call != NULL)
    /* call already in progress */
    return;

  call = org_moblin_connman_Manager_get_services_async(priv->connman_manager,
						       get_services_cb,
						       self);

  priv->services_call = call;
}

void wireless_connect_dialog_show(WirelessConnectDialog *self)
{
  WirelessConnectDialogPriv *priv;
  GtkWidget *entry;

  g_return_if_fail(self != NULL);
  g_return_if_fail(IS_WIRELESS_CONNECT_DIALOG(self));

  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  if (gtk_widget_get_visible(GTK_WIDGET(self)))
    return;

  /*
   * Delibretely updating services only when dialog is set to the visible
   * until we have solved service update usability issues.
   */
  get_services(self);

  priv->service_selected = FALSE;

  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->network_combo), -1);
  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo), 0);

  /* clear the entry inside GtkComboBoxEditor */
  entry = gtk_bin_get_child(GTK_BIN(priv->network_combo));

  g_return_if_fail(entry != NULL);
  g_return_if_fail(GTK_IS_ENTRY(entry));

  gtk_entry_set_text(GTK_ENTRY(entry), "");

  gtk_widget_show_all(GTK_WIDGET(self));
}

WirelessConnectDialog *wireless_connect_dialog_new(void)
{
  GtkWidget *vbox, *hbox, *label;
  WirelessConnectDialogPriv *priv;
  WirelessConnectDialog *self;
  GtkSizeGroup *group;

  self = g_object_new(WIRELESS_CONNECT_DIALOG_TYPE, NULL);
  priv = WIRELESS_CONNECT_DIALOG_GET_PRIVATE(self);

  gtk_window_set_title(GTK_WINDOW(self), _("Connect to Wireless Network"));

  /* FIXME: pre-select connect button once user has made a selection */
  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			 GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT,
			 NULL);

  vbox = gtk_dialog_get_content_area(GTK_DIALOG(self));

  group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 5);

  label = gtk_label_new(_("Network name:"));
  gtk_size_group_add_widget(group, label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  priv->store = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING,
				   G_TYPE_STRING);
  priv->network_combo = gtk_combo_box_entry_new_with_model(GTK_TREE_MODEL(priv->store),
							   COLUMN_NAME);
  g_signal_connect(G_OBJECT(priv->network_combo), "changed",
		   G_CALLBACK(network_changed), self);
  gtk_box_pack_start(GTK_BOX(hbox), priv->network_combo, TRUE, TRUE, 5);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 5);

  label = gtk_label_new("");
  gtk_size_group_add_widget(group, label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  priv->count_label = gtk_label_new(_("0 networks detected"));
  update_count_label(self);
  gtk_box_pack_start(GTK_BOX(hbox), priv->count_label, TRUE, TRUE, 5);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 5);

  label = gtk_label_new(_("Wireless security:"));
  gtk_size_group_add_widget(group, label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  priv->security_combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(priv->security_combo),
			    CONNMAN_SECURITY_NONE);
  gtk_combo_box_append_text(GTK_COMBO_BOX(priv->security_combo),
			    CONNMAN_SECURITY_WEP);
  gtk_combo_box_append_text(GTK_COMBO_BOX(priv->security_combo),
			    CONNMAN_SECURITY_PSK);
  gtk_combo_box_set_active(GTK_COMBO_BOX(priv->security_combo), 0);
  g_signal_connect(G_OBJECT(priv->security_combo), "changed",
		   G_CALLBACK(security_changed), self);
  gtk_box_pack_start(GTK_BOX(hbox), priv->security_combo, TRUE, TRUE, 5);

  g_signal_connect(G_OBJECT(self), "response", G_CALLBACK(responded),
		   self);

  return self;
}
