#!/usr/bin/env python

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#  DigiClockScreenlet- (c) 2008 by Helder Fraga aka Whise <helder.fraga@hotmail.com>

import screenlets
from screenlets.options import ColorOption , BoolOption, FontOption
from screenlets import DefaultMenuItem , sensors
import pango
import gobject
import random
import gtk
import cairo
import gconf
#use gettext for translation
import gettext

_ = screenlets.utils.get_translator(__file__)

def tdoc(obj):
	obj.__doc__ = _(obj.__doc__)
	return obj

@tdoc

class DigiClockScreenlet (screenlets.Screenlet):
	"""A digital clock screenlet."""
	
	# default meta-info for Screenlets (should be removed and put into metainfo)
	__name__	= 'DigiClockScreenlet'
	__version__	= '0.2.1'
	__author__	= 'Whise and Aantn'
	__desc__	= __doc__	# set description to docstring of class
	
	# initialize attributes
	gradient = None
	time = ''
	date = ''
	period = _('AM')

	use_gradient = True
	use_ampm = True

	font = "Sans Normal 12"
	color_text = (1,1,1,0.8)
	frame_color = (1,1,1,0.8)
	frame_color_selected = (1,1,1,0.8)

	def __init__ (self, **keyword_args):
		"""Constructor"""
		#call super (width/height MUST match the size of graphics in the theme)
		screenlets.Screenlet.__init__(self, width=150, height=70, 
			uses_theme=True,ask_on_option_override=False, **keyword_args)
		
		# initialize attributes
		self.gradient = None
		self.time = ''
		self.date = ''
		self.period = _('AM')
		
		# add Editable Options (options that are editable through the UI and automatically load and save themselves)
		self.add_options_group(_('Options'), _('Options'))
		
		self.add_option(BoolOption(_('Options'),'use_gradient', 
			True, _('Use Gradient'), _('Use a gradient for the background'), callback=self.create_gradient))
		self.add_option(ColorOption(_('Options'),'frame_color', 
			(1, 1, 1, 1), _('Frame color'), _('Frame color'), callback=self.create_gradient))
		self.add_option(BoolOption(_('Options'), 'use_ampm', 
			False, _('Use am/pm'),_('Use am/pm')))
		self.add_option(FontOption(_('Options'),'font', 
			_("FreeSans"), _('Text Font'), _('Text font')))
		self.add_option(ColorOption(_('Options'),'color_text', 
			(0, 0, 0, 0.6), _('Text color'), ''))
		
		# add the default menuitems
		self.add_default_menuitems()
		
		# create the cairo gradient for the first time
		if self.use_gradient:
			self.create_gradient()

		# add a one second timer
		self.update()
		self.timer = gobject.timeout_add(1000, self.update)
	
	def create_gradient (self):
		"""Create the cairo gradient used for drawing the screenlet."""
		self.gradient = cairo.LinearGradient(0, self.height*2, 0, 0)
		self.gradient.add_color_stop_rgba(1,*self.frame_color)
		self.gradient.add_color_stop_rgba(0.7, self.frame_color[0], self.frame_color[1],
			self.frame_color[2], 1.5-self.frame_color[3])
	
	def update (self):
		"""Timeout called every second to update the time and redraw the screenlet."""
		if self.use_ampm:
			self.time = str(sensors.cal_get_hour12()) + ':' + str(sensors.cal_get_minute()) + ':' + str(sensors.cal_get_second())
			hour = sensors.cal_get_hour()
			if int(hour)>12: 
				self.period = _('PM')
			else:
				self.period = _('AM')
		else:
			self.time = str(sensors.cal_get_hour24()) + ':' + str(sensors.cal_get_minute()) + ':' + str(sensors.cal_get_second())
		self.date = str(sensors.cal_get_local_date())
		self.redraw_canvas()
		return True # keep on running this event
	
	def on_draw (self, ctx):
		"""Called automatically to (re)draw the screenlet."""
		ctx.scale(self.scale, self.scale)
		
		if self.use_gradient and self.gradient is not None:
			ctx.set_source(self.gradient)
		else:
			ctx.set_source_rgba(*self.frame_color)
			
		self.draw_rectangle_advanced (ctx, 0, 0, self.width-12, self.height-12,
			rounded_angles=(5,5,5,5), fill=True, border_size=2, border_color=(0,0,0,0.5),
			shadow_size=6, shadow_color=(0,0,0,0.5))
		ctx.set_source_rgba(self.color_text[0], self.color_text[0], self.color_text[0], 0.05)
		
		self.draw_text(ctx,'88:88:88', 0, 20, self.font.split(' ')[0], 21, self.width, pango.ALIGN_CENTER)
		ctx.set_source_rgba(*self.color_text)

		if not self.use_ampm:
			self.draw_text(ctx, self.time, 0, 20, self.font.split(' ')[0], 21, self.width, pango.ALIGN_CENTER)
		else:
			self.draw_text(ctx, self.time, 0, 20, self.font.split(' ')[0], 21, self.width, pango.ALIGN_CENTER)
			self.draw_text(ctx, self.period, -8, 36, self.font.split(' ')[0], 6, self.width, pango.ALIGN_RIGHT)
		
		ctx.set_source_rgba(self.color_text[0],self.color_text[0],self.color_text[0],0.05)
		self.draw_text(ctx, '88:88:88', 0, 46, self.font.split(' ')[0], 9, self.width, pango.ALIGN_CENTER)
		ctx.set_source_rgba(*self.color_text)
		self.draw_text(ctx, self.date, 0, 46, self.font.split(' ')[0], 9, self.width, pango.ALIGN_CENTER)		

	def on_draw_shape (self, ctx):
		self.on_draw(ctx)
	
# If the program is run directly or passed as an argument to the python
# interpreter then create a Screenlet instance and show it
if __name__ == "__main__":
	# create new session
	import screenlets.session
	screenlets.session.create_session(DigiClockScreenlet)

