/***************************************************************************
                         ac3dfile.cpp  -  description
                            -------------------
   begin                : Sun Aug 20 2000
   copyright            : (C) 2000 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ac3dfile.h"
#include <vector>
#include <iostream.h>

extern "C"
{
   FilePlugin * i3d_createInstance()
   {
      return new AC3DFile();
   }
}

AC3DFile::AC3DFile() : FilePlugin( "AC3D", "ac" )
{
   setImportable( true );
   setExportable( true );

}

AC3DFile::~AC3DFile()
{}

void AC3DFile::importData( ifstream &in )
{
   char buffer[ 256 ];
   char line[ 1024 ];

   in >> buffer >> ws;  //read in the header

   getToken( buffer, in );

   while ( !in.eof() )
   {
      if ( strcmp( buffer, "MATERIAL" ) == 0 )
         readMaterial( in );
      else if ( strcmp( buffer, "OBJECT" ) == 0 )
         readObject( in );
      else
         in.getline( line, sizeof( line ), '\n' ); //ignore

      getToken( buffer, in );
   }

}

void AC3DFile::readObject( ifstream &in )
{
   Mesh * o = new Mesh();
   char buffer[ 256 ];
   char line[ 1024 ];

   in >> buffer >> ws; //read in the object tag;
   in >> buffer >> ws; //read in the object id tag;

   //reset any texrepeats or offsets;
   m_texRepeatU = 1;
   m_texRepeatV = 1;
   m_texOffsetU = 0;
   m_texOffsetV = 0;
   //ignore the world;
   if ( strcmp( buffer, "world" ) == 0 || strcmp( buffer, "group" ) == 0 )
   {
      in.getline( line, sizeof( line ), '\n' );
      return ;
   }

   getToken( buffer, in );

   while ( !in.eof() && ( strcmp( buffer, "OBJECT" ) != 0 ) )
   {
      if ( strcmp( buffer, "name" ) == 0 )
         readObjectName( o, in );
      else if ( strcmp( buffer, "data" ) == 0 )
         readObjectData( o, in );
      else if ( strcmp( buffer, "texture" ) == 0 )
         readObjectTexture( o, in );
      else if ( strcmp( buffer, "texrep" ) == 0 )
         readObjectTextureRep( o, in );
      else if ( strcmp( buffer, "texoff" ) == 0 )
         readObjectTextureOff( o, in );
      else if ( strcmp( buffer, "rot" ) == 0 )
         readObjectRotation( o, in );
      else if ( strcmp( buffer, "loc" ) == 0 )
         readObjectLocation( o, in );
      else if ( strcmp( buffer, "url" ) == 0 )
         readObjectURL( o, in );
      else if ( strcmp( buffer, "numvert" ) == 0 )
         readObjectVerts( o, in );
      else if ( strcmp( buffer, "numsurf" ) == 0 )
         readObjectSurfaces( o, in );
      else if ( strcmp( buffer, "kids" ) == 0 )
         readObjectKids( o, in );
      else
         in.getline( line, sizeof( line ), '\n' ); //ignore the tag.

      getToken( buffer, in );
   }

   o->normalize();
   addEntity( o );

}

void AC3DFile::readObjectName( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in.getline( line, sizeof( line ), '\n' );

}

void AC3DFile::readObjectData( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in.getline( line, sizeof( line ), '\n' );
}

void AC3DFile::readObjectTexture( Object *o, ifstream &in )
{
   //just ignore for now
   char * line;
   char buffer[ 1024 ];


   line = ( char * ) malloc( 1024 * sizeof( char ) );

   in >> buffer >> line >> ws;
   //strip off the prefix and suffix quotes.
   int i = strlen( line );
   line[ i - 1 ] = '\0';

   //increment past the first "
   line++;

   cerr << "Reading texture:" << line << endl;
   o->setTextureMaterial( I3D::getMatEditor() ->createMaterial( line ) );

}

void AC3DFile::readObjectTextureRep( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in >> line >> m_texRepeatU >> m_texRepeatV >> ws;
}

void AC3DFile::readObjectTextureOff( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in >> line >> m_texOffsetU >> m_texOffsetV >> ws;
}

void AC3DFile::readObjectRotation( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in.getline( line, sizeof( line ), '\n' );
}

void AC3DFile::readObjectLocation( Object *o, ifstream &in )
{
   char buffer[ 256 ];
   float x, y, z;
   in >> buffer >> x >> y >> z >> ws;
   o->setPosition( x, y, z );
}

void AC3DFile::readObjectURL( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in.getline( line, sizeof( line ), '\n' );
}

void AC3DFile::readObjectVerts( Object *o, ifstream &in )
{
   char buffer[ 256 ];
   int n;
   float x, y, z;
   in >> buffer >> n >> ws; //read in the number of vertices;
   for ( int i = 0; i < n; i++ )
   {
      in >> x >> y >> z >> ws;
      o -> createVertex( x, y, z );
   }
}

void AC3DFile::readObjectSurfaces( Object *o, ifstream &in )
{
   char buffer[ 256 ];
   char line[ 1024 ];
   int n;

   in >> buffer >> n >> ws; //read in the number of surfaces.

   for ( int i = 0; i < n; i++ )
   {
      getToken( buffer, in );

      while ( strcmp( buffer, "kids" ) != 0 )
      {
         if ( strcmp( buffer, "SURF" ) == 0 )
            in.getline( line, sizeof( line ), '\n' );
         else if ( strcmp( buffer, "mat" ) == 0 )
            in.getline( line, sizeof( line ), '\n' );
         else if ( strcmp( buffer, "refs" ) == 0 )
            readObjectSurfaceRefs( o, in );
         else
            in.getline( line, sizeof( line ), '\n' );

         getToken( buffer, in );
      }
   }
}

void AC3DFile::readObjectSurfaceRefs( Object *o, ifstream &in )
{
   char buffer[ 256 ];
   int index, n;

   vector<int> verts;
   vector<Vector4> uvs;

   in >> buffer >> n >> ws; //read in the number of verts.

   verts.reserve( n );
   uvs.reserve( n );

   Vector4 uv;

   for ( int i = 0; i < n; i++ )
   {
      in >> index >> uv.x >> uv.y >> ws;
      uv.x = uv.x * m_texRepeatU + m_texOffsetU;
      uv.y = uv.y * m_texRepeatV + m_texOffsetV;
      verts.push_back( index );
      uvs.push_back( uv );
   }

   Face *f = o -> createFace( verts );

   for ( int i = 0; i < ( int ) verts.size(); i++ )
   {
      f->setUVCoord( i, uvs[ i ] );
   }

}

void AC3DFile::readObjectKids( Object *o, ifstream &in )
{
   //just ignore for now
   char line[ 1024 ];
   in.getline( line, sizeof( line ), '\n' );
}

/**Returns the next token while leaving the stream position the same */
void AC3DFile::getToken( char *buffer, ifstream &in )
{
   streampos pos = in.tellg();
   in >> buffer >> ws;
   in.seekg( pos );
}


void AC3DFile::exportData( ofstream &out )
{
   vector<Selectable *> *list;
   list = getMeshes();

   int size = ( int ) list->size();

   out << "AC3Db" << endl;

   writeMaterials( out );

   out << "OBJECT world" << endl;
   out << "kids " << size << endl;

   for ( int i = 0; i < size; i++ )
   {
      writeObject( ( Object * ) ( *list ) [ i ], out );

   }
}

void AC3DFile::writeObject( Object *o, ofstream &out )
{
   out << "OBJECT poly" << endl;
   writeObjectName( o, out );
   writeObjectData( o, out );
   writeObjectTexture( o, out );
   writeObjectTextureRep( o, out );
   writeObjectRotation( o, out );
   writeObjectLocation( o, out );
   writeObjectURL( o, out );
   writeObjectVerts( o, out );
   writeObjectSurfaces( o, out );
   writeObjectKids( o, out );
}

void AC3DFile::writeObjectVerts( Object *o, ofstream &out )
{
   VertexList * vlist = o->getVerts();
   int size = ( int ) vlist->size();
   Vector4 pos;

   out << "numvert " << size << endl;

   for ( int i = 0; i < size; i++ )
   {
      pos = ( *vlist ) [ i ] ->getPosition();
      out << pos.x << " " << pos.y << " " << pos.z << endl;
   }
}

void AC3DFile::writeObjectSurfaces( Object * o, ofstream & out )
{
   int size = ( int ) o->numFaces();
   // Vector4 pos;

   out << "numsurf " << size << endl;

   for ( int i = 0; i < size; i++ )
   {
      out << "SURF 0x10" << endl;

      if ( o->getTextureMaterial() != 0 )
      {
         out << "mat " << I3D::getDB() ->getMaterialIndex( o->getTextureMaterial() ) << endl;
      }

      writeObjectSurfaceRefs( o->getFace( i ), out );
   }
}

void AC3DFile::writeObjectSurfaceRefs( Face *f, ofstream &out )
{
   vector<int> *flist = f->getVerts();
   vector<int> *uvlist = f->getUVs();

   int size = ( int ) flist->size();

   out << "refs " << size << endl;
   Vector4 uv;

   for ( int i = 0; i < size; i++ )
   {
      uv = f->getParentObject() ->getUVCoord( ( *uvlist ) [ i ] ) ->getPosition();
      out << ( *flist ) [ i ] << " " << uv.x << " " << uv.y << endl;
   }
}

void AC3DFile::writeObjectLocation( Object *o, ofstream &out )
{
   Vector4 pos;
   pos = o->getPosition();
   out << "loc " << pos.x << " " << pos.y << " " << pos.z << endl;
}

void AC3DFile::writeObjectRotation( Object *o, ofstream &out )
{}

void AC3DFile::writeObjectTexture( Object *o, ofstream &out )
{
   TextureMaterial * tm = o->getTextureMaterial();

   if ( tm != 0 && tm->texture != 0 )
   {
      out << "texture \"" << tm->texture->getFilename() ->ascii() << "\"" << endl;
   }
}

void AC3DFile::writeObjectTextureRep( Object *o, ofstream &out )
{}

void AC3DFile::writeObjectKids( Object *o, ofstream &out )
{
   out << "kids 0" << endl;
}

void AC3DFile::writeObjectName( Object *o, ofstream &out )
{
   out << "name \"jon\"" << endl;
}

void AC3DFile::writeObjectData( Object *o, ofstream &out )
{
   out << "data \"jon\"" << endl;
}

void AC3DFile::writeObjectURL( Object *o, ofstream &out )
{
}

void AC3DFile::readMaterial( ifstream &in )
{
   char line[ 1024 ];

   in.getline( line, sizeof( line ), '\n' );
   /*char buffer[256];

   TextureMaterial *tm = new TextureMaterial();

   float r, g, b;

   in >> buffer >> ws; //read in the material tag
   in >> buffer >> ws; //read in the name.

   in >> buffer >> ws; //read in the color tag;
   in >> r >> g >> b;
   tm->cDiffuse.r = r*255;
   tm->cDiffuse.g = g*255;
   tm->cDiffuse.b = b*255;

   in >> buffer >> ws; //read in the color tag;
   in >> r >> g >> b;
   tm->cAmbient.r = r*255;
   tm->cAmbient.g = g*255;
   tm->cAmbient.b = b*255;

   in >> buffer >> ws; //read in the color tag;
   in >> r >> g >> b;
   tm->cEmissive.r = r*255;
   tm->cEmissive.g = g*255;
   tm->cEmissive.b = b*255;

   in >> buffer >> ws; //read in the color tag;
   in >> r >> g >> b;
   tm->cSpecular.r = r*255;
   tm->cSpecular.g = g*255;
   tm->cSpecular.b = b*255;

   in >> buffer >> ws; //read in shininess tag;
   in >> tm->shininess >> ws;

   in >> buffer >> ws; //read in shininess tag;
   in >> r >> ws;
   tm->alpha = r * 255;

            */
}

void AC3DFile::writeMaterials( ofstream &out )
{
   //save materials here.

   ObjectDB* odb = I3D::getDB();
   assert(odb);

   TextureMaterial *m;

   for ( int i = 0; i < odb->numMaterials(); i++ )
   {
      m = odb->getMaterial(i);
      out << "MATERIAL \"\""; // << (*materials)[i]->getName()<<" ";
      out << "rgb " << ( float ) m->cDiffuse.r / 255 << " " << ( float ) m->cDiffuse.g / 255 << " " << ( float ) m->cDiffuse.b / 255 << " ";
      out << "amb " << ( float ) m->cAmbient.r / 255 << " " << ( float ) m->cAmbient.g / 255 << " " << ( float ) m->cAmbient.b / 255 << " ";
      out << "emis " << ( float ) m->cEmission.r / 255 << " " << ( float ) m->cEmission.g / 255 << " " << ( float ) m->cEmission.b / 255 << " ";
      out << "spec " << ( float ) m->cSpecular.r / 255 << " " << ( float ) m->cSpecular.g / 255 << " " << ( float ) m->cSpecular.b / 255 << " ";
      out << "shi " << ( float ) m->shininess << " ";
      out << "trans " << ( float ) ( 255 - m->alpha ) / 255 << endl;
   }

}
