/***************************************************************************
                         ribplugin.cpp  -  description
                            -------------------
   begin                : Sun Aug 20 2000
   copyright            : (C) 2000 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "ribplugin.h"
#include <qfiledialog.h>
#include <qvalidator.h>
#include <string>

extern "C"
{
   DialogPlugin * i3d_createInstance()
   {
      return new RIBPlugin();
   }
}

RIBPlugin::RIBPlugin() : DialogPlugin( "Renderman" ), CRIBDialog()
{
   fldWidth -> setValidator( new QIntValidator( 0, 2000, fldWidth ) );
   fldHeight -> setValidator( new QIntValidator( 0, 2000, fldHeight ) );
}

RIBPlugin::~RIBPlugin()
{
   // no need to delete child widgets, Qt does it all for us
}

void RIBPlugin::slotHelp()
{
   qWarning( "RIBPlugin::help(): Not implemented yet!" );
}

void RIBPlugin::slotRender()
{
   qWarning( "RIBPlugin::render(): Not implemented yet!" );

   getOptions();

   // saveTexturesAsTiffs();


}

void RIBPlugin::getOptions()
{

   height = fldHeight -> text().toInt();
   width = fldWidth -> text().toInt();

   toScreen = chScreen -> isChecked();
   toFile = chFile -> isChecked();

   strRendribPath = fldRendRibPath -> text();
   strShaderPath = fldShaderPath -> text();
   strSaveToPath = fldSaveToPath -> text();

   strName = fldName -> text();


}

/** Saves the current options as the default
 */
void RIBPlugin::slotSave()
{

   getOptions();
   //saveTexturesAsTiffs();

   QString filename;
   filename = strSaveToPath + "/" + strName + ".rib";

   ofstream out( filename.ascii(), ios::out );

   if ( !out )
   {
      cerr << "Cannot open file '" << filename.ascii() << "'" << endl;
      return ;
   }

   out << "##RenderMan RIB-Structure 10\n";
   out << "version 3.03\n\n";
   out << "Option \"searchpath\" \"shader\" [\".:";
   out << strShaderPath.ascii() << ":/usr/local/BMRT2.5/shaders:&\"]\n";
   out << "Display \"" << strName.ascii() << ".tif\" \"file\" \"rgba\"\n";
   out << "Format " << width << " " << height << " -1\n";
   out << "PixelSamples 1 1\n";
   out << "Projection \"perspective\" \"fov\" 60\n";


   /**
     * For now, the camera and the lights are hard coded
     */

   //set up the camera
   out << "Rotate -45 1 0 0\n";
   out << "Translate 0 -10 10\n";

   out << "WorldBegin\n\n";

   out << "LightSource \"ambientlight\" 1 \"intensity\" 0.5\n\n";


   out << "Declare \"shadows\" \"string\"\n";
   out << "Attribute \"light\" \"shadows\" \"on\"\n";
   out << "LightSource \"distantlight\" 1 \"from\" [0 -1 -4] \"to\" [0 0 0] \"intensity\" 0.8\n\n";

   cerr << "saved header/camera/lights" << endl;

   vector<Selectable *> list;

   if ( getMeshes() != 0 )
   {
      list = * getMeshes();

      for ( uint i = 0; i < list.size(); i++ )
      {
         cerr << "Saving mesh" << endl;
         saveMesh( static_cast<Mesh *>( list[ i ] ), out );
      }

      cerr << "Done with meshes" << endl;
   }


   if ( getSurfaces() != 0 )
   {
      list = * getSurfaces();

      for ( uint i = 0; i < list.size(); i++ )
      {
         cerr << "Saving surface" << endl;
         saveSurface( static_cast<NurbsSurface *>( list[ i ] ), out );
      }

      cerr << "Done with surfaces" << endl;
   }

   out << "\nWorldEnd\n";
   cerr << "Done with renderman" << endl;
   out.close();

}



void RIBPlugin:: startPlugin()
{
   show();
}

void RIBPlugin::stopPlugin()
{

   hide();
}

void RIBPlugin::saveMesh( Mesh *m, ofstream &out )
{
   Vector4 p;
   Quat q;
   Vector4 r;
   p = m -> getPosition();
   q = m -> getOrientation();

   r = q.getEuler();

   out << "AttributeBegin\n";
   out << "Surface \"supertexmap\"\n";
   out << "Color [1 1 1 ]\n";
   out << "Rotate " << r.x << " 1 0 0\n";
   out << "Rotate " << r.y << " 0 1 0\n";
   out << "Rotate " << r.z << " 0 0 1\n";
   out << "Translate " << p.x << " " << p.y << " " << - p.z << "\n";
   // out << "Rotate 20, 0, 0, 1\n";

   int num_faces = m -> numFaces();
   cerr << "writing faces:" << num_faces << endl;

   for ( int i = 0; i < num_faces; i++ )
   {
      cerr << i << ".";

      out << "Polygon ";

      Face *f = m -> getFace( i );
      int num_verts = f -> getNumVerts();
      //write out the points.
      out << "\"P\" [";

      for ( int j = 0; j < num_verts; j++ )
      {
         Vertex *v = f -> getVertPtr( j ) ;
         Vector4 p = v -> getPosition();
         out << p.x << " " << p.y << " " << -p.z << " ";
      }

      out << "]\n";

      //write out the normals.
      out << "\"N\" [";

      for ( int j = 0; j < num_verts; j++ )
      {
         Vertex *v = f -> getVertPtr( j ) ;
         Vector4 p = v -> getNormal();
         out << p.x << " " << p.y << " " << p.z << " ";
      }

      out << "]\n";

      //write out the UVs.
      out << "\"st\" [";

      for ( int j = 0; j < num_verts; j++ )
      {
         UVCoord *v = f -> getUVPtr( j ) ;
         Vector4 p = v -> getPosition();
         out << p.x << " " << 1 - p.y << " ";
      }

      out << "]\n";




   }

   out << "AttributeEnd\n\n";
}

void RIBPlugin::saveSurface( NurbsSurface *s, ofstream &out )
{
   Vector4 p;
   Quat q;
   Vector4 r;

   p = s -> getPosition();
   q = s -> getOrientation();
   r = q.getEuler();

   out << "AttributeBegin\n";
   out << "Surface \"plastic\"\n";
   out << "Color [1 1 1 ]\n";

   out << "Rotate " << r.x << " 1 0 0\n";
   out << "Rotate " << r.y << " 0 1 0\n";
   out << "Rotate " << r.z << " 0 0 1\n";
   out << "Translate " << p.x << " " << p.y << " " << - p.z << "\n";

   int nu, nv;
   int udegree, vdegree;
   int uorder, vorder;
   vector<float> uknots, vknots;

   nu = s -> getNumUPoints();
   nv = s -> getNumVPoints();

   udegree = s -> getUDegree();
   vdegree = s -> getVDegree();

   uknots = s -> getUKnots();
   vknots = s -> getVKnots();

   uorder = udegree + 1;
   vorder = vdegree + 1;
   //write out u information
   out << "NuPatch " << nu << " " << uorder << "[";

   for ( uint i = 0; i < uknots.size(); i++ )
      out << uknots[ i ] << " ";

   out << "] 0 1\n" ;

   //write out v information
   out << nv << " " << vorder << "[";

   for ( uint i = 0; i < vknots.size(); i++ )
      out << vknots[ i ] << " ";

   out << "] 0 1\n";

   //write out control points;
   out << "\"Pw\" [";

   for ( int i = 0; i < nu; i++ )
   {
      for ( int j = 0; j < nv; j++ )
      {
         Vertex *v = s -> getVertex( i * nu + j );
         Vector4 p = v -> getPosition();
         out << p.x << " " << p.y << " " << -p.z << " 1 " ;
      }

      out << "\n";
   }

   out << "]\n";

   out << "AttributeEnd\n\n";

}

void RIBPlugin::saveHeader( ofstream &out )
{
}

void RIBPlugin::saveTexturesAsTiffs( )
{
   cerr << "saveTextures: not implemented" << endl;
}

void saveCamera( Camera *, ofstream & )
{
   cerr << "saveCamera: not implemented" << endl;
}

void saveLight( Light *, ofstream & )
{
   cerr << "saveLight: not implemented" << endl;
}

void RIBPlugin::slotRendRibBrowse()
{
   QString dir = QFileDialog::getExistingDirectory( "" );

   if ( dir.isEmpty() )
      return ;

   fldRendRibPath -> setText( dir );
}

void RIBPlugin::slotShaderPathBrowse()
{

   QString dir = QFileDialog::getExistingDirectory( "" );

   if ( dir.isEmpty() )
      return ;

   fldShaderPath -> setText( dir );

}

void RIBPlugin::slotSavePathBrowse()
{
   QString dir = QFileDialog::getExistingDirectory( "" );

   if ( dir.isEmpty() )
      return ;

   fldSaveToPath -> setText( dir );

}
