// --------------------------------------------------------------------
// AppUi
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipevisitor.h"
#include "ipepage.h"
#include "iperef.h"
#include "ipestyle.h"
#include "ipeiml.h"
#include "ipepath.h"
#include "ipeutils.h"

#include "ipepdfdoc.h"
#include "ipeversion.h"
#include "ipeq.h"
#include "ipefilter.h"
#include "ipefontpool.h"

#include "appui.h"
#include "dialogs.h"
#include "debugwin.h"
#include "props.h"
#include "styles.h"

#include "ipeoverlay.h"
#include "ipecreatetext.h"
#include "ipeeditpath.h"

#include <qaccel.h>
#include <qaction.h>
#include <qapplication.h>
#include <qbuttongroup.h>
#include <qclipboard.h>
#include <qcolordialog.h>
#include <qcombobox.h>
#include <qdatetime.h>
#include <qdragobject.h>
#include <qvalidator.h>
#include <qfiledialog.h>
#include <qinputdialog.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qmainwindow.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qtextbrowser.h>
#include <qtextstream.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qwidgetstack.h>

QPixmap ipeIcon(const char* name);
QPixmap penguinIcon(int width);

// --------------------------------------------------------------------

class IpeSpinBox : public QSpinBox {
public:
  IpeSpinBox(QWidget *parent, const char *name);
  void Set(int value, int maxValue);
};

IpeSpinBox::IpeSpinBox(QWidget *parent, const char *name)
  : QSpinBox(1, 1, 1, parent, name)
{
  // nothing yet
}

//! This does not emit any valueChanged() signal.
void IpeSpinBox::Set(int val, int maxVal)
{
  // set value to 1 first, in case maxVal is less than current value
  directSetValue(1);
  setMaxValue(maxVal);
  directSetValue(val);
  updateDisplay();
}

// --------------------------------------------------------------------

class DecimalSpinBox : public QSpinBox {
public:
  DecimalSpinBox(int minValue, int maxValue, int step, QWidget *parent = 0);
  void setValueFromAttribute(int value);
protected:
  virtual QString mapValueToText(int value);
  virtual int mapTextToValue(bool *ok);
};

DecimalSpinBox::DecimalSpinBox(int minValue, int maxValue, int step,
			       QWidget *parent)
  : QSpinBox(minValue, maxValue, step, parent)
{
  setValidator(new QDoubleValidator(minValue / 10.0,
				    maxValue / 10.0, 1, this));
}

QString DecimalSpinBox::mapValueToText(int value)
{
  return QString("%1.%2").arg(value / 10).arg(value % 10, 1);
}

int DecimalSpinBox::mapTextToValue(bool *ok)
{
  *ok = false;
  QString str = cleanText();
  IpeLex lex(IpeQ(str));
  double d = lex.GetDouble();
  lex.SkipWhitespace();
  if (lex.Eos()) {
    *ok = true;
    return int(d * 10.0);
  }
  return 0;
}

void DecimalSpinBox::setValueFromAttribute(int value)
{
  directSetValue(value);
  updateDisplay();
}

// --------------------------------------------------------------------

ZoomSpinBox::ZoomSpinBox(QWidget *parent)
  : QSpinBox(10, 1000, 1, parent)
{
  // nothing
}

void ZoomSpinBox::stepUp()
{
  int res = value();
  res = int(res * 1.3);
  if (res > maxValue())
    res = maxValue();
  setValue(res);
}

void ZoomSpinBox::stepDown()
{
  int res = value();
  res = int(res / 1.3);
  if (res < minValue())
    res = minValue();
  setValue(res);
}

bool ZoomSpinBox::eventFilter(QObject* obj, QEvent* ev)
{
  // if ( obj != vi )
  // return FALSE;

  if ( ev->type() == QEvent::KeyPress ) {
    QKeyEvent* k = (QKeyEvent*) ev;
    int key = k->key();

    if (!(key == Key_Up || key == Key_Down || key == Key_Return ||
	  key == Key_Backspace || key == Key_Delete ||
	  (Key_0 <= key && key <= Key_9)))
      // stop event from being delivered
      return TRUE;
  }
  // else leave it to spin box to decide
  return QSpinBox::eventFilter(obj, ev);
}

// --------------------------------------------------------------------

class LayerBox : public QListBox {
public:
  LayerBox(QWidget *parent, const char *name) :
    QListBox(parent, name) { /* nothing else */ }
  virtual void mousePressEvent(QMouseEvent *e);
};

void LayerBox::mousePressEvent(QMouseEvent *e)
{
  QListBoxItem *i = itemAt(e->pos());
  // ignore mouse outside items
  if (!i)
    return;
  if (e->button() == LeftButton) {
    QListBox::mousePressEvent(e);
  } else
    emit rightButtonPressed(i, e->globalPos());
}

// --------------------------------------------------------------------

const char * fileNewText = QT_TR_NOOP(
"<qt><img source=\"new\"> "
"This button creates a new document. "
"You can also select the <b>New</b> command from the File menu.</qt>");
const char * fileOpenText = QT_TR_NOOP(
"<qt><img source=\"open\"> "
"This button opens a document from a file. "
"You can also select the <b>Open</b> command from the File menu.</qt>");
const char * fileSaveText = QT_TR_NOOP(
"<qt><img source=\"save\"> "
"This button saves the current document. If it is a new document, "
"you will be prompted for a file name. "
"You can also select the <b>Save</b> command from the File menu.</qt>");
const char * cutcopyText = QT_TR_NOOP(
"<qt>Cut <img source=\"cut\">, copy <img source=\"copy\">, and paste "
"<img source=\"paste\"> pass Ipe objects to and from the system clipboard. "
"When pasting objects, they are inserted into the active layer. "
"You can examine Ipe objects by copying them to the clipboard, and "
"pasting into your favorite text editor.</qt>");
const char * strokeColorText = QT_TR_NOOP(
"<qt>Select the <b>stroke color</b> here."
"<p>The stroke color is used to outline the shape of objects, to draw "
"lines, curves, arcs, marks, and for text.  To draw filled objects without "
"an outline, set the stroke color to <tt>void</tt>.</p></qt>");
const char * fillColorText = QT_TR_NOOP(
"<qt>Select the <b>fill color</b> here."
"<p>The fill color is used for the interior of objects such as polygons or "
"circles.  It is not used for text or marks.  To draw unfilled objects, "
"that is, polygons or circles with an outline only, set the fill "
"color to <tt>void</tt>.</p></qt>");
const char * resolutionText = QT_TR_NOOP(
"<qt>You control the resolution of the screen display here. "
"The number indicates the number of screen pixels being used to "
"display one inch in the document."
"<p>By setting this to the actual screen resolution (that is, "
"the number of pixels per inch of your monitor), you can view a "
"document in its original size "
"(the size it will have when printed).</p></qt>");
const char * pageNumberText = QT_TR_NOOP(
"<qt>The current page number.</qt>");
const char * viewNumberText = QT_TR_NOOP(
"<qt>Each page of an Ipe document can consist of several <em>views</em>. "
"Each view becomes one PDF/Postscript page when saving in that format. "
"A view is defined by selecting some layers of the page that will be "
"presented together.</qt>");
const char * lineWidthText = QT_TR_NOOP(
"<qt>This setting determines the width of lines you create. "
"If you change the setting while you have objects "
"selected, the line width of the selected objects will be changed "
"accordingly.</qt>");
const char * lineDashText = QT_TR_NOOP(
"<qt>This setting determines the dash-dot pattern of lines that you create. "
"If you change the setting while you have objects "
"selected, the dash-dot pattern of the selected objects will be changed "
"accordingly.</qt>");
const char * textSizeText = QT_TR_NOOP(
"<qt>This setting determines the font size of text objects that you create. "
"If you change the setting while you have text objects "
"selected, their font size will be changed "
"accordingly.</qt>");
const char * arrowText = QT_TR_NOOP(
"<qt>This setting determines the arrow heads on polyline, spline, or arc "
"objects that you create.  If you change the setting while you have objects "
"selected, the arrows on the selected objects will be changed "
"accordingly.</qt>");
const char * arrowSizeText = QT_TR_NOOP(
"<qt>This setting determines the size of arrow heads for objects that you "
"create.  If you change the setting while you have objects selected, "
"the arrow heads on the selected objects will be resized.</qt>");
const char * markShapeText = QT_TR_NOOP(
"<qt>This setting determines the shape of mark objects that you "
"create. If you change the setting while you have mark objects selected, "
"their shape will be changed.</qt>");
const char * markSizeText = QT_TR_NOOP(
"<qt>This setting determines the size of mark objects that you "
"create.  If you change the setting while you have mark objects selected, "
"the marks will be resized.</qt>");
const char * gridSizeText = QT_TR_NOOP(
"<qt>This setting determines the distance between grid points in "
"the snapping grid.</qt");
const char * angleSizeText = QT_TR_NOOP(
"<qt>This setting determines the angle between consecutive lines "
"in the angular snapping mode.</qt>");
const char * layerListText = QT_TR_NOOP(
"<qt>This list shows the layers of the current page. "
"<p>The selected layers are in the current <em>view</em>. "
"When you create new objects, they are inserted into the <em>active layer</em>, "
"indicated by the angular brackets. "
"<p>The letters H, D, L, S stand for <em>hidden</em>, <em>dimmed</em>, "
"<em>locked</em>, and <em>no snapping</em>.</p>"
"</qt>");
const char * penguinText = QT_TR_NOOP(
"<qt>This is a picture of penguins."
"<p>It is not related to Tux, the Linux-penguin. "
"In fact, Ipe has used a penguin icon since 1993, "
"long before I had ever heard of Linux.</p></qt>");
const char * snapText = QT_TR_NOOP (
"<qt>These buttons select the various snapping modes.  When the button "
"is pressed in, the mode is on, otherwise it is off. From left to right, "
"the buttons enable the following modes:"
"<ul><li>Snapping to object vertices,"
"<li>Snapping to path boundaries,"
"<li>Snapping to intersection points (currently only of straight segments),"
"<li>Snapping to the grid,"
"<li>Angular snapping,"
"<li>Automatic angular snapping."
"</ul>See the manual for more details about snapping.</qt>");
const char * canvasText = QT_TR_NOOP(
"<qt>This is the canvas, where you create your drawing."
"<p>Most operations can be done with the left mouse button. Its "
"function depends on the current mode.</p>"
"<p>The right mouse button pops up the object menu (you can change it "
"to select objects in the <em>Preferences</em> dialog - in that case "
"you get the object menu by holding the Control key with the right "
"mouse button).</p>"
"<p>The following mouse shortcuts are available:"
"<ul><li>Ctrl+Left mouse: Stretch"
"<li>Ctrl+Shift+Left mouse: Scale"
"<li>Middle mouse: Move"
"<li>Ctrl+Middle mouse: Rotate"
"<li>Shift+Middle mouse: Pan"
"<li>Shift+Ctrl+Middle mouse: Horizontal/vertical move"
"</ul></p></qt>");

// --------------------------------------------------------------------

const char * aboutText =
"<qt><h1>%1</h1>"
"<p>(c) 1993-2004 Otfried Cheong</p>"
"<p>The extensible drawing editor <b>Ipe</b> creates figures "
"in Postscript and PDF format, "
"using <b>PdfLaTeX</b> to format the text in the figures.</p>"
"<p>%1 includes code from <b>Xpdf</b> by Derek B. Noonburg to "
"parse PDF documents (www.foolabs.com/xpdf).</p>"
"<p>Furthermore, Ipe relies on the following fine libraries:"
"<ul>"
"<li> The GUI toolkit <b>Qt</b> (www.trolltech.com)"
"<li> The font rendering library <b>freetype 2</b> (www.freetype.org)"
"<li> The compression library <b>zlib</b> (www.gzip.org/zlib)"
"</ul>"
"<p>%1 is released under the GNU Public License.</p>"
"<p>See <tt>http://ipe.compgeom.org</tt> for details.</p>"
"</qt>";

// --------------------------------------------------------------------

/*! \class AppUi
  \brief AppUi encapsulates the Ipe user interface.

*/
/*! The constructor does not create an Ipe document.
  Clients need to call either Load or NewDoc after construction,
  before calling show. */
AppUi::AppUi(QWidget* parent, const char* name, WFlags f)
  : QMainWindow(parent,name,f), iKeyTranslator(0)
{
  QPixmap newIcon = ipeIcon("new");
  QPixmap openIcon = ipeIcon("open");
  QPixmap saveIcon = ipeIcon("save");
  QPixmap undoIcon = ipeIcon("undo");
  QPixmap redoIcon = ipeIcon("redo");
  QPixmap cutIcon = ipeIcon("cut");
  QPixmap copyIcon = ipeIcon("copy");
  QPixmap pasteIcon = ipeIcon("paste");

  QMimeSourceFactory::defaultFactory()->setPixmap("new", newIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("open", openIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("save", saveIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("undo", undoIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("redo", undoIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("cut", cutIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("copy", copyIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("paste", pasteIcon);

  IpePreferences *prefs = IpePreferences::Static();

  // --------------------------------------------------------------------
  // File tool bar
  // --------------------------------------------------------------------

  setUsesBigPixmaps(prefs->iBigToolButtons);

  iFileTools = new QToolBar( this, "file tools" );
  iFileTools->setLabel(tr("File tools"));

  QToolButton *fileNew
    = new QToolButton(newIcon, tr("New document"), QString::null,
		      this, SLOT(NewDoc()), iFileTools, "new document" );
  QWhatsThis::add(fileNew, tr(fileNewText));

  QToolButton *fileOpen
    = new QToolButton(openIcon, tr("Open file"), QString::null,
		      this, SLOT(Load()), iFileTools, "open file" );
  QWhatsThis::add(fileOpen, tr(fileOpenText));

  QToolButton *fileSave
    = new QToolButton(saveIcon, tr("Save file"), QString::null,
		      this, SLOT(Save()), iFileTools, "save file" );
  QWhatsThis::add(fileSave, tr(fileSaveText));

  iFileTools->addSeparator();

  QToolButton *editCut
    = new QToolButton(cutIcon, tr("Cut"), QString::null,
		      this, SLOT(Cut()), iFileTools, "cut" );
  QWhatsThis::add(editCut, tr(cutcopyText));

  QToolButton *editCopy
    = new QToolButton(copyIcon, tr("Copy"), QString::null,
		      this, SLOT(Copy()), iFileTools, "copy" );
  QWhatsThis::add(editCopy, tr(cutcopyText));

  QToolButton *editPaste
    = new QToolButton(pasteIcon, tr("Paste"), QString::null,
		      this, SLOT(Paste()), iFileTools, "paste" );
  QWhatsThis::add(editPaste, tr(cutcopyText));

  iFileTools->addSeparator();

  (void) QWhatsThis::whatsThisButton(iFileTools);

  // --------------------------------------------------------------------
  // Resolution tool bar
  // --------------------------------------------------------------------

  iResolutionTools = new QToolBar(this, "resolution tools");
  iResolutionTools->setLabel(tr("Resolution tool"));

  iResolution = new ZoomSpinBox(iResolutionTools);
  QToolTip::add(iResolution, tr("Screen resolution"));
  QWhatsThis::add(iResolution, tr(resolutionText));
  iResolution->setSuffix(tr(" dpi"));
  connect(iResolution, SIGNAL(valueChanged(int)),
	  this, SLOT(ResolutionChanged(int)));
  iResolution->setFocusPolicy(NoFocus);

  // --------------------------------------------------------------------
  // Page tool bar
  // --------------------------------------------------------------------

  iPageTools = new QToolBar(this, "page tools");
  iPageTools->setLabel(tr("Page tools"));

  iPageNumber = new IpeSpinBox(iPageTools, "page number");
  QToolTip::add(iPageNumber, tr("Current page"));
  QWhatsThis::add(iPageNumber, tr(pageNumberText));
  iPageNumber->setPrefix(tr("Page "));
  connect(iPageNumber, SIGNAL(valueChanged(int)),
	  this, SLOT(PageChanged(int)));
  iPageNumber->setFocusPolicy(NoFocus);

  iViewNumber = new IpeSpinBox(iPageTools, "view number");
  QToolTip::add(iViewNumber, tr("Current view"));
  QWhatsThis::add(iViewNumber, tr(viewNumberText));
  iViewNumber->setPrefix(tr("View "));
  connect(iViewNumber, SIGNAL(valueChanged(int)),
	  this, SLOT(ViewChanged(int)));
  iViewNumber->setFocusPolicy(NoFocus);

  // --------------------------------------------------------------------
  // Color tool bar
  // --------------------------------------------------------------------

  iColorTools = new QToolBar(this, "color tools");
  iColorTools->setLabel(tr("Color tools"));

  iStrokeColorStack = new QWidgetStack(iColorTools);
  QToolTip::add(iStrokeColorStack, tr("Stroke color"));
  QWhatsThis::add(iStrokeColorStack, tr(strokeColorText));
  iStrokeColor = new QComboBox(false);
  connect(iStrokeColor, SIGNAL(activated(int)),
	  SLOT(SetStrokeColorName(int)));
  iAbsStrokeColor = new QPushButton(tr("Stroke"), 0);
  connect(iAbsStrokeColor, SIGNAL(clicked()),
	  this, SLOT(SetStrokeColor()));
  iStrokeColorStack->addWidget(iStrokeColor, 0);
  iStrokeColorStack->addWidget(iAbsStrokeColor, 1);

  iFillColorStack = new QWidgetStack(iColorTools);
  QToolTip::add(iFillColorStack, tr("Fill color"));
  QWhatsThis::add(iFillColorStack, tr(fillColorText));
  iFillColor = new QComboBox(false);
  connect(iFillColor, SIGNAL(activated(int)),
	  SLOT(SetFillColorName(int)));
  iAbsFillColor = new QPushButton(tr("Fill"), 0);
  connect(iAbsFillColor, SIGNAL(clicked()),
	  this, SLOT(SetFillColor()));
  iFillColorStack->addWidget(iFillColor, 0);
  iFillColorStack->addWidget(iAbsFillColor, 1);

  // --------------------------------------------------------------------
  // Line width/style/arrow tool bar
  // --------------------------------------------------------------------

  iLineTools = new QToolBar(this, "line tools");
  iLineTools->setLabel(tr("Line style settings"));

  iLineWidthStack = new QWidgetStack(iLineTools);
  QToolTip::add(iLineWidthStack, tr("Line width"));
  QWhatsThis::add(iLineWidthStack, tr(lineWidthText));
  iLineWidth = new QComboBox(false);
  connect(iLineWidth, SIGNAL(activated(int)),
	  SLOT(LineWidthChanged(int)));
  iAbsLineWidth = new DecimalSpinBox(0, 1000, 2);
  connect(iAbsLineWidth, SIGNAL(valueChanged(int)),
	  SLOT(AbsLineWidthChanged(int)));
  iLineWidthStack->addWidget(iLineWidth, 0);
  iLineWidthStack->addWidget(iAbsLineWidth, 1);

  iDashStyle = new QComboBox(false, iLineTools);
  QToolTip::add(iDashStyle, tr("Line dash pattern"));
  QWhatsThis::add(iDashStyle, tr(lineDashText));
  connect(iDashStyle, SIGNAL(activated(int)),
	  SLOT(DashStyleChanged(int)));

  QPixmap arrowIcon1 = ipeIcon("arrowNone");
  QPixmap arrowIcon2 = ipeIcon("arrowRight");
  QPixmap arrowIcon3 = ipeIcon("arrowLeft");
  QPixmap arrowIcon4 = ipeIcon("arrowBoth");
  iArrow = new QComboBox(false, iLineTools);
  QToolTip::add(iArrow, tr("Arrow"));
  QWhatsThis::add(iArrow, tr(arrowText));
  iArrow->insertItem(arrowIcon1);
  iArrow->insertItem(arrowIcon2);
  iArrow->insertItem(arrowIcon3);
  iArrow->insertItem(arrowIcon4);
  connect(iArrow, SIGNAL(activated(int)),
	  SLOT(ArrowChanged(int)));
  iAttributes.iForwardArrow = false;
  iAttributes.iBackwardArrow = false;

  iArrowSizeStack = new QWidgetStack(iLineTools);
  QToolTip::add(iArrowSizeStack, tr("Arrow size"));
  QWhatsThis::add(iArrowSizeStack, tr(arrowSizeText));
  iArrowSize = new QComboBox(false);
  connect(iArrowSize, SIGNAL(activated(int)),
	  SLOT(ArrowSizeChanged(int)));
  iAbsArrowSize = new DecimalSpinBox(2, 1000, 10);
  connect(iAbsArrowSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsArrowSizeChanged(int)));
  iArrowSizeStack->addWidget(iArrowSize, 0);
  iArrowSizeStack->addWidget(iAbsArrowSize, 1);


  // --------------------------------------------------------------------
  // Mark and text sizes
  // --------------------------------------------------------------------

  iSizeTools = new QToolBar(this, "size tools");
  iSizeTools->setLabel(tr("Mark and text size settings"));

  QPixmap markIcon = ipeIcon("marks");
  iMarkShape = new QComboBox(false, iSizeTools, "mark shape");
  QToolTip::add(iMarkShape, tr("Mark shape"));
  QWhatsThis::add(iMarkShape, tr(markShapeText));
  connect(iMarkShape, SIGNAL(activated(int)),
	  SLOT(MarkShapeChanged(int)));
  iMarkShape->insertItem(markIcon, tr("Circle"));
  iMarkShape->insertItem(markIcon, tr("Disc"));
  iMarkShape->insertItem(markIcon, tr("Box"));
  iMarkShape->insertItem(markIcon, tr("Square"));
  iMarkShape->insertItem(markIcon, tr("Cross"));
  iAttributes.iMarkShape = iMarkShape->currentItem() + 1;

  iMarkSizeStack = new QWidgetStack(iSizeTools);
  QToolTip::add(iMarkSizeStack, tr("Mark size"));
  QWhatsThis::add(iMarkSizeStack, tr(markSizeText));
  iMarkSize = new QComboBox(false);
  connect(iMarkSize, SIGNAL(activated(int)),
	  SLOT(MarkSizeChanged(int)));
  iAbsMarkSize = new DecimalSpinBox(2, 1000, 10);
  connect(iAbsMarkSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsMarkSizeChanged(int)));
  iMarkSizeStack->addWidget(iMarkSize, 0);
  iMarkSizeStack->addWidget(iAbsMarkSize, 1);

  iTextSizeStack = new QWidgetStack(iSizeTools);
  QToolTip::add(iTextSizeStack, tr("Font size"));
  QWhatsThis::add(iTextSizeStack, tr(textSizeText));
  iTextSize = new QComboBox(false);
  connect(iTextSize, SIGNAL(activated(int)),
	  SLOT(TextSizeChanged(int)));
  iAbsTextSize = new DecimalSpinBox(40, 5000, 1);
  connect(iAbsTextSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsTextSizeChanged(int)));
  iTextSizeStack->addWidget(iTextSize, 0);
  iTextSizeStack->addWidget(iAbsTextSize, 1);

  // --------------------------------------------------------------------
  // Snap tool bar
  // --------------------------------------------------------------------

  QPixmap snapVtxIcon = ipeIcon("snapvtx");
  QPixmap snapBdIcon = ipeIcon("snapbd");
  QPixmap snapIntIcon = ipeIcon("snapint");
  QPixmap snapGridIcon = ipeIcon("snapgrid");
  QPixmap snapAngleIcon = ipeIcon("snapangle");
  QPixmap snapAutoIcon = ipeIcon("snapauto");

  QMimeSourceFactory::defaultFactory()->setPixmap("snapvtx", snapVtxIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("snapbd", snapVtxIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("snapint", snapVtxIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("snapgrid", snapGridIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("snapangle", snapAngleIcon);
  QMimeSourceFactory::defaultFactory()->setPixmap("snapauto", snapAutoIcon);

  iSnapTools = new QToolBar(this, "snap tools");
  iSnapTools->setLabel(tr("Snap tools"));

  iSnapActionGroup = new QActionGroup(this, 0, false);

  iSnapAction[0] = new QAction(iSnapActionGroup);
  iSnapAction[0]->setText(tr("Snap to vertices"));
  iSnapAction[0]->setAccel(Key("F4", "Snap|Vertices"));
  iSnapAction[0]->setIconSet(snapVtxIcon);
  iSnapAction[0]->addTo(iSnapTools);

  iSnapAction[1] = new QAction(iSnapActionGroup);
  iSnapAction[1]->setText(tr("Snap to boundary"));
  iSnapAction[1]->setAccel(Key("F5", "Snap|Boundary"));
  iSnapAction[1]->setIconSet(snapBdIcon);
  iSnapAction[1]->addTo(iSnapTools);

  iSnapAction[2] = new QAction(iSnapActionGroup);
  iSnapAction[2]->setText(tr("Snap to intersection"));
  iSnapAction[2]->setAccel(Key("F6", "Snap|Intersections"));
  iSnapAction[2]->setIconSet(snapIntIcon);
  iSnapAction[2]->addTo(iSnapTools);

  iSnapTools->addSeparator();

  iSnapAction[3] = new QAction(iSnapActionGroup);
  iSnapAction[3]->setText(tr("Snap to grid"));
  iSnapAction[3]->setAccel(Key("F7", "Snap|Grid"));
  iSnapAction[3]->setIconSet(snapGridIcon);
  iSnapAction[3]->addTo(iSnapTools);

  iGridSizeStack = new QWidgetStack(iSnapTools);
  QToolTip::add(iGridSizeStack, tr("Grid size"));
  QWhatsThis::add(iGridSizeStack, tr(gridSizeText));
  iGridSize = new QComboBox(false);
  connect(iGridSize, SIGNAL(activated(int)),
	  SLOT(GridSizeChanged(int)));
  iAbsGridSize = new DecimalSpinBox(10, 1000, 20);
  connect(iAbsGridSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsGridSizeChanged(int)));
  iGridSizeStack->addWidget(iGridSize, 0);
  iGridSizeStack->addWidget(iAbsGridSize, 1);

  iSnapTools->addSeparator();

  iSnapAction[4] = new QAction(iSnapActionGroup);
  iSnapAction[4]->setText(tr("Angular snap"));
  iSnapAction[4]->setAccel(Key("F8", "Snap|Angular"));
  iSnapAction[4]->setIconSet(snapAngleIcon);
  iSnapAction[4]->addTo(iSnapTools);

  iSnapAction[5] = new QAction(iSnapActionGroup);
  iSnapAction[5]->setText(tr("Automatic angular snap"));
  iSnapAction[5]->setAccel(Key("F9", "Snap|Automatic angular"));
  iSnapAction[5]->setIconSet(snapAutoIcon);
  iSnapAction[5]->addTo(iSnapTools);

  iAngleSizeStack = new QWidgetStack(iSnapTools);
  QToolTip::add(iAngleSizeStack, tr("Angular snapping angle"));
  QWhatsThis::add(iAngleSizeStack, tr(angleSizeText));
  iAngleSize = new QComboBox(false);
  connect(iAngleSize, SIGNAL(activated(int)),
	  SLOT(AngleSizeChanged(int)));
  iAbsAngleSize = new DecimalSpinBox(30, 3600, 20);
  connect(iAbsAngleSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsAngleSizeChanged(int)));
  iAngleSizeStack->addWidget(iAngleSize, 0);
  iAngleSizeStack->addWidget(iAbsAngleSize, 1);

  for (int i = 0; i < 6; ++i) {
    iSnapAction[i]->setToggleAction(true);
    iSnapAction[i]->setWhatsThis(tr(snapText));
    connect(iSnapAction[i], SIGNAL(toggled(bool)),
	    this, SLOT(SnapChanged(bool)));
  }

  // ------------------------------------------------------------
  // Layer toolbar
  // ------------------------------------------------------------

  iLayerTools = new QToolBar(tr("Layer tools"), this,
			     QMainWindow::Left, false, "layer tools");
  iLayerTools->setVerticalStretchable(true);
  iLayerList = new LayerBox(iLayerTools, "layer list box");
  iLayerList->setSelectionMode(QListBox::Multi);
  iLayerList->setFocusPolicy(NoFocus);
  QToolTip::add(iLayerList, tr("Layers of this page"));
  QWhatsThis::add(iLayerList, tr(layerListText));

  QLabel *penguins = new QLabel(iLayerTools);
  penguins->setPixmap(penguinIcon(iLayerList->sizeHint().width()));
  QWhatsThis::add(penguins, tr(penguinText));

  iLayerTools->setStretchableWidget(iLayerList);

  // --------------------------------------------------------------------
  // Object creators tool bar
  // --------------------------------------------------------------------

  iObjectTools = new QToolBar(this, "mode tools");
  iObjectTools->setLabel(tr("Mode tools"));

  iModeActionGroup = new QActionGroup(this);
  connect(iModeActionGroup, SIGNAL(selected(QAction*)),
	  SLOT(SetCreator(QAction*)));

  for (int i = 0; i < IpeOverlayFactory::Num; ++i) {
    QPixmap icon = ipeIcon(IpeOverlayFactory::PixmapName(i));
    iModeAction[i] = new QAction(iModeActionGroup);
    iModeAction[i]->setText(IpeOverlayFactory::Tooltip(i));
    iModeAction[i]->setIconSet(icon);
    iModeAction[i]->setWhatsThis(tr(IpeOverlayFactory::WhatsThis(i)));
    iModeAction[i]->setAccel(Key(IpeOverlayFactory::Shortcut(i),
				 IpeOverlayFactory::Context(i)));
    iModeAction[i]->setToggleAction(true);
    iModeAction[i]->addTo(iObjectTools);
    if (i == 4)
      iObjectTools->addSeparator();
  }

  // ------------------------------------------------------------
  // Menu
  // ------------------------------------------------------------

  QMenuBar* menu = menuBar();
  int id;

  iFileMenu = new QPopupMenu;
  iFileMenu->insertItem(tr("New &Window"), this, SLOT(NewWindow()),
			Key("none", "File|New Window"));

  id = iFileMenu->insertItem(newIcon, tr("&New"), this, SLOT(NewDoc()),
			     Key("Ctrl+N", "File|New"));
  iFileMenu->setWhatsThis(id, fileNewText );

  id = iFileMenu->insertItem(openIcon, tr("&Open"),
			     this, SLOT(Load()),
			     Key("Ctrl+O", "File|Open"));
  iFileMenu->setWhatsThis(id, fileOpenText );

  id = iFileMenu->insertItem(saveIcon, tr("&Save"), this, SLOT(Save()),
			     Key("Ctrl+S", "File|Save"));
  iFileMenu->setWhatsThis(id, fileSaveText );

  id = iFileMenu->insertItem(tr("Save &as..."), this, SLOT(SaveAs()),
			     Key("none", "File|Save as"));
  iFileMenu->setWhatsThis(id, fileSaveText );

  iFileMenu->insertItem(tr("Save as &bitmap"), this, SLOT(SaveAsBitmap()),
			Key("none", "File|Save as bitmap"));
  iFileMenu->insertSeparator();

  iFileMenu->insertItem(tr("Run &Latex"), this, SLOT(RunLatex(int)),
			Key("Ctrl+L", "File|Run Latex"), 1);
  iFileMenu->insertSeparator();

  iFileMenu->insertItem(tr("&Close"), this, SLOT(close()),
			Key("Ctrl+Q", "File|Close"));
  iFileMenu->insertItem(tr("E&xit"), qApp, SLOT(closeAllWindows()),
			Key("none", "File|Exit"));
  menu->insertItem(tr("&File"), iFileMenu);

  iEditMenu = new QPopupMenu;
  iEditMenu->insertItem(tr("&Absolute attributes"),
			this, SLOT(AbsoluteAttributes()),
			Key("Ctrl+T", "Edit|Absolute attributes"),
			EAbsoluteAttributesMenuId);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(undoIcon, tr("&Undo"),
			this, SLOT(UndoCmd(int)),
			Key("Ctrl+Z", "Edit|Undo"), ECmdUndo);
  iEditMenu->insertItem(redoIcon, tr("&Redo"),
			this, SLOT(UndoCmd(int)),
			Key("Ctrl+Y", "Edit|Redo"), ECmdRedo);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(cutIcon, tr("Cu&t"),
			this, SLOT(Cmd(int)),
			Key("Ctrl+X", "Edit|Cut"), ECmdCut);
  iEditMenu->insertItem(copyIcon, tr("&Copy"),
			this, SLOT(Cmd(int)),
			Key("Ctrl+C", "Edit|Copy"), ECmdCopy);
  iEditMenu->insertItem(pasteIcon, tr("&Paste"),
			this, SLOT(Cmd(int)),
			Key("Ctrl+V", "Edit|Paste"), ECmdPaste);
  iEditMenu->insertItem(tr("&Delete"),
			this, SLOT(Cmd(int)),
			Key("Del", "Edit|Delete"), ECmdDelete);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(tr("&Group"), this, SLOT(Cmd(int)),
			Key("Ctrl+G", "Edit|Group"), ECmdGroup);
  iEditMenu->insertItem(tr("&Ungroup"), this, SLOT(Cmd(int)),
			Key("Ctrl+U", "Edit|Ungroup"), ECmdUngroup);
  iEditMenu->insertItem(tr("&Front"), this, SLOT(Cmd(int)),
			Key("Ctrl+F", "Edit|Front"), ECmdFront);
  iEditMenu->insertItem(tr("&Back"), this, SLOT(Cmd(int)),
			Key("Ctrl+B", "Edit|Back"), ECmdBack);
  iEditMenu->insertItem(tr("&Duplicate"),
			this, SLOT(Cmd(int)),
			Key("d", "Edit|Duplicate"), ECmdDuplicate);
  iEditMenu->insertItem(tr("Select &all"), this, SLOT(Cmd(int)),
			Key("Ctrl+A", "Edit|Select all"), ECmdSelectAll);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(tr("Co&mpose paths"), this, SLOT(Cmd(int)),
			Key("none", "Edit|Compose paths"), ECmdComposePaths);
  iEditMenu->insertItem(tr("&Join paths"), this, SLOT(Cmd(int)),
			Key("Ctrl+J", "Edit|Join paths"), ECmdJoinPaths);
  //iEditMenu->insertItem(tr("Dec&ompose path"), this, SLOT(Cmd(int)),
  //Key_O, ECmdDecomposePath);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(tr("Insert text box"), this, SLOT(InsertTextBox()),
			Key("F10", "Edit|Insert text box"));
  iEditMenu->insertItem(tr("Insert item box"), this, SLOT(InsertItemBox()),
			Key("Shift+F10", "Edit|Insert text box"));
  iEditMenu->insertItem(tr("Edit text"), this, SLOT(EditObject()),
			Key("Ctrl+E", "Edit|Edit"), ECmdEdit);
  iEditMenu->insertSeparator();
  iEditMenu->insertItem(tr("D&ocument properties"),
			this, SLOT(EditDocProps()),
			Key("Ctrl+Shift+P", "Edit|Document properties"));
  iEditMenu->insertItem(tr("St&yle sheets"), this, SLOT(StyleSheets()),
			Key("Ctrl+Shift+S", "Edit|Style sheets"));
  connect(iEditMenu, SIGNAL(aboutToShow()),
	  this, SLOT(AboutToShowEditMenu()));
  menu->insertItem(tr("&Edit"), iEditMenu);

  iModeMenu = new QPopupMenu;
  for (int i = 0; i < IpeOverlayFactory::Num; ++i)
    iModeAction[i]->addTo(iModeMenu);

  menu->insertItem(tr("&Mode"), iModeMenu);

  iSnapMenu = new QPopupMenu;
  iSnapMenu->insertItem(tr("&Absolute grid/angle size"),
			this, SLOT(AbsoluteSnapping()),
			Key("Ctrl+Shift+T", "Snap|Absolute"),
			EAbsoluteSnappingMenuId);
  iSnapMenu->insertSeparator();
  iSnapActionGroup->addTo(iSnapMenu);
  iSnapMenu->insertSeparator();
  iSnapMenu->insertItem(tr("Set origin"), this,
			SLOT(SnapCmd(int)),
			Key("F1", "Snap|Set origin"), ECmdSetOrigin);
  iSnapMenu->insertItem(tr("Set origin && snap"), this,
			SLOT(SnapCmd(int)),
			Key("none", "Snap|Set origin & snap"),
			ECmdSetOriginSnap);
  iSnapMenu->insertItem(tr("Hide axes"), this,
			SLOT(SnapCmd(int)),
			Key("Shift+F1", "Snap|Hide axes"), ECmdResetOrigin);
  iSnapMenu->insertItem(tr("Set direction"), this,
			SLOT(SnapCmd(int)),
			Key("F2", "Snap|Set direction"), ECmdSetDirection);
  iSnapMenu->insertItem(tr("Set line"), this,
			SLOT(SnapCmd(int)),
			Key("F3", "Snap|Set line"), ECmdSetLine);
  iSnapMenu->insertItem(tr("Set line && snap"), this,
			SLOT(SnapCmd(int)),
			Key("Shift+F3", "Snap|Set line & snap"),
			ECmdSetLineSnap);
  menu->insertItem(tr("&Snap"), iSnapMenu, ESnapMenuId);

  iZoomMenu = new QPopupMenu;
  iZoomMenu->insertItem(tr("&Grid visible"),
			this, SLOT(GridVisible()),
			Key("F12", "Zoom|Grid"),
			EGridVisibleMenuId);
  iZoomMenu->insertSeparator();
  iZoomMenu->insertItem(tr("Zoom &in"), iResolution, SLOT(stepUp()),
			Key("Ctrl+PgUp", "Zoom|Zoom in"));
  iZoomMenu->insertItem(tr("Zoom &out"), iResolution, SLOT(stepDown()),
			Key("Ctrl+PgDown", "Zoom|Zoom out"));
  iZoomMenu->insertItem(tr("&Normal size"), this, SLOT(NormalSize()),
			Key("/", "Zoom|Normal size"));
  iZoomMenu->insertItem(tr("Fit &page"), this, SLOT(FitPage()),
			Key(",", "Zoom|Fit page"));
  iZoomMenu->insertItem(tr("&Fit objects"),
			this, SLOT(FitObjects()),
			Key("=", "Zoom|Fit objects"));
  iZoomMenu->insertItem(tr("Fit &selection"),
			this, SLOT(FitSelection()),
			Key("F11", "Zoom|Fit selection"));
  iZoomMenu->insertSeparator();
  iZoomMenu->insertItem(tr("Pan &here"), this,
			SLOT(SnapCmd(int)),
			Key(".", "view|Pan here"), ECmdHere);
  menu->insertItem(tr("&Zoom"), iZoomMenu, EZoomMenuId);

  iLayerMenu = new QPopupMenu;
  iLayerMenu->insertItem(tr("&New layer"), this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+N", "Layers|New"),
			 ECmdNewLayer);
  iLayerMenu->insertItem(tr("&Rename layer"), this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+R", "Layers|Rename"),
			 ECmdRenameLayer);
  iLayerMenu->insertSeparator();
  iLayerMenu->insertItem(tr("&Select all in layer"),
			 this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+A", "Layers|Select all"),
			 ECmdSelectLayer);
  iLayerMenu->insertItem(tr("&Move objects to layer"),
			 this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+M", "Layers|Move objects"),
			 ECmdMoveToLayer);
#if 0
  iLayerMenu->insertSeparator();
  iLayerMenu->insertItem(tr("&Hide layer"),
			 this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+H", "Layers|Hide"),
			 ECmdHideLayer);
  iLayerMenu->insertItem(tr("D&im layer"),
			 this, SLOT(Cmd(int)),
			 Key("none", "Layers|Dim"), ECmdDimLayer);
  iLayerMenu->insertItem(tr("&Lock layer"),
			 this, SLOT(Cmd(int)),
			 Key("Ctrl+Shift+L", "Layers|Lock"), ECmdLockLayer);
  iLayerMenu->insertItem(tr("Disable &snap for layer"),
			 this, SLOT(Cmd(int)),
			 Key("none", "Layers|Disable snap"),
			 ECmdUnsnapLayer);
#endif
  menu->insertItem(tr("&Layers"), iLayerMenu);

  iViewMenu = new QPopupMenu;
  iViewMenu->insertItem(tr("Next vie&w"), this, SLOT(NextView()),
			Key("PgDown", "Views|Next view"));
  iViewMenu->insertItem(tr("&Previous view"), this, SLOT(PreviousView()),
			Key("PgUp", "Views|Previous view"));
  iViewMenu->insertItem(tr("&First view"), this, SLOT(Cmd(int)),
			Key("Home", "Views|First view"),
			ECmdFirstView);
  iViewMenu->insertItem(tr("&Last view"), this, SLOT(Cmd(int)),
			Key("End", "Views|Last view"),
			ECmdLastView);
  iViewMenu->insertSeparator();
  iViewMenu->insertItem(tr("&New layer, new view"), this, SLOT(Cmd(int)),
			Key("Ctrl+Shift+I", "Views|New layer, new view"),
			ECmdNewLayerNewView);
  iViewMenu->insertItem(tr("New &view"), this, SLOT(Cmd(int)),
			Key("none", "Views|New view"),
			ECmdNewView);
  iViewMenu->insertItem(tr("&Delete view"), this, SLOT(Cmd(int)),
			Key("none", "Views|Delete view"),
			ECmdDeleteView);
  iViewMenu->insertSeparator();
  iViewMenu->insertItem(tr("&Hide layers not in view"),
			this, SLOT(HideLayersNotInView()),
			Key("none", "Views|Hide layers not in view"),
			EHideLayersMenuId);
  iViewMenu->insertSeparator();
  iViewMenu->insertItem(tr("&Edit views"), this,
			SLOT(EditViews()),
			Key("Ctrl+P", "Views|Edit views"));
  menu->insertItem(tr("&Views"), iViewMenu);

  iPageMenu = new QPopupMenu;
  iPageMenu->insertItem(tr("&Next page"), iPageNumber, SLOT(stepUp()),
		       Key("Shift+PgDown", "Page|Next"));
  iPageMenu->insertItem(tr("&Previous page"), iPageNumber, SLOT(stepDown()),
		       Key("Shift+PgUp", "Page|Previous"));
  iPageMenu->insertItem(tr("&First page"), this, SLOT(FirstPage()),
			Key("Shift+Home", "Page|First"));
  iPageMenu->insertItem(tr("&Last page"), this, SLOT(LastPage()),
			Key("Shift+End", "Page|Last"));
  iPageMenu->insertSeparator();
  iPageMenu->insertItem(tr("&Insert page"), this, SLOT(CreatePage()),
		       Key("Ctrl+I", "Page|Insert"));
  iPageMenu->insertItem(tr("&Cut page"), this, SLOT(CopyPage(int)),
		       Key("Ctrl+Shift+X", "Page|Cut"), 1);
  iPageMenu->insertItem(tr("Cop&y page"), this, SLOT(CopyPage(int)),
		       Key("Ctrl+Shift+C", "Page|Copy"), 0);
  iPageMenu->insertItem(tr("P&aste page"), this, SLOT(PastePage()),
		       Key("Ctrl+Shift+V", "Page|Paste"));
  iPageMenu->insertItem(tr("&Delete page"), this, SLOT(DeletePage()),
		       Key("none", "Page|Delete"));
  menu->insertItem(tr("&Pages"), iPageMenu);

  iIpeletMenu = new QPopupMenu;
  iPasteBitmapId = 0;
  for (uint i = 0; i < iIpeletMaster.size(); ++i) {
    Ipelet *ipelet = iIpeletMaster[i];
    if (ipelet->NumFunctions() > 1) {
      QPopupMenu *pop = new QPopupMenu;
      for (int j = 0; j < ipelet->NumFunctions(); ++j) {
	int id = pop->insertItem(ipelet->SubLabel(j),
				 this, SLOT(RunIpelet(int)),
				 Key(ipelet->KeySequence(j), "Ipelet"));
	pop->setItemParameter(id, i * 0x1000 + j);
      }
      if (!qstrcmp(ipelet->Label(), "Insert image")) {
	// personal treatment
	iFileMenu->insertItem(ipelet->Label(), pop, -1, 6);
	iFileMenu->insertSeparator(6);
	iPasteBitmapId = i * 0x1000 + 2;
      } else
	iIpeletMenu->insertItem(ipelet->Label(), pop);
    } else {
      int id = iIpeletMenu->insertItem(ipelet->Label(),
				       this, SLOT(RunIpelet(int)),
				       Key(ipelet->KeySequence(0), "Ipelet"));
      iIpeletMenu->setItemParameter(id, i * 0x1000 + 1);
    }
  }
  menu->insertItem(tr("&Ipelets"), iIpeletMenu);

  menu->insertSeparator();

  QPopupMenu* help = new QPopupMenu;
  help->insertItem(tr("Ipe &manual"), this, SLOT(Manual()));
  help->insertItem(tr("&What's this"), this, SLOT(whatsThis()));
  help->insertItem(tr("Message w&indow"), this, SLOT(ShowDebugWindow()));
  help->insertSeparator();
  help->insertItem(tr("Pre&ferences"), this, SLOT(EditPrefs()));
  help->insertItem(tr("Save &keys"), this, SLOT(SaveKeys()));
  help->insertSeparator();
  help->insertItem(tr("&About Ipe..."), this, SLOT(About()));
  help->insertItem(tr("About &Qt..."), this, SLOT(AboutQt()) );
  menu->insertItem(tr("&Help"), help);

  // --------------------------------------------------------------------

  iCanvas = new IpeCanvas(this, statusBar(), this);
  setCentralWidget(iCanvas);
  statusBar();
  QWhatsThis::add(iCanvas, tr(canvasText));

  iResolution->setValue(72);

  iAbsoluteAttributes = false;
  iAbsoluteSnapping = false;
  iHideLayers = true;
  iViewMenu->setItemChecked(EHideLayersMenuId, iHideLayers);
  SwitchAttributeUi();
  SwitchSnapUi();

  iSnapData.iSnap = IpeSnapData::ESnapNone;
  iSnapData.iGridSize = prefs->iGridSize;
  iSnapData.iGridVisible = prefs->iGridVisible;
  iSnapData.iSnapDistance = prefs->iSnapDistance;
  iSnapData.iSelectDistance = prefs->iSelectDistance;
  iSnapData.iWithAxes = false;
  iSnapData.iOrigin = IpeVector::Zero;
  iSnapData.iDir = 0.0;
  iSnapData.iView = -1; // no restriction

  iDoc = 0;
  SetCreator(iModeAction[0]);

  connect(prefs, SIGNAL(Changed()),
	  this, SLOT(PreferencesChanged()));
}

// ----------------------------------------------------------

//! Destructor.
AppUi::~AppUi()
{
  delete iDoc;
}

// --------------------------------------------------------------------

//! All windows closed, terminate Ipe.
void AppUi::closeEvent(QCloseEvent* ce)
{
  IpePreferences::Static()->Save();

  if (!iDoc->IsEdited()) {
    ce->accept();
    return;
  }

  switch (QMessageBox::information
	  (this, "Ipe",
	   tr("The document has been changed since the last save."),
	   tr("Save now"), tr("Cancel"), tr("Leave Anyway"), 0, 1)) {
  case 0:
    if (Save())
      ce->accept();
    else
      ce->ignore();
    break;
  case 1:
  default: // just for sanity
    ce->ignore(); break;
  case 2:
    ce->accept(); break;
  }
}

void AppUi::keyPressEvent(QKeyEvent *ev)
{
  qDebug("AppUi::keyPressEvent(%x, %x)", ev->key(), ev->state());
  ev->ignore();
}

void AppUi::AboutToShowEditMenu()
{
  bool objs = Page()->HasSelection();
  bool istext = false;
  bool isgroup = false;
  IpePage::iterator it = Page()->PrimarySelection();
  if (it != Page()->end()) {
    istext = (it->Object()->AsText() != 0);
    isgroup = (it->Object()->AsGroup() != 0);
  }
  bool pathObjs = true;
  bool singlePathObjs = true;
  int objCount = 0;
  for (IpePage::iterator it = Page()->begin(); it != Page()->end(); ++it) {
    if (it->Select()) {
      ++objCount;
      IpePath *p = it->Object()->AsPath();
      // must be a single open subpath
      if (!p) {
	pathObjs = false;
	singlePathObjs = false;
      } else if (p->NumSubPaths() > 1 || p->SubPath(0)->Closed())
	singlePathObjs = false;
    }
  }
  iEditMenu->setItemEnabled(ECmdCut, objs);
  iEditMenu->setItemEnabled(ECmdCopy, objs);
  iEditMenu->setItemEnabled(ECmdDelete, objs);
  iEditMenu->setItemEnabled(ECmdGroup, objs);
  iEditMenu->setItemEnabled(ECmdFront, objs);
  iEditMenu->setItemEnabled(ECmdBack, objs);
  iEditMenu->setItemEnabled(ECmdDuplicate, objs);
  iEditMenu->setItemEnabled(ECmdEdit, istext);
  iEditMenu->setItemEnabled(ECmdUngroup, isgroup);
  iEditMenu->setItemEnabled(ECmdComposePaths,
			    objs && pathObjs && objCount > 1);
  iEditMenu->setItemEnabled(ECmdJoinPaths,
			    objs && singlePathObjs && objCount > 1);

  if (iUndo.CanUndo()) {
    iEditMenu->setItemEnabled(ECmdUndo, true);
    iEditMenu->changeItem(ECmdUndo, tr("Undo %1").arg(QIpe(iUndo.UndoText())));
  } else {
    iEditMenu->setItemEnabled(ECmdUndo, false);
    iEditMenu->changeItem(ECmdUndo, tr("Undo"));
  }

  if (iUndo.CanRedo()) {
    iEditMenu->setItemEnabled(ECmdRedo, true);
    iEditMenu->changeItem(ECmdRedo, tr("Redo %1").arg(QIpe(iUndo.RedoText())));
  } else {
    iEditMenu->setItemEnabled(ECmdRedo, false);
    iEditMenu->changeItem(ECmdRedo, tr("Redo"));
  }
}

// --------------------------------------------------------------------
// New document, load, and save
// --------------------------------------------------------------------

//! Slot to create new document.
void AppUi::NewDoc()
{
  if (iDoc && iDoc->IsEdited()) {
    switch (QMessageBox::information
	    (this, "Ipe",
	     tr("The document has been changed since the last save."),
	     tr("Save now"), tr("Cancel"), tr("Discard old document"),
	     0, 1 )) {
    case 0:
      Save();
    case 2:
      break;
    case 1:
    default: // just for sanity
      return;
    }
  }
  delete iDoc;

  IpePreferences *prefs = IpePreferences::Static();

  // reset grid size to default
  iSnapData.iGridSize = prefs->iGridSize;

  iDoc = new IpePdfDocument;  // loads "standard" sheet

  if (!prefs->iStyleSheet.isNull()) {
    if (!iDoc->AddStyleSheet(prefs->iStyleSheet))
      ErrMsg(tr("Could not load or parse style sheet '%1', not added.")
	     .arg(prefs->iStyleSheet));
  }
  if (!prefs->iStyleSheet2.isNull()) {
    if (!iDoc->AddStyleSheet(prefs->iStyleSheet2))
      ErrMsg(tr("Could not load or parse style sheet '%1', not added.")
	     .arg(prefs->iStyleSheet2));
  }

  iDoc->push_back(NewPage());
  ShowStyleInUi();

  IpeDocument::SProperties props = iDoc->Properties();
  // set media size
  props.iMedia = IpeRect(IpeVector::Zero, prefs->iMedia);
  props.iCropBox = true;
  iDoc->SetProperties(props);
  iDoc->SetEdited(false);
  iUndo.Clear();
  iFileName = QString::null;
  iPno = 0;
  PageChanged();
  iNeedLatex = false;
  DefaultZoom();
  statusBar()->message(tr("New document"));
}

static IpePdfDocument::TFormat FormatFromFileName(QString fn)
{
  if (fn.length() < 5)
    return IpePdfDocument::EUnknown;
  fn = fn.lower();
  QString s = fn.right(4);
  if (s == ".xml")
    return IpePdfDocument::EXml;
  else if (s == ".pdf")
    return IpePdfDocument::EPdf;
  else if (fn.right(3) == ".ps")
    return IpePdfDocument::EPs;
  else if (s == ".eps")
    return IpePdfDocument::EEps;
  else if (s == ".ipe" || fn.right(5) == ".mipe")
    return IpePdfDocument::EIpe5;
  return IpePdfDocument::EUnknown;
}

//! Save document
bool AppUi::Save(QString fn, bool warnOverwrite)
{
  IpePdfDocument::TFormat fm = FormatFromFileName(fn);
  if (fm == IpePdfDocument::EUnknown) {
    ErrMsg(tr("You must save as *.xml, *.pdf, *.eps, or *.ps\n"
	      "File not saved."));
    return false;
  }

  if (fm == IpePdfDocument::EEps && iDoc->size() > 1) {
    ErrMsg(tr("Only single-page documents can be saved in EPS format."));
    return false;
  }

  // make sure latex has been run if format is not XML
  if (fm != IpePdfDocument::EXml && !RunLatex(0)) {
    statusBar()->message(tr("Latex error - not saved"));
    return false;
  }

  // check whether file exists
  if (warnOverwrite && QFile::exists(fn)) {
    if (QMessageBox::information
	(0, "Ipe", tr("The file '%1' exists already.\n"
		      "Do you wish to overwrite it?").arg(fn),
	 tr("Overwrite"), tr("Cancel save"), 0, 1) == 1) {
      statusBar()->message(tr("Not saved"));
      return false;
    }
  }
  IpeDocument::SProperties props = iDoc->Properties();
  QDateTime dt = QDateTime::currentDateTime();
  QString mod;
  mod.sprintf("D:%04d%02d%02d%02d%02d%02d",
	      dt.date().year(), dt.date().month(), dt.date().day(),
	      dt.time().hour(), dt.time().minute(), dt.time().second());
  props.iModified = IpeQ(mod);
  if (props.iCreated.empty())
    props.iCreated = props.iModified;
  iDoc->SetProperties(props);
  if (!iDoc->Save(fn, fm)) {
    ErrMsg(tr("Error saving the document.\nFile is not saved."));
    return false;
  }
  iDoc->SetEdited(false);
  statusBar()->message(tr("Saved document '%1'").arg(fn));
  iFileName = fn;
  SetCaption();
  return true;
}

static QString CheckAlternativeSave(const QString &fn)
{
  QString other;
  QFileInfo fi(fn);
  if (fi.extension().lower() == "pdf")
    other = "eps";
  else if (fi.extension().lower() == "eps")
    other = "pdf";
  else
    return QString::null;

  QString ofn = fn.left(fn.length() - 3) + other;
  QFileInfo ofi(ofn);
  if (!ofi.exists())
    return QString::null;

  if (QMessageBox::information
      (0, "Ipe", QObject::tr("<qt>You have saved as '%1'. "
			     "Would you also like to save as '%2'?</qt>").
       arg(fn).arg(ofn),
       QObject::tr("Yes"), QObject::tr("No"), 0, 1) == 0) {
    return ofn;
  }
  return QString::null;
}

//! Slot to save current document.
bool AppUi::Save()
{
  if (iFileName.isNull()) {
    return SaveAs();
  } else {
    bool ret = Save(iFileName, false);
    if (ret) {
      QString s = CheckAlternativeSave(iFileName);
      if (!s.isNull())
	ret = Save(s, false);
    }
    return ret;
  }
}

// --------------------------------------------------------------------
class IpeSaveFileDialog : public QFileDialog {
public:
  IpeSaveFileDialog(const QString &dir, const QString &filter,
		    QWidget *parent);
};

IpeSaveFileDialog::IpeSaveFileDialog(const QString &dir,
				     const QString &filter,
				     QWidget *parent)
  : QFileDialog(dir,
		tr("PDF files (*.pdf);;EPS files (*.eps);;Postscript files (*.ps);;XML files (*.xml)"),
		parent, "saveAsDlg", TRUE)
{
  setMode(QFileDialog::AnyFile);
  if (!filter.isNull())
    addFilter(filter);
}

// --------------------------------------------------------------------

//! Slot to save current document under a fresh name.
bool AppUi::SaveAs()
{
  IpePreferences *prefs = IpePreferences::Static();
  IpeSaveFileDialog dlg(prefs->iDialogDir, iSaveFilter, this);
  if (dlg.exec() == QDialog::Accepted) {
    QString fn = dlg.selectedFile();
    iSaveFilter = dlg.selectedFilter();
    int i = iSaveFilter.findRev(".");
    assert(i >= 0);
    QString ext = iSaveFilter.mid(i+1, iSaveFilter.length() - i - 2);
    IPEDBG << "Selected name: " << fn << " Filter: " << ext << "\n";
    QFileInfo fi(fn);
    if (fi.extension().isEmpty())
      fn += "." + ext;
    IPEDBG << "Saving as: " << fn << "\n";
    return Save(fn, true);
  }
  statusBar()->message(tr("Not saved"));
  return false;
}

//! Set window caption to reflect filename and page.
void AppUi::SetCaption()
{
  QString s = "Ipe ";
  if (!iFileName.isNull())
    s += "\"" + iFileName + "\" ";
  if (iDoc->IsEdited())
    s += "[Unsaved] ";
  if (Page()->Views().size() > 1) {
    s += tr("(View %1 of %2 on page %3 of %4) ")
      .arg(iVno + 1).arg(Page()->Views().size())
      .arg(iPno + 1).arg(iDoc->size());
  } else if (iDoc->size() > 1)
    s += tr("(Page %1 of %2) ").arg(iPno + 1).arg(iDoc->size());
  setCaption(s);
}

// --------------------------------------------------------------------

static void CheckAlternative(QString &fn, IpePdfDocument::TFormat &fm)
{
  QString other;
  QFileInfo fi(fn);
  if (fi.extension().lower() == "pdf")
    other = "eps";
  else if (fi.extension().lower() == "eps")
    other = "pdf";
  else
    return;

  QString ofn = fn.left(fn.length() - 3) + other;
  QFileInfo ofi(ofn);
  if (!ofi.exists() || fi.lastModified() > ofi.lastModified())
    return;

  if (QMessageBox::information
      (0, "Ipe", QObject::tr("<qt>You are trying to open the file '%1', "
			     "while '%2' appears to be more recent."
			     "<br/>Open '%3' instead?</qt>").
       arg(fn).arg(ofn).arg(ofn),
       QObject::tr("Yes"), QObject::tr("No"), 0, 1) == 0) {
    fn = ofn;
    fm = IpePdfDocument::FileFormat(ofn);
  }
}


//! Load Ipe document from \c fileName.
/*! Will discard the current document, even if IpeDocument::IsEdited
  is true! */
bool AppUi::Load(QString fileName)
{
  IpePdfDocument::TFormat fm = IpePdfDocument::FileFormat(fileName);
  if (fm == IpePdfDocument::EUnknown) {
    ErrMsg(tr("I don't know the format of this file."));
    return false;
  }

  if (fm == IpePdfDocument::EIpe5) {
    QFileInfo finfo(fileName);
    QString iml = finfo.dir().filePath(finfo.baseName() + ".xml");
    if (QMessageBox::information
	(0, "Ipe",
	 tr("<qt>You have selected to open the file"
	    "<blockquote><font color=\"red\">'%1'</font></blockquote>"
	    "This file was created by an ancient version of Ipe, and "
	    "I cannot open it directly.<br>"
	    "I will therefore convert it to a new file"
	    "<blockquote><font color=\"red\">'%2'</font></blockquote>"
	    "in Ipe 6.0 format.<br>"
	    "The conversion preserves all drawing information in the file. "
	    "You may only need to check the <em>Latex preamble</em> and "
	    "'strange' text objects.</qt>").arg(fileName).arg(iml),
	 tr("Convert file"), tr("Cancel load"), 0, 0, 1) == 1)
      return false;
    // check whether XML file exists
    if (QFile::exists(iml)) {
      if (QMessageBox::information
	  (0, "Ipe", tr("The file '%1' exists already.\n"
			"Do you wish to overwrite it?").arg(iml),
	   tr("Overwrite"), tr("Cancel conversion"), 0, 1) == 1) {
	statusBar()->message(tr("Conversion and loading canceled"));
	return false;
      }
    }
    QDir dir = IpePreferences::Static()->iBinDir;
    QString app = dir.filePath("ipe5toxml.exe");
#ifdef WIN32
    QCString cl =
      "call \"" + QFile::encodeName(QDir::convertSeparators(app)) + "\" \"" +
      QFile::encodeName(QDir::convertSeparators(fileName)) + "\" \"" +
      QFile::encodeName(QDir::convertSeparators(iml)) + "\"";
    qDebug("system('%s')", cl.data());
    system(cl.data());
#else
    QCString cl = QFile::encodeName(app) + " " +
      QFile::encodeName(fileName) + " " + QFile::encodeName(iml);
    system(cl.data());
#endif
    fileName = iml;
    fm = IpePdfDocument::EXml;
  }

  CheckAlternative(fileName, fm);

  IpePdfDocument *oldDoc = iDoc;
  int reason;
  iDoc = IpePdfDocument::New(fileName, fm, reason);
  if (!iDoc) {
    // need to reestablish document before showing the popup
    iDoc = oldDoc;
    if (reason >= 0)
      ErrMsg(tr("Error parsing document '%1' at character offset %2")
	     .arg(fileName).arg(reason));
    else if (reason < -60000)
      ErrMsg(tr("Document '%1' has been created by a version of Ipe "
		"more recent than the one you are running. "
		"Please upgrade Ipe at <tt>ipe.compgeom.org</tt>")
	     .arg(fileName));
    else
      ErrMsg(tr("Error loading document '%1' (reason %2)")
	     .arg(fileName).arg(reason));
    return false;
  }

  // for the moment, need to clear undo stack first because of bitmaps
  iUndo.Clear();
  delete oldDoc;

  ShowStyleInUi(); // why does this set the edited flag??
  iDoc->SetEdited(false);

  iFileName = fileName;
  iPno = 0;
  PageChanged();
  iNeedLatex = true;
  statusBar()->message(tr("Loaded document '%1'").arg(iFileName));

  IpeAttributeSeq seq;
  if (!iDoc->CheckStyle(seq)) {
    for (IpeAttributeSeq::iterator it = seq.begin(); it != seq.end(); ++it)
      IPEDBG << "Undefined symbolic attribute '" <<
	iDoc->Repository()->String(*it) << "'\n";
    ErrMsg(tr("The document you loaded contains symbolic attributes that "
	      "are not defined in its style sheet. You can find a list of "
	      "these attributes in the message window."));
  }
  return true;
}

//! Slot to load document.
void AppUi::Load()
{
  if (iDoc && iDoc->IsEdited()) {
    switch (QMessageBox::information
	    (this, "Ipe",
	     tr("The document has been changed since the last save."),
	     tr("Save now"), tr("Cancel load"), tr("Load anyway"),
	     0, 1 )) {
    case 0:
      Save();
    case 2:
      break;
      break;
    case 1:
    default: // just for sanity
      statusBar()->message(tr("Loading canceled"));
      return;
    }
  }
  IpePreferences *prefs = IpePreferences::Static();
  QString fn = QFileDialog::getOpenFileName
    (prefs->iDialogDir,
     tr("Ipe files (*.xml *.pdf *.ipe *.mipe *.ps *.eps);;All files (*.*)"),
     this);
  if (!fn.isEmpty()) {
    prefs->iDialogDir = QFileInfo(fn).dirPath();
    prefs->Save();
    Load(fn);
    DefaultZoom();
  } else
    statusBar()->message(tr("Loading canceled"));
}

extern QPixmap toPng(const IpePdfDocument *doc, const IpePage *page,
		     double zoom);

void AppUi::SaveAsBitmap()
{
  QString fn = QFileDialog::getSaveFileName
    (QString::null, tr("PNG files (*.png)"), this);
  if (fn.isEmpty()) {
    statusBar()->message(tr("No bitmap saved"));
    return;
  }

  double zoom = double(iResolution->value()) / 72.0;
  QPixmap pixmap = toPng(iDoc, Page(), zoom);
  if (!pixmap.save(fn, "PNG")) {
    ErrMsg(tr("Could not save bitmap"));
  }
}

void AppUi::SaveKeys()
{
  QString fn = QFileDialog::getSaveFileName
    (QString::null, tr("Qt localization files (*.qm)"), this);
  if (fn.isEmpty()) {
    statusBar()->message(tr("Keys not saved"));
    return;
  }
  iKeyTranslator.save(fn);
  statusBar()->message(tr("Keys saved"));
}

// --------------------------------------------------------------------
// Public slots
// --------------------------------------------------------------------

//! This handles all Ipe commands that operate on an IpePage.
void AppUi::Cmd(int cmd)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;

  statusBar()->clear();
  IpePage *page = Page();
  IpeLayer layer = page->Layer(iLayer);

  // check whether command requires a select object
  switch (cmd) {
  case ECmdGroup:
  case ECmdUngroup:
  case ECmdCut:
  case ECmdCopy:
  case ECmdFront:
  case ECmdBack:
  case ECmdDuplicate:
  case ECmdDelete:
  case ECmdComposePaths:
  case ECmdJoinPaths:
  case ECmdDecomposePath:
  case ECmdMoveToLayer:
    if (!page->HasSelection()) {
      statusBar()->message(tr("No selected object"));
      return;
    }
  default:
    break;
  }

  IpeAutoPtr<IpePage> originalPage(new IpePage(*page));
  page->SetEdited(false);
  QString undo;
  bool doUndo = false;

  switch (cmd) {
  case ECmdSelectAll:
    page->SelectAll();
    undo = tr("select all");
    // doUndo = true;
    break;
  case ECmdSelectLayer:
    page->SelectAllInLayer(iLayer);
    undo = tr("select all in layer");
    // doUndo = true;
    break;
  case ECmdFirstView:
    iViewNumber->setValue(1);
    break;
  case ECmdLastView:
    iViewNumber->setValue(page->CountViews() > 1 ? page->CountViews() : 1);
    break;
  case ECmdNewView:
  case ECmdNewLayerNewView:
    {
      if (page->CountViews() == 0) {
	// need to create full view first
	IpeView view;
	for (int i = 0; i < page->CountLayers(); ++i)
	  view.iLayers.push_back(page->Layer(i).Name());
	view.iActive = page->Layer(iLayer).Name();
	page->AddView(view);
      }
      IpeView view;
      undo = tr("view creation");
      if (cmd == ECmdNewLayerNewView) {
	undo = tr("layer and view creation");
	iLayer = page->NewLayer(iLayer + 1);
	for (int i = 0; i <= iLayer; ++i)
	  view.iLayers.push_back(page->Layer(i).Name());
      }
      view.iActive = page->Layer(iLayer).Name();
      page->AddView(view, ++iVno);
      ViewChanged();
      break;
    }
  case ECmdDeleteView:
    if (page->CountViews() > 0) {
      page->DeleteView(iVno);
      if (iVno > 0 && iVno >= page->CountViews())
	--iVno;
      undo = tr("view deletion");
      ViewChanged();
    }
    break;
  case ECmdMoveToLayer:
    page->MoveToLayer(iLayer);
    undo = tr("move to layer");
    break;
  case ECmdGroup:
    page->Group(iLayer);
    undo = tr("grouping");
    break;
  case ECmdUngroup:
    if (!page->Ungroup(iLayer))
      return;
    undo = tr("ungrouping");
    break;
  case ECmdCut:
  case ECmdCopy:
    {
      IpeString data;
      IpeStringStream stream(data);
      page->Copy(stream, iDoc->StyleSheet());
      if (cmd == ECmdCut)
	page->Delete();
      else
	statusBar()->message(tr("Copied"));
      QClipboard *cb = QApplication::clipboard();
      // cb->setData(new IpeDrag(data, this));
      cb->setText(QIpe(data));
      undo = tr("cut");
      break;
    }
  case ECmdPaste:
    {
      QClipboard *cb = QApplication::clipboard();
      //IpeString data;
      // if (!IpeDrag::decode(cb->data(), data) ||
      QString data = cb->text();
      if (data.length() == 0) {
#ifndef WIN32
	if (iPasteBitmapId > 0 && !cb->image().isNull()) {
	  // paste image
	  RunIpelet(iPasteBitmapId);
	} else
#endif
	  statusBar()->message(tr("Nothing to paste"));
      } else if (data.left(14) != "<ipeselection>") {
	IpeRect r = iDoc->Properties().iMedia;
	IpeVector pos = 0.5 * (r.Min() + r.Max());
	IpeText *obj = new IpeText(iAttributes, IpeQ(data), pos,
				   IpeText::ELabel);
	OvSvcAddObject(obj);
      } else {
	XmlQStringSource source(data);
	if (!page->Paste(iLayer, source, iDoc->Repository()))
	  ErrMsg(tr("Cannot parse Ipe objects in clipboard"));
      }
      undo = tr("paste");
      break;
    }
  case ECmdFront:
    page->Front();
    undo = tr("move to front");
    break;
  case ECmdBack:
    page->Back();
    undo = tr("move to back");
    break;
  case ECmdDuplicate:
    page->Duplicate(iLayer);
    undo = tr("duplication");
    statusBar()->message(tr("Object duplicated"));
    break;
  case ECmdDelete:
    page->Delete();
    undo = tr("deletion");
    break;
  case ECmdNewLayer:
    iLayer = page->NewLayer(iLayer + 1);
    UpdateLayers();
    undo = tr("adding layer");
    break;
  case ECmdRenameLayer:
    {
      bool ok = false;
      QString text = QInputDialog::getText
	(tr("Rename layer"),
	 tr("Enter new layer name"),
	 QLineEdit::Normal,
	 QIpe(layer.Name()), &ok, this);
      if (ok && !text.isEmpty()) {
	if (page->FindLayer(IpeQ(text)) >= 0) {
	  statusBar()->message(tr("Layer '%1' already exists").arg(text));
	  break;
	}
	layer.SetName(IpeQ(text));
	page->SetLayer(iLayer, layer);
	UpdateLayers();
      }
    }
    undo = tr("renaming layer");
    break;
  case ECmdLineWidth:
    page->SetLineWidth(iAttributes.iLineWidth);
    undo = tr("setting line width");
    break;
  case ECmdDashStyle:
    page->SetDashStyle(iAttributes.iDashStyle);
    undo = tr("setting dash style");
    break;
  case ECmdTextSize:
    page->SetTextSize(iAttributes.iTextSize);
    undo = tr("setting text size");
    break;
  case ECmdMarkShape:
    page->SetMarkShape(iAttributes.iMarkShape);
    undo = tr("setting mark shape");
    break;
  case ECmdMarkSize:
    page->SetMarkSize(iAttributes.iMarkSize);
    undo = tr("setting mark size");
    break;
  case ECmdSetArrows:
    page->SetArrows(iAttributes.iForwardArrow,
		    iAttributes.iBackwardArrow,
		    iAttributes.iArrowSize);
    undo = tr("setting arrows");
    break;
  case ECmdArrowSize:
    page->SetArrowSize(iAttributes.iArrowSize);
    undo = tr("setting arrow size");
    break;
  case ECmdStroke:
    page->SetStroke(iAttributes.iStroke);
    undo = tr("setting stroke");
    break;
  case ECmdFill:
    page->SetFill(iAttributes.iFill);
    undo = tr("setting fill");
    break;
  case ECmdComposePaths:
    if (!page->ComposePaths(iLayer)) {
      statusBar()->message(tr("Only path objects can be composed"));
      break;
    }
    undo = tr("path composition");
    break;
  case ECmdJoinPaths:
    if (!page->JoinPaths(iLayer)) {
      statusBar()->message(tr("Incorrect objects in selection"));
      break;
    }
    undo = tr("joining paths");
    break;
  case ECmdDecomposePath:
    if (!page->DecomposePath(iLayer)) {
      statusBar()->message(tr("Primary selection is not a path object"));
      break;
    }
    undo = tr("path decomposition");
    break;
  }
  bool isEdited = page->IsEdited();
  if (originalPage->IsEdited())
    page->SetEdited(true);
  if (doUndo || isEdited) {
    // page has been modified, save in Undo stack
    AddUndo(new IpeUndoPageEdit(iPno, originalPage.Take(), IpeQ(undo)));
  }
  iCanvas->Update();
  SetCaption();
}

void AppUi::AddUndo(IpeUndoItem *item)
{
  iUndo.Add(item);
}

void AppUi::UndoCmd(int cmd)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;
  statusBar()->clear();
  int pno = -1;
  switch (cmd) {
  case ECmdUndo:
    if (iUndo.CanUndo())
      pno = iUndo.Undo(iDoc);
    break;
  case ECmdRedo:
    if (iUndo.CanRedo())
      pno = iUndo.Redo(iDoc);
    break;
  }
  if (pno >= 0)
    iPno = pno;
  if (iPno >= int(iDoc->size()))
    iPno = iDoc->size() - 1;
  iDoc->SetEdited(true);
  PageChanged();
}

void AppUi::SnapCmd(int id)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;

  statusBar()->clear();

  // compute snapped mouse position WITHOUT angular snapping
  IpeVector pos = iCanvas->UnsnappedPos();
  iSnapData.SimpleSnap(pos, Page(), iSnapData.iSnapDistance / iCanvas->Zoom());

  switch (id) {
  case ECmdHere:
    iCanvas->SetPan(iCanvas->Pos());
    iCanvas->Update();
    return;
  case ECmdSetOriginSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setOn(true);
  case ECmdSetOrigin:
    iSnapData.iWithAxes = true;
    iSnapData.iOrigin = pos;
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdResetOrigin:
    iSnapData.iWithAxes = false;
    iSnapData.iSnap &= ~IpeSnapData::ESnapAngle;
    iSnapAction[4]->setOn(false);
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdSetDirectionSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setOn(true);
  case ECmdSetDirection:
    iSnapData.iWithAxes = true;
    iSnapData.iDir = (pos - iSnapData.iOrigin).Angle();
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdSetLineSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setOn(true);
  case ECmdSetLine:
    if (!iSnapData.SetEdge(pos, Page())) {
      statusBar()->message(tr("Mouse position is not on an edge"));
    } else {
      iSnapData.iWithAxes = true;
      iCanvas->SetSnap(iSnapData);
      iCanvas->Update();
    }
    break;
  }
}

// --------------------------------------------------------------------

// Slots to connect to tool buttons

void AppUi::Cut()
{
  Cmd(ECmdCut);
}

void AppUi::Copy()
{
  Cmd(ECmdCopy);
}

void AppUi::Paste()
{
  Cmd(ECmdPaste);
}

void AppUi::EditObject()
{
  IpePage::iterator it = Page()->PrimarySelection();
  if (it == Page()->end())
    return;
  IpePgObject original(*it);
  if (it->Object()->AsText() &&
      IpeCreateText::Edit(it->Object()->AsText(), iDoc->StyleSheet())) {
    OvSvcAddUndoItem(it, Page(), original, tr("text object edit"));
    Page()->SetEdited(true);
    iCanvas->Update();
  } else if (it->Object()->AsPath())
    iCanvas->SetOverlay(new IpeEditPath(iCanvas, it->Object()->AsPath(),
					Page(), this, it));
  statusBar()->clear();
}

// --------------------------------------------------------------------

void AppUi::NewWindow()
{
  AppUi *appui = new AppUi;
  appui->NewDoc();
  if (IpePreferences::Static()->iMaximize)
    appui->showMaximized();
  else
    appui->show();
  appui->FitPage();
  statusBar()->clear();
}

// --------------------------------------------------------------------

void AppUi::NextView()
{
  if (iVno + 1 >= Page()->CountViews()) {
    if (iPno + 1 < int(iDoc->size())) {
      iPageNumber->stepUp();
    }
  } else
    iViewNumber->stepUp();
}

void AppUi::PreviousView()
{
  if (iVno == 0) {
    if (iPno > 0) {
      iPageNumber->stepDown();
      Cmd(ECmdLastView);
    }
  } else
    iViewNumber->stepDown();
}

void AppUi::FirstPage()
{
  iPageNumber->setValue(1);
  statusBar()->clear();
}

void AppUi::LastPage()
{
  iPageNumber->setValue(iDoc->size() - 1);
  statusBar()->clear();
}

static double AdjustPan(double cmin, double cmax,
			double omin, double omax,
			double pmin, double pmax)
{
  double dx = 0;

  // if objects stick out on both sides, there is nothing we can do
  if (omin <= cmin &&  omax >= cmax)
    return dx;

  if (omax > cmax && omin > cmin) {
    // we can see more objects if we shift canvas right
    dx = IpeMin(omin - cmin, omax - cmax);
  } else if (omin < cmin && omax < cmax) {
    // we can see more objects if we shift canvas left
    dx = -IpeMin(cmin - omin, cmax - omax);
  }

  // shift canvas
  cmin += dx;
  cmax += dx;

  // if canvas fully contained in media, done
  if (pmin <= cmin && pmax >= cmax)
    return dx;
  // if media contained in canvas, can't improve
  if (cmin < pmin && pmax < cmax)
    return dx;

  if (pmin > cmin) {
    // improvement possible by shifting canvas right
    if (omin > cmin)
      dx += IpeMin(omin - cmin, IpeMin(pmin - cmin, pmax - cmax));
  } else {
    // improvement possible by shifting canvas left
    if (omax < cmax)
      dx -= IpeMin(cmax - omax, IpeMin(cmax - pmax, cmin - pmin));
  }
  return dx;
}

//! Change resolution to 72 dpi and maximize interesting visible area.
/*! As suggested by Rene:

1) scale to the proper size, with the center of the canvas as the
   origin of the scaling.
2) If there is a horizontal and/or vertical translation that makes a
   larger part of the *bounding box* of the objects visible, then
   translate (and maximize the part of the bounding box that is
   visible).
3) If there is a horizontal and/or vertical translation that makes a
   larger part of the paper visible, then translate (and maximize the
   part of the paper that is visible), under the restriction that no
   part of the bounding box of the objects may be moved `out of sight'
   in this step. (Note that there may be objects outside the paper).
*/
void AppUi::NormalSize()
{
  iResolution->setValue(72);

  IpeRect media = iDoc->Properties().iMedia;

  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    bbox.AddRect(it->BBox());
  }

  // size of canvas in user coordinates
  IpeVector s = (1.0 / iCanvas->Zoom()) *
    IpeVector(iCanvas->width(), iCanvas->height());

  IpeRect canvas(iCanvas->Pan() - 0.5 * s, iCanvas->Pan() + 0.5 * s);

  IpeVector pan;
  pan.iX = AdjustPan(canvas.Min().iX, canvas.Max().iX,
		     bbox.Min().iX, bbox.Max().iX,
		     media.Min().iX, media.Max().iX);
  pan.iY = AdjustPan(canvas.Min().iY, canvas.Max().iY,
		     bbox.Min().iY, bbox.Max().iY,
		     media.Min().iY, media.Max().iY);
  iCanvas->SetPan(iCanvas->Pan() + pan);
  statusBar()->clear();
}

void AppUi::FitBox(const IpeRect &box)
{
  if (box.IsEmpty())
    return;
  double xfactor = iCanvas->width() / box.Width();
  double yfactor = iCanvas->height() / box.Height();
  double zoom = (xfactor > yfactor) ? yfactor : xfactor;
  int resolution = int(zoom * 72);
  if (resolution > 1000)
    resolution = 1000;
  else if (resolution < 10)
    resolution = 10;
  iCanvas->SetPan(0.5 * (box.Min() + box.Max()));
  iResolution->setValue(resolution);
  statusBar()->clear();
}

//! Do standard zoom (after loading new document)
void AppUi::DefaultZoom()
{
  if (iDoc->size() > 1)
    FitPage();
  else
    NormalSize();
}

//! Change resolution so that page is displayed fully.
void AppUi::FitPage()
{
  IpeRect box = iDoc->Properties().iMedia;
  FitBox(box);
}

void AppUi::FitObjects()
{
  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    bbox.AddRect(it->BBox());
  }
  FitBox(bbox);
}

void AppUi::FitSelection()
{
  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    if (it->Select() != IpePgObject::ENone)
      bbox.AddRect(it->BBox());
  }
  FitBox(bbox);
}

IpePage *AppUi::NewPage()
{
  // Create one empty page
  IpePage *page = new IpePage;
  IpeAttribute bga =
    iDoc->Repository()->MakeSymbol(IpeAttribute::ETemplate, "background");
  if (iDoc->StyleSheet()->FindTemplate(bga)) {
    IpeLayer bglayer("background");
    bglayer.SetLocked(true);
    page->AddLayer(bglayer);
    IpeObject *obj = new IpeReference(iDoc->StyleSheet(), bga);
    page->push_back(IpePgObject(IpePgObject::ENone, 0, obj));
  }
  page->AddLayer(IpeLayer("alpha"));
  page->SetGridSize(iSnapData.iGridSize);
  return page;
}

//! Slot to create a new page after the current one.
void AppUi::CreatePage()
{
  ++iPno;
  IpePage *page = NewPage();
  iDoc->insert(iDoc->begin() + iPno, page);
  PageChanged();
  iDoc->SetEdited(true);
  statusBar()->clear();
  AddUndo(new IpeUndoPageIns(iPno, IpeQ(tr("page creation"))));
}

//! Slot to delete current page --- if it is empty.
void AppUi::DeletePage()
{
  IpePage *page = Page();
  if (page->size() > 0) {
    statusBar()->message(tr("Page is not empty, not deleted."));
    return;
  }
  // undo stack takes ownership of page
  AddUndo(new IpeUndoPageDel(iPno, (*iDoc)[iPno], IpeQ(tr("page deletion"))));
  // erase pointer only
  iDoc->erase(iDoc->begin() + iPno);
  if (iPno == int(iDoc->size()))
    --iPno;
  PageChanged();
  iDoc->SetEdited(true);
  statusBar()->clear();
}

void AppUi::CopyPage(int cut)
{
  IpeString data;
  IpeStringStream stream(data);
  Page()->CopyPage(stream, iDoc->StyleSheet());
  QClipboard *cb = QApplication::clipboard();
  // cb->setData(new IpeDrag(data, this));
  cb->setText(QIpe(data));

  if (cut) {
    IpePage *original = Page();
    iDoc->erase(iDoc->begin() + iPno);
    if (iPno == int(iDoc->size()))
      --iPno;
    PageChanged();
    iDoc->SetEdited(true);
    // undo stack takes ownership of original page
    AddUndo(new IpeUndoPageDel(iPno, original, IpeQ(tr("cut page"))));
  } else
    statusBar()->message(tr("Page copied"));
}

void AppUi::PastePage()
{
  QClipboard *cb = QApplication::clipboard();
  QString data = cb->text();
  // if (!IpeDrag::decode(cb->data(), data) ||
  if (data.left(9) != "<ipepage>") {
    statusBar()->message(tr("No Ipe page to paste"));
    return;
  }
  XmlQStringSource source(data);
  IpeImlParser parser(source, iDoc->Repository());
  IpePage *page = parser.ParsePageSelection();
  if (!page) {
    ErrMsg(tr("Could not parse page on clipboard"));
    return;
  }
  ++iPno;
  iDoc->insert(iDoc->begin() + iPno, page);
  PageChanged();
  iDoc->SetEdited(true);
  AddUndo(new IpeUndoPageIns(iPno, IpeQ(tr("paste page"))));
  statusBar()->clear();
}

//! Slot to edit document properties.
void AppUi::EditDocProps()
{
  IpeDocument::SProperties props = iDoc->Properties();
  DialogDocumentProperties *dialog =
    new DialogDocumentProperties(this, props, iDoc->StyleSheet());
  if (dialog->exec() == QDialog::Accepted) {
    if (props.iPreamble != iDoc->Properties().iPreamble)
      iNeedLatex = true;
    iDoc->SetProperties(props);
    // paper size may have changed
    PageChanged();
  }
  statusBar()->clear();
}

// --------------------------------------------------------------------

//! Slot to set mode (connected to object buttons and menu).
void AppUi::SetCreator(QAction *action)
{
  statusBar()->clear();
  for (iMode = 0; iMode < IpeOverlayFactory::Num; iMode++) {
    if (action == iModeAction[iMode])
      return;
  }
}

// --------------------------------------------------------------------

//! Slot to run Latex on text objects.
/*! Returns true if document can now be saved to PDF.  Does not run
  Latex if \a force is \c false and no text objects have been
  modified.
 */
bool AppUi::RunLatex(int force)
{
  statusBar()->message(tr("Running Latex ..."));
  QStringList sd;
  sd.append(QDir::current().path());
  if (!iFileName.isNull()) {
    QFileInfo fi(iFileName);
    sd.append(fi.dirPath(true));
  }
  switch (iDoc->RunLatex(force ? true : iNeedLatex, sd)) {
  case IpePdfDocument::ErrNoText:
    statusBar()->message(tr("No text objects in document, "
			    "no need to run Pdflatex."));
    iNeedLatex = false;
    return true;
  case IpePdfDocument::ErrNoDir:
    ErrMsg(tr("<qt>Directory '%1' does not exist and cannot be created.</qt>")
	   .arg(IpePreferences::Static()->iLatexDir));
    statusBar()->clear();
    return false;
  case IpePdfDocument::ErrWritingSource:
    ErrMsg(tr("<qt>Error writing Latex source</qt>"));
    statusBar()->clear();
    return false;
  case IpePdfDocument::ErrOldPdfLatex:
    ErrMsg(tr("<qt>Your installed version of Pdflatex is too old."
	      "Please install a more recent version (at least version 0.14f)."
	      "Until you do so, Ipe cannot convert text objects.</qt>"));
    statusBar()->clear();
    return false;
  case IpePdfDocument::ErrLatexOutput:
    ErrMsg(tr("<qt>Something is wrong with the PDF file generated by "
	      "Pdflatex.  Please consult the message window to identify the "
	      "problem."
#ifdef WIN32
	      "<hr/>You may also perform a <em>Refresh Filename Database</em> "
	      "in <u>MikTeX Options</u> and try again."
#endif
	      "</qt>"));
    DebugWindow::Show();
    statusBar()->clear();
    return false;
  case IpePdfDocument::ErrLatex:
    {
      DialogLatexError *dialog = new DialogLatexError(0, iDoc->LogFile());
      dialog->exec();
    }
    statusBar()->clear();
    return false;
  case IpePdfDocument::ErrNone:
    statusBar()->message(tr("Latex run completed successfully"));
    PageChanged(); // update
    iNeedLatex = false;
    return true;
  case IpePdfDocument::ErrAlreadyHaveForm:
    statusBar()->clear();
    return true;
  default:
    assert(false);
    return false;
  };
}

// --------------------------------------------------------------------

//! Slot to edit Ipe preferences.
void AppUi::EditPrefs()
{
  statusBar()->clear();
  IpePreferences *prefs = IpePreferences::Static();

  DialogPreferences *dialog =
    new DialogPreferences(this, *prefs, iSnapData.iGridSize);
  if (dialog->exec() == QDialog::Accepted)
    prefs->Save(); // will cause call to PreferencesChanged
}

void AppUi::PreferencesChanged()
{
  IpePreferences *prefs = IpePreferences::Static();
  iAttributes.iTransformable = prefs->iTransformable;
  iSnapData.iSnapDistance = prefs->iSnapDistance;
  iSnapData.iSelectDistance = prefs->iSelectDistance;
  iCanvas->SetSnap(iSnapData);
  iCanvas->Update();
  setUsesBigPixmaps(prefs->iBigToolButtons);
  QApplication::setFont(prefs->iFont, true);
  statusBar()->clear();
}

void AppUi::EditViews()
{
  statusBar()->clear();
  IpePage *page = Page();
  QStringList layers;
  for (int i = 0; i < page->CountLayers(); ++i) {
    layers.append(QIpe(page->Layer(i).iName));
  }
  IpeViewSeq views = page->Views();

  DialogPageViews *dialog = new DialogPageViews(this, layers, views);
  if (dialog->exec() == QDialog::Accepted) {
    AddUndo(new IpeUndoViews(iPno, page->Views(), IpeQ("view change")));
    page->SetViews(views);
    PageChanged();
  }
}

//! Slot to show Ipe manual.
void AppUi::Manual()
{
  QDir dir(IpePreferences::Static()->iDocDir);
  QString url = dir.filePath("manual.html");
  StartManual(url);
  statusBar()->clear();
}

//! Slot to show debug window.
void AppUi::ShowDebugWindow()
{
  DebugWindow::Show();
  statusBar()->clear();
}

void AppUi::StyleSheets()
{
  statusBar()->clear();
  DialogStyles *dialog = new DialogStyles(this, iDoc);
  dialog->exec();
  // need to force update of canvas information (i.e. stylesheet)
  PageChanged();
  ShowStyleInUi();
  // AddUndo(new IpeUndoViews(iPno, page->Views(),
  // IpeQ("style sheet change")));
}

// --------------------------------------------------------------------

void AppUi::RunIpelet(int id)
{
  statusBar()->clear();
  int ipelet = id >> 12;
  int function = id & 0xfff;

  IpePage *original = new IpePage(*Page());
  iIpeletNoUndo = false;
  iIpeletMaster[ipelet]->Run(function, Page(), this);
  Page()->SetEdited(true);
  if (iIpeletNoUndo) {
    delete original;
    iUndo.Clear();
  } else
    AddUndo(new IpeUndoPageEdit(iPno, original, IpeQ(tr("Ipelet result"))));
  iCanvas->Update();
  UpdateLayers();
  SetCaption();
}

// --------------------------------------------------------------------

//! Slot to display about window.
void AppUi::About()
{
  static QMessageBox* about =
    new QMessageBox("Ipe",
		    QString(aboutText).arg(IPE_VERSION)
		    .arg(IPE_VERSION).arg(IPE_VERSION),
		    QMessageBox::Information, 1, 0, 0, this, 0, false);
  about->setButtonText(1, tr("Dismiss"));
  about->show();
  statusBar()->clear();
}

//! Slot to display about Qt window.
void AppUi::AboutQt()
{
  QMessageBox::aboutQt(this, "Ipe");
  statusBar()->clear();
}

// --------------------------------------------------------------------
// Private slots
// --------------------------------------------------------------------

void AppUi::AbsoluteAttributes()
{
  iAbsoluteAttributes = !iAbsoluteAttributes;
  iEditMenu->setItemChecked(EAbsoluteAttributesMenuId, iAbsoluteAttributes);
  SwitchAttributeUi();
  statusBar()->clear();
  if (iAbsoluteAttributes) {
    // when switching to absolute, set absolute values from symbolic ones
    const IpeStyleSheet *sheet = iDoc->StyleSheet();
    iAttributes.iStroke = sheet->Find(iAttributes.iStroke);
    iAbsStrokeColor->setPixmap
      (ColorPixmap(iAttributes.iStroke, sheet));
    iAttributes.iFill = sheet->Find(iAttributes.iFill);
    iAbsFillColor->setPixmap
      (ColorPixmap(iAttributes.iFill, sheet));
    ConvertAbsolute(iAbsLineWidth, iAttributes.iLineWidth);
    ConvertAbsolute(iAbsMarkSize, iAttributes.iMarkSize);
    ConvertAbsolute(iAbsArrowSize, iAttributes.iArrowSize);
  } else {
    // when switching to symbolic, use previous symbolic settings
    // and update iAttributes
    iAttributes.iStroke =
      iSyms[IpeAttribute::EColor][iStrokeColor->currentItem()];
    iAttributes.iFill =
      iSyms[IpeAttribute::EColor][iFillColor->currentItem()];
    iAttributes.iLineWidth =
      iSyms[IpeAttribute::ELineWidth][iLineWidth->currentItem()];
    iAttributes.iTextSize =
      iSyms[IpeAttribute::ETextSize][iTextSize->currentItem()];
    iAttributes.iMarkSize =
      iSyms[IpeAttribute::EMarkSize][iMarkSize->currentItem()];
    iAttributes.iArrowSize =
      iSyms[IpeAttribute::EArrowSize][iArrowSize->currentItem()];
  }
}

void AppUi::AbsoluteSnapping()
{
  iAbsoluteSnapping = !iAbsoluteSnapping;
  iSnapMenu->setItemChecked(EAbsoluteSnappingMenuId, iAbsoluteSnapping);
  SwitchSnapUi();
  statusBar()->clear();
  if (iAbsoluteSnapping) {
    // when switching to absolute, set absolute values from symbolic ones
    iAbsGridSize->setValueFromAttribute(int(iSnapData.iGridSize * 10.0));
    iAbsAngleSize->setValueFromAttribute
      (int(IpeAngle(iSnapData.iAngleSize).Degrees() * 10.0));
  } else {
    // when switching to symbolic, use previous symbolic settings
    iSnapData.iGridSize =
      AbsValue(iSyms[IpeAttribute::EGridSize][iGridSize->currentItem()]);
    iSnapData.iAngleSize = IpeAngle::Degrees
      (AbsValue(iSyms[IpeAttribute::EAngleSize][iAngleSize->currentItem()]));
    iCanvas->SetSnap(iSnapData);
    if (iSnapData.iWithAxes || iSnapData.iGridVisible)
      iCanvas->Update();
  }
}

void AppUi::PageChanged(int no)
{
  iPno = no - 1;
  PageChanged();
}

void AppUi::PageChanged()
{
  iVno = 0;
  iLayer = 0;
  iPageNumber->Set(iPno + 1, iDoc->size());
  iCanvas->SetPage(Page(), iDoc->StyleSheet(),
		   iDoc->Properties().iMedia, PageColor());
  SetGridSizeFromPage();
  ViewChanged();
}

void AppUi::ViewChanged(int no)
{
  iVno = no - 1;
  ViewChanged();
}

void AppUi::ViewChanged()
{
  iViewNumber->Set(iVno + 1,
		   Page()->CountViews() > 1 ? Page()->CountViews() : 1);
  IpePage *page = Page();
  iSnapData.iView = -1;
  if (page->CountViews() > 0) {
    int i = page->FindLayer(page->View(iVno).iActive);
    if (i >= 0)
      iLayer = i;
    if (iHideLayers) {
      iSnapData.iView = iVno;
      page->DeselectNotInView(iVno);
      page->EnsurePrimarySelection();
    }
  }
  iCanvas->SetSnap(iSnapData);
  iCanvas->Update();
  UpdateLayers();
  SetCaption();
  statusBar()->clear();
}

void AppUi::HideLayersNotInView()
{
  iHideLayers = !iHideLayers;
  iViewMenu->setItemChecked(EHideLayersMenuId, iHideLayers);
  ViewChanged();
  iCanvas->Update();
}

void AppUi::ResolutionChanged(int resolution)
{
  // compute zoom factor from resolution: 72 = 1.0
  double zoom = resolution / 72.0;
  iCanvas->SetZoom(zoom);
  statusBar()->clear();
}

void AppUi::LineWidthChanged(int id)
{
  iAttributes.iLineWidth = iSyms[IpeAttribute::ELineWidth][id];
  Cmd(ECmdLineWidth);
}

void AppUi::AbsLineWidthChanged(int val)
{
  iAttributes.iLineWidth =
    iDoc->Repository()->ToAttribute(IpeAttribute::ELineWidth, val / 10.0);
  Cmd(ECmdLineWidth);
}

void AppUi::DashStyleChanged(int id)
{
  iAttributes.iDashStyle = iSyms[IpeAttribute::EDashStyle][id];
  Cmd(ECmdDashStyle);
}

void AppUi::TextSizeChanged(int id)
{
  iAttributes.iTextSize = iSyms[IpeAttribute::ETextSize][id];
  Cmd(ECmdTextSize);
}

void AppUi::AbsTextSizeChanged(int val)
{
  iAttributes.iTextSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::ETextSize, val / 10.0);
  Cmd(ECmdTextSize);
}

void AppUi::MarkShapeChanged(int id)
{
  iAttributes.iMarkShape = id + 1;
  Cmd(ECmdMarkShape);
}

void AppUi::MarkSizeChanged(int id)
{
  iAttributes.iMarkSize = iSyms[IpeAttribute::EMarkSize][id];
  Cmd(ECmdMarkSize);
}

void AppUi::AbsMarkSizeChanged(int val)
{
  iAttributes.iMarkSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::EMarkSize, val / 10.0);
  Cmd(ECmdMarkSize);
}

void AppUi::ArrowChanged(int id)
{
  iAttributes.iForwardArrow = (id & 1) != 0;
  iAttributes.iBackwardArrow = (id & 2) != 0;
  Cmd(ECmdSetArrows);
}

void AppUi::ArrowSizeChanged(int id)
{
  iAttributes.iArrowSize = iSyms[IpeAttribute::EArrowSize][id];
  Cmd(ECmdArrowSize);
}

void AppUi::AbsArrowSizeChanged(int val)
{
  iAttributes.iArrowSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::EArrowSize, val / 10.0);
  Cmd(ECmdArrowSize);
}

void AppUi::GridSizeChanged(int id)
{
  iSnapData.iGridSize = AbsValue(iSyms[IpeAttribute::EGridSize][id]);
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iGridVisible)
    iCanvas->Update();
  if (Page())
    Page()->SetGridSize(iSnapData.iGridSize);
  statusBar()->clear();
}

void AppUi::AbsGridSizeChanged(int val)
{
  iSnapData.iGridSize = val / 10.0;
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iGridVisible)
    iCanvas->Update();
  if (Page())
    Page()->SetGridSize(iSnapData.iGridSize);
  statusBar()->clear();
}

void AppUi::AngleSizeChanged(int id)
{
  iSnapData.iAngleSize =
    IpeAngle::Degrees(AbsValue(iSyms[IpeAttribute::EAngleSize][id]));
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iWithAxes)
    iCanvas->Update();
  statusBar()->clear();
}

void AppUi::AbsAngleSizeChanged(int val)
{
  iSnapData.iAngleSize = IpeAngle::Degrees(val / 10.0);
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iWithAxes)
    iCanvas->Update();
  statusBar()->clear();
}

//! Snapping was changed.
void AppUi::SnapChanged(bool)
{
  iSnapData.iSnap = 0;
  for (int i = 0; i < 6; ++i) {
    if (iSnapAction[i]->isOn())
      iSnapData.iSnap |= (1 << i);
  }
  if (iSnapData.iSnap & IpeSnapData::ESnapAngle)
    iSnapData.iWithAxes = true;
  iCanvas->SetSnap(iSnapData);
  statusBar()->clear();
}

void AppUi::GridVisible()
{
  iSnapData.iGridVisible = !iSnapData.iGridVisible;
  iZoomMenu->setItemChecked(EGridVisibleMenuId, iSnapData.iGridVisible);
  iCanvas->SetSnap(iSnapData);
  iCanvas->Update();
  statusBar()->clear();
}

//! Slot for the absolute stroke color button.
void AppUi::SetStrokeColor()
{
  QColor stroke = QColorDialog::getColor(Qt::red, 0);
  SetColorFromQColor(iAttributes.iStroke, stroke, iAbsStrokeColor);
  Cmd(ECmdStroke);
}

//! Slot for the absolute fill color button.
void AppUi::SetFillColor()
{
  QColor fill = QColorDialog::getColor(Qt::red, 0);
  SetColorFromQColor(iAttributes.iFill, fill, iAbsFillColor);
  Cmd(ECmdFill);
}

//! Slot for the stroke color comboBox.
void AppUi::SetStrokeColorName(int item)
{
  iAttributes.iStroke = iSyms[IpeAttribute::EColor][item];
  Cmd(ECmdStroke);
}

//! Slot for the fill color comboBox.
void AppUi::SetFillColorName(int item)
{
  iAttributes.iFill = iSyms[IpeAttribute::EColor][item];
  Cmd(ECmdFill);
}

// ----------------------------------------------------------
// Private utility functions
// ----------------------------------------------------------

void AppUi::UpdateLayers()
{
  // make sure it doesn't try to change layers
  iLayerList->disconnect(this);
  iLayerList->clear();
  bool selAll = (Page()->CountViews() == 0);
  std::vector<IpeString> layers;
  if (!selAll)
    layers = Page()->View(iVno).iLayers;
  for (int i = 0; i < Page()->CountLayers(); ++i) {
    const IpeLayer &l = Page()->Layer(i);
    QString s;
    if (!l.IsVisible())
      s += "H";
    else if (l.IsDimmed())
      s += "D";
    if (l.IsLocked())
      s += "L";
    if (!l.IsSnapping())
      s += "S";
    if (!s.isEmpty())
      s += " ";
    if (i == iLayer)
      s += "<<< " + QIpe(l.iName) + " >>>";
    else
      s += QIpe(l.iName);
    iLayerList->insertItem(s);
    iLayerList->setSelected
      (i, selAll ||
       std::find(layers.begin(), layers.end(), l.iName) != layers.end());
  }
  // reconnect signals
  connect(iLayerList,
	  SIGNAL(rightButtonPressed(QListBoxItem *, const QPoint &)),
	  this,
	  SLOT(LayerRightPress(QListBoxItem *, const QPoint &)));
  connect(iLayerList, SIGNAL(selectionChanged()),
	  this, SLOT(LayerSelectionChanged()));
#if 0
  // update menu
  const IpeLayer &l = Page()->Layer(iLayer);
  iLayerMenu->setItemChecked(ECmdHideLayer, !l.IsVisible());
  iLayerMenu->setItemChecked(ECmdDimLayer, l.IsDimmed());
  iLayerMenu->setItemChecked(ECmdLockLayer, l.IsLocked());
  iLayerMenu->setItemChecked(ECmdUnsnapLayer, !l.IsSnapping());
#endif
}

void AppUi::LayerRightPress(QListBoxItem *item, const QPoint &pos)
{
  enum { EActivate, EDelete, EHide, EUnhide, EDim, EUndim,
	 ELock, EUnlock, ESnap, EUnsnap };

  IpePage *page = Page();

  int index = iLayerList->index(item);
  if (index < 0)
    return;
  IpeLayer layer = page->Layer(index);

  QPopupMenu *menu = new QPopupMenu;
  if (index != iLayer) {
    if (!layer.IsLocked())
      menu->insertItem(tr("Set as active"), EActivate);
    menu->insertItem(tr("Delete"), EDelete);
  }
  if (layer.IsVisible()) {
    menu->insertItem(tr("Hide"), EHide);
    if (layer.IsDimmed())
      menu->insertItem(tr("Undim"), EUndim);
    else
      menu->insertItem(tr("Dim"), EDim);
  } else
    menu->insertItem(tr("Unhide"), EUnhide);
  if (layer.IsLocked())
    menu->insertItem(tr("Unlock layer"), EUnlock);
  else if (index != iLayer && !page->IsLayerActiveInView(index))
    menu->insertItem(tr("Lock layer"), ELock);
  if (layer.IsSnapping())
    menu->insertItem(tr("Disable snapping"), EUnsnap);
  else
    menu->insertItem(tr("Enable snapping"), ESnap);

  switch (menu->exec(pos)) {
  case EActivate:
    iLayer = index;
    if (page->CountViews() > 0) {
      IpeView view = page->View(iVno);
      view.iActive = page->Layer(iLayer).Name();
      IpePage *originalPage = new IpePage(*page);
      page->SetView(iVno, view);
      AddUndo(new IpeUndoPageEdit(iPno, originalPage,
				  IpeQ(tr("active layer change"))));
    }
    // return immediately, don't use layer undo below
    UpdateLayers();
    return;
  case EDelete:
    if (page->CountLayers() > 1 && index != iLayer) {
      IpePage::const_iterator it = page->begin();
      while (it != page->end() && it->Layer() != iLayer)
	++it;
      if (it != page->end()) {
	statusBar()->message(tr("Layer is not empty, not deleted"));
	return;
      }
      if (iLayer > index)
	--iLayer;
      IpePage *originalPage = new IpePage(*page);
      page->DeleteLayer(index);
      AddUndo(new IpeUndoPageEdit(iPno, originalPage,
				  IpeQ(tr("deleting layer"))));
      UpdateLayers();
    }
    // return immediately, don't use layer undo below
    return;
  case EHide:
    layer.SetVisible(false);
    break;
  case EUnhide:
    layer.SetVisible(true);
    break;
  case EDim:
    layer.SetDimmed(true);
    break;
  case EUndim:
    layer.SetDimmed(false);
    break;
  case ELock:
    layer.SetLocked(true);
    page->DeselectLayer(index); // no object in this layer must be selected
    break;
  case EUnlock:
    layer.SetLocked(false);
    break;
  case ESnap:
    layer.SetSnapping(true);
    break;
  case EUnsnap:
    layer.SetSnapping(false);
    break;
  default:
    // nothing
    return;
  }
  IpePage *originalPage = new IpePage(*page);
  page->SetLayer(index, layer);
  AddUndo(new IpeUndoPageEdit(iPno, originalPage,
			      IpeQ(tr("layer modification"))));
  UpdateLayers();
  iCanvas->Update();
  SetCaption();
}

void AppUi::LayerSelectionChanged()
{
  IpePage *page = Page();
  if (page->CountViews() == 0) {
    // it does not yet have a view
    // check if all selected
    bool all = true;
    for (uint i = 0; i < iLayerList->count(); ++i) {
      if (!iLayerList->isSelected(i))
	all = false;
    }
    if (all)
      return; // okay
    // need to create the first view
    IpeView view; // default w/o any layers
    page->AddView(view);
  }
  IpeView view = page->View(iVno);
  view.iLayers.clear();
  for (int i = 0; i < page->CountLayers(); ++i) {
    if (iLayerList->isSelected(i))
      view.iLayers.push_back(page->Layer(i).Name());
  }
  view.iActive = page->Layer(iLayer).Name();
  IpePage *original = new IpePage(*page);
  page->SetView(iVno, view);
  AddUndo(new IpeUndoPageEdit(iPno, original, IpeQ(tr("view modification"))));
  if (iHideLayers)
    iCanvas->Update();
  statusBar()->message(tr("View modified"));
}

double AppUi::AbsValue(IpeAttribute attr)
{
  IpeAttribute abs = iDoc->StyleSheet()->Find(attr);
  return iDoc->Repository()->ToScalar(abs);
}

void AppUi::ConvertAbsolute(DecimalSpinBox *spin, IpeAttribute &attr)
{
  IpeAttribute abs = iDoc->StyleSheet()->Find(attr);
  double value = iDoc->Repository()->ToScalar(abs);
  spin->setValueFromAttribute(int(10 * value));
  attr = abs;
}

//! Switch to displaying absolute or symbolic values
void AppUi::SwitchAttributeUi()
{
  bool a = (iAbsoluteAttributes ? 1 : 0);
  iStrokeColorStack->raiseWidget(a);
  iFillColorStack->raiseWidget(a);
  iLineWidthStack->raiseWidget(a);
  iArrowSizeStack->raiseWidget(a);
  iTextSizeStack->raiseWidget(a);
  iMarkSizeStack->raiseWidget(a);
}

//! Switch to displaying absolute or symbolic snap values
void AppUi::SwitchSnapUi()
{
  bool a = (iAbsoluteSnapping ? 1 : 0);
  iGridSizeStack->raiseWidget(a);
  iAngleSizeStack->raiseWidget(a);
}

void AppUi::ResetCombo(IpeKind kind,
		       QComboBox *combo,
		       IpeAttribute &attr,
		       QPixmap *pixmap)
{
  combo->clear();
  for (IpeAttributeSeq::const_iterator it = iSyms[kind].begin();
       it != iSyms[kind].end(); ++it) {
    if (pixmap)
      combo->insertItem(*pixmap, QIpe(iDoc->Repository()->String(*it)));
    else
      combo->insertItem(QIpe(iDoc->Repository()->String(*it)));
  }
  combo->setCurrentItem(0);
  attr = iSyms[kind][0];
}

void AppUi::ShowStyleInUi()
{
  iDoc->StyleSheet()->AllNames(iSyms);

  iFillColor->clear();
  iStrokeColor->clear();
  for (IpeAttributeSeq::const_iterator
	 it = iSyms[IpeAttribute::EColor].begin();
       it != iSyms[IpeAttribute::EColor].end(); ++it) {
    InsertColor(iStrokeColor, *it);
    InsertColor(iFillColor, *it);
  }
  iStrokeColor->setCurrentItem(1);
  iAttributes.iStroke = iSyms[IpeAttribute::EColor][1];
  iAbsStrokeColor->setPixmap(ColorPixmap(iAttributes.iStroke,
					 iDoc->StyleSheet()));

  iFillColor->setCurrentItem(0);
  iAttributes.iFill = iSyms[IpeAttribute::EColor][0];
  iAbsFillColor->setPixmap(ColorPixmap(iAttributes.iFill, iDoc->StyleSheet()));

  iDashStyle->clear();
  for (IpeAttributeSeq::const_iterator
	 it = iSyms[IpeAttribute::EDashStyle].begin();
       it != iSyms[IpeAttribute::EDashStyle].end(); ++it) {
    iDashStyle->insertItem(QIpe(iDoc->Repository()->String(*it)));
  }
  iDashStyle->setCurrentItem(0);
  iAttributes.iDashStyle = iSyms[IpeAttribute::EDashStyle][0];

  QPixmap lwIcon = ipeIcon("lineWidth");
  QPixmap arrowIcon = ipeIcon("arrow");
  QPixmap abcIcon = ipeIcon("paragraph");
  QPixmap markIcon = ipeIcon("marks");

  ResetCombo(IpeAttribute::ELineWidth, iLineWidth,
	     iAttributes.iLineWidth, &lwIcon);
  ResetCombo(IpeAttribute::ETextSize, iTextSize,
	     iAttributes.iTextSize, &abcIcon);
  ResetCombo(IpeAttribute::EArrowSize, iArrowSize,
	     iAttributes.iArrowSize, &arrowIcon);
  ResetCombo(IpeAttribute::EMarkSize, iMarkSize,
	     iAttributes.iMarkSize, &markIcon);

  IpeAttribute grid;
  ResetCombo(IpeAttribute::EGridSize, iGridSize, grid);
  iSnapData.iGridSize = AbsValue(grid);

  IpeAttribute angle;
  ResetCombo(IpeAttribute::EAngleSize, iAngleSize, angle);
  iSnapData.iAngleSize = IpeAngle::Degrees(AbsValue(angle));
  iCanvas->SetSnap(iSnapData);
}

void AppUi::InsertColor(QComboBox *combo, IpeAttribute sym)
{
  combo->insertItem(ColorPixmap(sym, iDoc->StyleSheet()),
		    QIpe(iDoc->Repository()->String(sym)));
}

// Set gridsize box from page information
void AppUi::SetGridSizeFromPage()
{
  IpeScalar gs = Page()->GridSize();
  if (gs > 0.0) {
    // find correct symbolic gridsize
    int sym = -1;
    for (uint i = 0; i < iSyms[IpeAttribute::EGridSize].size(); ++i) {
      if (IpeAbs(AbsValue(iSyms[IpeAttribute::EGridSize][i]) - gs) < 1e-10) {
	sym = i;
	break;
      }
    }
    // change absolute snapping mode
    if (iAbsoluteSnapping && sym >= 0 || !iAbsoluteSnapping && sym < 0)
      AbsoluteSnapping();
    if (iAbsoluteSnapping)
      iAbsGridSize->setValueFromAttribute(int(gs * 10.0));
    else
      iGridSize->setCurrentItem(sym);
    iSnapData.iGridSize = gs;
    iCanvas->SetSnap(iSnapData);
    if (iSnapData.iGridVisible)
      iCanvas->Update();
  }
}

/*! Set \c colorToSet from \c qColor, and update the color in the \c
  button.
*/
void AppUi::SetColorFromQColor(IpeAttribute &colorToSet, QColor &qColor,
			       QPushButton *button)
{
  QPixmap pixmap;
  pixmap.resize(14, 14);
  pixmap.fill(qColor);
  button->setPixmap(pixmap);
  int r, g, b;
  qColor.rgb(&r, &g, &b);
  colorToSet = iDoc->Repository()->
    ToAttribute(IpeColor(r / 255.0, g / 255.0, b / 255.0));
}

// --------------------------------------------------------------------

void AppUi::ErrMsg(QString str)
{
  QMessageBox::warning(this, "Ipe", "<qt>" + str + "</qt>", tr("Dismiss"));
}

void AppUi::InsertTextBox()
{
  // need to use media and margins
  IpeRect r = Page()->TextBox(iDoc->Properties().iMedia, iDoc->StyleSheet());
  IpeCreateText::New(0, iCanvas, this, IpeCreateText::EMinipage, &r);
}

void AppUi::InsertItemBox()
{
  // need to use media and margins
  IpeRect r = Page()->TextBox(iDoc->Properties().iMedia, iDoc->StyleSheet());
  IpeCreateText::New(0, iCanvas, this, IpeCreateText::EMinipage, &r,
		     "\\begin{ITEM}\n\\end{ITEM}\n");
}

#if QT_VERSION >= 300
QKeySequence AppUi::Key(const char *src, const char *comment)
{
  QString transl = qApp->translate("Key", src, comment);
  QTranslatorMessage m("Key", src, comment, transl);
  iKeyTranslator.insert(m);
  return QKeySequence(transl);
}
#else
int AppUi::Key(const char *src, const char *comment)
{
  QString transl = qApp->translate("Key", src, comment);
  QTranslatorMessage m("Key", src, comment, transl);
  iKeyTranslator.insert(m);
  int key = QAccel::stringToKey(transl);
  if ((key & CTRL) && (key & 0xffff) < 0x100)
    key &= ~UNICODE_ACCEL;
  return key;
}
#endif

IpeAttribute AppUi::PageColor()
{
  if (IpePreferences::Static()->iWhitePaper)
    return IpeAttribute::White();
  return iDoc->Repository()->ToAttribute(IpeColor(1.0, 1.0, 0.5));
}

// --------------------------------------------------------------------
// IpeCanvasServices interface
// --------------------------------------------------------------------

void AppUi::CvSvcRequestOverlay(QMouseEvent *ev)
{
  IpeOverlayFactory f(ev, iCanvas, Page(), iSnapData.iView, this);
  // need to select right Overlay
  if (ev->button() == RightButton) {
    if (IpePreferences::Static()->iRightMouseSelects &&
	!(ev->state() & ControlButton)) {
      f.CreateOverlay(IpeOverlayFactory::ESelecting);
    } else {
      double d = IpePreferences::Static()->iSelectDistance / iCanvas->Zoom();
      if (Page()->UpdateCloseSelection(iCanvas->Pos(), d, true)) {
	AttributePopup *pop =
	  new AttributePopup(Page()->PrimarySelection(), Page(),
			     iCanvas, iLayer,
			     iDoc->StyleSheet(), iSyms, this);
	pop->Exec();
      }
      SetCaption();
    }
  } else if (ev->button() == MidButton) {
    if ((ev->state() & ControlButton) && (ev->state() & ShiftButton))
      f.CreateOverlay(IpeOverlayFactory::EMoving);
    else if (ev->state() & ControlButton)
      f.CreateOverlay(IpeOverlayFactory::ERotating);
    else if (ev->state() & (AltButton|ShiftButton))
      f.CreateOverlay(IpeOverlayFactory::EPanning);
    else
      f.CreateOverlay(IpeOverlayFactory::EMoving);
  } else {
    // Left button
    if (ev->state() & (AltButton|ControlButton))
      f.CreateOverlay(IpeOverlayFactory::EStretching);
    else
      f.CreateOverlay(iMode);
  }
}

void AppUi::CvSvcSetDrawingMode(bool mode)
{
  for (uint i = 0; i < menuBar()->count(); ++i) {
    menuBar()->setItemEnabled(menuBar()->idAt(i), !mode);
  }
  menuBar()->setItemEnabled(EZoomMenuId, true);
  menuBar()->setItemEnabled(ESnapMenuId, true);
  iPageTools->setEnabled(!mode);
  iFileTools->setEnabled(!mode);
  iModeActionGroup->setEnabled(!mode);
}

void AppUi::CvSvcWheelZoom(int delta)
{
  if (delta > 0)
    iResolution->stepUp();
  else
    iResolution->stepDown();
}

void AppUi::CvSvcSetRenderData(IpeBitmap &bitmap)
{
  QImage img = IpeImageFilter::Decode(bitmap);
  if (img.width() > 200 || img.height() > 200) {
    int wid = img.width();
    int ht = img.height();
    double factor;
    if (wid > ht)
      factor = wid / 200.0;
    else
      factor = ht / 200.0;
    wid = int(wid / factor + 0.5);
    ht = int(ht / factor + 0.5);
    bitmap.SetRenderData(new QImage(img.smoothScale(wid, ht)));
  } else
    bitmap.SetRenderData(new QImage(img));
}

void AppUi::CvSvcRenderCharacter(QPixmap *pixmap, int fontObject,
			    const IpeMatrix &m, const IpeVector &pos,
			    QRgb cRgb, uint charCode, uint uniCode)
{
  if (iDoc->FontPool())
    iDoc->FontPool()->RenderCharacter(pixmap, fontObject, m, pos,
				      cRgb, charCode, uniCode);
}

// --------------------------------------------------------------------
// IpeOverlayServices interface
// --------------------------------------------------------------------

void AppUi::OvSvcAddObject(IpeObject *obj)
{
  IpePage *page = Page();
  page->DeselectAll();
  page->push_back(IpePgObject(IpePgObject::EPrimary, iLayer, obj));
  page->SetEdited(true);
  AddUndo(new IpeUndoObjInsertion(iPno, IpeQ(tr("object insertion"))));
  SetCaption();
}

const IpeAllAttributes &AppUi::OvSvcAttributes()
{
  return iAttributes;
}

const IpeStyleSheet *AppUi::OvSvcStyleSheet()
{
  return iDoc->StyleSheet();
}

void AppUi::OvSvcAddUndoItem(IpePage *page, QString s)
{
  AddUndo(new IpeUndoPageEdit(iPno, page, IpeQ(s)));
}

void AppUi::OvSvcAddUndoItem(IpePage::iterator it, IpePage *page,
			     const IpePgObject &original, QString s)
{
  AddUndo(new IpeUndoObjectEdit(iPno, it, page, original, IpeQ(s)));
}

// --------------------------------------------------------------------
// IpeletHelper interface
// --------------------------------------------------------------------

void AppUi::Message(const char *msg)
{
  statusBar()->message(msg);
}

int AppUi::MessageBox(const char *text, const char *button1,
		      const char *button2, const char *button3)
{
  return (QMessageBox::information(this, "Ipelet",
				   QString("<qt>") + text + "</qt>",
				   button1, button2, button3));
}

bool AppUi::GetString(const char *prompt, IpeString &str)
{
  bool ok = false;
  QString text = QInputDialog::getText("Ipelet", prompt,
				       QLineEdit::Normal,
				       QIpe(str), &ok, this);
  if (ok)
    str = IpeQ(text);
  return ok;
}

const IpeStyleSheet *AppUi::StyleSheet()
{
  return iDoc->StyleSheet();
}

const IpeDocument *AppUi::Document()
{
  return iDoc;
}

IpeDocument *AppUi::EditDocument()
{
  iDoc->SetEdited(true);
  iIpeletNoUndo = true;
  return iDoc;
}

int AppUi::CurrentPage() const
{
  return iPno;
}

int AppUi::CurrentView() const
{
  return iVno;
}

int AppUi::CurrentLayer() const
{
  return iLayer;
}

const IpeAllAttributes &AppUi::Attributes() const
{
  return iAttributes;
}

const IpeSnapData &AppUi::SnapData() const
{
  return iSnapData;
}

// --------------------------------------------------------------------
