// --------------------------------------------------------------------
// Various utility classes
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeutils.h"
#include "ipepage.h"
#include "ipegroup.h"
#include "iperef.h"
#include "ipeimage.h"
#include "ipetext.h"

// --------------------------------------------------------------------

/*\class IpeBitmapFinder
  \ingroup high
  \brief A visitor that recursively scans objects and collects all bitmaps.
*/

void IpeBitmapFinder::ScanPage(const IpePage *page)
{
  for (IpePage::const_iterator it = page->begin(); it != page->end(); ++it)
    (*this)(*it);
}

void IpeBitmapFinder::VisitGroup(const IpeGroup *obj)
{
  for (IpeGroup::const_iterator it = obj->begin(); it != obj->end(); ++it)
    (*it)->Accept(*this);
}

void IpeBitmapFinder::VisitReference(const IpeReference *obj)
{
  if (obj->Object())
    obj->Object()->Accept(*this);
}

void IpeBitmapFinder::VisitImage(const IpeImage *obj)
{
  iBitmaps.push_back(obj->Bitmap());
}

// --------------------------------------------------------------------

IpeBBoxPainter::IpeBBoxPainter(const IpeStyleSheet *style)
  : IpePainter(style)
{
  // nothing else
}

void IpeBBoxPainter::BeginPath(const IpeVector &v)
{
  iV = Matrix() * v;
  Add(iV);
}

void IpeBBoxPainter::BeginClosedPath(const IpeVector &v)
{
  iV = Matrix() * v;
  Add(iV);
}

void IpeBBoxPainter::LineTo(const IpeVector &v)
{
  iV = Matrix() * v;
  Add(iV);
}

void IpeBBoxPainter::CurveTo(const IpeVector &v1, const IpeVector &v2,
			    const IpeVector &v3)
{
  IpeBezier bez(iV, Matrix() * v1, Matrix() * v2, Matrix() * v3);
  IpeRect bb = bez.BBox();
  double lw = 0.0;
  if (LineWidth())
    lw = Repository()->ToScalar(LineWidth());
  iBBox.AddPoint(bb.Min() - IpeVector(lw, lw));
  iBBox.AddPoint(bb.Max() + IpeVector(lw, lw));
  iV = Matrix() * v3;
}

void IpeBBoxPainter::DrawBitmap(IpeBitmap)
{
  iBBox.AddPoint(Matrix() * IpeVector(0.0, 0.0));
  iBBox.AddPoint(Matrix() * IpeVector(0.0, 1.0));
  iBBox.AddPoint(Matrix() * IpeVector(1.0, 1.0));
  iBBox.AddPoint(Matrix() * IpeVector(1.0, 0.0));
}

void IpeBBoxPainter::DrawText(const IpeText *text)
{
  iBBox.AddPoint(Matrix() * IpeVector(0,0));
  iBBox.AddPoint(Matrix() * IpeVector(0, text->TotalHeight()));
  iBBox.AddPoint(Matrix() * IpeVector(text->Width(), text->TotalHeight()));
  iBBox.AddPoint(Matrix() * IpeVector(text->Width(), 0));
}

void IpeBBoxPainter::Add(const IpeVector &pos)
{
  double lw = 0.0;
  if (LineWidth())
    lw = Repository()->ToScalar(LineWidth());
  iBBox.AddPoint(IpeVector(pos.iX - lw, pos.iY - lw));
  iBBox.AddPoint(IpeVector(pos.iX + lw, pos.iY + lw));
}

// --------------------------------------------------------------------
