// --------------------------------------------------------------------
// The pool of embedded fonts.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipefontpool.h"
#include "ipeprefs.h"

#include "ipefontpool_p.h"
#include "ipeftfont_p.h"

#include <qdir.h>
#include <qstring.h>
#include <qbuffer.h>

// --------------------------------------------------------------------

/*! \class IpeFontPool
  \brief The PDF fonts taken from Pdflatex output.

  IpeFontPool is created by IpeLatex and stores each font both as a
  string (for embedding into PDF files to be created by Ipe) and as a
  Freetype face object FTFontFile (for rendering to the screen).

  After the IpeFontPool has been created, the Pdflatex output PDF file
  is no longer needed.

  Code for PDF font caching taken from XPDF.
*/

//! Create empty font pool.
IpeFontPool::IpeFontPool()
{
  iPriv = new IpeFontPoolPriv;
}

//! Destructor.
IpeFontPool::~IpeFontPool()
{
  delete iPriv;
}

void IpeFontPool::RenderCharacter(QPixmap *pixmap, int fontObject,
				  const IpeMatrix &m,
				  const IpeVector &pos,
				  QRgb cRgb, uint charCode, uint uniCode) const
{
  Ref id;
  id.num = fontObject;
  id.gen = 0;
  FTFont *font = iPriv->GetFont(&id, m.iA[0], m.iA[1], m.iA[2], m.iA[3]);
  if (font)
    font->DrawChar(pixmap, int(pos.iX), int(pos.iY), cRgb, charCode, uniCode);
}

// --------------------------------------------------------------------

IpeFontPoolPriv::IpeFontPoolPriv()
{
  // with anti-aliasing
  iEngine = new FTFontEngine(IpePreferences::Static()->iAntiAlias);
  if (!iEngine->IsOk()) {
    delete iEngine;
    iEngine = 0;
  }
}

//! Destructor.
IpeFontPoolPriv::~IpeFontPoolPriv()
{
  for (FTFontSeq::iterator it = iFTFonts.begin(); it != iFTFonts.end(); ++it)
    delete *it;
  for (FTFaceSeq::iterator it = iFTFaces.begin(); it != iFTFaces.end(); ++it)
    delete *it;
  delete iEngine;
}

//! Load a face.
FTFace *IpeFontPoolPriv::GetFace(XRef *xref, GfxFont *gfxFont)
{
  FTFace *face;

  // try for a cached FontFile, embedded font, or an external font file
  face = 0;
  switch (gfxFont->getType()) {
  case fontType1:
  case fontType1C:
    face = TryGetFTFace(xref, gfxFont);
    break;
  case fontTrueType:
  case fontCIDType2:
    face = TryGetFTFace(xref, gfxFont);
    break;
  case fontCIDType0C:
    face = TryGetFTFace(xref, gfxFont);
    break;
  default:
    break;
  }

  // get standard Postscript font
  if (!face && gfxFont->getType() == fontType1) {
    IpeBuffer buffer = IpePreferences::Static()->
      StandardFont(gfxFont->getName()->getCString());
    if (buffer.size() > 0)
      face = TryGetFTFaceFromBuffer(xref, buffer.data(), buffer.size(),
				    gfxFont);
  }

  // check for error
  if (!face) {
    // This will happen if the user specifies a bogus font in the
    // config file (a non-existent font file or a font that requires a
    // rasterizer that is disabled or wasn't built in), or if a CID
    // font has no associated font in the config file.
    if (gfxFont->isCIDFont()) {
      error(-1, "Couldn't find a font for the '%s' character collection",
	    ((GfxCIDFont *)gfxFont)->getCollection()->getCString());
    } else {
      error(-1, "Couldn't find a font for '%s'",
	    gfxFont->getName() ?
	    gfxFont->getName()->getCString() : "[unnamed]");
    }
    return 0;
  }
  return face;
}

//! Get a font.
/*! The face must already have been loaded!
  Note that this function works also on a const IpeFontPoolPriv,
  as it only modifies a "hidden" (and mutable) font cache. */
FTFont *IpeFontPoolPriv::GetFont(Ref *id, double m11, double m12,
				 double m21, double m22) const
{
  // is it the most recently used font?
  if (!iFTFonts.empty() &&
      iFTFonts.front()->Matches(id, m11, m12, m21, m22)) {
    return iFTFonts.front();
  }

  // is it in the cache?
  FTFontSeq::iterator it = iFTFonts.begin();
  while (++it != iFTFonts.end()) {
    if ((*it)->Matches(id, m11, m12, m21, m22)) {
      // Reinsert it at front
      FTFont *font = *it;
      iFTFonts.erase(it);
      iFTFonts.push_front(font);
      return font;
    }
  }

  // get the face
  const FTFace *face = 0;
  for (FTFaceSeq::const_iterator it = iFTFaces.begin();
       it != iFTFaces.end(); ++it) {
    if ((*it)->IsID(id)) {
      face = *it;
      break;
    }
  }
  if (!face)
    return 0;

  // create the Font
  FTFont *font = new FTFont(face, m11, m12, m21, m22);
  if (!font->IsOk()) {
    delete font;
    return 0;
  }

  // avoid cache becoming too big
  if (iFTFonts.size() == xOutFontCacheSize) {
    delete iFTFonts.back();
    iFTFonts.pop_back();
  }

  // insert font in cache
  iFTFonts.push_front(font);
  return font;
}

FTFace *IpeFontPoolPriv::TryGetFTFace(XRef *xref, GfxFont *gfxFont)
{
  // check the already available faces
  Ref *id = gfxFont->getID();
  for (FTFaceSeq::iterator it = iFTFaces.begin(); it != iFTFaces.end(); ++it) {
    if ((*it)->IsID(id))
      return *it;
  }

  FTFace *face = 0;

  // check for an embedded font
  Ref embRef;
  if (gfxFont->getEmbeddedFontID(&embRef)) {
    // Create memory buffer
    IpeBuffer buffer;
    if (0 &&
	(gfxFont->getType() == fontTrueType ||
	 gfxFont->getType() == fontCIDType2)) {
      /*  Apparently FT cannot handle TTF fonts with missing tables,
	  so Xpdflib can create a temporary TTF font with missing parts
	  added --- test this! */
      char *fontBuf;
      int fontLen;
      TrueTypeFontFile *ff;
      if (!(fontBuf = gfxFont->readEmbFontFile(xref, &fontLen)))
	return 0;
      // create the font file
      QByteArray ttfArr;
      QBuffer *ttfBuf = new QBuffer(ttfArr);
      ttfBuf->open(IO_WriteOnly);
      OCFILE *f = ocfopen(ttfBuf);
      ff = new TrueTypeFontFile(fontBuf, fontLen);
      ff->writeTTF(f);
#ifdef IPE_DEBUG_FONTPOOL
      IPEDBG << "Embedded TTF font has " << fontLen << " bytes, "
	  << "rewritten TTF has " << int(ttfArr.size()) << " bytes\n";
#endif
      delete ff;
      gfree(fontBuf);
      fclose(f);  // deletes the ttfBuf object
      // save data
      buffer = IpeBuffer(ttfArr.data(), ttfArr.size());
    } else {
      Object refObj, strObj;
      refObj.initRef(embRef.num, embRef.gen);
      refObj.fetch(xref, &strObj);
      refObj.free();
      // copy stream to buffer
      std::vector<char> buf;
      strObj.streamReset();
      int ch;
      while ((ch = strObj.streamGetChar()) != EOF)
	buf.push_back(char(ch));
      strObj.streamClose();
      strObj.free();
      buffer = IpeBuffer(&buf[0], buf.size());
    }
    face = TryGetFTFaceFromBuffer(xref, buffer.data(), buffer.size(), gfxFont);
    EmbeddableFont font;
    font.iLatexNumber = -1;
    font.iObjectNumber = gfxFont->getID()->num;
    font.iStreamData = buffer;
    iFonts.push_back(font);
  }
  return face;
}

FTFace *IpeFontPoolPriv::TrySettingFTFaceMode(FTFace *face, GfxFont *gfxFont)
{
  if (!face->IsOk()) {
    error(-1, "Couldn't create FreeType font '%s'",
	  gfxFont->getName()->getCString());
    delete face;
    return 0;
  }

  if (gfxFont->isCIDFont()) {
    if (gfxFont->getType() == fontCIDType2) {
      face->SetMode(((GfxCIDFont *)gfxFont)->getCIDToGID(),
		    ((GfxCIDFont *)gfxFont)->getCIDToGIDLen());
    } else { // fontCIDType0C
      face->SetMode();
    }
  } else {
    face->SetMode(((Gfx8BitFont *)gfxFont)->getEncoding(),
		  ((Gfx8BitFont *)gfxFont)->getHasEncoding(),
		  (gfxFont->getType() == fontType1 ||
		   gfxFont->getType() == fontType1C));
  }

  if (!face->IsOk()) {
    error(-1, "Couldn't find encoding for FreeType font from '%s'",
	  gfxFont->getName()->getCString());
    delete face;
    return 0;
  }

  ipeDebug("Loaded Face (%s)", gfxFont->getName()->getCString());
  // add to list
  iFTFaces.push_back(face);
  return face;
}


FTFace *IpeFontPoolPriv::TryGetFTFaceFromBuffer(XRef *,
						const char *buffer,
						int size,
						GfxFont *gfxFont)
{
  Ref *id = gfxFont->getID();
  return TrySettingFTFaceMode(new FTFace(id, iEngine, buffer, size),
			      gfxFont);
}

// --------------------------------------------------------------------
