// --------------------------------------------------------------------
// The Ipe document, with PDF support
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeiml.h"
#include "ipepdfdoc.h"
#include "ipestyle.h"

#include "ipefontpool.h"
#include "ipeprefs.h"
#include "ipeq.h"
#include "ipeversion.h"

#include "ipepdfwriter.h"
#include "ipepdfloader_p.h"
#include "ipelatex_p.h"
#include "ipepdflib_p.h"

#include <qbuffer.h>
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

// --------------------------------------------------------------------

/*! \class IpePdfDocument
  \brief The model for an Ipe document, with added PDF support.

  Adds reading Ipe documents from PDF files (but only if they have
  been created by Ipe in the first place), writing PDF files, and
  running pdflatex to generate the PDF output for text objects.

 */

//! Construct an empty document.
IpePdfDocument::IpePdfDocument() : IpeDocument()
{
  iFontPool = 0;
}

//! Destructor.
IpePdfDocument::~IpePdfDocument()
{
  delete iFontPool;
}

//! Determine format of file \a fname.
IpePdfDocument::TFormat IpePdfDocument::FileFormat(QString fname)
{
  QFile f(fname);
  if (!f.open(IO_ReadOnly))
    return EUnknown;
  QTextStream str(&f);
  QString l = str.readLine();
  QString l2 = str.readLine();
  f.close();
  if (l.left(5) == "<?xml" || l.left(4) == "<ipe")
    return EXml;
  if (l.left(4) == "%PDF")
    return EPdf; // assume it contains an Ipe stream
  if (l.left(4) == "%!PS") {
    if (l2.left(10) == "%%Creator:" && l2.find("Ipe") >= 0)
      return EIpe5;
    return (l.find("EPS") >= 0) ? EEps : EPs;
  }
  if (l.left(5) == "%\\Ipe" || l.left(6) == "%\\MIPE")
    return EIpe5;
  return EUnknown;
}

IpePdfDocument *DoParse(IpePdfDocument *self, IpeImlParser &parser,
			int &reason)
{
  int requires;
  if (!parser.ParseDocument(*self, requires) || requires > IPELIB_VERSION) {
    delete self;
    self = 0;
    if (requires > IPELIB_VERSION)
      reason = -requires;
    else
      reason = parser.ParsePosition();
  }
  return self;
}

IpePdfDocument *DoParseXml(IpeXmlDataSource &source, int &reason)
{
  IpePdfDocument *self = new IpePdfDocument;
  IpeImlParser parser(source, self->Repository());
  return DoParse(self, parser, reason);
}

IpePdfDocument *DoParsePdf(IpePdfLoader &loader, int &reason)
{
  IpePdfDocument *self = new IpePdfDocument;
  IpePdfParser parser(loader, self->Repository());
  return DoParse(self, parser, reason);
}

//! Create a document from file.
/*! Returns 0 if file could not be read, or was not created by Ipe. */
IpePdfDocument *IpePdfDocument::New(QString fname, TFormat format,
				    int &reason)
{
  if (format == EXml) {
    QFile file(fname);
    reason = -1;
    if (!file.open(IO_ReadOnly))
      return 0;
    XmlQSource source(&file);
    IpePdfDocument *self = DoParseXml(source, reason);
    file.close();
    return self;
  }

  if (format == EPdf) {
    IpePdfLoader loader(fname);
    reason = -1;
    if (!loader.IsOk())
      return 0;
    reason = -2;
    if (!loader.IsIpeFile())
      return 0;
    /*
    QFile tmp("tmp.xml");
    assert(tmp.open(IO_WriteOnly));
    int ch;
    while ((ch = loader.GetChar()) != EOF)
      tmp.putch(ch);
    tmp.close();
    return 0;
    */
    return DoParsePdf(loader, reason);
  }

  if (format == EPs || format == EEps) {
    QFile file(fname);
    reason = -1;
    if (!file.open(IO_ReadOnly))
      return 0;
    QString s;
    do {
      file.readLine(s, 200);
    } while (s.left(10) != "%%EndSetup");
    XmlQSource source(&file, true);
    source.GetChar(); // skip '%'
    IpePdfDocument *self = DoParseXml(source, reason);
    file.close();
    return self;
  }

  return 0;
}

static void outputToFile(void *stream, char *data, int len)
{
  fwrite(data, 1, len, (OCFILE *)stream);
}

//! Save document to file.
bool IpePdfDocument::Save(QString fname, TFormat format, bool doExport)
{
  if (format == EXml) {
    QFile file(fname);
    if (!file.open(IO_WriteOnly))
      return false;
    IpeQStream stream(&file);
    SaveAsXml(stream, IPE_VERSION);
    file.close();
    return true;
  }

  if (format == EPdf) {
    QFile file(fname);
    if (!file.open(IO_WriteOnly))
      return false;
    IpePdfWriter writer(&file, StyleSheet(),
			IpePreferences::Static()->iCompressLevel,
			iProperties.iCropBox, iProperties.iMedia);
    writer.EmbedFonts(iFontPool);
    for (iterator it = begin(); it != end(); ++it)
      writer.CreatePage(*it);
    if (!doExport) {
      IpeString xmlData;
      // all bitmaps have been embedded and carry correct object number
      IpeStringStream stream(xmlData);
      SaveAsXml(stream, IPE_VERSION, true);
      writer.CreateXmlStream(xmlData);
    }
    writer.CreateTrailer(iProperties);
    file.close();
    return true;
  }

  if (format == EPs || format == EEps) {
    QBuffer *pdfBuf = new QBuffer;
    pdfBuf->open(IO_WriteOnly);
    IpePdfWriter writer(pdfBuf, StyleSheet(), 0, true, iProperties.iMedia);
    writer.EmbedFonts(iFontPool);
    for (iterator it = begin(); it != end(); ++it)
      writer.CreatePage(*it);
    writer.CreateTrailer(iProperties);
    // That's it. Now convert to PS.
    pdfBuf->close();
    pdfBuf->open(IO_ReadOnly);
    OCFILE *file = ocfopen(pdfBuf); // takes ownership of pdfBuf
    Object obj;
    obj.initNull();
    FileStream *str = new FileStream(file, 0, gFalse, 0, &obj);
    PDFDoc *doc = new PDFDoc(str);  // takes ownership of str
    if (!doc->isOk()) {
      delete doc;
      fclose(file); // deletes pdfBuf
      return false;
    }
    QFile *psFile = new QFile(fname);
    if (!psFile->open(IO_WriteOnly)) {
      delete doc;
      delete psFile;
      return false;
    }
    OCFILE *file2 = ocfopen(psFile); // takes ownership of psFile
    PSOutputDev *psOut =
      new PSOutputDev(outputToFile, file2, doc->getXRef(), doc->getCatalog(),
		      1, doc->getNumPages(),
		      (format == EEps) ? psModeEPS : psModePS);
    if (psOut->isOk()) {
      if (!doExport) {
	IpeString xmlData;
	IpeStringStream stream(xmlData);
	SaveAsXml(stream, IPE_VERSION);
	fputc('%', file2);
	for (int i = 0; i < xmlData.size() - 1; ++i) {
	  fputc(xmlData[i], file2);
	  if (xmlData[i] == '\n')
	    fputc('%', file2);
	}
	fputc('\n', file2);
      }
      doc->displayPages(psOut, 1, doc->getNumPages(), 72, 0, gFalse);
    }
    bool ok = psOut->isOk();
    delete psOut;
    delete doc;
    fclose(file);
    fclose(file2);
    return ok;
  }

  return false;
}

//! Update the font pool (after running Pdflatex).
/*! Takes ownership of the font pool. */
void IpePdfDocument::SetFontPool(IpeFontPool *fontPool)
{
  delete iFontPool;
  iFontPool = fontPool;
}

//! Run PdfLatex
/*! Prepend \a styleDirs to Latex include path. */
int IpePdfDocument::RunLatex(bool needLatex, const QStringList &styleDirs)
{
  IpePreferences *prefs = IpePreferences::Static();
  iLogFile = QString::null;
  IpeLatex converter;
  int count = 0;
  for (iterator it = begin(); it != end(); ++it)
    count += converter.ScanPage(*it);
  if (count == 0)
    return ErrNoText;
  // First we need a temporary directory
  QDir temp = QDir(prefs->iLatexDir);
  if (!temp.exists())
    // create Pdflatex directory if it doesn't exist
    QDir().mkdir(temp.absPath());
  if (!temp.exists())
    return ErrNoDir;

  QString texFile = temp.filePath("text.tex");
  QString pdfFile = temp.filePath("text.pdf");
  QString logFile = temp.filePath("text.log");
  QFile::remove(logFile);
  int err = converter.CreateLatexSource(QIpe(Properties().iPreamble),
					StyleSheet(), texFile);
  if (err < 0)
    return ErrWritingSource;
  if (err == 0 && !needLatex)
    return ErrAlreadyHaveForm;
  // file prepared correctly, now run Pdflatex
  if (styleDirs.count() > 0) {
    QCString ti = "TEXINPUTS=";
    for (uint i = 0; i < styleDirs.count(); ++i)
      ti.append((styleDirs[i] + ":").local8Bit());
    const char *texinputs = getenv("TEXINPUTS");
    if (texinputs)
      ti.append(texinputs);
    putenv(ti.data());
  }
#ifdef WIN32
  QString s = temp.filePath("runlatex.bat");
  QFile f(s);
  f.open(IO_WriteOnly);
  QTextStream stream(&f);
  QString dirPath = QDir::convertSeparators(temp.path());
  if (dirPath.length() > 2 && dirPath[1] == ':')
    stream << dirPath.left(2) << "\r\n";
  stream << "cd \"" << dirPath << "\"\r\n";
  stream << "\"" << prefs->iPdfLatex << "\"";
  if (prefs->iMikTeX) {
    for (uint i = 0; i < styleDirs.count(); ++i) {
      stream << " -include-directory=\""
	     << QDir::convertSeparators(styleDirs[i]) << "\"";
    }
  }
  stream << " text.tex\r\n";
  f.close();
  s = "call \"" + QDir::convertSeparators(s) + "\"";
  system(s.local8Bit());
#else
  QCString s = "cd " + QFile::encodeName(temp.path()) + "; " +
    QFile::encodeName(prefs->iPdfLatex) + " text.tex";
  system(s.data());
#endif
  // Check log file for Pdflatex version and errors
  QFile log(logFile);
  if (!log.open(IO_ReadOnly))
    return ErrLatex;
  iLogFile = QString(log.readAll());
  log.close();
  if (iLogFile.left(14) != "This is pdfTeX" &&
      iLogFile.left(15) != "This is pdfeTeX")
    return ErrLatex;
  prefs->iMikTeX = (iLogFile.left(80).find("MikTeX") >= 0);
  int i = iLogFile.find('-');
  if (i < 0)
    return ErrLatex;
  QString version = iLogFile.mid(i+1, 30);
  ipeDebug("pdfTeX version %s", version.latin1());
  if (version[0] == '1' && '0' <= version[1] && version[1] <= '3')
    return ErrOldPdfLatex;
  if (version.left(2) == "14" && version.left(3) < "14f")
    return ErrOldPdfLatex;
  // Check for error
  if (iLogFile.find("\n!") >= 0)
    return ErrLatex;
  if (converter.ReadPdf(pdfFile) && converter.UpdateTextObjects()) {
    SetFontPool(converter.TakeFontPool());
    ipeDebug("/RunLatex");
    return ErrNone;
  }
  return ErrLatexOutput;
}

//! Load a style sheet and add at top of cascade.
bool IpePdfDocument::AddStyleSheet(const QString &fn)
{
  QFile file(fn);
  if (!file.open(IO_ReadOnly))
    return false;
  XmlQSource source(&file);
  IpeImlParser parser(source, Repository());
  IpeStyleSheet *sheet = parser.ParseStyleSheet();
  file.close();
  if (sheet) {
    sheet->SetCascade(GetStyleSheet());
    SetStyleSheet(sheet);
    return true;
  } else
    return false;
}

// --------------------------------------------------------------------
