// --------------------------------------------------------------------
// ipetoipe
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipepdfdoc.h"
#include "ipeprefs.h"

#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qtextcodec.h>

#include <cstdio>
#include <cstdlib>

using namespace std;

#ifdef WIN32
#include <windows.h>
#endif

#ifdef __BORLANDC__
#ifdef main
#undef main
#endif
#endif

extern QPixmap toPng(const IpePdfDocument *doc, const IpePage *page,
		     double zoom);

static void ipeDebugImpl(const char *msg)
{
  fprintf(stderr, "%s\n", msg);
}

static int toxml(const char *src, const char *xml)
{
  int reason;
  IpePdfDocument *doc =
    IpePdfDocument::New(src, IpePdfDocument::FileFormat(src), reason);

  if (!doc) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    src, reason);
    return 1;
  }

  doc->Save(xml, IpePdfDocument::EXml);
  delete doc;
  return 0;
}

static int runlatex(IpePdfDocument *doc)
{
  switch (doc->RunLatex(true, QDir::currentDirPath())) {
  case IpePdfDocument::ErrNoText:
    fprintf(stderr, "No text objects in document, no need to run Pdflatex\n");
    return 0;
  case IpePdfDocument::ErrNoDir:
    fprintf(stderr, "Directory '%s' does not exist and cannot be created\n",
	    ((const char *) IpePreferences::Static()->iLatexDir.local8Bit()));
    return 1;
  case IpePdfDocument::ErrWritingSource:
    fprintf(stderr, "Error writing Latex source.\n");
    return 1;
  case IpePdfDocument::ErrOldPdfLatex:
    fprintf(stderr, "Your installed version of Pdflatex is too old.\n");
    return 1;
  case IpePdfDocument::ErrLatex:
    fprintf(stderr, "There were errors trying to run Pdflatex\n");
    return 1;
  case IpePdfDocument::ErrNone:
  default:
    fprintf(stderr, "Pdflatex was run sucessfully.\n");
    return 0;
  }
}

static int topng(const char *src, const char *dst, int pageNum, double zoom)
{
  int reason;
  IpePdfDocument *doc =
    IpePdfDocument::New(src, IpePdfDocument::FileFormat(src), reason);

  if (!doc) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    src, reason);
    return 1;
  }
  if (pageNum < 1 || pageNum > int(doc->size())) {
    fprintf(stderr,
	    "The document contains %d pages, cannot convert page %d.\n",
	    doc->size(), pageNum);
    delete doc;
    return 1;
  }

  if (runlatex(doc)) {
    delete doc;
    return 1;
  }

  const IpePage *page = (*doc)[pageNum - 1];
  QPixmap pixmap = toPng(doc, page, zoom);
  delete doc;

  fprintf(stderr, "src: %s dst: %s res: %g\n", src, dst, zoom);
  if (!pixmap.save(dst, "PNG")) {
    fprintf(stderr, "Error saving bitmap '%s'\n", dst);
    return 1;
  }
  return 0;
}

static int topdforps(const char *src, const char *dst,
		     IpePdfDocument::TFormat fm, bool doExport)
{
  int reason;
  IpePdfDocument *doc =
    IpePdfDocument::New(src, IpePdfDocument::FileFormat(src), reason);

  if (!doc) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    src, reason);
    return 1;
  }
  if (fm == IpePdfDocument::EEps && doc->size() > 1) {
    fprintf(stderr,
	    "The document contains %d pages, cannot convert to EPS format.\n",
	    doc->size());
    delete doc;
    return 1;
  }

  fprintf(stderr, "Document %s has %d pages\n", src, doc->size());

  int exitCode = runlatex(doc);
  if (!exitCode)
    doc->Save(dst, fm, doExport);
  delete doc;
  if (doExport)
    fprintf(stderr, "Warning: the exported file contains no Ipe markup!\n");
  return exitCode;
}

static void usage()
{
  fprintf(stderr, "Usage:\n (1) ipetoipe ( -xml | -eps | -pdf | -ps ) [ -export ] infile outfile\n");
  fprintf(stderr, " (2) ipe -png <page> <resolution> infile file.png\n");
  fprintf(stderr, " (3) ipe -help\n");
  fprintf(stderr, "Form (1) converts between Ipe formats.\n");
  fprintf(stderr, " (With -export output contains no Ipe markup.)\n");
  fprintf(stderr, "Form (2) saves a single page as a bitmap.\n");
  fprintf(stderr, " (<resolution> is the number of pixels per inch.)\n");
  fprintf(stderr, "Form (3) prints this  message.\n");
  exit(1);
}

int main(int argc, char *argv[])
{
  // Check Ipelib version mismatch
  if (IpelibVersion() != IPELIB_VERSION) {
    fprintf(stderr,
	    "Ipe has been compiled with header files for Ipelib %d\n"
	    "but is (probably dynamically) linked against libipe %d.\n"
	    "Check with 'ldd ipe' which libipe is being loaded, and either "
	    "replace it by the correct version or set LD_LIBRARY_PATH.\n",
	    IPELIB_VERSION, IpelibVersion());
    exit(99);
  }

  // set ipeDebug handler
  ipeDebugHandler = ipeDebugImpl;

  // ensure at least three arguments (handles -help as well :-)
  if (argc < 4)
    usage();

  QString s(argv[1]);

  if (s == "-png") {
    if (argc != 6)
      usage();
    int pageNum = IpeLex(IpeString(argv[2])).GetInt();
    double dpi = IpeLex(IpeString(argv[3])).GetDouble();
    const char *src = argv[4];
    const char *dst = argv[5];
    // topng needs a QApplication (it creates a pixmap)
    QApplication a(argc, argv);
    return topng(src, dst, pageNum, dpi / 72.0);
  }

  bool doExport = false;
  int i = 2;
  if (!qstrcmp(argv[i], "-export")) {
    doExport = true;
    ++i;
  }

  // remaining arguments must be two filenames
  if (argc != i + 2)
    usage();

  if (s == "-xml") {
    if (doExport)
      fprintf(stderr, "An XML file without Ipe markup would be empty, "
	      "ignoring -export flag.\n");
    return toxml(argv[i], argv[i+1]);
  } else if (s == "-pdf") {
    return topdforps(argv[i], argv[i+1], IpePdfDocument::EPdf, doExport);
  } else if (s == "-ps") {
    return topdforps(argv[i], argv[i+1], IpePdfDocument::EPs, doExport);
  } else if (s == "-eps") {
    return topdforps(argv[i], argv[i+1], IpePdfDocument::EEps, doExport);
  } else
    usage();
  return 0;
}

// --------------------------------------------------------------------
