// --------------------------------------------------------------------
// Pdftoipe: convert PDF file to editable Ipe XML file
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include "ocfile.h"
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include "parseargs.h"
#include "gstring.h"
#include "gmem.h"
#include "globalparams.h"
#include "object.h"
#include "stream.h"
#include "array.h"
#include "dict.h"
#include "xref.h"
#include "catalog.h"
#include "page.h"
#include "pdfdoc.h"
#include "xmloutputdev.h"
#include "error.h"

static int firstPage = 1;
static int lastPage = 0;
static char ownerPassword[33] = "";
static char userPassword[33] = "";
static GBool quiet = gFalse;
static GBool printHelp = gFalse;
static GBool math = gFalse;

static ArgDesc argDesc[] = {
  {"-f",      argInt,      &firstPage,      0,
   "first page to convert"},
  {"-l",      argInt,      &lastPage,       0,
   "last page to convert"},
  {"-opw",    argString,   ownerPassword,   sizeof(ownerPassword),
   "owner password (for encrypted files)"},
  {"-upw",    argString,   userPassword,    sizeof(userPassword),
   "user password (for encrypted files)"},
  {"-q",      argFlag,     &quiet,          0,
   "don't print any messages or errors"},
  {"-math",   argFlag,     &math,           0,
   "turn all text objects into math formulas"},
  {"-h",      argFlag,     &printHelp,      0,
   "print usage information"},
  {"-help",   argFlag,     &printHelp,      0,
   "print usage information"},
  {"--help",  argFlag,     &printHelp,      0,
   "print usage information"},
  {"-?",      argFlag,     &printHelp,      0,
   "print usage information"},
  {NULL, argFlag, 0, 0, 0}
};

int main(int argc, char *argv[])
{
  // parse args
  GBool ok = parseArgs(argDesc, &argc, argv);
  if (!ok || argc < 2 || argc > 3 || printHelp) {
    fprintf(stderr, "pdftoipe version %s\n", PDFTOIPE_VERSION);
    printUsage("pdftoipe", "<PDF-file> [<XML-file>]", argDesc);
    return 1;
  }

  GString *fileName = new GString(argv[1]);

  // read config file
  globalParams = new GlobalParams(0);
  if (quiet) {
    globalParams->setErrQuiet(quiet);
  }

  GString *ownerPW, *userPW;
  if (ownerPassword[0]) {
    ownerPW = new GString(ownerPassword);
  } else {
    ownerPW = 0;
  }
  if (userPassword[0]) {
    userPW = new GString(userPassword);
  } else {
    userPW = 0;
  }

  // open PDF file
  PDFDoc *doc = new PDFDoc(fileName, ownerPW, userPW);
  delete userPW;
  delete ownerPW;

  if (!doc->isOk())
    return 1;

  // check for print permission
  if (!doc->okToPrint()) {
    error(-1, "Printing this document is not allowed.");
    return 3;
  }

  // construct XML file name
  GString *xmlFileName;
  if (argc == 3) {
    xmlFileName = new GString(argv[2]);
  } else {
    char *p = fileName->getCString() + fileName->getLength() - 4;
    if (!strcmp(p, ".pdf") || !strcmp(p, ".PDF")) {
      xmlFileName = new GString(fileName->getCString(),
				fileName->getLength() - 4);
    } else {
      xmlFileName = fileName->copy();
    }
    xmlFileName->append(".xml");
  }

  // get page range
  if (firstPage < 1) {
    firstPage = 1;
  }
  if (lastPage < 1 || lastPage > doc->getNumPages()) {
    lastPage = doc->getNumPages();
  }

  // write XML file
  XmlOutputDev *xmlOut =
    new XmlOutputDev(xmlFileName->getCString(), doc->getXRef(),
		     doc->getCatalog(), firstPage, lastPage,
		     math);
  int exitCode = 2;
  if (xmlOut->isOk()) {
    doc->displayPages(xmlOut, firstPage, lastPage, 72, 0, gFalse);
    exitCode = 0;
  }

  // clean up
  delete xmlOut;
  delete xmlFileName;
  delete doc;
  delete globalParams;

  // check for memory leaks
  Object::memCheck(stderr);
  gMemReport(stderr);

  return exitCode;
}

// --------------------------------------------------------------------
