/***************************************************************************
 *   Copyright (C) 2004 by Michael Schulze                                 *
 *   mike.s@genion.de                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qcstring.h>
#include <qsocket.h>
#include <qdatetime.h>
#include <qbitarray.h>
#include <qbuffer.h>
#include <qregexp.h>
#include <qfileinfo.h>

#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include <kapplication.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kinstance.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kurl.h>
#include <ksock.h>
#include <kmimetype.h>
#include <kmountpoint.h>

#include "ipodslave.h"

#include "ipodutility.h"
// #include "propertyutility.h"
#include "statisticsutility.h"
#include "syncutility.h"
#include "itunesdb/itunesdbparser.h"


using namespace KIO;

extern "C"
{
    int kdemain(int argc, char **argv)
    {
        KInstance instance( "kio_ipodslave" );
        
        kdDebug() << "*** Starting kio_ipodslave " << endl;
        
        if (argc != 4) {
            kdDebug(7101) << "Usage: kio_ipodslave  protocol domain-socket1 domain-socket2" << endl;
            exit(-1);
        }
        
        kio_ipodslaveProtocol slave(argv[2], argv[3]);
        slave.dispatchLoop();
        
        kdDebug(7101) << "*** kio_ipodslave Done" << endl;
        return 0;
    }
} 

extern int errno;

kio_ipodslaveProtocol::kio_ipodslaveProtocol(const QCString &pool_socket, const QCString &app_socket)
    : SlaveBase("kio_ipodslave", pool_socket, app_socket)
{
    kdDebug() << "kio_ipodslaveProtocol::kio_ipodslaveProtocol()" << endl;
    
    // fill the utility map
    IPodUtility * utility= new StatisticsUtility(ipod);
    utilitymap.insert( utility->getName(), utility);
    // utility= new PropertyUtility(ipod);
    // utilitymap.insert( utility->getName(), utility);*/
    utility= new SyncUtility(ipod);
    utilitymap.insert( utility->getName(), utility);
    
    ensureConsistency();
}


kio_ipodslaveProtocol::~kio_ipodslaveProtocol()
{
    kdDebug() << "kio_ipodslaveProtocol::~kio_ipodslaveProtocol()" << endl;
    // cleanup utility map
    UtilityMap::iterator utility_it= utilitymap.begin();
    for( ; utility_it!= utilitymap.end(); ++utility_it) {
        delete *utility_it;
        delete utility_it.key();
    }
    utilitymap.clear();
}


void kio_ipodslaveProtocol::get(const KURL &url )
{
    ipod.lock(false);
    
    kdDebug() << "ipodslave::get()" << url.path() << endl;
    DirectoryModel dirmodel( url);    
    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__get_cleanup;
    
    if( dirmodel.type == DirectoryModel::UNKNOWN) {
        error(ERR_DOES_NOT_EXIST, url.path());
        goto kio_ipodslaveProtocol__get_cleanup;
    }

    if( !dirmodel.isFile) {
        error(  ERR_IS_DIRECTORY, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__get_cleanup;
    }
    
    switch( dirmodel.type) {
    case DirectoryModel::TRACK: {
        TrackMetadata * track= findTrack( dirmodel);
        if( track == NULL) {
            error( ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__get_cleanup;
        }
        
        // redirect to file
        redirection( KURL( QString("file:")+ ipod.getRealPath( track->getPath())));
        }
        break;
    case DirectoryModel::UTILITY: {
        QByteArray databuf;
        QString mimetypebuf;

        // get the utility
        UtilityMap::iterator utility_it= utilitymap.find( dirmodel.getFilename());
        if( utility_it == utilitymap.end()) {
            error(  ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__get_cleanup;
        }
        
        (*utility_it)->handleRequest( url, databuf, mimetypebuf);
        
        mimeType( mimetypebuf);
        data( databuf);
        }
        break;
    default:
        // we shouldn't come 'round here
        kdDebug() << "ipodslave::get() error: icky URL " << url.path() << endl;
        error(  ERR_INTERNAL, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__get_cleanup;
    }
    data(QByteArray()); // empty array means we're done sending the data
    kdDebug() << "ipodslave::get()" << url.path() << "finished." << endl;
    finished();
    
kio_ipodslaveProtocol__get_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::mimetype(const KURL &url)
{
    ipod.lock(false);
    kdDebug() << "ipodslave::mimetype()" << url.path() << endl;
    DirectoryModel dirmodel( url);    
    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__mimetype_cleanup;
    
    if( dirmodel.type == DirectoryModel::UNKNOWN) {
        error(ERR_DOES_NOT_EXIST, url.path());
        goto kio_ipodslaveProtocol__mimetype_cleanup;
    }
    
    if( !dirmodel.isFile){
        mimeType( "inode/directory");
    } else {
        switch( dirmodel.type) {
        case DirectoryModel::TRACK: {
            TrackMetadata * track= findTrack( dirmodel);
            if( track == NULL) {
                error( ERR_DOES_NOT_EXIST, url.path());
                goto kio_ipodslaveProtocol__mimetype_cleanup;
            }
            mimeType( KMimeType::findByPath( ipod.getRealPath( track->getPath()))->name());
            }
            break;
        case DirectoryModel::UTILITY: {
                // get the utility
                UtilityMap::iterator utility_it= utilitymap.find( dirmodel.getFilename());
                if( utility_it == utilitymap.end()) {
                    error(  ERR_DOES_NOT_EXIST, dirmodel.getFilename());
                    goto kio_ipodslaveProtocol__mimetype_cleanup;
                }
                
                mimeType( (*utility_it)->getDefaultMimeType());
            }
            break;
        default:
            kdDebug() << "ipodslave::get() icky default section: " << url.path() << endl;
            get(url);
        }
    }
    
    kdDebug() << "ipodslave::mimetype()" << url.path() << "finished." << endl;
    finished();
    
kio_ipodslaveProtocol__mimetype_cleanup:
    ipod.unlock();
}



/*!
    \fn kio_ipodslaveProtocol::listDir (const KURL &url)
 */
void kio_ipodslaveProtocol::listDir(const KURL &url)
{
    ipod.lock(false);
    kdDebug() << "ipodslave::listDir()" << url.path() << endl;
    DirectoryModel dirmodel( url);
    UDSEntry direntry;

    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__listDir_cleanup;
    
    if( dirmodel.type == DirectoryModel::UNKNOWN) {
        error(ERR_DOES_NOT_EXIST, url.path());
        goto kio_ipodslaveProtocol__listDir_cleanup;
    }
    
    if( dirmodel.isFile) {
        error(  ERR_IS_FILE, unsupportedActionErrorString(mProtocol, CMD_LISTDIR));
        goto kio_ipodslaveProtocol__listDir_cleanup;
    }
    
    switch( dirmodel.type) {
    case DirectoryModel::ROOT: {
        // list categories
        direntry.clear();
        fillUDSEntry( direntry, DirectoryModel::getCategoryName( DirectoryModel::Artists), 0, S_IFDIR, ipod.isChanged());
        listEntry( direntry, false);
        direntry.clear();
        fillUDSEntry( direntry, DirectoryModel::getCategoryName( DirectoryModel::Playlists), 0, S_IFDIR, ipod.isChanged());
        listEntry( direntry, false);
        direntry.clear();
        fillUDSEntry( direntry, DirectoryModel::getCategoryName( DirectoryModel::Utilites), 0, S_IFDIR, false);
        listEntry( direntry, false);
        }
        break;
    case DirectoryModel::CATEGORY:
        switch( dirmodel.getCategory()) {
        case DirectoryModel::Artists: {
            // list artists
            QStringList artists;
            if (!ipod.getArtists(artists)) {
                listEntry(direntry, true);
                goto kio_ipodslaveProtocol__listDir_cleanup;
            }
            for( QStringList::iterator artistiterator= artists.begin(); artistiterator!= artists.end(); ++artistiterator) {
                bool album_changed = false;
                direntry.clear();
                Artist * artist = ipod.getArtistByName(*artistiterator);
                if(artist == NULL)
                    continue;
                
                // check if there's a changed album for this artist
                if (!artist->isEmpty()) {
                    for(ArtistIterator albums(*artist); !album_changed && albums.current(); ++albums)
                        album_changed |= (*albums)->unsavedChanges();
                } else {
                    album_changed = true;
                }
                
                fillUDSEntry( direntry, *artistiterator, 0, S_IFDIR, album_changed);
                listEntry( direntry, false);
            }
            }
            break;
        case DirectoryModel::Playlists: {
            // list playlists
            QStringList playlisttitles;
            if(!ipod.getPlaylistTitles(playlisttitles)) {
                listEntry(direntry, true);
                goto kio_ipodslaveProtocol__listDir_cleanup;
            }
            for(QStringList::iterator iter = playlisttitles.begin(); iter != playlisttitles.end(); ++iter) {
                TrackList * tracklist = ipod.getPlaylistByTitle(*iter);
                direntry.clear();
                if(tracklist != NULL) {
                    fillUDSEntry( direntry, tracklist->getTitle(), 0, S_IFDIR, tracklist->unsavedChanges());
                } else {
                    kdDebug() << "ipodslave::listDir() " << url.path() << ": can't find playlist " << *iter << endl;
                    fillUDSEntry( direntry, *iter, 0, S_IFDIR, true);
                }
                
                listEntry( direntry, false);
            }
            }
            break;
        case DirectoryModel::Utilites: {
            // list utilities
            UtilityMap::iterator utility_it= utilitymap.begin();
            for( ; utility_it!= utilitymap.end(); ++utility_it) {
                direntry.clear();
                fillUDSEntry( direntry, utility_it.key(), 0, S_IFREG, false, &((*utility_it)->getDefaultMimeType()));
                listEntry( direntry, false);
            }
            }
            break;
        default:
            error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__listDir_cleanup;
        }
        break;
    case DirectoryModel::PLAYLIST: {
        // list tracks
        TrackList * playlist= ipod.getPlaylistByTitle( dirmodel.getFilename());
        if( playlist == NULL) {
            error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__listDir_cleanup;
        }
        int tracknum= 0;
        unsigned short trackdigits= (unsigned short)log10( playlist->getNumTracks())+ 1;
        TrackList::Iterator trackiterator = playlist->getTrackIDs();
        while( trackiterator.hasNext()) {
            Q_UINT32 trackid= trackiterator.next();
            if( trackid == LISTITEM_DELETED) {    // deleted playlist element
                ++tracknum;
                continue;
            }
            TrackMetadata * track= ipod.getTrackByID( trackid);
            if( track == NULL) {
                ++tracknum;
                continue;    // shouldn't happen - ignore this entry
            }
            direntry.clear();
            QString trackname= formatTrackname( *track, ++tracknum, trackdigits, true);
            fillUDSEntry( direntry, trackname, *track, S_IFREG, false);
            listEntry( direntry, false);
        }
        }
        break;
    case DirectoryModel::ARTIST: {
        // list albums based on given artist
        Artist * artist;
        if( ( artist= ipod.getArtistByName( dirmodel.getFilename())) == NULL) {
            error( ERR_DOES_NOT_EXIST, dirmodel.getFilename());
        } else {
            for( ArtistIterator albumiterator(*artist); albumiterator.current(); ++albumiterator) {
                direntry.clear();
                fillUDSEntry( direntry, albumiterator.currentKey(), 0, S_IFDIR, albumiterator.current()->unsavedChanges());
                listEntry( direntry, false);
            }
        }
        }
        break;
    case DirectoryModel::ALBUM: {
        // List tracks
        TrackList * album= ipod.getAlbum( dirmodel.getArtist(), dirmodel.getAlbum());
        if (album != NULL) {
            int tracknum= 0;
            unsigned short trackdigits= (unsigned short)log10( album->getMaxTrackNumber())+ 1;
            TrackList::Iterator trackiterator= album->getTrackIDs();
            while (trackiterator.hasNext()) {
                TrackMetadata * track= ipod.getTrackByID(trackiterator.next());
                if( track == NULL)
                    continue;    // this shouldn't happen - just ignore
                
                direntry.clear();
                QString trackname= formatTrackname( *track, ++tracknum, trackdigits, false);
                fillUDSEntry( direntry, trackname, *track, S_IFREG, false);
                listEntry( direntry, false);
            }
        } else {
            error( ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__listDir_cleanup;
        }
        }
        break;
    default:
        // we shouldn't come 'round here
        kdDebug() << "ipodslave::listDir() Don't know how to handle directory " << url.path() << endl;
        error(ERR_INTERNAL, url.path());
        goto kio_ipodslaveProtocol__listDir_cleanup;
    }
    
    kdDebug() << "ipodslave::listDir()" << url.path() << " finished." << endl;
    listEntry( direntry, true);
    finished();
    
kio_ipodslaveProtocol__listDir_cleanup:
    ipod.unlock();
}

/*!
    \fn kio_ipodslaveProtocol::stat(const KURL &url)
 */
void kio_ipodslaveProtocol::stat(const KURL &url)
{
    ipod.lock(false);
    kdDebug() << "ipodslave::stat() " << url.path() << endl;
    DirectoryModel dirmodel(url);
    UDSEntry direntry;
    
    if(!ensureConsistency()) {
        goto kio_ipodslaveProtocol__stat_cleanup;
    }
    if( dirmodel.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::stat() don't know how to handle URL " << url.path() << endl;
        error(ERR_DOES_NOT_EXIST, url.path());
        goto kio_ipodslaveProtocol__stat_cleanup;
    }
    if( !dirmodel.isFile) {
        // directory
        fillUDSEntry( direntry, dirmodel.getFilename(), 0, S_IFDIR, false);
        statEntry( direntry);
    } else {
        // file
        switch( dirmodel.type) {
        case DirectoryModel::TRACK: {
            TrackMetadata * track= findTrack(dirmodel);
            if( track == NULL) {
                error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
                goto kio_ipodslaveProtocol__stat_cleanup;
            }
            fillUDSEntry( direntry, dirmodel.getFilename(), *track, S_IFREG, false);
            statEntry( direntry);
            }
            break;
        case DirectoryModel::UTILITY: {
            // get the utility
            UtilityMap::iterator utility_it= utilitymap.find( dirmodel.getUtilityName());
            if( utility_it == utilitymap.end()) {
                error(  ERR_DOES_NOT_EXIST, dirmodel.getUtilityName());
                goto kio_ipodslaveProtocol__stat_cleanup;
            }
            IPodUtility * util = *utility_it;
            fillUDSEntry( direntry, util->getName(), 0, S_IFREG, false, &(util->getDefaultMimeType()));
            statEntry( direntry);
            }
            break;
        default:
            kdDebug() << "ipodslave::stat() don't know how to handle " << dirmodel.getFilename() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__stat_cleanup;
        }
    }
    
    kdDebug() << "ipodslave::stat()" << url.path() << " finished." << endl;
    finished();
    
kio_ipodslaveProtocol__stat_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::mkdir( const KURL & url, int permissions) {
    DirectoryModel dirmodel( url);
    bool ipodunchanged;
    
    ipod.lock(true);
    kdDebug() << "ipodslave::mkdir() " << url.path() << endl;
    if (!ensureConsistency())
        goto kio_ipodslaveProtocol__mkdir_cleanup;
    
    ipodunchanged = !ipod.isChanged();
    
    if (dirmodel.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::mkdir() don't know how to handle " << dirmodel.getFilename() << endl;
        error(ERR_COULD_NOT_MKDIR, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__mkdir_cleanup;
    }
    
    switch (dirmodel.type) {
    case DirectoryModel::PLAYLIST: {    // create playlist
        if (ipod.getPlaylistByTitle( dirmodel.getFilename())) {
            kdDebug() << "ipodslave::mkdir() directory already exists " << dirmodel.getFilename() << endl;
            error(ERR_DIR_ALREADY_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__mkdir_cleanup;
        }
        ipod.createPlaylist(dirmodel.getFilename());
        }
        break;
    default:
        kdDebug() << "ipodslave::mkdir() could not mkdir " << dirmodel.getFilename() << endl;
        error(ERR_COULD_NOT_MKDIR, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__mkdir_cleanup;
    }
    
    if (ipodunchanged)
        showSyncInfoMessage();
        
    kdDebug() << "ipodslave::mkdir() " << url.path() << " finished" << endl;
    finished();
    
kio_ipodslaveProtocol__mkdir_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::del( const KURL& url, bool isFile) {
    bool ipodunchanged;
    ipod.lock(true);
    
    kdDebug() << "ipodslave::del() " << url.path() << endl;
    DirectoryModel dirmodel(url);
    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__del_cleanup;
    
    ipodunchanged = !ipod.isChanged();
    
    if( dirmodel.type == DirectoryModel::UNKNOWN) {
        if( !dirmodel.isFile)
            error( ERR_COULD_NOT_RMDIR, dirmodel.getFilename());
        else
            error( ERR_CANNOT_DELETE, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__del_cleanup;
    }
    
    switch( dirmodel.type) {
    case DirectoryModel::PLAYLIST:    // delete playlist
        if (ipod.deletePlaylist(dirmodel.getFilename()) == IPod::Err_DoesNotExist) {
            error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__del_cleanup;
        }
        break;
    case DirectoryModel::TRACK: {
        int tracknum= -1;
        Track * pTrack = findTrack(dirmodel, &tracknum);
        if( pTrack == NULL) {
            kdDebug() << "ipodslave::del() : track doesn't exist " << dirmodel.getFilename() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel.getFilename());
            goto kio_ipodslaveProtocol__del_cleanup;
        }
        
        if( dirmodel.getCategory() == DirectoryModel::Artists) {    // remove Track
            QString trackfilename = ipod.getRealPath(pTrack->getPath());
            if (QFile::exists(trackfilename) && !QFile::remove(trackfilename)) {
                kdDebug() << "ipodslave::del() : track could not be deleted " << dirmodel.getFilename() << endl;
                error(ERR_CANNOT_DELETE, dirmodel.getFilename());
                goto kio_ipodslaveProtocol__del_cleanup;
            }
            ipod.deleteTrack(pTrack->getID());
        }
        else if (dirmodel.getCategory() == DirectoryModel::Playlists) {    // remove Track from playlist
            if (ipod.removeFromPlaylist(tracknum - 1, dirmodel.getCurrentDirectory()) == IPod::Err_DoesNotExist) {
                kdDebug() << "ipodslave::del() : playlist doesn't exist " << dirmodel.getCurrentDirectory() << endl;
                error(ERR_DOES_NOT_EXIST, dirmodel.getCurrentDirectory());
                goto kio_ipodslaveProtocol__del_cleanup;
            }
        } else {
            // error for now. There may be other locations for tracks in the future
            // TODO if there are more possibilities here replace the if ... else if with a switch ... case
            kdDebug() << "ipodslave::del() : unknown directory" << endl;
            error(ERR_CANNOT_DELETE, url.path());
            goto kio_ipodslaveProtocol__del_cleanup;
        }
        
        }
        break;
    case DirectoryModel::ALBUM: {
        switch (ipod.deleteAlbum(dirmodel.getArtist(), dirmodel.getAlbum())) {
        case IPod::Err_DoesNotExist:
            kdDebug() << "ipodslave::del() : album doesn't exist " << url.path() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel.getAlbum());
            goto kio_ipodslaveProtocol__del_cleanup;
        default:
            break;
        }
        }
        break;
    case DirectoryModel::ARTIST: {
        if (ipod.deleteArtist(dirmodel.getArtist()) != IPod::Err_None) {
            kdDebug() << "ipodslave::del() : artist not empty " << url.path() << endl;
            error(ERR_CANNOT_DELETE, dirmodel.getArtist());
            goto kio_ipodslaveProtocol__del_cleanup;
        }
        }
        break;
    default:
        error(ERR_CANNOT_DELETE, dirmodel.getFilename());
        goto kio_ipodslaveProtocol__del_cleanup;
    }
    
    if (ipodunchanged)
        showSyncInfoMessage();
        
    kdDebug() << "ipodslave::del() " << url.path() << "finished." << endl;
    finished();
    
kio_ipodslaveProtocol__del_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::rename( const KURL & src, const KURL & dest, bool overwrite) {
    bool ipodunchanged;
    ipod.lock(true);
    
    kdDebug() << "ipodslave::rename() " << src.path() << "->" << dest.path() << endl;
    DirectoryModel dirmodel_src(src);
    DirectoryModel dirmodel_dest(dest);
    
    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__rename_cleanup;
    
    ipodunchanged = !ipod.isChanged();
    
    if( dirmodel_src.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::rename() : unknown source type" << endl;
        error(ERR_CANNOT_RENAME, src.path());
        goto kio_ipodslaveProtocol__rename_cleanup;
    }
    if( dirmodel_dest.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::rename() : unknown dest type" << endl;
        error(ERR_UNKNOWN, dest.path());
        goto kio_ipodslaveProtocol__rename_cleanup;
    }
    
    switch( dirmodel_src.type) {
    case DirectoryModel::PLAYLIST:    // rename playlist
        if( dirmodel_dest.type != DirectoryModel::PLAYLIST) {    // is "dest" a playlist?
            kdDebug() << "ipodslave::rename() : destination not a playlist" << endl;
            error(ERR_COULD_NOT_WRITE, dirmodel_dest.getFilename());
            goto kio_ipodslaveProtocol__rename_cleanup;
        }
        
        switch(ipod.renamePlaylist(dirmodel_src.getFilename(), dirmodel_dest.getFilename())) {
        case IPod::Err_AlreadyExists:
            kdDebug() << "ipodslave::rename() : playlist already exists " << dirmodel_dest.getFilename() << endl;
            error(ERR_DIR_ALREADY_EXIST, dirmodel_dest.getFilename());
            goto kio_ipodslaveProtocol__rename_cleanup;
        case IPod::Err_DoesNotExist:
            kdDebug() << "ipodslave::rename() : playlist doesn't exist " << dirmodel_src.getFilename() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel_src.getFilename());
            goto kio_ipodslaveProtocol__rename_cleanup;
        case IPod::Err_None:
            break;
        default:
            error(ERR_INTERNAL, "ipodslave::rename");
            goto kio_ipodslaveProtocol__rename_cleanup;
        }
        break;
    case DirectoryModel::TRACK: {
        if (dirmodel_src.getFilename() != dirmodel_dest.getFilename())
            error(ERR_CANNOT_RENAME, dirmodel_dest.getFilename());    // renaming tracks is not possible
        
        switch (dirmodel_src.getCategory()) {
        case DirectoryModel::Playlists:
            error(ERR_UNSUPPORTED_ACTION, dirmodel_src.getFilename());    // redirect to copy and del
            goto kio_ipodslaveProtocol__rename_cleanup;
        case DirectoryModel::Artists:
            if (dirmodel_dest.getCategory() == DirectoryModel::Artists && dirmodel_dest.type == DirectoryModel::TRACK) {
                TrackMetadata * track = findTrack(dirmodel_src);
                if (track == NULL) {
                    error(ERR_DOES_NOT_EXIST, dirmodel_src.getFilename());
                    goto kio_ipodslaveProtocol__rename_cleanup;
                }
                ipod.moveTrack(*track, dirmodel_dest.getArtist(), dirmodel_dest.getAlbum());
            } else {
                error(ERR_SLAVE_DEFINED, "<b>Moving</b> tracks here (" + dest.path() + ") doesn't make sense - use <b>copy</b> instead");
                goto kio_ipodslaveProtocol__rename_cleanup;
            }
            break;
        default:
            error(ERR_DOES_NOT_EXIST, dirmodel_src.getFilename());
            goto kio_ipodslaveProtocol__rename_cleanup;
        }
        }
        break;
    case DirectoryModel::ALBUM: {
        if (dirmodel_dest.type != DirectoryModel::ALBUM) {
            kdDebug() << "ipodslave::rename() : destination not an album " << dirmodel_dest.getFilename() << endl;
            error(ERR_CANNOT_RENAME, dirmodel_src.getFilename());
            goto kio_ipodslaveProtocol__rename_cleanup;
        }
        switch(ipod.renameAlbum(dirmodel_src.getArtist(), dirmodel_src.getAlbum(), dirmodel_dest.getArtist(), dirmodel_dest.getAlbum())) {
        case IPod::Err_AlreadyExists:
            kdDebug() << "ipodslave::rename() : album already exists " << dirmodel_dest.getFilename() << endl;
            error(ERR_DIR_ALREADY_EXIST, dest.path());
            goto kio_ipodslaveProtocol__rename_cleanup;
        case IPod::Err_DoesNotExist:
            kdDebug() << "ipodslave::rename() : album doesn't exist " << dirmodel_src.getFilename() << endl;
            error(ERR_DOES_NOT_EXIST, src.path());
            goto kio_ipodslaveProtocol__rename_cleanup;
        case IPod::Err_None:
            break;
        default:
            error(ERR_INTERNAL, "ipodslave::rename");
            goto kio_ipodslaveProtocol__rename_cleanup;
        }
        }
        break;
    default:
        kdDebug() << "ipodslave::rename() : cannot handle " << src.path() << endl;
        error( ERR_UNSUPPORTED_ACTION, dirmodel_src.getFilename());
        goto kio_ipodslaveProtocol__rename_cleanup;
    }

    if (ipodunchanged)
        showSyncInfoMessage();

    kdDebug() << "ipodslave::rename() " << src.path() << "->" << dest.path() << " finished." << endl;
    finished();
    
kio_ipodslaveProtocol__rename_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::copy( const KURL & src, const KURL & dest, int permissions, bool overwrite) {
    bool ipodunchanged;
    ipod.lock(true);
    
    kdDebug() << "ipodslave::copy() " << src.path() << "->" << dest.path() << endl;
    DirectoryModel dirmodel_src(src);
    DirectoryModel dirmodel_dest(dest);
    
    if(!ensureConsistency())
        goto kio_ipodslaveProtocol__copy_cleanup;
    
    ipodunchanged = !ipod.isChanged();

    if( dirmodel_src.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::copy() : unknown source type" << endl;
        error(ERR_UNKNOWN, src.path());
        goto kio_ipodslaveProtocol__copy_cleanup;
    }
    if( dirmodel_dest.type == DirectoryModel::UNKNOWN) {
        kdDebug() << "ipodslave::copy() : unknown dest type" << endl;
        error(ERR_UNKNOWN, dest.path());
        goto kio_ipodslaveProtocol__copy_cleanup;
    }
    
    switch( dirmodel_src.type) {
    case DirectoryModel::TRACK: {
        if (dirmodel_dest.getCategory() == DirectoryModel::Artists) {
            kdDebug() << "ipodslave::copy() : Only one instance of this track is allowed here" << endl;
            error(ERR_SLAVE_DEFINED, "Copying tracks to another album is not allowed - use <b>move</b> instead");
            goto kio_ipodslaveProtocol__copy_cleanup;
        } else if (dirmodel_dest.getCategory() != DirectoryModel::Playlists || !dirmodel_dest.isFile) {
            kdDebug() << "ipodslave::copy() : destination not a playlist" << endl;
            error(ERR_CANNOT_OPEN_FOR_WRITING, dest.path());
            goto kio_ipodslaveProtocol__copy_cleanup;
        }
        
        TrackMetadata * track= findTrack( dirmodel_src);
        if (track == NULL) {
            kdDebug() << "ipodslave::copy() : track doesn't exist " << dirmodel_src.getFilename() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel_src.getFilename());
            goto kio_ipodslaveProtocol__copy_cleanup;
        }
        
        switch(ipod.addTrackToPlaylist(track->getID(), dirmodel_dest.getPlaylist())) {
        case IPod::Err_DoesNotExist:
            kdDebug() << "ipodslave::copy() : playlist doesn't exist " << dirmodel_dest.getPlaylist() << endl;
            error(ERR_DOES_NOT_EXIST, dirmodel_src.getPlaylist());
            goto kio_ipodslaveProtocol__copy_cleanup;
            break;
        case IPod::Err_None:
            break;
        default:
            error(ERR_INTERNAL, "ipodslave::copy");
            goto kio_ipodslaveProtocol__copy_cleanup;
        }
        }
        break;
    default:
        kdDebug() << "ipodslave::copy() : cannot handle " << src.path() << endl;
        error( ERR_ACCESS_DENIED, dirmodel_dest.getFilename());
        goto kio_ipodslaveProtocol__copy_cleanup;
    }
    
    if (ipodunchanged)
        showSyncInfoMessage();
    
    kdDebug() << "ipodslave::copy() " << src.path() << "->" << dest.path() << " finished." << endl;
    finished();
    
kio_ipodslaveProtocol__copy_cleanup:
    ipod.unlock();
}


/*!
    \fn kio_ipodslaveProtocol::put (const KURL &url, int permissions, bool overwrite, bool resume)
*/ 
void kio_ipodslaveProtocol::put(const KURL &url, int permissions, bool overwrite, bool resume)
{
    // TODO implement overwrite when we're able to check for already existent tracks
    if (resume) {
        error(ERR_CANNOT_RESUME, url.path());
        return;
    }
    canResume(0);
    
    ipod.lock(true);
    kdDebug() << "ipodslave::put() " << url.path() << endl;
    
    if (ensureConsistency()) {
        DirectoryModel dirmodel(url);
        TrackMetadata track = ipod.createNewTrackMetadata();
        // append the real file extension to the path
        track.setPath(track.getPath() + dirmodel.getFilename().section('.', -1, -1, QString::SectionIncludeLeadingSep));
        QString trackpath = ipod.getRealPath(track.getPath());    // real path in the filesystem
        
        // copy track to iPod (file wise)
        QFile outfile( trackpath);
        outfile.open(IO_WriteOnly);
        int result= 0;
        do {
            QByteArray buffer;
            int remaining = 0;
            dataReq();
            result = readData( buffer );
            if (result > 0) {
                remaining = buffer.size();
                char * ptr = buffer.data();
                while (remaining > 0) {
                    int byteswritten = write(outfile.handle(), ptr, remaining);
                    if (byteswritten != -1) {
                        ptr += byteswritten;
                        remaining -= byteswritten;
                    } else {
                        remaining = -1;
                    }
                }
            }
            
            if (remaining < 0 || result < 0 || wasKilled()) {    // some error happened
                outfile.close();
                QFile::remove(trackpath);
                switch (errno) {
                case 0:
                    break;    // doesn't seem to be critical
                case ENOSPC:
                    error(ERR_DISK_FULL, dirmodel.getFilename());
                    break;
                default:
                    error(ERR_COULD_NOT_WRITE, dirmodel.getFilename());
                    break;
                }
                goto kio_ipodslaveProtocol__put_cleanup;
            }
        } while( result > 0);
        outfile.close();
        
        kdDebug() << "ipodslave::put() writing file " << trackpath << " done." << endl;
        
        // add the track to the TrackMap
        
        // read id3 information and fill in these into the track object
        if( !track.readFromFile( trackpath)) {
            // if insufficient id3 information is given try to guess from the URL
            // if even that is not possible ask the user what to do
            //     idea: one option could be to open a track information editor utiliy
            //        that gets the trackID
            //        another option would be "cancel"
        
            kdDebug() << "ipodslave::put() could not read id3tags in file " << dirmodel.getFilename() << endl;
            error(ERR_SLAVE_DEFINED, "Insufficient ID3 meta information found for <br><b>" + dirmodel.getFilename() + "</b>. Please add at least artist, album and title and try again.");
            QFile::remove( trackpath);
            goto kio_ipodslaveProtocol__put_cleanup;
        }
        
        kdDebug() << "ipodslave::put() parsed Track(" << track.getArtist() << "/" << track.getAlbum() << "/" << track.getTitle() << ")" << endl;
        
        if( !ipod.isChanged())
            showSyncInfoMessage();
        
        // add the track object to the database 
        ipod.addTrack(track);
        
        kdDebug() << "ipodslave::put() " << url.path() << " finished." << endl;
        finished();
    }
    
kio_ipodslaveProtocol__put_cleanup:
    ipod.unlock();
}


void kio_ipodslaveProtocol::showSyncInfoMessage() {
    if( !messageBox( Information, QString("Changes will NOT be saved automatically to the iPod. To save your changes you need to use the Sync Utility at ipod:/Utilities."))) {
        kdDebug() << "ipodslave::showSyncInfoMessage() messageBox communication failure" << endl;
    }
}

/*!
    \fn kio_ipodslaveProtocol::validateData()
 */
bool kio_ipodslaveProtocol::ensureConsistency()
{
    bool success = true;
    
    // check if ipod is opened und consistent?
    if(!ipod.isOpen() || !ipod.isStillConnected()) {    // NO!
        if(ipod.isOpen()) {
            kdDebug() << "ipodslave::ensureConsistency():  closing iPod." << endl;
            ipod.close();
        }
        
        // try to find a mounted ipod
        IPodMountPoint::List currentmountpoints = IPodMountPoint::mountedIPods();
        IPodMountPoint::List::Iterator mountiter = currentmountpoints.begin();
        for (; mountiter != currentmountpoints.end(); ++mountiter) {
            QString device = (*mountiter).getDevice();
            
            if (ipod.open(*mountiter)) {
                break;
            } else {
                kdDebug() << "ipodslave::ensureConsistency(): " << (*mountiter).getMountPoint().ascii() << " is not an iPod." << endl;
            }
        }
        
        // no iPod found
        if(!ipod.isOpen()) {
            error(ERR_DOES_NOT_EXIST, "Apple iPod");
            success = false;
            goto kio_ipodslaveProtocol_ensureConsistency_cleanup;
        }
        
        // some parse error
        if( !ipod.getItunesDBError().isEmpty()) {
            error( ERR_COULD_NOT_STAT, ipod.getItunesDBError());
            success = false;
            goto kio_ipodslaveProtocol_ensureConsistency_cleanup;
        }
    }
    
    success &= ipod.ensureConsistency();
    
    if(!success)
        error(ERR_INTERNAL, "Apple iPod");
    
kio_ipodslaveProtocol_ensureConsistency_cleanup:
    
    return success;
}



/*!
    \fn kio_ipodslaveProtocol::createUDSEntry( QString &name, size_t size, long type)
 */
void kio_ipodslaveProtocol::fillUDSEntry( UDSEntry &entry, const QString &name, TrackMetadata& track, long type, bool changed)
{
    QString filename= QFile::decodeName(name.local8Bit());
    appendUDSAtom( entry, KIO::UDS_NAME, 0, &filename);
    appendUDSAtom( entry, KIO::UDS_FILE_TYPE, type);
    appendUDSAtom( entry, KIO::UDS_SIZE, track.getFileSize());
    // appendUDSAtom( entry, KIO::UDS_MODIFICATION_TIME, track.lastmodified);
    if (type == S_IFDIR) {
        appendUDSAtom( entry, KIO::UDS_ACCESS, 0755);
        
        if (changed) {
            QString iconname("folder_important");
            appendUDSAtom( entry, KIO::UDS_ICON_NAME, 0, &iconname);
        }
    } else {
        appendUDSAtom( entry, KIO::UDS_ACCESS, 0644);
    }
    
    // appendUDSAtom( entry, KIO::UDS_EXTRA, 0, &track.getArtist());
    // appendUDSAtom( entry, KIO::UDS_EXTRA, 0, &track.getAlbum());
}


/*!
    \fn kio_ipodslaveProtocol::createUDSEntry( QString &name, size_t size, long type)
 */
void kio_ipodslaveProtocol::fillUDSEntry( UDSEntry &entry, const QString &name, size_t size, long type, bool changed, const QString *mimetype)
{
    QString filename= QFile::decodeName(name.local8Bit());
    appendUDSAtom( entry, KIO::UDS_NAME, 0, &filename);
    appendUDSAtom( entry, KIO::UDS_FILE_TYPE, type);
    appendUDSAtom( entry, KIO::UDS_SIZE, size);
    // appendUDSAtom( entry, KIO::UDS_EXTRA, 0, &filename);
    // appendUDSAtom( entry, KIO::UDS_EXTRA, 0, &filename);
    if (type == S_IFDIR) {
        appendUDSAtom( entry, KIO::UDS_ACCESS, 0755);
        
        if (changed) {
            QString iconname("folder_important");
            appendUDSAtom( entry, KIO::UDS_ICON_NAME, 0, &iconname);
        }
    } else {    // normal file
        appendUDSAtom( entry, KIO::UDS_ACCESS, 0644);
    }
    
    if (mimetype)
        appendUDSAtom(entry, KIO::UDS_MIME_TYPE, 0, mimetype);
}

/*!
    \fn kio_ipodslaveProtocol::createUDSAtom( unsigned int uds, long type)
 */
void kio_ipodslaveProtocol::appendUDSAtom( UDSEntry &entry, unsigned int uds, long longinfo,const QString *stringinfo)
{
    UDSAtom atom;
    atom.m_uds= uds;
    
    if( stringinfo!= NULL)
        atom.m_str= *stringinfo;
    else
        atom.m_long= longinfo;
    
    entry.append( atom);
}



/*!
    \fn kio_ipodslaveProtocol::findTrack( DirectoryModel& dirmodel)
 */
TrackMetadata * kio_ipodslaveProtocol::findTrack( DirectoryModel& dirmodel, int * tracknumber)
{
    TrackMetadata * track= NULL;
    TrackList * playlist;
    bool isplaylist = (dirmodel.getCategory() == DirectoryModel::Playlists);
    
    if( dirmodel.getTrack().isEmpty())
        return NULL;

    QString filename= dirmodel.getTrack();
    filename= filename.remove( QRegExp("^0+"));    // remove leading zeroes
    
    // which path?
    switch( dirmodel.getCategory()) {
    case DirectoryModel::Artists:
        // got here by artist/album
        playlist= ipod.getAlbum( dirmodel.getArtist(), dirmodel.getAlbum());
        break;
    case DirectoryModel::Playlists:
        // OK get the tracklist from the given playlist
        playlist= ipod.getPlaylistByTitle( dirmodel.getPlaylist());
        break;
    default:
        // no alternatives yet : stop here
        return NULL;
    }

    if( playlist == NULL)
        return NULL;    
    
    int tracknum= 0;
    TrackList::Iterator trackiterator= playlist->getTrackIDs();    
    while( trackiterator.hasNext()) {
        Q_UINT32 trackid= trackiterator.next();
        
        if( trackid == LISTITEM_DELETED) {    // this happens if a playlist has a "removed" element
            tracknum++;
            continue;
        }
        if( (track= ipod.getTrackByID( trackid))== NULL) {    // track not found
            tracknum++;
            continue;
        }
        
        QString trackname = formatTrackname( *track, ++tracknum, 1, isplaylist);
        if( filename.compare(trackname) == 0) {
            if( tracknumber != 0)
                *tracknumber= tracknum;
            break;
        } else {
            track = NULL;
        }
    }
    
    return track;
}

// does the reverse of formatTrackname()
QString kio_ipodslaveProtocol::stripTrackname( const QString& trackfilename) {
    QString tracktitle= trackfilename;
    // remove leading numbers, remove file extension, replace %2f with /
    return tracktitle.remove( QRegExp("^[0-9]+ - ")).remove( QRegExp( ".[^. ]+$")).replace( "%2f", "/");
}

/*!
    \fn kio_ipodslaveProtocol::formatTrackname( QString& buffer, Track& track, int tracknum)
 */
QString kio_ipodslaveProtocol::formatTrackname( TrackMetadata& track, int tracknum, unsigned short tracknumdigits, bool isPlaylist)
{
    QString buffer;
    QString title= track.getTitle();
    
    if( track.getFileExtension().isEmpty()) {
        QString extension= QFileInfo( ipod.getRealPath( track.getPath())).extension(FALSE);
        track.setFileExtension( extension);
    }
    
    QString numberformat= "%0"+ QString::number( tracknumdigits)+ "d - ";
    buffer.sprintf( numberformat.ascii(), (isPlaylist || track.getTrackNumber() == 0) ? tracknum : track.getTrackNumber());
    buffer.append( title.replace( "/", "%2f"));
    buffer.append( "."+ track.getFileExtension());
    
    // kdDebug() << "kio_ipodslaveProtocol::formatTrackname("<< tracknumdigits << ") trackname=" << buffer << endl;
    return buffer;
}

