# -*- coding: utf-8 -*-
"""Tools for inspecting Python objects.

Uses syntax highlighting for presenting the various information elements.

Similar in spirit to the inspect module, but all calls take a name argument to
reference the name under which an object is being read.

$Id: OInspect.py,v 1.18 2005/01/18 19:14:54 fperez Exp $
"""

#*****************************************************************************
#       Copyright (C) 2001-2004 Fernando Perez <fperez@colorado.edu>
#
#  Distributed under the terms of the BSD License.  The full license is in
#  the file COPYING, distributed as part of this software.
#*****************************************************************************

from IPython import Release
__author__  = '%s <%s>' % Release.authors['Fernando']
__license__ = Release.license

__all__ = ['Inspector','InspectColors']

# stdlib modules
import inspect,linecache,types,StringIO,string

# IPython's own
from IPython.Itpl import itpl
from IPython.genutils import page,indent,Term
from IPython import PyColorize
from IPython.ColorANSI import *

#****************************************************************************
# Builtin color schemes

Colors = TermColors  # just a shorthand

# Build a few color schemes
NoColor = ColorScheme(
    'NoColor',{
    'header' : Colors.NoColor,
    'normal' : Colors.NoColor  # color off (usu. Colors.Normal)
    }  )

LinuxColors = ColorScheme(
    'Linux',{
    'header' : Colors.LightRed,
    'normal' : Colors.Normal  # color off (usu. Colors.Normal)
    } )

LightBGColors = ColorScheme(
    'LightBG',{
    'header' : Colors.Red,
    'normal' : Colors.Normal  # color off (usu. Colors.Normal)
    }  )

# Build table of color schemes (needed by the parser)
InspectColors = ColorSchemeTable([NoColor,LinuxColors,LightBGColors],
                                 'Linux')

#****************************************************************************
# Auxiliary functions
def getdoc(obj):
    "Wrapper around inspect.getdoc which can't crash because of attribute problems"
    try:
        ds = inspect.getdoc(obj)
    except:
        # Harden against an inspect failure, which can occur with
        # SWIG-wrapped extensions.
        ds = None
    return ds

#****************************************************************************
# Class definitions

class myStringIO(StringIO.StringIO):
    """Adds a writeln method to normal StringIO."""
    def writeln(self,*arg,**kw):
        """Does a write() and then a write('\n')"""
        self.write(*arg,**kw)
        self.write('\n')

class Inspector:
    def __init__(self,color_table,code_color_table,scheme):
        self.color_table = color_table
        self.parser = PyColorize.Parser(code_color_table,out='str')
        self.format = self.parser.format
        self.set_active_scheme(scheme)

    def __getargspec(self,obj):
        """Get the names and default values of a function's arguments.

        A tuple of four things is returned: (args, varargs, varkw, defaults).
        'args' is a list of the argument names (it may contain nested lists).
        'varargs' and 'varkw' are the names of the * and ** arguments or None.
        'defaults' is an n-tuple of the default values of the last n arguments.

        Modified version of inspect.getargspec from the Python Standard
        Library."""

        if inspect.isfunction(obj):
            func_obj = obj
        elif inspect.ismethod(obj):
            func_obj = obj.im_func
        else:
            raise TypeError, 'arg is not a Python function'
        args, varargs, varkw = inspect.getargs(func_obj.func_code)
        return args, varargs, varkw, func_obj.func_defaults

    def __getdef(self,obj,oname=''):
        """Return the definition header for any callable object.

        If any exception is generated, None is returned instead and the
        exception is suppressed."""
        
        try:
            return oname + inspect.formatargspec(*self.__getargspec(obj))
        except:
            return None
 
    def __head(self,h):
        """Return a header string with proper colors."""
        return '%s%s%s' % (self.color_table.active_colors.header,h,
                           self.color_table.active_colors.normal)

    def set_active_scheme(self,scheme):
        self.color_table.set_active_scheme(scheme)
        self.parser.color_table.set_active_scheme(scheme)
    
    def noinfo(self,msg,oname):
        """Generic message when no information is found."""
        print 'No %s found' % msg,
        if oname:
            print 'for %s' % oname
        else:
            print
            
    def pdef(self,obj,oname=''):
        """Print the definition header for any callable object.

        If the object is a class, print the constructor information."""

        if not callable(obj):
            print 'Object is not callable.'
            return

        header = ''
        if type(obj) is types.ClassType:
            header = self.__head('Class constructor information:\n')
            obj = obj.__init__
        elif type(obj) is types.InstanceType:
            obj = obj.__call__

        output = self.__getdef(obj,oname)
        if output is None:
            self.noinfo('definition header',oname)
        else:
            print >>Term.cout, header,self.format(output),

    def pdoc(self,obj,oname='',formatter = None):
        """Print the docstring for any object.

        Optional:
        -formatter: a function to run the docstring through for specially
        formatted docstrings."""
        
        head = self.__head  # so that itpl can find it even if private
        if formatter is None:
            ds = getdoc(obj)
        else:
            ds = formatter(getdoc(obj))
        if type(obj) is types.ClassType:
            init_ds = getdoc(obj.__init__)
            output = itpl('$head("Class Docstring:")\n'
                          '$indent(ds)\n'
                          '$head("Constructor Docstring"):\n'
                          '$indent(init_ds)')
        elif type(obj) is types.InstanceType and hasattr(obj,'__call__'):
            call_ds = getdoc(obj.__call__)
            if call_ds:
                output = itpl('$head("Class Docstring:")\n$indent(ds)\n'
                              '$head("Calling Docstring:")\n$indent(call_ds)')
            else:
                output = ds
        else:
            output = ds
        if output is None:
            self.noinfo('documentation',oname)
            return
        page(output)

    def psource(self,obj,oname=''):
        """Print the source code for an object."""

        # Flush the source cache because inspect can return out-of-date source
        linecache.checkcache()
        try:
            src = inspect.getsource(obj) 
        except:
            self.noinfo('source',oname)
        else:
            page(self.format(src))

    def pfile(self,obj,oname=''):
        """Show the whole file where an object was defined."""
        try:
            sourcelines,lineno = inspect.getsourcelines(obj)
        except:
            self.noinfo('file',oname)
        else:
            # run contents of file through pager starting at line
            # where the object is defined            
            page(self.format(open(inspect.getabsfile(obj)).read()),lineno)

    def pinfo(self,obj,oname='',formatter=None,info=None,detail_level=0):
        """Show detailed information about an object.

        Optional arguments:
        
        - oname: name of the variable pointing to the object.
        - formatter: special formatter for docstrings (see pdoc)
        - info: a structure with some information fields (such as the
        docstring) which may have been precomputed already.
        - detail_level: if set to 1, more information is given.
        """

        obj_type = type(obj)

        header = self.__head
        if info is None:
            ds = getdoc(obj)
            ismagic = 0
            isalias = 0
            ospace = ''
        else:
            ds = info.docstring
            ismagic = info.ismagic
            isalias = info.isalias
            ospace = info.namespace
        if formatter is not None:
            ds = formatter(ds)

        # store output in a list which gets joined with \n at the end.
        out = myStringIO()
        
        string_max = 200 # max size of strings to show (snipped if longer)
        shalf = int((string_max -5)/2)

        if ismagic:
            obj_type_name = 'Magic function'
        elif isalias:
            obj_type_name = 'System alias'
        else:
            obj_type_name = obj_type.__name__
        out.writeln(header('Type:\t\t')+obj_type_name)

        try:
            bclass = obj.__class__
            out.writeln(header('Base Class:\t')+str(bclass))
        except: pass

        # String form, but snip if too long in ? form (full in ??)
        try:
            ostr = str(obj)
            str_head = 'String Form:'
            if not detail_level and len(ostr)>string_max:
                ostr = ostr[:shalf] + ' <...> ' + ostr[-shalf:]
                ostr = ("\n" + " " * len(str_head.expandtabs())).\
                       join(map(string.strip,ostr.split("\n")))
            if ostr.find('\n') > -1:
                # Print multi-line strings starting at the next line.
                str_sep = '\n'
            else:
                str_sep = '\t'
            out.writeln("%s%s%s" % (header(str_head),str_sep,ostr))
        except:
            pass

        if ospace:
            out.writeln(header('Namespace:\t')+ospace)

        # Length (for strings and lists)
        try:
            length = str(len(obj))
            out.writeln(header('Length:\t\t')+length)
        except: pass

        # Filename where object was defined
        try:
            file = inspect.getabsfile(obj)
            if file.endswith('<string>'):
                file = 'Dynamically generated function. No source code available.'
            out.writeln(header('File:\t\t')+file)
        except: pass

        # reconstruct the function definition and print it:
        defln = self.__getdef(obj,oname)
        if defln:
            out.write(header('Definition:\t')+self.format(defln))
 
        # Docstrings only in detail 0 mode, since source contains them (we
        # avoid repetitions).  If source fails, we add them back, see below.
        if ds and detail_level == 0:
                out.writeln(header('Docstring:\n') + indent(ds))

        # Original source code for any callable
        if detail_level:
            # Flush the source cache because inspect can return out-of-date source
            linecache.checkcache()
            try:
                source = self.format(inspect.getsource(obj))
                out.write(header('Source:\n')+source.rstrip())
            except:
                if ds:
                    out.writeln(header('Docstring:\n') + indent(ds))

        # Constructor docstring for classes
        if obj_type is types.ClassType:
            # reconstruct the function definition and print it:
            init_def = self.__getdef(obj.__init__,oname)
            init_ds  = getdoc(obj.__init__)

            if init_def or init_ds:
                out.writeln(header('\nConstructor information:'))
                if init_def:
                    out.write(header('Definition:\t')+ self.format(init_def))
                if init_ds:
                    out.writeln(header('Docstring:\n') + indent(init_ds))
        # and class docstring for instances:
        elif obj_type is types.InstanceType:

            # First, check whether the instance docstring is identical to the
            # class one, and print it separately if they don't coincide.  In
            # most cases they will, but it's nice to print all the info for
            # objects which use instance-customized docstrings.
            if ds:
                class_ds = getdoc(obj.__class__)
                if class_ds and ds != class_ds:
                    out.writeln(header('Class Docstring:\n') +
                                indent(class_ds))

            # Next, try to show constructor docstrings
            try:
                init_ds = getdoc(obj.__init__)
            except AttributeError:
                init_ds = None
            if init_ds:
                out.writeln(header('Constructor Docstring:\n') +
                            indent(init_ds))

            # Call form docstring for callable instances
            if hasattr(obj,'__call__'):
                out.writeln(header('Callable:\t')+'Yes')
                call_def = self.__getdef(obj.__call__,oname)
                if call_def is None:
                    out.write(header('Call def:\t')+
                              'Calling definition not available.')
                else:
                    out.write(header('Call def:\t')+self.format(call_def))
                call_ds = getdoc(obj.__call__)
                if call_ds:
                    out.writeln(header('Call docstring:\n') + indent(call_ds))

        # Finally send to printer/pager
        output = out.getvalue()
        if output:
            page(output)
        # end pinfo
