/* app.c
 *
 * Copyright (C) 2004-2007 Xin Zhen
 * Copyright (C) 2008 Daniele Napolitano
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <config.h>
#include "app.h"
#include "debug.h"

#include <libnotify/notification.h>
#include <libnotify/notify.h>

#include <gtk/gtkmenu.h>
#include <glib/gi18n-lib.h>
#include <string.h>
#include "eggtrayicon.h"
#include "resources.h"
#include "sendfile.h"
#include "recvfile.h"
#include "irda_misc.h"

/* Keep a list of all open application windows */
static GSList* app_list = NULL;

static gint delete_event_cb(GtkWidget* w, GdkEventAny* e, gpointer data);
static void button_click_cb(GtkWidget* w, gpointer data);

static void trayicon_init();
void discoverer_startpoll();
static GtkWidget * show_device_info_cb( void *data );
int clean_log_file( GtkWidget *w, gpointer data );
static GtkWidget * show_log_file( void *data );

GtkWidget* 
ircptray_app_new()
{
    GtkWidget* app = NULL;
    GtkWidget* button;
    GtkWidget* label;
    GtkWidget* status;
    GtkWidget* frame;

    frame = gtk_frame_new(NULL);

    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    gtk_frame_set_label(GTK_FRAME(frame), PACKAGE);

    app_list = g_slist_prepend(app_list, app);

    //sanity check
    if(!irda_check()) //no irda support
    {
        GtkWidget* dialog;

	dialog = gtk_message_dialog_new(NULL,
                                        GTK_DIALOG_MODAL,
                                        GTK_MESSAGE_ERROR,
                                        GTK_BUTTONS_CLOSE,
                                        _("You don't seem to have IrDA enabled in your kernel, quit."),
                                        NULL);
                                        
	gtk_dialog_run(GTK_DIALOG(dialog));
	
	return dialog;
    }

    trayicon_init();

    recvfile_listen(TRUE);
	
    discoverer_startpoll();

    return app;
}

EggTrayIcon* TrayIcon;
static GtkImage* TrayIconImage;
static GtkWidget* IconPopMenuItem;

#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 12
#else
static GtkTooltips* TrayToolTips;
#endif

static int TrayIconAnimationRefCount=0;
static int TrayIconTimerId = 0;
static char* TrayIconFileName = ICON_TRAY_INACTIVE;

static gboolean trayicon_onclick(GtkWidget *widget,
                                            GdkEventButton *event,
                                            gpointer user_data)
{
	if(event->button == 3) //right click
		gtk_menu_popup(GTK_MENU(user_data), NULL, NULL, NULL,
				NULL, event->button,
				gtk_get_current_event_time());
	else if(event->button == 1) //left click
		sendfile_launch_dialog();
}

static GtkTargetEntry target_table[] =
{
    {"text/uri-list",	0, 0},
    {"image/jpeg",	0, 0}
    //{"STRING",		0, 1},
    //{"text/plain",	0, 2}
};

static const guint n_targets = sizeof(target_table) / sizeof(target_table[0]);

static void trayicon_init()
{
    GtkWidget* icon;
    GtkWidget* label;
    GtkWidget* menu, *menuitem;
    GtkWidget* image;
    GtkWidget* eventbox;
    unsigned long rc;
                                                                            
    TrayIcon = egg_tray_icon_new("ircp-tray");
                                                                                
	//building icon popup menu
	menu = gtk_menu_new();
	
	menuitem = gtk_image_menu_item_new_with_label( _("Send Files") );
	image = gtk_image_new_from_stock (GTK_STOCK_GOTO_TOP, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM (menuitem), image);
	
	g_signal_connect(G_OBJECT(menuitem),
			"activate",
			G_CALLBACK(sendfile_onclick),
			NULL);
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
	
	IconPopMenuItem = menuitem = gtk_image_menu_item_new_with_label( _("Get Devices Info") );
	image = gtk_image_new_from_stock (GTK_STOCK_INFO, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM (menuitem), image);
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem),
                        "activate",
                        G_CALLBACK(show_device_info_cb),
                        NULL);
	gtk_widget_set_sensitive(menuitem, FALSE);

	menuitem = gtk_menu_item_new_with_label( _("Show History") );
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
        g_signal_connect(G_OBJECT(menuitem),
                        "activate",
                        G_CALLBACK(show_log_file),
                        NULL);

	menuitem = gtk_separator_menu_item_new();
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
	
	// About menu item
	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem),
			"activate",
			G_CALLBACK(version_onclick),
			NULL);
			
	// Exit menu item
	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
	gtk_container_add(GTK_CONTAINER(menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem),
			"activate",
			G_CALLBACK(gtk_main_quit),
			NULL);

	gtk_widget_show_all(menu);

	TrayIconImage = (GtkImage* )gtk_image_new_from_icon_name(TrayIconFileName, GTK_ICON_SIZE_SMALL_TOOLBAR);
	
	eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(eventbox), GTK_WIDGET(TrayIconImage));
	g_signal_connect(G_OBJECT(eventbox),
                   "button-press-event",
                   G_CALLBACK(trayicon_onclick),
                   menu);

	gtk_container_add(GTK_CONTAINER(TrayIcon), eventbox);
	
	//d'n'd
	gtk_drag_dest_set(
			GTK_WIDGET(TrayIcon),
			GTK_DEST_DEFAULT_ALL | GTK_DEST_DEFAULT_HIGHLIGHT,
			0,
			0,
			0
		);

	DEBUG(5,"%i\n",n_targets);
	
	
	/*	gtk_drag_dest_set(
			GTK_WIDGET(TrayIcon),
			GTK_DEST_DEFAULT_ALL | GTK_DEST_DEFAULT_HIGHLIGHT,
			target_table,
			n_targets,
			GDK_ACTION_COPY
		);*/
	
	// GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP
	//tooltips

#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 12
#else
	TrayToolTips = gtk_tooltips_new               ();
#endif


#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 12
	gtk_widget_set_tooltip_text (GTK_WIDGET(TrayIcon),  _("Send files via infrared"));
#else
	gtk_tooltips_set_tip(TrayToolTips, GTK_WIDGET(TrayIcon), _("Send files via infrared"), "ircp-tray");
#endif

	g_signal_connect(G_OBJECT(TrayIcon), "drag_data_received",
				G_CALLBACK(sendfile_ondragdrop), NULL);
	
	gtk_widget_show_all(GTK_WIDGET(TrayIcon));
}

void trayicon_set_text(char* str)
{
	int rc = egg_tray_icon_send_message(TrayIcon, 5, str, strlen(str));
}

void trayicon_set_icon(char* filename)
{
	TrayIconFileName = filename;
	gtk_image_set_from_icon_name(TrayIconImage, TrayIconFileName, GTK_ICON_SIZE_SMALL_TOOLBAR);
}

gboolean  trayicon_animation_ontimer (gpointer data)
{
	char* anim_table[] = {
		ICON_TRAY_ANIM1,
		ICON_TRAY_ANIM2,
		ICON_TRAY_ANIM3,
		ICON_TRAY_ANIM4
	};
	
	static anim_index = 0;
	int anim_num = sizeof(anim_table)/sizeof(char*);
	
	gtk_image_set_from_icon_name(TrayIconImage, anim_table[anim_index], GTK_ICON_SIZE_SMALL_TOOLBAR);
	
	anim_index = (anim_index+1) % anim_num;
	
	return TRUE;
}

void trayicon_animation_start()
{
	if(TrayIconAnimationRefCount == 0) {//setup a timer
		TrayIconTimerId = g_timeout_add_seconds_full(
							G_PRIORITY_LOW,
							1, //1 second
							trayicon_animation_ontimer,
							NULL, NULL);
	}
	TrayIconAnimationRefCount++;
	DEBUG(5,"Animation start %d\n", TrayIconAnimationRefCount);
}

void trayicon_animation_stop()
{
	TrayIconAnimationRefCount--;
	DEBUG(5,"Animation stop %d\n", TrayIconAnimationRefCount);
	
	if(TrayIconAnimationRefCount == 0) {
		//destroy timer
		g_source_remove(TrayIconTimerId);
		//restore default
		gtk_image_set_from_icon_name(TrayIconImage, TrayIconFileName, GTK_ICON_SIZE_MENU);
	}
}

void version_onclick(GtkMenuItem *menuitem, gpointer user_data)
{

	char* authors[] = {
		"Xin Zhen <xinzhen@pub.minidns.net>",
		"Daniele Napolitano <dnax88@gmail.com>",
		"",
		"Marcel Holtmann",
		"Pontus Fuchs",
		"Ragnar Henriksen",
		
		NULL
	};

	char* artists[] = {
		"Icon design: Jakub Steiner <jimmac@ximian.com>",
		"Animation: Dieter Vanderfaeillie <dieter.vanderfaeillie@gmail.com>",
		NULL
	};
	
	/* Feel free to put your names here translators :-) */
	char* translators = _("translator-credits");
		
#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 6

	gtk_show_about_dialog(GTK_WINDOW(TrayIcon),
                          "authors", authors,
			  "artists", artists,
			  "translator-credits", strcmp("translator-credits", translators) ? translators : NULL,
                          "comments", _("A utility for IrDA OBEX beaming and receiving"),
                          "copyright", "Copyright © 2004-2007 Xin Zhen \"LoneStar\"\nCopyright © 2008 Daniele Napolitano \"DnaX\"",
                          "version", VERSION,
                          "website", "https://launchpad.net/ircp-tray",
			  "logo-icon-name", MAIN_ICON,
                           NULL);

#endif
}

char* config_get_lastsavedir()
{
	char* lastsavedir = NULL;
	char* configfilename;

#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 6
	GKeyFile* config;

	configfilename = g_malloc(strlen(g_get_home_dir())+11);
	g_sprintf(configfilename, "%s/.ircp-tray", g_get_home_dir());

	config = g_key_file_new();
	g_key_file_load_from_file(config, configfilename,
		G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS, NULL);
	lastsavedir = g_key_file_get_string(config, "General", "LastSaveDir",
						NULL);
	g_key_file_free(config);
	g_free(configfilename);
#endif
	return lastsavedir;
}

void config_set_lastsavedir(char* dir)
{
	char* configfilename;
#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 6
	GKeyFile* config;
	char* configstr;
	FILE *configfile;
		
	configfilename = g_malloc(strlen(g_get_home_dir())+11);
	sprintf(configfilename, "%s/.ircp-tray", g_get_home_dir());

	config = g_key_file_new();
	g_key_file_load_from_file(config, configfilename,
		G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS, NULL);
	g_key_file_set_string(config, "General", "LastSaveDir", dir);
	configstr = g_key_file_to_data(config, NULL, NULL);
		
	configfile = fopen(configfilename, "w");
	fprintf(configfile, configstr);
	fclose(configfile);
		
	g_free(configstr);
	g_free(configfilename);
	g_key_file_free(config);
#endif
}

void log_file(const char* message)
{
	char* logfile;
	FILE* fp;
	char filename[] = "/.ircp-tray-log";
        char tmstr[16];
        time_t tm = time(NULL);

	printf("%s\n", message);
        strftime(tmstr, 16, "%X", localtime(&tm));

	logfile = g_malloc(strlen(g_get_home_dir())+strlen(filename)+1);
	g_sprintf(logfile, "%s%s", g_get_home_dir(), filename);
	fp = fopen(logfile, "a");
	if(fp) {
		fprintf(fp, "%s: %s\n", tmstr, message);
		fclose(fp);
	}
	g_free(logfile);
}

static int discoverer_poll_timer;
static const int discoverer_poll_interval = 3; //in seconds
#define discoverer_name_len 256
static char discoverer_name[discoverer_name_len];
static int discoverer_hints[2];

static gboolean discoverer_do(void*p)
{
	struct irda_device_list* list;

	list = irda_discover_device(-1);
		
	if(list && list->len)
	{
		if(strncmp(discoverer_name, list->dev[0].info,
				discoverer_name_len) != 0)
		{ 
			gchar *title = NULL;
			NotifyNotification *n = NULL;
			gchar tooltips[discoverer_name_len+32];
			
			title = g_strdup ( g_get_application_name() );
			
			//new device discovered
			g_sprintf(tooltips, _("Remote device discovered: %s"),
					 list->dev[0].info);
			
			notify_init(PACKAGE);

			n = notify_notification_new (title,
                                       tooltips,
                                       "phone",
                                       GTK_WIDGET(TrayIcon));

			// debug
			printf("%s\n", tooltips);
			
			// refresh tray icon
			trayicon_set_icon(ICON_TRAY);
			
			strncpy(discoverer_name, list->dev[0].info,
						discoverer_name_len);
			discoverer_name[discoverer_name_len-1] = '\0';
			discoverer_hints[0] = list->dev[0].hints[0];
			discoverer_hints[1] = list->dev[0].hints[1];
			
			
			g_sprintf(tooltips, _("Infrared File Transfer: %s"), discoverer_name);

			#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 12
        		gtk_widget_set_tooltip_text (GTK_WIDGET(TrayIcon),  tooltips);
			#else
			gtk_tooltips_set_tip(TrayToolTips, GTK_WIDGET(TrayIcon), tooltips, "ircp-tray");
			#endif
			
			if (n != NULL) {
				notify_notification_set_urgency(n, NOTIFY_URGENCY_NORMAL);
				notify_notification_set_timeout (n, NOTIFY_EXPIRES_DEFAULT);

				if (!notify_notification_show (n, NULL))
					g_warning("failed to send notification\n");
				g_object_unref(G_OBJECT(n));
			}

		        gtk_widget_set_sensitive(IconPopMenuItem, TRUE);
		        
		        
		}
	}
	else
	{
		if(strlen(discoverer_name))
		{
			gchar *title = NULL;
			NotifyNotification *n = NULL;
			char byemsg[128];
			
			title = g_strdup ( g_get_application_name() );
			
			sprintf(byemsg, _("Remote device %s disappeared"),
					 discoverer_name);
					 
			notify_init(PACKAGE);

			n = notify_notification_new (title,
                                       byemsg,
                                       "phone",
                                       GTK_WIDGET(TrayIcon));
                        
                        // debug
			printf("%s\n", byemsg);
			
			discoverer_name[0] = '\0';
			
			// refresh tray icon
			trayicon_set_icon(ICON_TRAY_INACTIVE);

			#if GTK_MAJOR_VERSION >= 2 && GTK_MINOR_VERSION >= 12
        		gtk_widget_set_tooltip_text (GTK_WIDGET(TrayIcon),  _("Infrared File Transfer"));
			#else
        		gtk_tooltips_set_tip(TrayToolTips, GTK_WIDGET(TrayIcon), _("Infrared File Transfer"), "ircp-tray");
			#endif
			
			if (n != NULL) {
				notify_notification_set_urgency(n, NOTIFY_URGENCY_LOW);
				notify_notification_set_timeout (n, 3000); // less then DEFAULT

				if (!notify_notification_show (n, NULL))
					g_warning("failed to send notification\n");
				g_object_unref(G_OBJECT(n));
			}

                        gtk_widget_set_sensitive(IconPopMenuItem, FALSE);

		}
	}
	g_free(list);
}

void discoverer_startpoll()
{
	printf("Start polling of devices in range, interval %d s\n",
				discoverer_poll_interval);
	discoverer_name[0] = '\0';
	discoverer_poll_timer = g_timeout_add_seconds(discoverer_poll_interval, discoverer_do, 0);
}

char* discoverer_get_name()
{
	return discoverer_name;
}

int* discoverer_get_hints()
{
	return discoverer_hints;
}

int discoverer_is_in_range()
{
	return strlen(discoverer_get_name()) > 0;
}

static void
on_dialog_response (GtkDialog *dialog, gint response, gpointer ok_handler_data)
{
	if (response == GTK_RESPONSE_CLOSE) {
		gtk_widget_destroy (GTK_WIDGET (dialog));
		
		DEBUG(5,"Dialog destroyed\n");
	}
}

static GtkWidget *
show_device_info_cb( void *data )
{
	GtkWidget *info;
	GtkWidget *vbox1;
	GtkWidget *close;
	GtkWidget *table1;
	GtkWidget *label1;
	GtkWidget *ck1;
	GtkWidget *aa1;
	gint i;
	gint y;
	gint numdev;
	struct _dev_info *devices;
	struct _dev_info *p;
	gchar *labels[] = {_("Hints"),
			   _("PNP"),
			   _("PDA"),
			   _("COMPUTER"),
			   _("PRINTER"),
			   _("MODEM"),
			   _("FAX"),
			   _("LAN"),
			   _("EXTENSION"),
			   _("TELEPHONY"),
			   _("FILESERVER"),
			   _("COMM"),
			   _("MESSAGE"),
			   _("HTTP"),
			   _("OBEX"),
			   NULL
	};

	devices = irda_get_hints( &numdev );
	printf("Devices: %d\n", numdev);
	if ( numdev <= 0 )
		return 0;

	info = gtk_dialog_new ();

	// Window title
	gtk_window_set_title (GTK_WINDOW (info), _("Information"));
	gtk_window_set_resizable( GTK_WINDOW (info), FALSE );
	
	// RGBA patch
	//GdkScreen *screen = gtk_widget_get_screen(info);
    	//GdkColormap *colormap = gdk_screen_get_rgba_colormap (screen);

    	//if (colormap)
	//	gtk_widget_set_default_colormap(colormap);

	gtk_window_set_destroy_with_parent (GTK_WINDOW (info), TRUE);
	gtk_dialog_set_default_response (GTK_DIALOG (info), GTK_RESPONSE_CLOSE);

	vbox1 =  GTK_DIALOG (info)->vbox;
	gtk_widget_show (vbox1);
	
	table1 = gtk_table_new (15, numdev+1, TRUE);
	gtk_widget_show (table1);
	gtk_box_pack_start (GTK_BOX (vbox1), table1, TRUE, TRUE, 0);

	/* Create labels */
	i = 0;
	while ( labels[i] ) {
		label1 = gtk_label_new (_(labels[i]));
		gtk_widget_show (label1);
		gtk_table_attach (GTK_TABLE (table1), label1, 0, 1, i, i+1,
				  (GtkAttachOptions) (GTK_FILL),
				  (GtkAttachOptions) (0), 5, 0);
		gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_LEFT);
		gtk_misc_set_alignment (GTK_MISC (label1), 0, 0.5);
		i++;
	}

	/* Adding checkboxes */
	p = devices;
	i = 0;
	while ( p ) {
		label1 = gtk_label_new( p->name );
		gtk_widget_show (label1);
		gtk_table_attach (GTK_TABLE (table1), label1, i+1, i+2, 0, 1,
				  (GtkAttachOptions) (GTK_FILL),
				  (GtkAttachOptions) (0), 5, 0);
		gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_LEFT);
		gtk_misc_set_alignment (GTK_MISC (label1), 0, 0.5);
		
		y = 1;
		while ( labels[y] ) {
			ck1 = gtk_check_button_new();
			GTK_WIDGET_UNSET_FLAGS (ck1, GTK_CAN_FOCUS);
			gtk_widget_set_sensitive (ck1, FALSE);

			if ( (y <= 8) && (p->hints[0] & 1<<(y-1)) )
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(ck1), 1 );

			if ( (y > 7) && (p->hints[1] & 1<<(y-9)) )
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(ck1), 1 );

			gtk_widget_show (ck1);
			gtk_table_attach (GTK_TABLE (table1), ck1, i+1, i+2, y, y+1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 5, 0);
			y++;
		}
		p = p->next;
		i++;
	}

	/*p = devices;
	while ( p ) {
		devices = p->next;
		g_free( p );
		p = devices;
	}*/
	free_dev_info(devices);

	/* Create close button */
	aa1 = GTK_DIALOG (info)->action_area;
	gtk_widget_show (aa1);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (aa1), GTK_BUTTONBOX_END);
	
	close = gtk_button_new_from_stock ("gtk-close");
	gtk_widget_show (close);
	gtk_dialog_add_action_widget (GTK_DIALOG (info), close, GTK_RESPONSE_CLOSE);
	GTK_WIDGET_SET_FLAGS (close, GTK_CAN_DEFAULT);

	g_signal_connect (GTK_OBJECT (info), "response",
			    G_CALLBACK (on_dialog_response),
			    NULL);

	gtk_widget_show(info);
	
	return info;
}

int clean_log_file( GtkWidget* w, gpointer data )
{
	GtkTextBuffer *buffer = data;
	char* logfile;
	
	char filename[] = "/.ircp-tray-log";
	logfile = g_malloc(strlen(g_get_home_dir())+strlen(filename)+1);
	g_sprintf(logfile, "%s%s", g_get_home_dir(), filename);
	
	if(g_remove (logfile))
	{	
		printf("Deleted log file: %s\n", logfile);
		gtk_text_buffer_set_text (buffer, "", -1);
		return TRUE;
	}
	else
		return FALSE;
}

static GtkWidget * show_log_file( void *data )
{
	GtkWidget *window, *control, *button, *vbox;
	GtkWidget *view, *scroll;
  	GtkTextBuffer *buffer;
  	
	//GtkTextBuffer* text;
	char* logfile;
	char* content;
	FILE* fp;
	int len;

	char filename[] = "/.ircp-tray-log";
        logfile = g_malloc(strlen(g_get_home_dir())+strlen(filename)+1);
        g_sprintf(logfile, "%s%s", g_get_home_dir(), filename);
	content = g_malloc0(64 * 1024);
	fp = fopen(logfile, "r");
	if (fp)
		len = fread(content, 1, 64 * 1024, fp);
	else
		len = 0;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_resizable( GTK_WINDOW (window), FALSE );
	gtk_window_set_title (GTK_WINDOW (window), _("History"));
	
	// RGBA patch
	GdkScreen *screen = gtk_widget_get_screen(window);
    	GdkColormap *colormap = gdk_screen_get_rgba_colormap (screen);

    	if (colormap)
		gtk_widget_set_default_colormap(colormap);
	
	scroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);	
  		
  	view = gtk_text_view_new ();
  	gtk_widget_set_size_request(view, 400, 200);
  	gtk_text_view_set_editable( GTK_TEXT_VIEW (view), FALSE );
  	gtk_text_view_set_border_window_size(GTK_TEXT_VIEW (view), GTK_TEXT_WINDOW_LEFT, 10);
  	
  	gtk_container_add (GTK_CONTAINER (scroll), view);  	  	

  	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));

  	gtk_text_buffer_set_text (buffer, content, -1);
     	
    	vbox = gtk_vbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (window), vbox);
	
	gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, FALSE, 5);
	
	button = gtk_button_new_from_stock(GTK_STOCK_CLEAR);
	gtk_container_add(GTK_CONTAINER(vbox), button);
	
	g_signal_connect(G_OBJECT (button), 
         "clicked",
		   G_CALLBACK (clean_log_file),
		   buffer);
	
	gtk_widget_show_all(window);

	g_free(content);
	if (fp)
		fclose(fp);
	g_free(logfile);
}

