#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>



#include "prototypes.h"

/*
   Copyright (C) 2003 Red Hat, Inc. All rights reserved.
                                                                                                             
   Usage and distribution of this file are subject to the Open Software License version 1.1
   that can be found at http://www.opensource.org/licenses/osl-1.1.txt and the COPYING file as
   distributed together with this file is included herein by reference.
                                                                                                                      
   Author: Arjan van de Ven   <arjanv@redhat.com>

*/

/*
 * This file contains the code that deals with the /proc/interrupts file and updating the
 * counts in the interrupts[] array.
 */


/* characters should remain in the parsed lines */
#define validchars "01234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-/"

/*
 * Calculate irq delta and reset the cpu for the distribution code to start working from
 * a clean slate.
 * In addition this cleans up the counts so that the next run starts fresh. 
 */
static void post_process_interrupts(int calcdelta)
{
	int i;
	for (i = 0; i < MAX_INTERRUPTS ; i++) {
		/* 
		 * use a running average for the delta to smoothen out peaks, and more 
		 * importantly, avoid short periods of inactivity to have too much effect.
		 */
		 
		if (calcdelta)		
			interrupts[i].delta = interrupts[i].delta/3 + 
			 	interrupts[i].count -  interrupts[i].oldcount;
		interrupts[i].oldcpu =  interrupts[i].cpu;
		interrupts[i].oldcount =interrupts[i].count;
		interrupts[i].count = 0;
		interrupts[i].cpu = MAX_CPU; /* none */
	}
}

static void reset_interrupt_info(void)
{
	int i;
	for (i = 0; i < MAX_INTERRUPTS ; i++) {
		 interrupts[i].count = 0;
		 interrupts[i].oldcount = 0;
		 interrupts[i].oldcpu = 0;
		 interrupts[i].cpu = MAX_CPU; /* all/none */
		 interrupts[i].type = IRQ_INACTIVE;
		 interrupts[i].number = i;
	}
}

int parse_proc_interrupts(int incremental) 
{
	FILE *file;
	char linebuffer[1024];
	
	file = fopen("/proc/interrupts","r");
	assert(file != NULL);
	/* first line is useless */
	fgets(linebuffer, 1024, file);
	
	if (!incremental)
		reset_interrupt_info();
	
	while (!feof(file)) {
		char *cursor, *word;
		int column = 0;
		int irqnumber = 0;
		fgets(linebuffer, 1024, file);
		if (strlen(linebuffer)==0)
			continue;
		/* squash all unwanted characters */
		word = cursor = linebuffer;
		while (cursor) {
			char *word;
			long long count;
			
			word = strsep(&cursor, " ");
			
			if (!strlen(word))
				continue;
				
		
			/* first column is the irq number */	
			if (column==0) {
				int ret;
				ret = sscanf(word,"%i",&irqnumber);
				if (!ret)  /* non numeric end stuff */
					irqnumber = MAX_INTERRUPTS-1; 
			/* then N columns of counts, where N is the number of cpu's */
			} else if (column <= cpucount) {
				sscanf(word,"%lli",&count);
				interrupts[irqnumber].count += count;
			/* and lastly the names of the drivers */
			} else if ( ( (incremental==0) || (interrupts[irqnumber].type==IRQ_INACTIVE) ) 
							&& column>cpucount+1)
				classify_type(irqnumber, word);
			column++;
		}
	}
	
	fclose(file);
	post_process_interrupts(incremental);	
	return 0;
}

/* 
 * Write a selected configuration back to the kernel to activate the new settings 
 */
void activate_irqtable(void)
{
	int i;
	FILE *file;
	char filename[256];
	for (i = 0; i < MAX_INTERRUPTS; i++) {
		/* shortcut: if old and new cpu are identical, don't do a thing */
		if (interrupts[i].oldcpu == interrupts[i].cpu)
			continue;
		if (interrupts[i].type == IRQ_INACTIVE)
			continue;
		snprintf(filename,255,"/proc/irq/%i/smp_affinity",i);
		file = fopen(filename, "w");
		if (file==NULL)
			continue;
		if (interrupts[i].cpu < MAX_CPU)
			fprintf(file,"%x",1<<interrupts[i].cpu);
		else
			fprintf(file,"ffffffff");
		fclose(file);	
	}
}

/* 
 * Read the current configuration back from the kernel 
 */
int get_current_irqcpu(int irq)
{
	unsigned int i;
	FILE *file;
	char filename[256];
	
	snprintf(filename,255,"/proc/irq/%i/smp_affinity",irq);
	file = fopen(filename, "r");
	if (file==NULL)
		return 0;
	if (fscanf(file, "%x", &i)<1) 
		return 0;
	fclose(file);	
	i = ffs(i);
	
	if ( (i>0) && (i<=MAX_CPU) )
		return i-1;
	else
		return 0;
}
