/***************************************************************************
                          qvncview.cpp  -  main widget
                             -------------------
    begin                : Thu Dec 20 15:11:42 CET 2001
    copyright            : (C) 2001-2003 by Tim Jansen
    email                : tim@tjansen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qvncview.h"
#include "qkey.h"

#include <qapplication.h>
#include <qmessagebox.h>

#include <qdatastream.h>
#include <qmutex.h>
#include <qvbox.h>
#include <qwaitcondition.h>
#include <qdir.h>

#include "vncviewer.h"
#include "rsa_crypt_lib.h"
#include "italc_rfb_ext.h"
#include "paths.h"
#include "vncviewer.h"
#include "vncauth.h"

#include "qvncview.moc"

//#include <X11/Xlib.h>

/*
 * appData is our application-specific data which can be set by the user with
 * application resource specs.  The AppData structure is defined in the header
 * file.
 */
AppData appData;
bool appDataConfigured = false;

Display* dpy;

static QVncView *qvncview;

static QCString password;
static QMutex passwordLock;
static QWaitCondition passwordWaiter;

const unsigned int MAX_SELECTION_LENGTH = 4096;


QVncView::QVncView(QWidget *parent,
		   const char *name,
		   const QString &_host,
		   int _port,
		   const QString &_password,
		   Quality quality,
		   DotCursorState dotCursorState,
		   const QString &encodings) :
  QRemoteView(parent, name, Qt::WResizeNoErase | Qt::WRepaintNoErase | Qt::WStaticContents),
  m_cthread(this, m_wthread, m_quitFlag),
  m_wthread(this, m_quitFlag),
  m_quitFlag(false),
  m_enableFramebufferLocking(false),
  m_scaling(false),
  m_remoteMouseTracking(false),
  m_viewOnly(false),
  m_buttonMask(0),
  m_host(_host),
  m_port(_port),
  m_cursorState(dotCursorState)
{
	qvncview = this;
	password = _password.latin1();
	dpy = qt_xdisplay();
	setFixedSize(16,16);
	setFocusPolicy(QWidget::StrongFocus);

	m_cursor = QCursor( Qt::ArrowCursor );

	if ((quality != QUALITY_UNKNOWN) ||
	    !encodings.isNull())
		configureApp(quality, encodings);
#ifdef RSA_CRYPT_AVAILABLE
	const QString PRIV_KEY_FILE = QDir::home().absPath() + "/" + ITALC_CONFIG_PATH + "/id_rsa";
	const QString PUB_KEY_FILE = PRIV_KEY_FILE + ".public";

	privateKey = readPrivateKey( PRIV_KEY_FILE );
	if( privateKey == NULL )
		exit( -1 );
#endif
}



void QVncView::showDotCursor(DotCursorState state) {
	if (state == m_cursorState)
		return;

	m_cursorState = state;
	showDotCursorInternal();
}

DotCursorState QVncView::dotCursorState() const {
	return m_cursorState;
}

void QVncView::showDotCursorInternal() {
	switch (m_cursorState) {
	case DOT_CURSOR_ON:
		setCursor(m_cursor);
		break;
	case DOT_CURSOR_OFF:
		setCursor(QCursor(Qt::BlankCursor));
		break;
	case DOT_CURSOR_AUTO:
		if (m_enableClientCursor)
			setCursor(QCursor(Qt::BlankCursor));
		else
			setCursor(m_cursor);
		break;
	}
}

QString QVncView::host() {
	return m_host;
}

int QVncView::port() {
	return m_port;
}

void QVncView::startQuitting() {
	m_quitFlag = true;
	m_wthread.kick();
	m_cthread.kick();
}

bool QVncView::isQuitting() {
	return m_quitFlag;
}

void QVncView::configureApp(Quality q, const QString specialEncodings) {
	appDataConfigured = true;
	appData.shareDesktop = 1;
	appData.viewOnly = 0;

	if (q == QUALITY_LOW) {
		appData.useBGR233 = 1;
		appData.encodingsString = "background copyrect softcursor tight zlib hextile raw";
		appData.compressLevel = -1;
		appData.qualityLevel = 1;
		appData.dotCursor = 1;
	}
	else if (q == QUALITY_MEDIUM) {
		appData.useBGR233 = 0;
		appData.encodingsString = "background copyrect softcursor tight zlib hextile raw";
		appData.compressLevel = -1;
		appData.qualityLevel = 7;
		appData.dotCursor = 1;
	}
	else if ((q == QUALITY_HIGH) || (q == QUALITY_UNKNOWN)) {
		appData.useBGR233 = 0;
		appData.encodingsString = "copyrect softcursor tight raw";
		appData.compressLevel = 3;
		appData.qualityLevel = 9;
		appData.dotCursor = 0;
	}

	if (!specialEncodings.isNull())
		appData.encodingsString = specialEncodings.latin1();

	appData.nColours = 256;
	appData.useSharedColours = 1;
	appData.requestedDepth = 0;

	appData.rawDelay = 0;
	appData.copyRectDelay = 0;

	if (!appData.dotCursor)
		m_cursorState = DOT_CURSOR_OFF;
	showDotCursorInternal();
}

bool QVncView::start() {

	setStatus(REMOTE_VIEW_CONNECTING);

	m_cthread.start();
	setBackgroundMode(Qt::NoBackground);
	return true;
}

QVncView::~QVncView()
{
	startQuitting();
	m_cthread.wait();
	m_wthread.wait();
	freeResources();
}

bool QVncView::supportsLocalCursor() const {
	return true;
}

bool QVncView::supportsScaling() const {
	return true;
}

bool QVncView::scaling() const {
	return m_scaling;
}

bool QVncView::viewOnly() {
	return m_viewOnly;
}

QSize QVncView::framebufferSize() {
	return m_framebufferSize;
}

void QVncView::setViewOnly(bool s) {
	m_viewOnly = s;

	if (s)
		setCursor(Qt::ArrowCursor);
	else
		showDotCursorInternal();
}

void QVncView::enableScaling(bool s) {
//	bool os = m_scaling;
	m_scaling = s;
//	if (s != os) {
		if (s) {
			setMaximumSize(m_framebufferSize);
			setMinimumSize(m_framebufferSize.width()/16,
				       m_framebufferSize.height()/16);
		}
		else
			setFixedSize(m_framebufferSize);
//	}
}

void QVncView::paintEvent(QPaintEvent *e) {
	drawRegion(e->rect().x(),
		   e->rect().y(),
		   e->rect().width(),
		   e->rect().height());
}

void QVncView::drawRegion(int x, int y, int w, int h) {
	if (m_scaling)
		DrawZoomedScreenRegionX11Thread(winId(), width(), height(),
						x, y, w, h);
	else
		DrawScreenRegionX11Thread(winId(), x, y, w, h);
}

void QVncView::customEvent(QCustomEvent *e)
{
	if (e->type() == ScreenRepaintEventType) {
		ScreenRepaintEvent *sre = (ScreenRepaintEvent*) e;
		drawRegion(sre->x(), sre->y(),sre->width(), sre->height());
	}
	else if (e->type() == ScreenResizeEventType) {
		ScreenResizeEvent *sre = (ScreenResizeEvent*) e;
		m_framebufferSize = QSize(sre->width(), sre->height());
		if( QApplication::desktop()->width() < sre->width() || QApplication::desktop()->height()-32 < sre->height() )
		{
			m_framebufferSize = QSize( QApplication::desktop()->width(), QApplication::desktop()->height()-32 );
			enableScaling( TRUE );
		}
		setFixedSize(m_framebufferSize);
		emit changeSize( m_framebufferSize.width(), m_framebufferSize.height());
	}
	else if (e->type() == DesktopInitEventType) {
		m_cthread.desktopInit();
	}
	else if (e->type() == StatusChangeEventType) {
		StatusChangeEvent *sce = (StatusChangeEvent*) e;
		setStatus(sce->status());
		if (m_status == REMOTE_VIEW_CONNECTED) {
			emit connected();
			setFocus();
			setMouseTracking(true);
		}
		else if (m_status == REMOTE_VIEW_DISCONNECTED) {
			setMouseTracking(false);
			emit disconnected();
		}
	}
	else if (e->type() == PasswordRequiredEventType) {
		qFatal( "pw required. giving up\n" );
	}
	else if (e->type() == FatalErrorEventType) {
		FatalErrorEvent *fee = (FatalErrorEvent*) e;
		setStatus(REMOTE_VIEW_DISCONNECTED);
		switch (fee->errorCode()) {
		case ERROR_CONNECTION:
			QMessageBox::critical(0,
					   tr("Connection attempt to host failed."),
					   tr("Connection Failure"));
			break;
		case ERROR_PROTOCOL:
			QMessageBox::critical(0,
					   tr("Remote host is using an incompatible protocol."),
					   tr("Connection Failure"));
			break;
		case ERROR_IO:
			QMessageBox::critical(0,
					   tr("The connection to the host has been interrupted."),
					   tr("Connection Failure"));
			break;
		case ERROR_SERVER_BLOCKED:
			QMessageBox::critical(0,
					   tr("Connection failed. The server does not accept new connections."),
					   tr("Connection Failure"));
			break;
		case ERROR_NAME:
			QMessageBox::critical(0,
					   tr("Connection failed. A server with the given name cannot be found."),
					   tr("Connection Failure"));
			break;
		case ERROR_NO_SERVER:
			QMessageBox::critical(0,
					   tr("Connection failed. No server running at the given address and port."),
					   tr("Connection Failure"));
			break;
		case ERROR_AUTHENTICATION:
			QMessageBox::critical(0,
					   tr("Authentication failed. Connection aborted."),
					   tr("Authentication Failure"));
			break;
		default:
			QMessageBox::critical(0,
					   tr("Unknown error."),
					   tr("Unknown Error"));
			break;
		}
		emit disconnectedError();
	}
	else if (e->type() == BeepEventType) {
		QApplication::beep();
	}
	else if (e->type() == ServerCutEventType) {
		//ServerCutEvent *sce = (ServerCutEvent*) e;
		//QString ctext = QString::fromUtf8(sce->bytes(), sce->length());
		//m_dontSendCb = true;
		//m_cb->setText(ctext, QClipboard::Clipboard);
		//m_cb->setText(ctext, QClipboard::Selection);
		//m_dontSendCb = false;
	}
	else if (e->type() == MouseStateEventType) {
		MouseStateEvent *mse = (MouseStateEvent*) e;
		emit mouseStateChanged(mse->x(), mse->y(), mse->buttonMask());
		bool show = m_plom.handlePointerEvent(mse->x(), mse->y());
		if (m_cursorState != DOT_CURSOR_ON)
			showDotCursor(show ? DOT_CURSOR_AUTO : DOT_CURSOR_OFF);
	}
}

void QVncView::mouseEvent(QMouseEvent *e) {
	if (m_status != REMOTE_VIEW_CONNECTED)
		return;
	if (m_viewOnly)
		return;

	if ( e->type() != QEvent::MouseMove ) {
		if ( (e->type() == QEvent::MouseButtonPress) ||
                     (e->type() == QEvent::MouseButtonDblClick)) {
			if ( e->button() & LeftButton )
				m_buttonMask |= 0x01;
			if ( e->button() & MidButton )
				m_buttonMask |= 0x02;
			if ( e->button() & RightButton )
				m_buttonMask |= 0x04;
		}
		else if ( e->type() == QEvent::MouseButtonRelease ) {
			if ( e->button() & LeftButton )
				m_buttonMask &= 0xfe;
			if ( e->button() & MidButton )
				m_buttonMask &= 0xfd;
			if ( e->button() & RightButton )
				m_buttonMask &= 0xfb;
		}
	}

	int x = e->x();
	int y = e->y();
	m_plom.registerPointerState(x, y);
	if (m_scaling) {
		x = (x * m_framebufferSize.width()) / width();
		y = (y * m_framebufferSize.height()) / height();
	}
	m_wthread.queueMouseEvent(x, y, m_buttonMask);

	if (m_enableClientCursor)
		DrawCursorX11Thread(x, y); // in rfbproto.c
}

void QVncView::mousePressEvent(QMouseEvent *e) {
	mouseEvent(e);
	e->accept();
}

void QVncView::mouseDoubleClickEvent(QMouseEvent *e) {
	mouseEvent(e);
	e->accept();
}

void QVncView::mouseReleaseEvent(QMouseEvent *e) {
	mouseEvent(e);
	e->accept();
}

void QVncView::mouseMoveEvent(QMouseEvent *e) {
	mouseEvent(e);
	e->ignore();
}

void QVncView::wheelEvent(QWheelEvent *e) {
	if (m_status != REMOTE_VIEW_CONNECTED)
		return;
	if (m_viewOnly)
		return;

	int eb = 0;
	if ( e->delta() < 0 )
		eb |= 0x10;
	else
		eb |= 0x8;

	int x = e->pos().x();
	int y = e->pos().y();
	if (m_scaling) {
		x = (x * m_framebufferSize.width()) / width();
		y = (y * m_framebufferSize.height()) / height();
	}
	m_wthread.queueMouseEvent(x, y, eb|m_buttonMask);
	m_wthread.queueMouseEvent(x, y, m_buttonMask);
	e->accept();
}

void QVncView::pressKey(XEvent *xe) {
	QKey k(xe);
	uint mod = k.mod();
	if (mod & QKey::modX(QKey::SHIFT))
		m_wthread.queueKeyEvent(XK_Shift_L, true);
	if (mod & QKey::modX(QKey::CTRL))
		m_wthread.queueKeyEvent(XK_Control_L, true);
	if (mod & QKey::modX(QKey::ALT))
		m_wthread.queueKeyEvent(XK_Alt_L, true);
	if (mod & QKey::modX(QKey::WIN))
		m_wthread.queueKeyEvent(XK_Meta_L, true);

	m_wthread.queueKeyEvent(k.sym(), true);
	m_wthread.queueKeyEvent(k.sym(), false);

	if (mod & QKey::modX(QKey::WIN))
		m_wthread.queueKeyEvent(XK_Meta_L, false);
	if (mod & QKey::modX(QKey::ALT))
		m_wthread.queueKeyEvent(XK_Alt_L, false);
	if (mod & QKey::modX(QKey::CTRL))
		m_wthread.queueKeyEvent(XK_Control_L, false);
	if (mod & QKey::modX(QKey::SHIFT))
		m_wthread.queueKeyEvent(XK_Shift_L, false);

	m_mods.clear();
}

bool QVncView::x11Event(XEvent *e) {
	bool pressed;
	if (e->type == KeyPress)
		pressed = true;
	else if (e->type == KeyRelease)
		pressed = false;
	else
		return QWidget::x11Event(e);

	if (!m_viewOnly) {
		unsigned int s = QKey( e ).sym();

		switch (s) {
		case XK_Meta_L:
		case XK_Alt_L:
		case XK_Control_L:
		case XK_Shift_L:
		case XK_Meta_R:
		case XK_Alt_R:
		case XK_Control_R:
		case XK_Shift_R:
			if (pressed)
				m_mods[s] = true;
			else if (m_mods.contains(s))
				m_mods.remove(s);
			else
				unpressModifiers();
		}
		m_wthread.queueKeyEvent(s, pressed);
	}
	return true;
}

void QVncView::unpressModifiers() {
	QValueList<unsigned int> keys = m_mods.keys();
	QValueList<unsigned int>::const_iterator it = keys.begin();
	while (it != keys.end()) {
		m_wthread.queueKeyEvent(*it, false);
		it++;
	}
	m_mods.clear();
}

void QVncView::focusOutEvent(QFocusEvent *) {
	unpressModifiers();
}

QSize QVncView::sizeHint() {
	return maximumSize();
}

void QVncView::setRemoteMouseTracking(bool s) {
	m_remoteMouseTracking = s;
}

bool QVncView::remoteMouseTracking() {
	return m_remoteMouseTracking;
}
/*
void QVncView::clipboardChanged() {
	if (m_status != REMOTE_VIEW_CONNECTED)
		return;

	if (m_cb->ownsClipboard() || m_dontSendCb)
		return;

	QString text = m_cb->text(QClipboard::Clipboard);
	if (text.length() > MAX_SELECTION_LENGTH)
		return;

	m_wthread.queueClientCut(text);
}

void QVncView::selectionChanged() {
	if (m_status != REMOTE_VIEW_CONNECTED)
		return;

	if (m_cb->ownsSelection() || m_dontSendCb)
		return;

	QString text = m_cb->text(QClipboard::Selection);
	if (text.length() > MAX_SELECTION_LENGTH)
		return;

	m_wthread.queueClientCut(text);
}
*/

void QVncView::lockFramebuffer() {
	if (m_enableFramebufferLocking)
		m_framebufferLock.lock();
}

void QVncView::unlockFramebuffer() {
	if (m_enableFramebufferLocking)
		m_framebufferLock.unlock();
}

void QVncView::enableClientCursor(bool enable) {
	if (enable) {
		m_enableFramebufferLocking = true; // cant be turned off
	}
	m_enableClientCursor = enable;
	showDotCursorInternal();
}

int getPassword(char *passwd, int pwlen) {
	int retV = 1;

	passwordLock.lock();
	if (password.isNull()) {
		QApplication::postEvent(qvncview, new PasswordRequiredEvent());
		passwordWaiter.wait(&passwordLock);
	}
	if (!password.isNull())
		strncpy(passwd, (const char*)password, pwlen);
	else {
		passwd[0] = 0;
		retV = 0;
	}
	passwordLock.unlock();

	if (!retV)
		qvncview->startQuitting();
	return retV;
}

extern int isQuitFlagSet() {
	return qvncview->isQuitting() ? 1 : 0;
}

extern void DrawScreenRegion(int x, int y, int width, int height) {
/*	KApplication::kApplication()->lock();
	qvncview->drawRegion(x, y, width, height);
	KApplication::kApplication()->unlock();
*/
	QApplication::postEvent(qvncview, new ScreenRepaintEvent(x, y, width, height));
}

// call only from x11 thread!
extern void DrawAnyScreenRegionX11Thread(int x, int y, int width, int height) {
	qvncview->drawRegion(x, y, width, height);
}

extern void EnableClientCursor(int enable) {
	qvncview->enableClientCursor(enable);
}

extern void LockFramebuffer() {
  	qvncview->lockFramebuffer();
}

extern void UnlockFramebuffer() {
  	qvncview->unlockFramebuffer();
}

extern void beep() {
	QApplication::postEvent(qvncview, new BeepEvent());
}

extern void newServerCut(char *bytes, int length) {
	QApplication::postEvent(qvncview, new ServerCutEvent(bytes, length));
}

extern void postMouseEvent(int x, int y, int buttonMask) {
	QApplication::postEvent(qvncview, new MouseStateEvent(x, y, buttonMask));
}

#ifdef RSA_CRYPT_AVAILABLE
int doiTALCAuth( int rfb_sock )
{
	// send user-name to server (server has to encrypt data with the appropriate key)
	s_rfb_authentication ra;
	QString user_name = getenv( "USER" );

	ra.user_name_len = Swap16IfLE( user_name.length() + 1 );
	if( !WriteExact( rfb_sock, (const char*) &ra, sizeof( ra ) ) )
		return( FALSE );
	if( !WriteExact( rfb_sock, user_name.ascii(), user_name.length() + 1 ) )
		return( FALSE );

	// challenge-response-authentication
	int ecs = RSA_size( privateKey );
	char * encrypted_challenge = new char[ecs];
	if( !ReadFromRFBServer( encrypted_challenge, ecs ) )
	{
		return( FALSE );
	}

	char * decrypted_response = new char[DEFAULT_CHALLENGE_SIZE];
			
	int bytes = RSA_private_decrypt( ecs, (unsigned char *)encrypted_challenge, (unsigned char *)decrypted_response, privateKey, RSA_PKCS1_PADDING );
	// TODO: add error handling if bytes != DEFAULT_CHALLENGE_SIZE

	if( !WriteExact( rfb_sock, decrypted_response, DEFAULT_CHALLENGE_SIZE ) )
	{
		return( FALSE );
	}
	delete[] encrypted_challenge;
	delete[] decrypted_response;
	return( TRUE );
}
#endif
