/****************************************************************************
 *                              DFIOTrace.h
 * 
 * Author: Matthew Ballance
 * Desc:   Describes a Dump-File IO trace
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#ifndef INCLUDED_DFIO_TRACE_H
#define INCLUDED_DFIO_TRACE_H
#include "types.h"
#include "HashEntry.h"
#include "DFIO.h"
#include "DFIOValChg.h"
#include "vector.h"
#include "ivi_String.h"

class DFIO;

class SdbSignal;
class DFIOTrace : public HashEntry {

    /****************************************************************
     * Public Types
     ****************************************************************/
    public:

       /*****************************************************
        * DFIO TraceType
        *****************************************************/
        enum {
            DFT_BitStr  = 0,
            DFT_Integer = 1,
            DFT_Double  = 2,
            DFT_String  = 4 
        };

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:


        /************************************************************
         * DFIOTrace()
         ************************************************************/
        DFIOTrace(
            Char     *name,
            Uint32    rows,
            Uint32    msb,
            Uint32    lsb,
            Uint32    traceType);

        /************************************************************
         * DFIOTrace()
         ************************************************************/
        DFIOTrace(
            Char     *name
        ) : HashEntry(name) { }

        /************************************************************
         * ~DFIOTrace()
         ************************************************************/
        virtual ~DFIOTrace();
   
        /************************************************************
         * AddClient()
         ************************************************************/
        void AddClient(SdbSignal *signal);

        /************************************************************
         * RemoteClient()
         ************************************************************/
        void RemoveClient(SdbSignal *signal);

    virtual void setIndepTraceMax(bool indep) { d_indepTraceMax = indep; }

    virtual Int32 setValueInt(Uint32 row, Uint32 value) { 
        return base_setValueInt(row, value);}
    virtual Int32 setValueStr(Uint32 row, Char *string){ 
        return base_setValueStr(row, string);}
    virtual Int32 setValueBitStr(Uint32 row, Char *bitStr){ 
        return base_setValueBitStr(row, bitStr);}
    virtual Int32 setValueABVal8(Uint32 row, DFIOVal8 *value) {
        return base_setValueABVal8(row, value); }
    virtual Int32 setValueABVal32(Uint32 row, DFIOVal32 *value) {
        return base_setValueABVal32(row, value); }

    Int32 base_setValueInt(Uint32 row, Uint32 value) { return -1; }
    Int32 base_setValueStr(Uint32 row, Char  *string) { return -1; }
    Int32 base_setValueBitStr(Uint32 row, Char *bitStr) { return -1; }
    Int32 base_setValueABVal8(Uint32 row, DFIOVal8 *bitStr) { return -1; }
    Int32 base_setValueABVal32(Uint32 row, DFIOVal32 *bitStr) { return -1; }

    virtual Int32 setTime(Uint32    newTime) {
        return base_setTime(newTime); }
    Int32 base_setTime(Uint32 newTime);

    inline Uint32 get_lsb()  { return lsb;          }
    inline Uint32 get_msb()  { return msb;          }
    inline Uint32 get_len()  { return len;          }
    inline Char  *get_name() { return name.value(); }
    inline String &get_name_str() { return name;   }


    /****************************************************************
     * clearTrace()
     *
     * Clears all data from the trace...
     ****************************************************************/
    virtual void clearTrace() { ; }

    /***********************************************************
     * Options
     * 
     * GVF_OnePlus
     *    If set, return the next value change past the value 
     *    requested (if possible). This means that if the 
     *    following request is entered:
     *    getValue(20, 20, GVF_OnePlus), two values will be 
     *    returned.
     ***********************************************************/
    enum getValFlags {
        GVF_Default    = 0,
        GVF_OnePlus    = 1,
        GVF_OneMinus   = 2,
        GVF_LastOption
    };

   /************************************************************
    * base_getValue()
    ************************************************************/
    virtual Vector<DFIOValChg> *getValue(
            Uint32      begin, 
            Uint32      end,
            Uint32      flags) {
        return base_getValue(begin, end, flags); }
    Vector<DFIOValChg> *base_getValue(Uint32 begin, Uint32 end, Uint32 flags);
    
   /************************************************************
    * getTraceMaxTime()
    ************************************************************/
    virtual Uint32 getTraceMaxTime();

   /*****************************************************
    * Member variables.
    *****************************************************/
    Uint32                 traceId;
    Uint32                 rows;
    Uint32                 msb;
    Uint32                 lsb;
    Uint32                 len;
    Uint32                 traceType;
    Uint32                 lastChangePos;
    Uint32                 d_traceTime;
    Uint64                 d_traceTime64;
    bool                   d_indepTraceMax;
    DFIO                  *parent; 
    String                 name;
    Vector<SdbSignal>      d_clients;
    voidPtr                userData[4];
};

#endif /* INCLUDED_DFIO_TRACE_H */

