/****************************************************************************
 *                              DialogMgr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a dialog-box manager. Creates dialog boxes from a
 *         description
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "DialogMgr.h"
#include "CmdSwitcher.h"
#include "TclListObj.h"

class DialogInfo {

    private:
        Tcl_Obj        *cmdList;

    public:
        Tcl_Obj *getCmdList(void) {return cmdList;}
        DialogInfo(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
            Uint32 i;

            cmdList = Tcl_NewListObj(0, 0);
            for (i=0; i<objc; i++) {
                Tcl_ListObjAppendElement(interp, cmdList, 
                        Tcl_DuplicateObj(objv[i]));
            }
        }
};

struct PackList {

    Uint32          vert;
    Tcl_Obj        *packList;

    PackList(Uint32 dir) {
        packList = Tcl_NewListObj(0, 0);
        vert = dir;
    }

    ~PackList(void) {
/*        Tcl_DecrRefCount(packList); */
    }
};

/********************************************************************
 * DialogMgr_InstCmd()
 ********************************************************************/
static int DialogMgr_InstCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                argc,
        Tcl_Obj          *const objv[])
{
    DialogMgr   *dlgm = (DialogMgr *)clientData;
    return dlgm->InstCmd(argc-1, &objv[1]);
}

/********************************************************************
 * DialogMgr_LoadFileHelper()
 ********************************************************************/
static int DialogMgr_LoadFileHelper(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[])
{
    DialogMgr   *dlgm = (DialogMgr *)clientData;
    return dlgm->LoadFileHelper(argc, objv);
}

typedef enum {
    LFC_Hbox   = 1,
    LFC_Vbox,
    LFC_Button,
    LFC_RadioBox,
        LFC_Radio,
    LFC_CheckBox,
        LFC_Check,
    LFC_Label,
    LFC_Dialog,
    LFC_Entry,
    LFC_Combo,
    LFC_NumCmds
};

static CmdSwitchStruct   lfc_cmds[] = {
    { "hbox",        LFC_Hbox },
    { "vbox",        LFC_Vbox },
    { "button",      LFC_Button },
    { "radiobox",    LFC_RadioBox },
    { "radio",       LFC_Radio    },
    { "checkbox",    LFC_CheckBox },
    { "check",       LFC_Check    },
    { "button",      LFC_Button   },
    { "label",       LFC_Label    },
    { "dialog",      LFC_Dialog   },
    { "entry",       LFC_Entry    },
    { "combo",       LFC_Combo    },
    { "",            0            }
};

/********************************************************************
 * DialogMgr()
 ********************************************************************/
DialogMgr::DialogMgr(
        Tcl_Interp *interp,
        Char         *inst_name) : 
    instName(inst_name), pathTmp(128), groupTmp(128), dlgPath(128), 
    widgetList(0)
{
    Int32    idx;
    this->interp = interp;

    loadMode   = 0;
    createMode = 0;

    packStack = new Stack<PackList>();
    currPackList = 0;

    Tcl_InitHashTable(&hashTable, TCL_STRING_KEYS);

    slaveInterp = Tcl_CreateInterp();
    Tcl_Init(slaveInterp);

    for (idx=0; lfc_cmds[idx].cmdString[0]; idx++) {
        Tcl_CreateObjCommand(slaveInterp, lfc_cmds[idx].cmdString,
                DialogMgr_LoadFileHelper, this, 0);
    }

    Tcl_CreateObjCommand(interp, inst_name, DialogMgr_InstCmd, this, 0);
}
/********************************************************************
 * getName
 ********************************************************************/
Char *DialogMgr::getName(void)
{
    return instName.value();
}

/********************************************************************
 * LoadFileHelper()
 ********************************************************************/
int DialogMgr::LoadFileHelper(
        int        objc,
        Tcl_Obj   *const objv[])
{
    Int32           cmd, i, itmp, ret, pre_idx = 0, exec = 0;
    Tcl_Obj        *l_copy, *l_opts, *res_obj = 0;
    Tcl_HashEntry  *hashEntry;
    DialogInfo     *dialogInfo;
    Uint32          packVert = 1, specified;
    Tcl_Obj        *obj_tmp = 0;
    Int32           didText = 0;
    Char            buf[64], *str;
    TclListObj      cmd_obj(interp);
    Uint32          setLen = 0;

    cmd = CmdSwitch(lfc_cmds, Tcl_GetString(objv[0]));

    if (cmd != LFC_Dialog) {
        pre_idx = pathTmp.length();
    }

    switch (cmd) {

        /**** hbox <options> { } ****/
        case LFC_Hbox:
            pathTmp += ".hbox";

            Tcl_ListObjLength(interp, currPackList->packList, &itmp);
            sprintf(buf, "%d", itmp);
            pathTmp += buf;

            cmd_obj << "TitleFrame::TitleFrame" << pathTmp;

            if (ProcessOpts(interp, 0, cmd_obj, 1, objc-2, &objv[1],
                        OPT_TEXT, &specified) != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return TCL_ERROR;
            }

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }

            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            pathTmp += ".f";

            packVert = 0;
            exec = 1;
            break;

        /**** vbox <options> { } ****/
        case LFC_Vbox:
            pathTmp += ".vbox";
            Tcl_ListObjLength(interp, currPackList->packList, &itmp);
            sprintf(buf, "%d", itmp);
            pathTmp += buf;

            cmd_obj << "TitleFrame::TitleFrame" << pathTmp;

            if (ProcessOpts(interp, 0, cmd_obj, 1, objc-2, &objv[1],
                        OPT_TEXT, &specified) != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return TCL_ERROR;
            }

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }

            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            pathTmp += ".f";

            exec = 1;
            break;

        /**** button <id> <options> { } ****/
        case LFC_Button:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            cmd_obj << "button" << pathTmp;

            if (ProcessOpts(interp, objv[1], cmd_obj, 1, objc-2, &objv[2], 
                        OPT_TEXT, &specified) != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return TCL_ERROR;
            }

            if (String::equal(Tcl_GetString(objv[1]), "cancel")) {
                cmd_obj << "-command" << 
                    (TclListObj(interp) << "destroy" << dlgPath);

            } else if (String::equal(Tcl_GetString(objv[1]), "ok")) {
                cmd_obj << "-command" <<
                    (TclListObj(interp) << dlgPath << "ok");
            }

            /**** Append options here... ****/
            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }

            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            if (String::equal(Tcl_GetString(objv[1]), "ok")) {
                cmd_obj << "bind" << dlgPath << "<Return>" <<
                    (TclListObj(interp) << dlgPath << "ok");
                Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            }
            break;

        case LFC_Combo:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            cmd_obj << "ComboBox" << pathTmp;
            cmd_obj.append_elems(objc-2, &objv[2]);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }
            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
            break;

        case LFC_RadioBox:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            cmd_obj << "TitleFrame::TitleFrame" << pathTmp;

            if (ProcessOpts(interp, 0, cmd_obj, 0, objc-3, &objv[2],
                        OPT_TEXT|OPT_VERT|OPT_HORIZ, &specified) != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return TCL_ERROR;
            }

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }

            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            if ((specified & OPT_HORIZ) && (specified & OPT_VERT)) {
                Tcl_AppendResult(interp, "cannot specify both -vert and -horiz",
                        0);
                return TCL_ERROR;
            }

            pathTmp += ".f";
            exec = 1;
            break;

        /**** radio <id> <options> ****/
        case LFC_Radio:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            Tcl_ListObjLength(interp, currPackList->packList, &itmp);
            if (!itmp) {
                groupTmp = pathTmp;
            }

            cmd_obj << "radiobutton" << pathTmp 
                    << "-variable" << groupTmp
                    << "-value" << objv[1];
         
            if (ProcessOpts(interp, 0, cmd_obj, 1, objc-2, &objv[2],
                        OPT_TEXT|OPT_VERT|OPT_HORIZ, &specified) != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return TCL_ERROR;
            }

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }

            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            break;

        case LFC_CheckBox:
            break;

        case LFC_Check:
            break;

        case LFC_Label:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            cmd_obj << "label" << pathTmp;
            cmd_obj.append_elems(objc-2, &objv[2]);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);

            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }
            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
            break;

        case LFC_Entry:
            pathTmp += ".";
            pathTmp += Tcl_GetString(objv[1]);

            widgetList << objv[1] << pathTmp;

            cmd_obj << "Entry::Entry" << pathTmp;
            cmd_obj.append_elems(objc-2, &objv[2]);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                return ret;
            }
            Tcl_ListObjAppendElement(interp, currPackList->packList,
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));

            break;


            /**** dialog <name> <options> <description> ****/
        case LFC_Dialog:
            if (!createMode) {
                dialogInfo = new DialogInfo(interp, objc, objv);

                if (!(hashEntry = Tcl_FindHashEntry(&hashTable, 
                                Tcl_GetString(objv[1])))) {
                    hashEntry = Tcl_CreateHashEntry(&hashTable, 
                            Tcl_GetString(objv[1]), &itmp);
                }
                Tcl_SetHashValue(hashEntry, dialogInfo);
            } else {
                currPackList = 0;

                /**** Create a window... ****/
                cmd_obj << "DialogBox::DialogBox" << pathTmp;
                dlgPath = pathTmp;

                for (i=2; i<objc-1; i++) {
                    str = Tcl_GetString(objv[i]);
                    if (String::equal(str, "-title") || 
                            String::equal(str, "-text")) {
                        cmd_obj << objv[i] << objv[i+1];
                        i++;
                    } 
                }


                /**** Create a new widget list... ****/
                widgetList(interp);
                widgetList << pathTmp << "widget_list";

                ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);

                res_obj = Tcl_DuplicateObj(Tcl_GetObjResult(interp));

                if (ret != TCL_OK) {
                    return ret;
                }
                
                /**** Now, withdraw the box before building the components
                 ****/
                cmd_obj << dlgPath << "withdraw";
                ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
                if (ret != TCL_OK) {
                    fprintf(stderr, "Dialog withdraw command failed - %s\n",
                            Tcl_GetStringResult(interp));
                    return ret;
                }

                exec = 1;
            }
            break;

        default:
            break;
    }

    if (exec) {

        /**** Push the current pack-list onto the stack... ****/
        if (currPackList) {
            packStack->push(currPackList);
        }
        currPackList = new PackList(packVert);

        ret = Tcl_EvalObjEx(slaveInterp, objv[objc-1], TCL_EVAL_GLOBAL);
        if (ret != TCL_OK) {
            Tcl_SetObjResult(slaveInterp, 
                    Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
            return ret;
        }

        /**** Okay, now deal pack what was created at this level... ****/
        Tcl_ListObjLength(interp, currPackList->packList, &itmp);
        if (itmp > 0) {
            if (currPackList->vert) {
                /**** For vertical elements, must call grid repeatedly...
                 ****/
                for (i=0; i<itmp; i++) {
                    cmd_obj << "grid" 
                            << TclListObj(interp, currPackList->packList)[i]
                            << "-padx" << 2
                            << "-pady" << 2;

                    if (cmd == LFC_RadioBox) {
                        cmd_obj << "-sticky" << "w";
                    } else {
                        cmd_obj << "-sticky" << "ew";
                    }

                    ret = Tcl_EvalObjEx(interp,cmd_obj.end(),TCL_EVAL_GLOBAL);
                    if (ret != TCL_OK) {
                        Tcl_SetObjResult(slaveInterp, 
                            Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                        return TCL_ERROR;
                    }


#ifdef UNDEFINED
                    cmd_obj << "grid" << "columnconfigure" 
                        << pathTmp << 0<< "-weight" << 1;

                    ret = Tcl_EvalObjEx(interp,cmd_obj.end(),TCL_EVAL_GLOBAL);
                    if (ret != TCL_OK) {
                        Tcl_SetObjResult(slaveInterp, 
                            Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                        return TCL_ERROR;
                    }
#endif /* UNDEFINED */
                }
            } else {
                /**** For horizontal, just need to create grid command:
                 ****    grid <packList>
                 ****/
                cmd_obj << "grid";
                cmd_obj.append_elems(currPackList->packList);
                cmd_obj << "-sticky" << "nw"
                        << "-padx"   << 2
                        << "-pady"   << 2;

                ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
                if (ret != TCL_OK) {
                    Tcl_SetObjResult(slaveInterp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(interp)));
                    return TCL_ERROR;
                }
            }
        }

        /**** Now, delete the current packList and pop off the stack
         ****/
        delete currPackList;
        currPackList = packStack->pop();

        /**** Okay, do some final work if we just completed building
         **** a dialog-box...
         ****/
        if (cmd == LFC_Dialog) {
            ret = Tcl_EvalObjEx(interp, widgetList.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                return ret;
            }

#if 0
            cmd_obj << pathTmp << "popup";

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) {
                return ret;
            }
#endif
        }
    }

    if (res_obj) {
        Tcl_SetObjResult(slaveInterp, res_obj);
    }

    if (cmd != LFC_Dialog && !setLen) {
        pathTmp.setLen(pre_idx);
    }

    return TCL_OK;
}

typedef enum {
    DMC_Create = 1,
    DMC_Load,
    DMC_NumCmds
};

static CmdSwitchStruct dlgmgr_cmds[] = {
    {"create",              DMC_Create},
    {"load",                DMC_Load},
    {"",                    0}
};

/********************************************************************
 * InstCmd()
 ********************************************************************/
int DialogMgr::InstCmd(int argc, Tcl_Obj *const objv[])
{
    Int32           cmd, ret, i, x;
    Tcl_HashEntry  *hashEntry;
    DialogInfo     *dInfo;
    TclListObj      cmd_obj(interp);
    Tcl_Obj        *unset = 0;

    cmd = CmdSwitch(dlgmgr_cmds, Tcl_GetString(objv[0]));
    switch (cmd) {

        /**** $dlgm create <DialogType> <PathName> ****/
        case DMC_Create:
            if (!(hashEntry = Tcl_FindHashEntry(&hashTable, 
                            Tcl_GetString(objv[1])))) {
                Tcl_AppendResult(interp, "no dialog ", 
                        Tcl_GetString(objv[1]), 0);
                return TCL_ERROR;
            }
            dInfo = (DialogInfo *)Tcl_GetHashValue(hashEntry);

            createMode = 1;
            pathTmp = Tcl_GetString(objv[2]);
            
            if (argc > 3) {
                for (i=3; i<argc; i++) {
                    if (String::equal(Tcl_GetString(objv[i]), "-var")) {
                        i++;
                        TclListObj  it(interp, objv[i]);
                        /**** Loop through each pair in the attached 
                         **** list...
                         ****/
                        unset = objv[i];
                        if ((it.length() % 2)) {
                            Tcl_AppendResult(interp, 
                                    "var-list has un-even length", 0);
                            return TCL_ERROR;
                        }
                        for (x=0; x<it.length(); x+=2) {
                            /**** Vars in {var val} pairs ****/
                            Tcl_ObjSetVar2(slaveInterp, it[x], 0, 
                                    Tcl_DuplicateObj(it[x+1]), TCL_GLOBAL_ONLY);
                        }
                    }
                }
            }

            ret = Tcl_EvalObjEx(slaveInterp, 
                    Tcl_DuplicateObj(dInfo->getCmdList()), 
                    TCL_EVAL_GLOBAL);
            createMode = 0;

            if (ret != TCL_OK) {
                Tcl_SetObjResult(interp, 
                        Tcl_DuplicateObj(Tcl_GetObjResult(slaveInterp)));
                return TCL_ERROR;
            }

            Tcl_SetObjResult(interp, 
                    Tcl_DuplicateObj(Tcl_GetObjResult(slaveInterp)));

            if (unset) {
                TclListObj   it(interp, unset);

                for (x=0; x<it.length(); x+=2) {
                    Tcl_UnsetVar(slaveInterp, Tcl_GetString(it[x+1]),
                            TCL_GLOBAL_ONLY);
                }
            }
            break;

        case DMC_Load:
            if (argc < 2) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            loadMode=1;
            Tcl_EvalFile(slaveInterp, Tcl_GetString(objv[1]));
            loadMode=0;
            break;

        default:
            break;
    }

    return TCL_OK;
}

/********************************************************************
 *
 ********************************************************************/
int DialogMgr::ProcessOpts(
        Tcl_Interp          *interp,
        Tcl_Obj             *id, 
        TclListObj          &cmd_list,
        Uint32               in_spec,
        Uint32               objc,
        Tcl_Obj       *const objv[],
        Uint32               valid, 
        Uint32              *spec)
{
    Int32     len, i;
    Char     *str;

    *spec = 0;

    for (i=0; i<objc; i++) {
        str = Tcl_GetStringFromObj(objv[i], &len);

        if (String::equal(str, "-text")) {
            if (!(valid & OPT_TEXT)) {
                Tcl_AppendResult(interp, "-text is invalid option", 0);
                return TCL_ERROR;
            }
            *spec |= OPT_TEXT;
            if (in_spec) {
                cmd_list << objv[i] << objv[i+1];
                i++;
            }
        } else if (String::equal(str, "-horiz")) {
            *spec |= OPT_HORIZ;
        } else if (String::equal(str, "-vert")) {
            *spec |= OPT_VERT;
        } else if (String::equal(str, "-relief")) {
            cmd_list << objv[i] << objv[i+1];
            i++;
        } else if (String::equal(str, "-command")) {
            cmd_list << objv[i] << objv[i+1];
            i++;
        } else {
            Tcl_AppendResult(interp, "Unknown opt ", 
                    Tcl_GetString(objv[i]), 0);
            return TCL_ERROR;
        }
    }

    if (!(*spec & OPT_TEXT) && id && in_spec) {
        cmd_list << "-text" << id;
    }

    return TCL_OK;
}

static DialogMgr *Globl_DialogMgr = 0;

/********************************************************************
 * DialogMgr_TclCmd()
 ********************************************************************/
static int DialogMgr_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    DialogMgr *dlgm;
    Uint32     globl = 0, i;

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few arguments", 0);
        return TCL_ERROR;
    }

    for (i=1; i<argc; i++) {
        if (String::equal(argv[i], "-global")) {
            globl = 1;
        }
    }

    if (globl) {
        if (!Globl_DialogMgr) {
            dlgm = new DialogMgr(interp, argv[1]);
            Globl_DialogMgr = dlgm;
        }
        Tcl_AppendResult(interp, Globl_DialogMgr->getName(), 0);
    } else {
        dlgm = new DialogMgr(interp, argv[1]);
        Tcl_AppendResult(interp, argv[1], 0);
    }

    return TCL_OK;
}

/********************************************************************
 * DialogMgr_Init()
 ********************************************************************/
extern "C" int DialogMgr_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "dialog_mgr", 
            (Tcl_CmdProc *)DialogMgr_TclCmd, 0, 0);
    return TCL_OK;
}


