/****************************************************************************
 *                       RemoteAppConnection.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements the low-level details of communication across 
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "RemoteAppConnection.h"
#include "RemoteAppConnectionListener.h"
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#ifndef __MINGW32__
#include <sys/socket.h>
#endif

#undef   DEBUG_REMOTE_CONNECTION

#define FP stderr
#ifdef DEBUG_REMOTE_CONNECTION
#define DBG_MSG(x) fprintf x ; fflush(FP)
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * RemoteAppConnection()
 ********************************************************************/
RemoteAppConnection::RemoteAppConnection()
{
    int i;

    d_hdrBuf[0] = 0x9A;
    d_hdrBuf[1] = 0xA5;
    d_hdrBuf[2] = 0x73;
    d_hdrBuf[3] = 0x37;

    d_dataBufIdx = 0;
    d_dataBufMax = 0;
    d_hdrBufIdx  = 0;
    d_lenBufIdx  = 0;
    d_idxBufIdx  = 0;

    d_state = Rcv_ScanHdr;

    d_connected       = false;
    d_connectComplete = false;

    for (i=0; i<32; i++) {
        d_listeners[i] = 0;
    }

    d_xmitBuf    = 0;
    d_xmitBufLen = 0;
}

/********************************************************************
 * ~RemoteAppConnection()
 ********************************************************************/
RemoteAppConnection::~RemoteAppConnection()
{
    if (d_xmitBufLen) {
        delete [] d_xmitBuf;
    }
}

/********************************************************************
 * HandleDisconnect()
 ********************************************************************/
void RemoteAppConnection::HandleDisconnect()
{

}

/********************************************************************
 * AddListener()
 ********************************************************************/
void RemoteAppConnection::AddListener(RemoteAppConnectionListener *listener)
{
    Uint32 idx = listener->getIdx();

    d_listeners[idx] = listener;

    listener->Connect(this);
}

/********************************************************************
 * DeleteListener()
 ********************************************************************/
void RemoteAppConnection::DeleteListener(RemoteAppConnectionListener *listener)
{
    d_listeners[listener->getIdx()] = 0;
}

/********************************************************************
 * DispatchPacket()
 ********************************************************************/
void RemoteAppConnection::DispatchPacket(Uint32 idx, Uint32 len, Uchar *data) 
{
    if (d_listeners[idx]) {
        d_listeners[idx]->Receive(idx, len, data);
    } else {
        fprintf(stderr, "ERROR: Data Packet for idx %d (len %d) unhandled\n",
                idx, len);
        fprintf(stderr, "\tpid is %d\n", getpid());
    }
}

/********************************************************************
 * DataRecv()
 *
 * This function called to 
 ********************************************************************/
void RemoteAppConnection::DataRecv(Uint32 len, Uchar *buf)
{
    int bidx = 0;

    DBG_MSG((FP, "DataRecv(%d)\n", len));
#ifdef DEBUG_REMOTE_CONNECTION
    for (int i=0; i<len; i++) {
        DBG_MSG((FP, "%d ", buf[i]));
    }
    DBG_MSG((FP, "\n"));
#endif

    while (bidx < len) {
        switch (d_state) {
 
            /************************************************************
             * Rcv_ScanHdr
             *
             * Scan, looking for the header bytes: 0x9A 0xA5 0x73 0x37
             ************************************************************/
            case Rcv_ScanHdr:
                DBG_MSG((FP, "Rcv_ScanHdr\n"));

                while ((bidx<len) && (d_hdrBufIdx < 4)) {
                    if (buf[bidx++] == d_hdrBuf[d_hdrBufIdx]) {
                        DBG_MSG((FP, "\thdr match @ %d\n", d_hdrBufIdx));
                        d_hdrBufIdx++;
                    } else {
                        DBG_MSG((FP, "\thdr mis-match @ %d\n", d_hdrBufIdx));
                        d_hdrBufIdx = 0;
                    }
                }

                if (d_hdrBufIdx == 4) {
                    DBG_MSG((stderr, "Done with Hdr scan\n"));
                    d_state = Rcv_ScanLen;
                    d_hdrBufIdx = 0;
                } else {
                    d_state = Rcv_ScanHdr;
                }
                break;

            /************************************************************
             * Rcv_ScanLen
             *
             * Scan the data-buffer length. 4-byte quantity 
             ************************************************************/
            case Rcv_ScanLen:
                DBG_MSG((stderr, "Rcv_ScanLen\n"));

                while ((bidx<len) && (d_lenBufIdx<4)) {
                    d_lenBuf[d_lenBufIdx++] = buf[bidx++];
                }

                if (d_lenBufIdx == 4) {
                    d_len = d_lenBuf[0]|(d_lenBuf[1]<<8)|
                        (d_lenBuf[2]<<16)|(d_lenBuf[3]<<24);
                    d_lenBufIdx=0;
                    d_dataBufIdx=0;
                    d_state = Rcv_ScanIdx;
                    DBG_MSG((stderr, "\tlength=%d\n", d_len));
                } 
                break;

            /************************************************************
             * Rcv_ScanIdx
             ************************************************************/
            case Rcv_ScanIdx:
                DBG_MSG((stderr, "Rcv_ScanIdx\n"));

                while ((bidx<len) && (d_idxBufIdx<4)) {
                    d_idxBuf[d_idxBufIdx++] = buf[bidx++];
                }

                if (d_idxBufIdx == 4) {
                    d_idx = d_idxBuf[0]|(d_idxBuf[1]<<8)|
                        (d_idxBuf[2]<<16)|(d_idxBuf[3]<<24);
                    d_idxBufIdx=0;
                    d_dataBufIdx=0;
                    d_state = Rcv_ScanData;
                } 
                break;

            /************************************************************
             * Scan the data in the buffer. 
             ************************************************************/
            case Rcv_ScanData:

                DBG_MSG((stderr, "Rcv_ScanData\n"));

                if (d_len > d_dataBufMax) {
                    if (d_dataBufMax) {
                        delete [] d_dataBuf;
                    }
                    d_dataBuf = new Uchar[d_len];
                    d_dataBufMax = d_len;
                }

                while ((bidx<len) && (d_dataBufIdx<d_len)) {
                    d_dataBuf[d_dataBufIdx++] = buf[bidx++];
                }

                if (d_dataBufIdx == d_len) {
                    Uint32 idx, len;

                    DBG_MSG((stderr, "...done\n"));

                    idx = d_idxBuf[0] | (d_idxBuf[1] << 8) |
                        (d_idxBuf[2] << 16) | (d_idxBuf[3] << 24);

                    len = d_lenBuf[0] | (d_lenBuf[1] << 8) |
                        (d_lenBuf[2] << 16) | (d_lenBuf[3] << 24);

                    /**** Send completed packet... ****/
                    DBG_MSG((stderr, "Dispatching packet "
                            "(idx=%d, len=%d, d_dataBuf=%x)\n",
                            idx, len, d_dataBuf));
                    DispatchPacket(idx, len, d_dataBuf);

                    d_dataBufIdx = 0;
                    d_state = Rcv_ScanHdr;
                }
                break;
        }
    }
}

/********************************************************************
 * EventProc()
 ********************************************************************/
void RemoteAppConnection::EventProc(ClientData clientData, int mask)
{
    return ((RemoteAppConnection *)clientData)->EventProc(mask);
}

/********************************************************************
 * EventProc()
 ********************************************************************/
void RemoteAppConnection::EventProc(int mask)
{
    char buf[1024];
    int  bytes;

    DBG_MSG((FP, "---> EventProc(%d) channel=0x%08x\n", mask, d_channel));

    if (mask & TCL_READABLE) {
       
        do {
            DBG_MSG((FP, "\t----> Tcl_Read()\n"));
            bytes = Tcl_Read(d_channel, buf, 1023);
            DBG_MSG((FP, "\t----> Tcl_Read() => %d\n", bytes));

            if (bytes > 0) {
                DataRecv(bytes, (Uchar *)buf);
            } else if (bytes == 0) {
                if (Tcl_Eof(d_channel)) {
                    d_connected = 0;
                    HandleDisconnect();
                }
            } else if (bytes < 0) {
                fprintf(stderr, "Tcl_Read returns error\n");
                fflush(stderr);
            }
        } while (bytes > 0);
    }

    if (mask & TCL_EXCEPTION) {
        fprintf(stderr, "Exception on socket\n");
    }

    DBG_MSG((FP, "<--- EventProc(%d)\n", mask));
}

/********************************************************************
 * Wait()
 ********************************************************************/
int RemoteAppConnection::Wait(Uint32 idx, Uint32 *len, Uchar **data)
{
    int ret;

    do {
        ret = Poll(REMOTE_POLL_FOREVER);
    } while (1);
}

/********************************************************************
 * Poll()
 ********************************************************************/
int RemoteAppConnection::Poll(Uint32 timeout)
{
#if 0
    struct timeval    tv;
    struct timeval   *tvp = 0;
    fd_set rset;
    int    ret;

    if (timeout != ~0) {
        tv.tv_sec  = timeout/1000;
        tv.tv_usec = (timeout%1000)*1000;
        tvp = &tv;
    }

    FD_ZERO(&rset);
    FD_SET(d_skt, &rset);

    ret = select(d_skt+1, &rset, 0, 0, tvp);

    if (ret > 0) {
        EventProc(TCL_READABLE);
    } else if (ret < 0) {
        fprintf(stderr, "ERROR: select returned error\n");
    }

    return ret;
//    return Tcl_DoOneEvent(TCL_FILE_EVENTS|TCL_TIMER_EVENTS);
#endif
    return Tcl_DoOneEvent(0);
}

/********************************************************************
 * Send()
 ********************************************************************/
int RemoteAppConnection::Send(Uint32 idx, Uint32 len, Uchar *data)
{
    int ret;
    Uchar lenBuf[4], idxBuf[4];

    lenBuf[0] = len;
    lenBuf[1] = len >> 8;
    lenBuf[2] = len >> 16;
    lenBuf[3] = len >> 24;

    idxBuf[0] = idx;
    idxBuf[1] = idx >> 8;
    idxBuf[2] = idx >> 16;
    idxBuf[3] = idx >> 24;

    if ((12+len) > d_xmitBufLen) {
        if (d_xmitBuf) {
            delete [] d_xmitBuf;
        }

        d_xmitBuf = new Uchar[len+12];
        d_xmitBufLen = len+12;
    }

    memcpy(d_xmitBuf, d_hdrBuf, 4);
    memcpy(&d_xmitBuf[4], lenBuf, 4);
    memcpy(&d_xmitBuf[8], idxBuf, 4);
    memcpy(&d_xmitBuf[12], data, len);

    Tcl_Write(d_channel, (char *)d_xmitBuf, len+12);
    Tcl_Flush(d_channel);

    DBG_MSG((FP, "NOTE: Sending data: data_len=%d pkt_length=%d\n", 
                (len+12), len));

    return 0;
}

/********************************************************************
 * WriteUint32()
 ********************************************************************/
void RemoteAppConnection::WriteUint32(Uint32 val, Uchar *buf, Uint32 &idx)
{
    buf[idx++] = val;
    buf[idx++] = (val >> 8);
    buf[idx++] = (val >> 16);
    buf[idx++] = (val >> 24);
}

/********************************************************************
 * ReadUint32()
 ********************************************************************/
Uint32 RemoteAppConnection::ReadUint32(Uchar *buf, Uint32 &idx)
{
    Uint32 ret = 0;

    ret |= buf[idx++];
    ret |= buf[idx++]<<8;
    ret |= buf[idx++]<<16;
    ret |= buf[idx++]<<24;

    return ret;
}


